package org.somda.dsl.biceps

import org.somda.dsl.biceps.base.Handle
import org.somda.dsl.biceps.base.tree.buildComponentTree
import org.somda.dsl.biceps.component.Mds
import org.somda.dsl.biceps.validator.AlertSourceValidator
import org.somda.dsl.biceps.validator.HandleValidator
import org.somda.dsl.biceps.validator.OperationTargetValidator
import org.somda.dsl.mpkp.MpkpExtensionsRenderer
import org.somda.dsl.rendering.jaxb.installExtensionRenderer
import org.somda.dsl.sdpi.SdpiExtensionsRenderer
import java.math.BigInteger
import java.net.URI
import java.util.*

internal inline fun <reified T : Any, reified V> T.checkUnset(
    value: V?,
    name: String,
    setter: T.() -> Unit
) {
    if (value != null) {
        currentScope().also {
            it.logger.info {
                "Overwrite assigned value for property ${this::class.simpleName}.$name at '${it.cursor()}'"
            }
        }
    }
    setter()
}

internal val installedValidators: MutableSet<MdibValidator> = mutableSetOf()

internal val installedPostProcessors: MutableSet<MdibPostProcessor> = mutableSetOf()

public fun installValidator(validator: MdibValidator) {
    installedValidators.add(validator)
}

public fun installPostProcessor(postProcessor: MdibPostProcessor) {
    installedPostProcessors.add(postProcessor)
}

public data class MdibVersion(
    val version: BigInteger,
    val sequenceId: String,
    val instanceId: BigInteger?
)

@MdibDsl
public class Mdib internal constructor(public val name: String) {
    public var mdibVersion: MdibVersion? = null
        private set

    private val mdsList: MutableList<Mds> = mutableListOf()

    public fun mds(handle: Handle = uniqueHandle(), init: Mds.() -> Unit = {}): Mds {
        return Mds(handle).preProcess {
            it.apply(init)
            mdsList.add(it)
        }
    }

    public fun mdibVersion(
        version: BigInteger,
        sequenceId: String = randomUuidUri(),
        instanceId: BigInteger? = null
    ): MdibVersion {
        return MdibVersion(version, sequenceId, instanceId).also {
            checkUnset(mdibVersion, "mdibVersion") {
                mdibVersion = it
            }
        }
    }

    public fun mdibVersion(
        version: Int,
        sequenceId: String = randomUuidUri(),
        instanceId: BigInteger? = null
    ): MdibVersion {
        return mdibVersion(version.toBigInteger(), sequenceId, instanceId)
    }

    public fun mdibVersion(
        version: Long,
        sequenceId: String = randomUuidUri(),
        instanceId: BigInteger? = null
    ): MdibVersion {
        return mdibVersion(version.toBigInteger(), sequenceId, instanceId)
    }

    private fun randomUuidUri() = URI.create("urn:uuid:${UUID.randomUUID()}").toString()

    public val mdss: List<Mds> = mdsList
}

public fun mdib(name: String, init: Mdib.() -> Unit): Mdib {
    installExtensionRenderer(SdpiExtensionsRenderer)
    installExtensionRenderer(MpkpExtensionsRenderer)
    installValidator(HandleValidator)
    installValidator(OperationTargetValidator)
    installValidator(AlertSourceValidator)

    return scoped(name) {
        currentScope().let { scope ->
            scope.logger.info { "Start new MDIB scope" }
            scope.pushCursor(name) {
                for (installedValidator in installedValidators) {
                    installedValidator.init()
                }
                Mdib(name).apply {
                    runCatching {
                        init()
                    }.onFailure { err ->
                        scope.logError(err.message ?: "No error message recorded (unknown error)")
                        handleErrors()
                    }
                }.also { mdib ->
                    val tree = buildComponentTree(mdib)
                    for (installedValidator in installedValidators) {
                        runCatching {
                            installedValidator.validate(mdib, tree)
                        }.onFailure { err ->
                            scope.logError(err.message ?: "No error message recorded (unknown error)")
                        }
                    }

                    for (installedPostProcessor in installedPostProcessors) {
                        runCatching {
                            installedPostProcessor.run(mdib, tree)
                        }.onFailure { err ->
                            scope.logError(err.message ?: "Unknown error during post processing")
                        }
                    }
                }
            }.also {
                handleErrors()
                scope.logger.info { "Finish MDIB scope" }
            }
        }
    }
}

private fun handleErrors() {
    currentScope().also { scope ->
        if (scope.errors.isNotEmpty()) {
            scope.errors.forEach { scope.logger.error(it) }
            error(
                "Validation of scope '${scope.name}' failed. See previous error log messages for a " +
                        "list of detected findings."
            )
        }
    }
}