package org.somda.dsl.biceps

import org.apache.logging.log4j.CloseableThreadContext
import org.apache.logging.log4j.kotlin.KotlinLogger
import org.apache.logging.log4j.kotlin.loggerOf
import org.apache.logging.log4j.spi.ExtendedLogger
import org.somda.dsl.biceps.base.tree.Descriptor
import org.somda.dsl.biceps.base.tree.DescriptorMultiState
import org.somda.dsl.biceps.base.tree.ExtensibleMdibComponent
import java.lang.Class
import java.lang.reflect.InvocationHandler
import java.lang.reflect.Method
import java.lang.reflect.Proxy
import kotlin.reflect.KProperty
import kotlin.use

public data class Cursor(public val items: List<String>) {
    override fun toString(): String {
        return items.joinToString(" -> ")
    }

    public fun withoutScope(): Cursor = copy(items = this.items.drop(1))
}

public data class MdibScope(
    val name: String
) {
    val logger: KotlinLogger by InstanceLogger(Pair("scope", name))

    private val cursorItems: MutableList<String> = mutableListOf()

    private val _errors: MutableList<String> = mutableListOf()

    public val errors: List<String> = _errors

    public fun cursor(): Cursor = Cursor(cursorItems)

    public fun <T> pushCursor(item: String, action: () -> T): T {
        cursorItems.add(item)
        val result = action()
        cursorItems.removeLast()
        return result
    }

    public fun logError(msg: String) {
        _errors.add(msg)
    }
}

private var currentScope: MdibScope? = null

public fun currentScope(): MdibScope {
    return requireNotNull(currentScope) {
        "Tried to setup a component that requires a handle scope without an active scope. See org.somda.dsl.biceps.scope()"
    }
}

private fun <T : ExtensibleMdibComponent> T.validate(action: (validator: MdibValidator) -> Unit) {
    for (installedValidator in installedValidators) {
        try {
            action(installedValidator)
        } catch (e: Exception) {
            error("Error at ${currentScope().cursor()} detected by ${installedValidator.name}: ${e.message}")
        }
    }
}

internal inline fun <reified T : Descriptor<*>> T.preProcess(
    crossinline action: (T) -> Unit
): T {
    currentScope().pushCursor(this.toString()) {
        validate { it.visitPreInit(this) }
        action(this)
        validate { it.visitPostInit(this) }
    }
    return this
}

internal inline fun <reified T : DescriptorMultiState<*>> T.preProcess(
    crossinline action: (T) -> Unit
): T {
    currentScope().pushCursor(this.toString()) {
        validate { it.visitPreInit(this) }
        action(this)
        validate { it.visitPostInit(this) }
    }
    return this
}

public fun <T : Any> scoped(name: String, init: () -> T): T {
    try {
        if (currentScope == null) {
            currentScope = MdibScope(name)
        }
        return init()
    } finally {
        currentScope = null
    }
}

private class InstanceLogger(vararg loggerVariables: Pair<String, String>) {
    private val loggerVariables = loggerVariables.associate { it.first to it.second }

    companion object {
        fun wrap(logger: KotlinLogger, loggerVariables: Map<String, String>) = KotlinLogger(
            Proxy.newProxyInstance(
                InstanceLogger::class.java.classLoader, arrayOf<Class<*>>(ExtendedLogger::class.java),
                InstanceLoggerInvocationHandler(logger.delegate, loggerVariables)
            ) as ExtendedLogger
        )

        private class InstanceLoggerInvocationHandler(
            private val logger: ExtendedLogger,
            private val loggerVariables: Map<String, String>
        ) : InvocationHandler {
            @Throws(Throwable::class)
            override fun invoke(proxy: Any, method: Method, args: Array<Any>): Any? {
                CloseableThreadContext
                    .putAll(loggerVariables.entries.associate { it.key to it.value })
                    .use {
                        return method.invoke(logger, *args)
                    }
            }
        }
    }

    operator fun getValue(thisRef: Any, property: KProperty<*>) =
        wrap(loggerOf(thisRef.javaClass), loggerVariables)

}