package org.somda.dsl.biceps.validator

import org.somda.dsl.biceps.MdibValidator
import org.somda.dsl.biceps.base.Handle
import org.somda.dsl.biceps.base.tree.Descriptor
import org.somda.dsl.biceps.base.tree.DescriptorMultiState
import org.somda.dsl.biceps.base.tree.ExtensibleMdibComponent
import org.somda.dsl.biceps.queueError

private typealias HandleSet = MutableSet<Handle>

internal object HandleValidator : MdibValidator {
    override val name: String = "HandleValidator"

    private val handleSet: HandleSet = mutableSetOf()

    fun contains(handle: Handle): Boolean {
        return handleSet.contains(handle)
    }

    override fun init() {
        handleSet.clear()
    }

    override fun visitPreInit(component: ExtensibleMdibComponent) {
        when (component) {
            is Descriptor<*> -> validateHandle(component.handle)
            is DescriptorMultiState<*> -> validateHandle(component.handle)
        }
    }

    private fun validateHandle(handle: Handle) {
        checkCharacters(handle)
        checkUniqueHandle(handle)
    }

    @OptIn(ExperimentalStdlibApi::class)
    private fun checkCharacters(handle: Handle) {
        val invalidChars = handle.name
            .filter { it < '!' || it > '~' }
            .toCharArray()
            .map { it.code.toHexString(HexFormat.UpperCase) }
            .map { it.substring(4) }
            .map { "U+$it" }

        if (invalidChars.isNotEmpty()) {
            val links = invalidChars.joinToString("\n") {
                "  https://www.compart.com/de/unicode/$it"
            }

            queueError {
                "Handle '${handle.name}' contains the following invalid characters: " +
                        "${invalidChars.joinToString(", ")}\n\nSee also:\n$links\n"
            }
        }
    }

    private fun checkUniqueHandle(handle: Handle) {
        check(!handleSet.contains(handle)) {
            "Found duplicated handle name '${handle.name}'"
        }
        handleSet.add(handle)
    }
}