package org.somda.dsl.rendering.jaxb.mapping

import org.somda.dsl.biceps.alert.AlertState
import org.somda.dsl.biceps.alert.AlertSystem
import org.somda.dsl.biceps.base.Handle
import org.somda.dsl.biceps.base.HandleRef
import org.somda.dsl.biceps.base.tree.ComplexDeviceComponent
import org.somda.dsl.biceps.base.tree.Descriptor
import org.somda.dsl.biceps.base.tree.DescriptorMultiState
import org.somda.dsl.biceps.base.tree.DeviceComponent
import org.somda.dsl.biceps.base.tree.DeviceComponentState
import org.somda.dsl.biceps.base.tree.VersionedMdibComponent
import org.somda.dsl.biceps.context.ContextState
import org.somda.dsl.biceps.metric.Metric
import org.somda.dsl.biceps.metric.MetricState
import org.somda.dsl.biceps.sco.Operation
import org.somda.dsl.biceps.sco.OperationState
import org.somda.dsl.biceps.sco.Sco
import org.somda.dsl.rendering.jaxb.mapping.base.makeOptionalExtension
import org.somda.dsl.rendering.jaxb.mapping.base.mapAccessLevel
import org.somda.dsl.rendering.jaxb.mapping.base.mapAlertActivation
import org.somda.dsl.rendering.jaxb.mapping.base.mapCodedValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapComponentActivation
import org.somda.dsl.rendering.jaxb.mapping.base.mapContextAssociation
import org.somda.dsl.rendering.jaxb.mapping.base.mapDerivationMethod
import org.somda.dsl.rendering.jaxb.mapping.base.mapInstanceIdentifier
import org.somda.dsl.rendering.jaxb.mapping.base.mapMetricAvailability
import org.somda.dsl.rendering.jaxb.mapping.base.mapMetricCategory
import org.somda.dsl.rendering.jaxb.mapping.base.mapOperatingMode
import org.somda.dsl.rendering.jaxb.mapping.base.mapOperationGroup
import org.somda.dsl.rendering.jaxb.mapping.base.mapPhysicalConnectorInfo
import org.somda.dsl.rendering.jaxb.mapping.base.mapProductionSpecification
import org.somda.dsl.rendering.jaxb.mapping.base.mapRelation
import org.somda.dsl.rendering.jaxb.mapping.base.mapSafetyClassification
import org.somda.sdc.biceps.model.participant.AbstractAlertState
import org.somda.sdc.biceps.model.participant.AbstractComplexDeviceComponentDescriptor
import org.somda.sdc.biceps.model.participant.AbstractContextState
import org.somda.sdc.biceps.model.participant.AbstractDescriptor
import org.somda.sdc.biceps.model.participant.AbstractDeviceComponentDescriptor
import org.somda.sdc.biceps.model.participant.AbstractDeviceComponentState
import org.somda.sdc.biceps.model.participant.AbstractMetricDescriptor
import org.somda.sdc.biceps.model.participant.AbstractMetricState
import org.somda.sdc.biceps.model.participant.AbstractMultiState
import org.somda.sdc.biceps.model.participant.AbstractOperationDescriptor
import org.somda.sdc.biceps.model.participant.AbstractOperationState
import org.somda.sdc.biceps.model.participant.AbstractState
import org.somda.sdc.biceps.model.participant.AlertActivation
import org.somda.sdc.biceps.model.participant.AlertSystemDescriptor
import org.somda.sdc.biceps.model.participant.ComponentActivation
import org.somda.sdc.biceps.model.participant.ContextAssociation
import org.somda.sdc.biceps.model.participant.OperatingMode
import org.somda.sdc.biceps.model.participant.ScoDescriptor
import org.somda.sdc.biceps.model.participant.ScoState
import java.math.BigInteger
import kotlin.time.toJavaDuration

internal class BaseMapping(
    private val mappedStates: MutableList<AbstractState>,
    private val mapAlertSystem: ((alertSystem: AlertSystem?) -> AlertSystemDescriptor?)? = null,
    private val mapOperation: ((src: Operation<*>) -> AbstractOperationDescriptor)? = null
) {

    fun mapIntoAbstractDeviceComponentDescriptor(
        src: DeviceComponent<*>,
        dest: AbstractDeviceComponentDescriptor
    ) {
        mapIntoAbstractDescriptor(src, dest)

        dest.productionSpecification = src.productionSpecifications.map { mapProductionSpecification(it) }
    }

    fun mapIntoAbstractComplexDeviceComponentDescriptor(
        src: ComplexDeviceComponent<*>,
        dest: AbstractComplexDeviceComponentDescriptor
    ) {
        mapIntoAbstractDeviceComponentDescriptor(src, dest)
        requireNotNull(mapAlertSystem) {
            "Required alert system mapper, but none was provided"
        }.also {
            dest.alertSystem = it(src.alertSystem)
        }

        dest.sco = mapSco(src.sco)
    }

    fun mapIntoAbstractDescriptor(src: Descriptor<*>, dest: AbstractDescriptor) {
        dest.handle = src.handle.name
        dest.descriptorVersion = src.version
        dest.type = mapCodedValue(src.type)
        dest.safetyClassification = src.safetyClassification?.let { mapSafetyClassification(it) }
        dest.extension = makeOptionalExtension(src.extensions)
    }

    fun mapIntoAbstractDescriptor(src: DescriptorMultiState<*>, dest: AbstractDescriptor) {
        dest.handle = src.handle.name
        dest.descriptorVersion = src.version
        dest.type = mapCodedValue(src.type)
        dest.safetyClassification = src.safetyClassification?.let { mapSafetyClassification(it) }
        dest.extension = makeOptionalExtension(src.extensions)
    }

    fun mapIntoAbstractState(
        src: VersionedMdibComponent?,
        handleRef: HandleRef,
        descriptorVersion: BigInteger?,
        dest: AbstractState
    ) {
        dest.descriptorHandle = handleRef.name
        dest.descriptorVersion = descriptorVersion
        src?.also {
            dest.extension = makeOptionalExtension(src.extensions)
            dest.stateVersion = src.version
        }
    }

    fun mapIntoAbstractAlertState(
        src: AlertState?,
        handleRef: HandleRef,
        descriptorVersion: BigInteger?,
        dest: AbstractAlertState
    ) {
        mapIntoAbstractState(src, handleRef, descriptorVersion, dest)

        dest.activationState = mapAlertActivation(src?.activationState) ?: AlertActivation.ON
    }

    fun mapIntoAbstractMetricDescriptor(src: Metric<*>, dest: AbstractMetricDescriptor) {
        mapIntoAbstractDescriptor(src, dest)
        dest.bodySite = src.bodySites.map { mapCodedValue(it) }
        dest.metricCategory = mapMetricCategory(src.metricCategory)
        dest.metricAvailability = mapMetricAvailability(src.metricAvailability)
        dest.unit = mapCodedValue(src.unit)
        dest.derivationMethod = mapDerivationMethod(src.derivationMethod)
        dest.lifeTimePeriod = src.lifeTimePeriod?.toJavaDuration()
        dest.activationDuration = src.activationDuration?.toJavaDuration()
        dest.determinationPeriod = src.determinationPeriod?.toJavaDuration()
        dest.maxMeasurementTime = src.maxMeasurementTime?.toJavaDuration()
        dest.maxDelayTime = src.maxDelayTime?.toJavaDuration()
        dest.relation = src.relations.map { mapRelation(it) }
    }

    fun mapIntoAbstractMetricState(src: MetricState?, handleRef: HandleRef, descriptorVersion: BigInteger?,dest: AbstractMetricState) {
        mapIntoAbstractState(src, handleRef, descriptorVersion, dest)

        dest.activationState = mapComponentActivation(src?.activationState) ?: ComponentActivation.ON
        dest.physicalConnector = mapPhysicalConnectorInfo(src?.physicalConnector)
        dest.bodySite = src?.bodySites?.map { mapCodedValue(it) }
        dest.activeDeterminationPeriod = src?.activeDeterminationPeriod?.toJavaDuration()
        dest.lifeTimePeriod = src?.lifeTimePeriod?.toJavaDuration()
    }

    fun mapIntoAbstractOperationDescriptor(src: Operation<*>, dest: AbstractOperationDescriptor) {
        mapIntoAbstractDescriptor(src, dest)
        dest.operationTarget = src.operationTarget.name
        dest.invocationEffectiveTimeout = src.invocationEffectiveTimeout?.toJavaDuration()
        dest.maxTimeToFinish = src.maxTimeToFinish?.toJavaDuration()
        dest.isRetriggerable = src.isRetriggerable()
        dest.accessLevel = mapAccessLevel(src.accessLevel)
    }

    fun mapIntoAbstractOperationState(src: OperationState?, handleRef: HandleRef, descriptorVersion: BigInteger?, dest: AbstractOperationState) {
        mapIntoAbstractState(src, handleRef, descriptorVersion, dest)
        dest.operatingMode = mapOperatingMode(src?.operatingMode) ?: OperatingMode.EN
    }

    fun mapIntoAbstractDeviceComponentState(
        src: DeviceComponentState?,
        handleRef: HandleRef,
        descriptorVersion: BigInteger?,
        dest: AbstractDeviceComponentState
    ) {
        mapIntoAbstractState(src, handleRef, descriptorVersion, dest)
        dest.activationState = mapComponentActivation(src?.activationState) ?: ComponentActivation.ON
        dest.physicalConnector = mapPhysicalConnectorInfo(src?.physicalConnector)
    }

    fun mapIntoAbstractContextState(
        src: ContextState?,
        handleRef: HandleRef,
        descriptorVersion: BigInteger?,
        dest: AbstractContextState
    ) {

        mapIntoAbstractState(src, handleRef, descriptorVersion, dest)

        dest.handle = src?.handle?.name
        dest.category = mapCodedValue(src?.category)

        dest.contextAssociation = mapContextAssociation(src?.contextAssociation) ?: ContextAssociation.NO
        dest.bindingMdibVersion = src?.bindingMdibVersion
        dest.unbindingMdibVersion = src?.unbindingMdibVersion
        dest.bindingStartTime = src?.bindingStartTime
        dest.bindingEndTime = src?.bindingEndTime

        dest.validator = src?.validators?.map { mapInstanceIdentifier(it) }?.toList()
        dest.identification = src?.identification?.map { mapInstanceIdentifier(it) }?.toList()
    }

    private fun mapSco(
        src: Sco?
    ): ScoDescriptor? {
        return src?.let {
            ScoDescriptor().apply {
                mapIntoAbstractDeviceComponentDescriptor(src, this)

                requireNotNull(mapOperation) {
                    "Required operation mapper, but none was provided"
                }.also { mapOp ->
                    operation = src.operations.map { mapOp(it) }
                }
                mappedStates.add(mapScoState(src))
            }
        }
    }

    private fun mapScoState(
        src: Sco,
    ): ScoState {
        return ScoState().apply {
            mapIntoAbstractDeviceComponentState(src.state, src.handle.ref, src.version, this)
            this.operationGroup = src.state?.operationGroups?.map { mapOperationGroup(it) }
            this.invocationRequested = src.state?.invocationRequestedHandles?.map { it.name }
            this.invocationRequired = src.state?.invocationRequiredHandles?.map { it.name }
        }
    }
}