package org.somda.dsl.rendering.jaxb.mapping

import org.somda.dsl.biceps.metric.DistributionSampleArrayMetric
import org.somda.dsl.rendering.jaxb.mapping.base.mapAllowedValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapNumericMetricValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapRange
import org.somda.dsl.rendering.jaxb.mapping.base.mapSampleArrayValue
import org.somda.dsl.rendering.jaxb.mapping.base.mapStringMetricValue
import org.somda.dsl.biceps.metric.EnumStringMetric
import org.somda.dsl.biceps.metric.Metric
import org.somda.dsl.biceps.metric.NumericMetric
import org.somda.dsl.biceps.metric.RealTimeSampleArrayMetric
import org.somda.dsl.biceps.metric.StringMetric
import org.somda.dsl.rendering.jaxb.mapping.base.mapCodedValue
import org.somda.sdc.biceps.model.participant.AbstractMetricDescriptor
import org.somda.sdc.biceps.model.participant.AbstractState
import org.somda.sdc.biceps.model.participant.DistributionSampleArrayMetricDescriptor
import org.somda.sdc.biceps.model.participant.DistributionSampleArrayMetricState
import org.somda.sdc.biceps.model.participant.EnumStringMetricDescriptor
import org.somda.sdc.biceps.model.participant.EnumStringMetricState
import org.somda.sdc.biceps.model.participant.NumericMetricDescriptor
import org.somda.sdc.biceps.model.participant.NumericMetricState
import org.somda.sdc.biceps.model.participant.RealTimeSampleArrayMetricDescriptor
import org.somda.sdc.biceps.model.participant.RealTimeSampleArrayMetricState
import org.somda.sdc.biceps.model.participant.StringMetricDescriptor
import org.somda.sdc.biceps.model.participant.StringMetricState
import kotlin.time.toJavaDuration

internal class MetricMapping(private val mappedStates: MutableList<AbstractState>) {
    private val baseMapping = BaseMapping(mappedStates)

    fun mapMetric(src: Metric<*>): AbstractMetricDescriptor {
        return when (src) {
            is StringMetric -> mapStringMetric(src)
            is EnumStringMetric -> mapEnumStringMetric(src)
            is NumericMetric -> mapNumericMetric(src)
            is RealTimeSampleArrayMetric -> mapRealTimeSampleArrayMetric(src)
            is DistributionSampleArrayMetric -> mapDistributionSampleArrayMetric(src)
        }
    }

    private fun mapStringMetric(src: StringMetric) = StringMetricDescriptor().apply {
        baseMapping.mapIntoAbstractMetricDescriptor(src, this)

        mappedStates.add(mapStringMetricState(src))
    }

    private fun mapStringMetricState(src: StringMetric): StringMetricState {
        return StringMetricState().apply {
            baseMapping.mapIntoAbstractMetricState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                metricValue = mapStringMetricValue(it.metricValue)
            }
        }
    }

    private fun mapEnumStringMetric(src: EnumStringMetric) = EnumStringMetricDescriptor().apply {
        baseMapping.mapIntoAbstractMetricDescriptor(src, this)
        this.allowedValue = src.allowedValues.map { mapAllowedValue(it) }

        mappedStates.add(mapEnumStringMetricState(src))
    }

    private fun mapEnumStringMetricState(
        src: EnumStringMetric
    ): EnumStringMetricState {
        return EnumStringMetricState().apply {
            baseMapping.mapIntoAbstractMetricState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                metricValue = mapStringMetricValue(it.metricValue)
            }
        }
    }

    private fun mapNumericMetric(src: NumericMetric) = NumericMetricDescriptor().apply {
        baseMapping.mapIntoAbstractMetricDescriptor(src, this)
        this.averagingPeriod = src.averagingPeriod?.toJavaDuration()
        this.resolution = src.resolution.value
        this.technicalRange = src.technicalRanges.map { mapRange(it) }

        mappedStates.add(mapNumericMetricState(src))
    }

    private fun mapNumericMetricState(src: NumericMetric): NumericMetricState {
        return NumericMetricState().apply {
            baseMapping.mapIntoAbstractMetricState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                metricValue = mapNumericMetricValue(it.metricValue)
                physiologicalRange = it.physiologicalRanges.mapNotNull { rng -> mapRange(rng) }
            }
        }
    }

    private fun mapRealTimeSampleArrayMetric(src: RealTimeSampleArrayMetric): RealTimeSampleArrayMetricDescriptor {
        return RealTimeSampleArrayMetricDescriptor().apply {
            baseMapping.mapIntoAbstractMetricDescriptor(src, this)
            this.samplePeriod = src.samplePeriod.toJavaDuration()
            this.resolution = src.resolution.value
            this.technicalRange = src.technicalRanges.map { mapRange(it) }
            mappedStates.add(mapRealTimeSampleArrayMetricState(src))
        }
    }

    private fun mapRealTimeSampleArrayMetricState(src: RealTimeSampleArrayMetric): RealTimeSampleArrayMetricState {
        return RealTimeSampleArrayMetricState().apply {
            baseMapping.mapIntoAbstractMetricState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                metricValue = mapSampleArrayValue(it.metricValue)
                physiologicalRange = it.physiologicalRanges.mapNotNull { rng -> mapRange(rng) }
            }
        }
    }

    private fun mapDistributionSampleArrayMetric(src: DistributionSampleArrayMetric): DistributionSampleArrayMetricDescriptor {
        return DistributionSampleArrayMetricDescriptor().apply {
            baseMapping.mapIntoAbstractMetricDescriptor(src, this)
            this.domainUnit = mapCodedValue(src.domainUnit)
            this.distributionRange = mapRange(src.distributionRange)
            this.resolution = src.resolution.value
            this.technicalRange = src.technicalRanges.map { mapRange(it) }
            mappedStates.add(mapDistributionSampleArrayMetricState(src))
        }
    }

    private fun mapDistributionSampleArrayMetricState(src: DistributionSampleArrayMetric): DistributionSampleArrayMetricState {
        return DistributionSampleArrayMetricState().apply {
            baseMapping.mapIntoAbstractMetricState(src.state, src.handle.ref, src.version, this)
            src.state?.also {
                metricValue = mapSampleArrayValue(it.metricValue)
                physiologicalRange = it.physiologicalRanges.mapNotNull { rng -> mapRange(rng) }
            }
        }
    }
}