package org.somda.dsl.sdpi

import org.somda.dsl.rendering.jaxb.ExtensionRenderer
import org.somda.dsl.rendering.jaxb.JaxbContext
import org.somda.dsl.rendering.jaxb.Namespace
import org.somda.dsl.rendering.jaxb.NamespacePrefix
import org.somda.dsl.rendering.jaxb.XmlSchemaPath
import org.somda.dsl.rendering.jaxb.mapping.base.mapMustUnderstand
import org.somda.sdpi.model.CodedDecimalAttributeType
import org.somda.sdpi.model.CodedIntegerAttributeType
import org.somda.sdpi.model.CodedStringAttributeType
import org.somda.sdpi.model.MdcAttributeType
import org.somda.sdpi.model.ObjectFactory

public object SdpiExtensionsRenderer : ExtensionRenderer {
    override val jaxbContextPath: JaxbContext = JaxbContext("org.somda.sdpi.model")
    override val xmlSchemaPath: XmlSchemaPath? = null

    override val namespaceMapping: Pair<NamespacePrefix, Namespace>
        get() = NamespacePrefix("sdpi") to Namespace("urn:oid:1.3.6.1.4.1.19376.1.6.2.10.1.1.1")

    override fun render(dslModel: Any): Any? {
        return when (dslModel) {
            is EquipmentIdentifierExtension -> mapEquipmentIdentifierExtension(dslModel)
            is CodedAttributesExtension -> mapCodedAttributesExtension(dslModel)
            is GenderExtension -> mapGenderExtension(dslModel)
            else -> null
        }
    }

    private fun mapEquipmentIdentifierExtension(
        src: EquipmentIdentifierExtension
    ): Any {
        return ObjectFactory().createEquipmentIdentifier().apply {
            value = src.equipmentIdentifier
            mapMustUnderstand(src) { isMustUnderstand = true }
        }
    }

    private fun mapCodedAttributesExtension(
        src: CodedAttributesExtension
    ): Any {
        return ObjectFactory().createCodedAttributes(ObjectFactory().createCodedAttributesType().apply {
            this.codedStringAttribute = src.codedStrings.map {
                CodedStringAttributeType().apply {
                    mdcAttribute = mapMdcAttribute(it.mdcAttribute)
                    value = it.value
                }
            }
            this.codedIntegerAttribute = src.codedIntegers.map {
                CodedIntegerAttributeType().apply {
                    mdcAttribute = mapMdcAttribute(it.mdcAttribute)
                    value = it.value
                }
            }
            this.codedDecimalAttribute = src.codedDecimals.map {
                CodedDecimalAttributeType().apply {
                    mdcAttribute = mapMdcAttribute(it.mdcAttribute)
                    value = it.value.value
                }
            }
            mapMustUnderstand(src) { isMustUnderstand = true }
        })
    }

    private fun mapMdcAttribute(src: MdcAttribute): MdcAttributeType {
        return MdcAttributeType().apply {
            code = src.type.code
            codingSystem = src.type.codingSystem
            codingSystemVersion = src.type.codingSystemVersion
        }
    }

    private fun mapGender(src: Gender): org.somda.sdpi.model.GenderType {
        return when (src) {
            Gender.MALE -> org.somda.sdpi.model.GenderType.MALE
            Gender.FEMALE -> org.somda.sdpi.model.GenderType.FEMALE
            Gender.OTHER -> org.somda.sdpi.model.GenderType.OTHER
            Gender.UNKNOWN -> org.somda.sdpi.model.GenderType.UNKNOWN
        }
    }

    private fun mapGenderExtension(src: GenderExtension): Any? {
        return ObjectFactory().createGender().apply {
            value = mapGender(src.gender)
            mapMustUnderstand(src) { isMustUnderstand = true }
        }
    }
}