/*
 * Decompiled with CFR 0.152.
 */
package org.somda.sdc.dpws.udp;

import com.google.common.util.concurrent.AbstractIdleService;
import com.google.inject.assistedinject.Assisted;
import com.google.inject.assistedinject.AssistedInject;
import com.google.inject.name.Named;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Collections;
import java.util.Random;
import javax.annotation.Nullable;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.somda.sdc.common.logging.InstanceLogger;
import org.somda.sdc.dpws.CommunicationLog;
import org.somda.sdc.dpws.DpwsConstants;
import org.somda.sdc.dpws.network.NetworkInterfaceUtil;
import org.somda.sdc.dpws.soap.ApplicationInfo;
import org.somda.sdc.dpws.soap.CommunicationContext;
import org.somda.sdc.dpws.soap.TransportInfo;
import org.somda.sdc.dpws.soap.exception.TransportException;
import org.somda.sdc.dpws.udp.UdpBindingService;
import org.somda.sdc.dpws.udp.UdpMessage;
import org.somda.sdc.dpws.udp.UdpMessageReceiverCallback;

public class UdpBindingServiceImpl
extends AbstractIdleService
implements UdpBindingService {
    private static final Logger LOG = LogManager.getLogger(UdpBindingServiceImpl.class);
    private final Random random = new Random();
    private final NetworkInterface networkInterface;
    private final InetAddress multicastGroup;
    private final Integer socketPort;
    private final Logger instanceLogger;
    private Thread multicastSocketRunner;
    private Thread unicastSocketRunner;
    private DatagramSocket incomingSocket;
    private MulticastSocket multicastSocket;
    private DatagramSocket outgoingSocket;
    private final int maxMessageSize;
    private NetworkInterfaceUtil networkInterfaceUtil;
    private final CommunicationLog communicationLog;
    private UdpMessageReceiverCallback receiver;
    private InetAddress networkInterfaceAddress;
    private InetSocketAddress multicastAddress;

    @AssistedInject
    UdpBindingServiceImpl(@Assisted NetworkInterface networkInterface, @Assisted @Nullable InetAddress multicastGroup, @Assisted(value="multicastPort") Integer multicastPort, @Assisted(value="maxMessageSize") Integer maxMessageSize, NetworkInterfaceUtil networkInterfaceUtil, CommunicationLog communicationLog, @Named(value="Common.InstanceIdentifier") String frameworkIdentifier) {
        this.instanceLogger = InstanceLogger.wrapLogger((Logger)LOG, (String)frameworkIdentifier);
        this.networkInterface = networkInterface;
        this.multicastGroup = multicastGroup;
        this.socketPort = multicastPort;
        this.maxMessageSize = maxMessageSize;
        this.networkInterfaceUtil = networkInterfaceUtil;
        this.communicationLog = communicationLog;
        this.multicastAddress = new InetSocketAddress(multicastGroup, (int)this.socketPort);
        this.multicastSocket = null;
        this.networkInterfaceAddress = null;
    }

    protected void startUp() throws Exception {
        this.instanceLogger.info("Start UDP binding on network interface {}", (Object)this);
        this.networkInterfaceAddress = this.networkInterfaceUtil.getFirstIpV4Address(this.networkInterface).orElseThrow(() -> new SocketException(String.format("Could not retrieve network interface address from: %s", this.networkInterface)));
        this.instanceLogger.info("Bind to address {}", (Object)this.networkInterfaceAddress);
        this.outgoingSocket = new DatagramSocket(0, this.networkInterfaceAddress);
        this.instanceLogger.info("Outgoing socket at {} is open", (Object)this.outgoingSocket.getLocalSocketAddress());
        if (this.multicastGroup != null) {
            if (!this.multicastGroup.isMulticastAddress()) {
                throw new Exception(String.format("Given address is not a multicast address: %s", this.multicastGroup));
            }
            this.multicastSocket = new MulticastSocket(this.socketPort);
            this.instanceLogger.info("Join to UDP multicast address group {}", (Object)this.multicastAddress);
            this.multicastSocket.joinGroup(this.multicastAddress, this.networkInterface);
            this.incomingSocket = this.multicastSocket;
        } else {
            this.incomingSocket = new DatagramSocket(0, this.networkInterfaceAddress);
            this.instanceLogger.info("Incoming socket is open: {}", (Object)this.incomingSocket.getLocalSocketAddress());
        }
        if (this.receiver == null) {
            this.instanceLogger.info("No data receiver configured; ignore incoming UDP messages");
        } else {
            this.instanceLogger.info("Data receiver configured; process incoming UDP messages");
            this.multicastSocketRunner = new Thread(() -> {
                while (!this.multicastSocketRunner.isInterrupted()) {
                    DatagramPacket packet = new DatagramPacket(new byte[this.maxMessageSize], this.maxMessageSize);
                    try {
                        this.incomingSocket.receive(packet);
                    }
                    catch (IOException e) {
                        this.instanceLogger.trace("Could not process UDP packet. Discard.");
                        continue;
                    }
                    CommunicationContext ctxt = new CommunicationContext(new ApplicationInfo(), new TransportInfo("soap.udp", this.incomingSocket.getLocalAddress().getHostAddress(), this.incomingSocket.getLocalPort(), packet.getAddress().getHostAddress(), packet.getPort(), Collections.emptyList()));
                    UdpMessage message = new UdpMessage(packet.getData(), packet.getLength(), ctxt);
                    this.logUdpPacket(CommunicationLog.Direction.INBOUND, packet);
                    this.receiver.receive(message);
                }
            });
            this.unicastSocketRunner = new Thread(() -> {
                while (!this.unicastSocketRunner.isInterrupted()) {
                    DatagramPacket packet = new DatagramPacket(new byte[this.maxMessageSize], this.maxMessageSize);
                    try {
                        this.outgoingSocket.receive(packet);
                    }
                    catch (IOException e) {
                        this.instanceLogger.trace("Could not process UDP packet. Discard.");
                        continue;
                    }
                    CommunicationContext ctxt = new CommunicationContext(new ApplicationInfo(), new TransportInfo("soap.udp", this.outgoingSocket.getLocalAddress().getHostAddress(), this.outgoingSocket.getLocalPort(), packet.getAddress().getHostAddress(), packet.getPort(), Collections.emptyList()));
                    UdpMessage message = new UdpMessage(packet.getData(), packet.getLength(), ctxt);
                    this.logUdpPacket(CommunicationLog.Direction.INBOUND, packet);
                    this.receiver.receive(message);
                }
            });
            this.multicastSocketRunner.start();
            this.unicastSocketRunner.start();
        }
        Thread.sleep(1000L);
        this.instanceLogger.info("UDP binding {} is running", (Object)this);
    }

    protected void shutDown() throws Exception {
        this.instanceLogger.info("Shut down UDP binding {}", (Object)this);
        this.multicastSocketRunner.interrupt();
        this.unicastSocketRunner.interrupt();
        if (this.multicastSocket != null && this.multicastGroup != null && this.multicastAddress != null) {
            this.multicastSocket.leaveGroup(this.multicastAddress, this.networkInterface);
        }
        this.incomingSocket.close();
        this.outgoingSocket.close();
        this.instanceLogger.info("UDP binding {} shut down", (Object)this);
    }

    @Override
    public void setMessageReceiver(UdpMessageReceiverCallback receiver) {
        this.receiver = receiver;
    }

    @Override
    public void sendMessage(UdpMessage message) throws IOException, TransportException {
        if (!this.isRunning()) {
            this.instanceLogger.warn("Try to send message, but service is not running. Skip.");
            return;
        }
        if (message.getLength() > this.maxMessageSize) {
            String msg = String.format("Exceed maximum UDP message size. Try to write %d Bytes, but only %d Bytes allowed.", message.getLength(), this.maxMessageSize);
            throw new IOException(msg);
        }
        DatagramPacket packet = new DatagramPacket(message.getData(), message.getLength());
        if (message.hasTransportData()) {
            packet.setAddress(InetAddress.getByName(message.getHost()));
            packet.setPort(message.getPort());
            this.logUdpPacket(CommunicationLog.Direction.OUTBOUND, packet);
        } else {
            if (this.multicastGroup == null) {
                throw new TransportException(String.format("No transport data in UDP message, which is required as no multicast group is available. Message: %s", message.toString()));
            }
            packet.setAddress(this.multicastGroup);
            packet.setPort(this.socketPort);
            this.logUdpPacket(CommunicationLog.Direction.OUTBOUND, packet);
        }
        this.sendMessageWithRetry(packet);
    }

    private void sendMessageWithRetry(DatagramPacket packet) throws IOException {
        this.outgoingSocket.send(packet);
        int udpMinDelay = (int)DpwsConstants.UDP_MIN_DELAY.toMillis();
        int udpMaxDelay = (int)DpwsConstants.UDP_MAX_DELAY.toMillis();
        int udpUpperDelay = (int)DpwsConstants.UDP_UPPER_DELAY.toMillis();
        int t = this.random.nextInt(udpMaxDelay - udpMinDelay + 1) + udpMinDelay;
        for (int udpRepeat = 1; udpRepeat > 0; --udpRepeat) {
            try {
                Thread.sleep(t);
            }
            catch (InterruptedException e) {
                this.instanceLogger.info("Thread interrupted");
                break;
            }
            this.outgoingSocket.send(packet);
            if ((t *= 2) <= udpUpperDelay) continue;
            t = udpUpperDelay;
        }
    }

    public String toString() {
        if (this.isRunning()) {
            return this.makeStringRepresentation();
        }
        return String.format("[%s]", this.networkInterface);
    }

    private String makeStringRepresentation() {
        String multicast = "w/o multicast";
        if (this.multicastSocket != null) {
            multicast = String.format("w/ multicast joined at %s:%s", this.multicastGroup.getHostName(), this.socketPort);
        }
        return String.format("[%s:[%s|%s] %s]", this.networkInterfaceAddress.toString(), this.incomingSocket.getLocalPort(), this.outgoingSocket.getLocalPort(), multicast);
    }

    private void logUdpPacket(CommunicationLog.Direction direction, DatagramPacket packet) {
        TransportInfo requestTransportInfo = new TransportInfo("soap.udp", this.outgoingSocket.getLocalAddress().getHostName(), this.outgoingSocket.getPort(), packet.getAddress().getHostAddress(), packet.getPort(), Collections.emptyList());
        CommunicationContext requestCommContext = new CommunicationContext(new ApplicationInfo(), requestTransportInfo);
        try (ByteArrayInputStream messageData = new ByteArrayInputStream(packet.getData(), packet.getOffset(), packet.getLength());){
            this.communicationLog.logMessage(direction, CommunicationLog.TransportType.UDP, CommunicationLog.MessageType.UNKNOWN, requestCommContext, messageData);
        }
        catch (IOException e) {
            this.instanceLogger.warn("Could not log udp message though the communication log", (Throwable)e);
        }
    }
}

