/*
 * Decompiled with CFR 0.152.
 */
package org.somda.sdc.dpws.udp;

import com.google.common.util.concurrent.AbstractIdleService;
import com.google.inject.assistedinject.Assisted;
import com.google.inject.assistedinject.AssistedInject;
import com.google.inject.name.Named;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.MulticastSocket;
import java.net.NetworkInterface;
import java.net.StandardProtocolFamily;
import java.net.StandardSocketOptions;
import java.nio.channels.DatagramChannel;
import java.util.Collections;
import java.util.Random;
import javax.annotation.Nullable;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.somda.sdc.common.logging.InstanceLogger;
import org.somda.sdc.dpws.CommunicationLog;
import org.somda.sdc.dpws.DpwsConstants;
import org.somda.sdc.dpws.factory.CommunicationLogFactory;
import org.somda.sdc.dpws.soap.ApplicationInfo;
import org.somda.sdc.dpws.soap.CommunicationContext;
import org.somda.sdc.dpws.soap.TransportInfo;
import org.somda.sdc.dpws.soap.exception.TransportException;
import org.somda.sdc.dpws.udp.UdpBindingService;
import org.somda.sdc.dpws.udp.UdpMessage;
import org.somda.sdc.dpws.udp.UdpMessageReceiverCallback;

public class UdpBindingServiceImpl
extends AbstractIdleService
implements UdpBindingService {
    private static final Logger LOG = LogManager.getLogger(UdpBindingServiceImpl.class);
    private final Random random = new Random();
    private final NetworkInterface networkInterface;
    private final InetAddress multicastGroup;
    private final Integer socketPort;
    private final Logger instanceLogger;
    private Thread unicastSocketRunner;
    private Thread multicastSocketRunner;
    private DatagramSocket unicastSocket;
    private MulticastSocket multicastSocket;
    private final int maxMessageSize;
    private final int multicastTtl;
    private final CommunicationLog communicationLog;
    private UdpMessageReceiverCallback receiver;
    private final InetSocketAddress multicastAddress;

    @AssistedInject
    UdpBindingServiceImpl(@Assisted NetworkInterface networkInterface, @Assisted @Nullable InetAddress multicastGroup, @Assisted(value="multicastPort") Integer multicastPort, @Assisted(value="maxMessageSize") Integer maxMessageSize, @Named(value="Dpws.MulticastTtl") Integer multicastTtl, CommunicationLogFactory communicationLogFactory, @Named(value="Common.InstanceIdentifier") String frameworkIdentifier) {
        if (multicastGroup != null && !multicastGroup.isMulticastAddress()) {
            throw new IllegalArgumentException("Given address is not a multicast address: " + multicastGroup);
        }
        this.instanceLogger = InstanceLogger.wrapLogger((Logger)LOG, (String)frameworkIdentifier);
        this.networkInterface = networkInterface;
        this.multicastGroup = multicastGroup;
        this.socketPort = multicastPort;
        this.maxMessageSize = maxMessageSize;
        this.multicastTtl = multicastTtl;
        this.communicationLog = communicationLogFactory.createCommunicationLog();
        this.multicastAddress = new InetSocketAddress(multicastGroup, (int)this.socketPort);
    }

    protected void startUp() throws Exception {
        this.instanceLogger.info("Start UDP binding on network interface {}", (Object)this);
        if (Runtime.version().feature() < 15) {
            this.unicastSocket = new MulticastSocket(0);
            ((MulticastSocket)this.unicastSocket).setNetworkInterface(this.networkInterface);
            ((MulticastSocket)this.unicastSocket).setTimeToLive(this.multicastTtl);
        } else {
            this.unicastSocket = DatagramChannel.open(StandardProtocolFamily.INET).socket();
            this.unicastSocket.setOption(StandardSocketOptions.IP_MULTICAST_IF, this.networkInterface);
            this.unicastSocket.setOption(StandardSocketOptions.IP_MULTICAST_TTL, this.multicastTtl);
        }
        this.instanceLogger.info("Unicast socket at {} is open", (Object)this.unicastSocket.getLocalSocketAddress());
        if (this.multicastGroup == null) {
            this.multicastSocket = new MulticastSocket(0);
            this.multicastSocket.setNetworkInterface(this.networkInterface);
            this.instanceLogger.info("Multicast socket is open: {}", (Object)this.multicastSocket.getLocalSocketAddress());
        } else {
            this.multicastSocket = new MulticastSocket(this.socketPort);
            this.instanceLogger.info("Join to UDP multicast address group {}", (Object)this.multicastAddress);
            this.multicastSocket.joinGroup(this.multicastAddress, this.networkInterface);
        }
        if (this.receiver == null) {
            this.instanceLogger.info("No data receiver configured; ignore incoming UDP messages");
        } else {
            this.instanceLogger.info("Data receiver configured; process incoming UDP messages");
            this.multicastSocketRunner = this.startThreadToReceiveFromSocket(this.multicastSocket);
            this.unicastSocketRunner = this.startThreadToReceiveFromSocket(this.unicastSocket);
        }
        Thread.sleep(1000L);
        this.instanceLogger.info("UDP binding {} is running", (Object)this);
    }

    private Thread startThreadToReceiveFromSocket(DatagramSocket socket) {
        Thread thread = new Thread(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                DatagramPacket packet = new DatagramPacket(new byte[this.maxMessageSize], this.maxMessageSize);
                try {
                    socket.receive(packet);
                }
                catch (IOException e) {
                    this.instanceLogger.trace("Could not process UDP packet. Discard.");
                    continue;
                }
                CommunicationContext ctxt = new CommunicationContext(new ApplicationInfo(), new TransportInfo("soap.udp", socket.getLocalAddress().getHostAddress(), socket.getLocalPort(), packet.getAddress().getHostAddress(), packet.getPort(), Collections.emptyList()), null);
                UdpMessage message = new UdpMessage(packet.getData(), packet.getLength(), ctxt);
                this.logUdpPacket(CommunicationLog.Direction.INBOUND, packet);
                this.receiver.receive(message);
            }
        });
        thread.start();
        return thread;
    }

    protected void shutDown() throws Exception {
        this.instanceLogger.info("Shut down UDP binding {}", (Object)this);
        this.multicastSocketRunner.interrupt();
        this.unicastSocketRunner.interrupt();
        if (this.multicastGroup != null && this.multicastAddress != null) {
            this.multicastSocket.leaveGroup(this.multicastAddress, this.networkInterface);
        }
        this.multicastSocket.close();
        this.unicastSocket.close();
        this.instanceLogger.info("UDP binding {} shut down", (Object)this);
    }

    @Override
    public void setMessageReceiver(UdpMessageReceiverCallback receiver) {
        this.receiver = receiver;
    }

    @Override
    public void sendMessage(UdpMessage message) throws IOException, TransportException {
        if (!this.isRunning()) {
            this.instanceLogger.warn("Try to send message, but service is not running. Skip.");
            return;
        }
        if (message.getLength() > this.maxMessageSize) {
            String msg = String.format("Exceed maximum UDP message size. Try to write %d Bytes, but only %d Bytes allowed.", message.getLength(), this.maxMessageSize);
            throw new IOException(msg);
        }
        DatagramPacket packet = new DatagramPacket(message.getData(), message.getLength());
        if (message.hasTransportData()) {
            packet.setAddress(InetAddress.getByName(message.getHost()));
            packet.setPort(message.getPort());
            this.logUdpPacket(CommunicationLog.Direction.OUTBOUND, packet);
        } else {
            if (this.multicastGroup == null) {
                throw new TransportException(String.format("No transport data in UDP message, which is required as no multicast group is available. Message: %s", message));
            }
            packet.setAddress(this.multicastGroup);
            packet.setPort(this.socketPort);
            this.logUdpPacket(CommunicationLog.Direction.OUTBOUND, packet);
        }
        this.sendMessageWithRetry(packet);
    }

    private void sendMessageWithRetry(DatagramPacket packet) throws IOException {
        this.unicastSocket.send(packet);
        int udpMinDelay = (int)DpwsConstants.UDP_MIN_DELAY.toMillis();
        int udpMaxDelay = (int)DpwsConstants.UDP_MAX_DELAY.toMillis();
        int udpUpperDelay = (int)DpwsConstants.UDP_UPPER_DELAY.toMillis();
        int t = this.random.nextInt(udpMaxDelay - udpMinDelay + 1) + udpMinDelay;
        for (int udpRepeat = 1; udpRepeat > 0; --udpRepeat) {
            try {
                Thread.sleep(t);
            }
            catch (InterruptedException e) {
                this.instanceLogger.info("Thread interrupted");
                break;
            }
            this.unicastSocket.send(packet);
            if ((t *= 2) <= udpUpperDelay) continue;
            t = udpUpperDelay;
        }
    }

    public String toString() {
        if (this.isRunning()) {
            return this.makeStringRepresentation();
        }
        return String.format("[%s]", this.networkInterface);
    }

    private String makeStringRepresentation() {
        String multicast = "w/o multicast";
        if (this.multicastGroup != null) {
            multicast = String.format("w/ multicast joined at %s:%s", this.multicastGroup.getHostAddress(), this.socketPort);
        }
        return String.format("[%s:[%s|%s] %s]", this.networkInterface.toString(), this.multicastSocket.getLocalPort(), this.unicastSocket.getLocalPort(), multicast);
    }

    private void logUdpPacket(CommunicationLog.Direction direction, DatagramPacket packet) {
        TransportInfo requestTransportInfo = new TransportInfo("soap.udp", this.unicastSocket.getLocalAddress().getHostName(), this.unicastSocket.getPort(), packet.getAddress().getHostAddress(), packet.getPort(), Collections.emptyList());
        CommunicationContext requestCommContext = new CommunicationContext(new ApplicationInfo(), requestTransportInfo, null);
        try (ByteArrayInputStream messageData = new ByteArrayInputStream(packet.getData(), packet.getOffset(), packet.getLength());){
            this.communicationLog.logMessage(direction, CommunicationLog.TransportType.UDP, CommunicationLog.MessageType.UNKNOWN, requestCommContext, messageData);
        }
        catch (IOException e) {
            this.instanceLogger.warn("Could not log udp message though the communication log", (Throwable)e);
        }
    }
}

