package com.example.provider1;

import com.example.BaseUtil;
import com.google.inject.Guice;
import com.google.inject.Injector;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.config.Configurator;
import org.apache.logging.log4j.core.config.DefaultConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.somda.sdc.biceps.guice.DefaultBicepsConfigModule;
import org.somda.sdc.biceps.guice.DefaultBicepsModule;
import org.somda.sdc.biceps.model.participant.AbstractMetricValue;
import org.somda.sdc.biceps.model.participant.GenerationMode;
import org.somda.sdc.biceps.model.participant.MeasurementValidity;
import org.somda.sdc.common.guice.DefaultHelperModule;
import org.somda.sdc.dpws.DpwsConfig;
import org.somda.sdc.dpws.crypto.CryptoConfig;
import org.somda.sdc.dpws.crypto.CryptoSettings;
import org.somda.sdc.dpws.guice.DefaultDpwsModule;
import org.somda.sdc.glue.GlueConstants;
import org.somda.sdc.glue.guice.DefaultGlueConfigModule;
import org.somda.sdc.glue.guice.DefaultGlueModule;
import org.somda.sdc.glue.guice.GlueDpwsConfigModule;

import javax.net.ssl.HostnameVerifier;
import java.net.URI;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.List;

/**
 * This class provides the configuration used for the provider instance.
 * <p>
 * Overwriting configuration steps allows customizing the behavior of the framework through
 * injection.
 */
public class ProviderUtil extends BaseUtil {
    private static final Logger LOG = LoggerFactory.getLogger(ProviderUtil.class);
    public static final String OPT_REPORT_INTERVAL = "report_interval";
    public static final String OPT_WAVEFORMS_INTERVAL = "waveform_interval";

    private static final String DEFAULT_REPORT_INTERVAL = "5000"; // millis
    private static final String DEFAULT_WAVEFORM_INTERVAL = "100"; // millis

    private final Injector injector;
    private final Duration reportInterval;
    private final Duration waveformInterval;

    public ProviderUtil(String[] args) {
        super(args);
        Configurator.initialize(new DefaultConfiguration());
        Configurator.setRootLevel(Level.INFO);

        reportInterval = Duration.ofMillis(
                Long.parseLong(getParsedArgs().getOptionValue(OPT_REPORT_INTERVAL, DEFAULT_REPORT_INTERVAL))
        );

        waveformInterval = Duration.ofMillis(
                Long.parseLong(getParsedArgs().getOptionValue(OPT_WAVEFORMS_INTERVAL, DEFAULT_WAVEFORM_INTERVAL))
        );


        injector = Guice.createInjector(
                new DefaultGlueModule(),
                new DefaultGlueConfigModule(),
                new DefaultBicepsModule(),
                new DefaultBicepsConfigModule(),
                new DefaultHelperModule(),
                new DefaultDpwsModule(),
                new GlueDpwsConfigModule() {
                    @Override
                    protected void customConfigure() {
                        super.customConfigure();
                        bind(CryptoConfig.CRYPTO_SETTINGS,
                                CryptoSettings.class,
                                createCustomCryptoSettings()
                        );
                        bind(DpwsConfig.HTTPS_SUPPORT, Boolean.class, isUseTls());
                        bind(DpwsConfig.HTTP_SUPPORT, Boolean.class, !isUseTls());
                        bind(CryptoConfig.CRYPTO_DEVICE_HOSTNAME_VERIFIER,
                                HostnameVerifier.class,
                                (hostname, session) -> {
                                    try {
                                        // since this is not a real implementation, we still want to allow all peers
                                        // which is why this doesn't really filter anything
                                        // returning false in this filter would reject an incoming request
                                        var peerCerts = session.getPeerCertificates();
                                        final X509Certificate x509 = (X509Certificate) peerCerts[0];
                                        List<String> extendedKeyUsage = x509.getExtendedKeyUsage();
                                        if (extendedKeyUsage == null || extendedKeyUsage.isEmpty()) {
                                            LOG.warn("No EKU in peer certificate");
                                            return true;
                                        }

                                        // find matching provider key purpose
                                        for (String key : extendedKeyUsage) {
                                            try {
                                                URI keyUri = URI.create(key);
                                                if (keyUri.equals(URI.create(GlueConstants.OID_KEY_PURPOSE_SDC_SERVICE_CONSUMER))) {
                                                    LOG.debug("SDC Service Consumer PKP found");
                                                    return true;
                                                }
                                            } catch (IllegalArgumentException e) {
                                                // don't care, was no uri
                                            }
                                        }
                                        return true;
                                    } catch (Exception e) {
                                        LOG.error("Error while validating client certificate: {}", e.getMessage());
                                        LOG.trace("Error while validating client certificate", e);
                                    }
                                    return false;
                                });

                    }
                });
    }

    public Injector getInjector() {
        return injector;
    }

    public static void addMetricQualityDemo(AbstractMetricValue val) {
        if (val.getMetricQuality() == null) {
            var qual = new AbstractMetricValue.MetricQuality();
            qual.setMode(GenerationMode.DEMO);
            qual.setValidity(MeasurementValidity.VLD);
            val.setMetricQuality(qual);
        }
    }

    @Override
    protected Options configureOptions() {
        var options = super.configureOptions();

        {
            String message = "Interval in ms in which reports are being generated."
                    + " Default: " + DEFAULT_REPORT_INTERVAL;
            Option reportIntervalOpt = new Option(null, OPT_REPORT_INTERVAL,
                    true, message);
            reportIntervalOpt.setType(Long.class);
            options.addOption(reportIntervalOpt);
        }

        {

            String message = "Interval in ms in which waveforms are being generated."
                    + " Default: " + DEFAULT_WAVEFORM_INTERVAL;
            Option waveformIntervalOpt = new Option(null, OPT_WAVEFORMS_INTERVAL,
                    true, message);
            waveformIntervalOpt.setType(Long.class);
            options.addOption(waveformIntervalOpt);
        }

        return options;
    }

    public Duration getReportInterval() {
        return reportInterval;
    }

    public Duration getWaveformInterval() {
        return waveformInterval;
    }
}
