package org.somda.sdc.glue.common;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Table;
import org.somda.sdc.glue.provider.localization.LocalizationException;
import org.somda.sdc.glue.provider.localization.helper.LocalizedData;

import java.math.BigInteger;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Utility functions for Localization service.
 */
public class LocalizationServiceFilterUtil {


    /**
     * Filters localized texts by provided languages.
     *
     * @param localizedTextTable represents table of all available localized texts.
     * @param languages          list used as filter. If empty list is provided localized texts of all languages are
     *                           returned.
     * @return a multimap of rows and values of the provided localizedTextTable.
     */
    public static <R, V> Multimap<R, V> filterByLanguage(
        Table<R, String, V> localizedTextTable, List<String> languages) {

        Multimap<R, V> referencesToTextMap = ArrayListMultimap.create();

        // if language list provided filter records by language, otherwise include all languages
        if (!languages.isEmpty()) {
            languages.forEach(language -> localizedTextTable.column(language).forEach(referencesToTextMap::put));
        } else {
            localizedTextTable.columnKeySet().forEach(
                key -> localizedTextTable.column(key).forEach(referencesToTextMap::put));
        }
        return referencesToTextMap;
    }


    /**
     * Filter localized texts by provided references.
     *
     * @param referencesToTextMap represents a map of reference as a key and localized text as a value.
     * @param references          list used as filter.
     * @return a filtered list of localized.
     */
    public static <T> List<T> filterByReferences(Multimap<String, T> referencesToTextMap,
                                                 List<String> references) {
        return references.stream()
            .filter(referencesToTextMap::containsKey)
            .map(referencesToTextMap::get)
            .flatMap(Collection::stream)
            .distinct()
            .collect(Collectors.toList());
    }

    /**
     * Filters localized data by provided number of lines.
     *
     * @param data          list of localized data to filter.
     * @param numberOfLines list used as filter. If empty no filtering is applied.
     * @return a filtered list of localized data.
     */
    public static List<LocalizedData> filterByNumberOfLines(List<LocalizedData> data, List<BigInteger> numberOfLines) {
        if (numberOfLines.isEmpty()) {
            return data;
        }
        // all smaller numbers of lines are included with the largest one when filtering
        var largestNumberOfLines = numberOfLines.stream()
            .max(BigInteger::compareTo)
            .orElseThrow(() -> new LocalizationException("Failed to process number of lines."));
        return data.stream().filter(datum -> datum.getLines().compareTo(largestNumberOfLines) < 1)
            .collect(Collectors.toList());
    }
}
