package org.somda.sdc.glue.provider.localization.helper

import com.google.common.collect.HashBasedTable
import com.google.common.collect.Table
import org.somda.sdc.biceps.model.participant.LocalizedTextWidth
import org.somda.sdc.glue.common.LocalizationServiceFilterUtil.filterByLanguage
import org.somda.sdc.glue.common.LocalizationServiceFilterUtil.filterByNumberOfLines
import org.somda.sdc.glue.provider.localization.LocalizationStorage
import java.math.BigInteger
import java.util.NavigableMap
import java.util.TreeMap

typealias Reference = String
typealias Language = String
typealias Version = BigInteger

/**
 * Default implementation of [LocalizationStorage].
 *
 * Localized texts are stored in heap map and can be added during runtime.
 */
class HeapBasedLocalizationStorage : LocalizationStorage {
    private val supportedLanguages = mutableSetOf<String>()
    private val localizationStorage =
        mutableMapOf<Reference, Table<LocalizedTextWidth, Language, NavigableMap<Version, LocalizedData>>>()

    @Synchronized
    override fun getSupportedLanguages(): List<String> = supportedLanguages.toList()

    @Synchronized
    override fun getLocalizedData(
        references: List<String>,
        version: BigInteger?,
        languages: List<String>,
        textWidths: List<LocalizedTextWidth>,
        numberOfLines: List<BigInteger>
    ): List<LocalizedData> {
        val refs = references.ifEmpty { localizationStorage.keys }
        return refs.mapNotNull { localizationStorage[it] }
            .map { filterByLanguage(it, languages.ifEmpty { it.columnKeySet().toList() }) }
            .flatMap { textWidthMaps ->
                val filteredMaps = when (val width = textWidths.maxOrNull()) {
                    null -> textWidthMaps.values()
                    else -> textWidthMaps.entries().filter { it.key <= width }.map { it.value }
                }
                filteredMaps.mapNotNull {
                    if (version == null)
                        it.lastEntry()?.value
                    else
                        it[version]
                }
            }
            .let {
                filterByNumberOfLines(it, numberOfLines)
            }
    }

    @Synchronized
    fun addLocalizedData(data: LocalizedData) {
        addToSupportedLanguages(data)
        addToStorage(data)
    }

    @Synchronized
    fun addAllLocalizedData(texts: List<LocalizedData>) = texts.forEach { addLocalizedData(it) }

    private fun addToStorage(data: LocalizedData) {
        val table = localizationStorage.getOrPut(data.reference) {
            HashBasedTable.create<LocalizedTextWidth, Language, NavigableMap<Version, LocalizedData>>()
        }
        val versionMap = table.get(data.textWidth, data.language)
            ?: TreeMap<Version, LocalizedData>().also { table.put(data.textWidth, data.language, it) }
        versionMap[data.version] = data
    }

    private fun addToSupportedLanguages(text: LocalizedData) {
        supportedLanguages.add(text.language)
    }
}
