package org.somda.sdc.glue.provider.localization.helper

import org.somda.sdc.biceps.model.participant.LocalizedText
import org.somda.sdc.biceps.model.participant.LocalizedTextWidth
import java.math.BigInteger
import kotlin.jvm.Throws

/**
 * Wrapper for localized text and its associated line count.
 */
data class LocalizedData private constructor(
    val value: String,
    val reference: String,
    val language: String,
    val version: BigInteger,
    val textWidth: LocalizedTextWidth,
    val lines: BigInteger
) {

    fun toLocalizedText(): LocalizedText {
        val text = LocalizedText().apply {
            value = this@LocalizedData.value
            ref = this@LocalizedData.reference
            lang = this@LocalizedData.language
            version = this@LocalizedData.version
            textWidth = this@LocalizedData.textWidth
        }
        return text
    }

    companion object {
        /**
         * Tries to create a [LocalizedData] from a [LocalizedText] and the number of lines.
         * Checks if the provided [LocalizedText] has all required attributes set and that the
         * number of lines is greater than zero.
         */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun tryFrom(localizedText: LocalizedText, lines: BigInteger): LocalizedData {
            return tryFrom(
                localizedText.value,
                localizedText.ref,
                localizedText.lang,
                localizedText.version,
                localizedText.textWidth,
                lines
            )
        }

        /**
         * Tries to create a [LocalizedData] from a [LocalizedText] and the number of lines.
         * Checks if the provided [LocalizedText] has all required attributes set and that the
         * number of lines is greater than zero.
         */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun tryFrom(
            value: String?,
            reference: String?,
            language: String?,
            version: BigInteger?,
            textWidth: LocalizedTextWidth?,
            lines: BigInteger
        ): LocalizedData =
            LocalizedData(
                value = requireNotNull(value) { "Value was null" },
                reference = requireNotNullOrEmpty(reference) { "Reference was null or empty" },
                language = requireNotNullOrEmpty(language) { "Language was null or empty" },
                version = requireNotNullAtLeast(version, BigInteger.ZERO) {
                    "Version was null or below zero"
                },
                textWidth = requireNotNull(textWidth) { "Width was null" },
                lines = requireNotNullAtLeast(lines, BigInteger.ONE) { "Lines was below one" }
            )

        private inline fun requireNotNullAtLeast(
            value: BigInteger?,
            minimum: BigInteger,
            lazyMessage: () -> Any
        ): BigInteger {
            val notNull: BigInteger = requireNotNull(value, lazyMessage)
            if (notNull < minimum) {
                val message = lazyMessage()
                throw IllegalArgumentException("$message - was $notNull, minimum $minimum")
            } else {
                return value
            }
        }

        private inline fun requireNotNullOrEmpty(value: String?, lazyMessage: () -> Any): String {
            val notNull = requireNotNull(value, lazyMessage)
            if (notNull.isEmpty()) {
                val message = lazyMessage()
                throw IllegalArgumentException(message.toString())
            } else {
                return value
            }
        }
    }
}
