/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.go.converter;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.go.converter.ExternalProcessStreamConsumer;
import org.sonarsource.slang.api.ASTConverter;
import org.sonarsource.slang.api.ParseException;
import org.sonarsource.slang.api.Tree;
import org.sonarsource.slang.persistence.JsonTree;

public class GoConverter
implements ASTConverter {
    private static final long MAX_SUPPORTED_SOURCE_FILE_SIZE = 1500000L;
    private static final Logger LOG = LoggerFactory.getLogger(GoConverter.class);
    private static final long PROCESS_TIMEOUT_MS = 5000L;
    private final ProcessBuilder processBuilder;
    private final ExternalProcessStreamConsumer errorConsumer;

    public GoConverter(File workDir) {
        this(new DefaultCommand(workDir));
    }

    GoConverter(Command command) {
        this.processBuilder = new ProcessBuilder(command.getCommand());
        this.errorConsumer = new ExternalProcessStreamConsumer();
    }

    @Override
    public Tree parse(String content) {
        if ((long)content.length() > 1500000L) {
            throw new ParseException("The file size is too big and should be excluded, its size is " + content.length() + " (maximum allowed is 1500000 bytes)");
        }
        try {
            return JsonTree.fromJson(this.executeGoToJsonProcess(content));
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new ParseException("Go parser external process interrupted: " + e.getMessage(), null, e);
        }
        catch (IOException e) {
            throw new ParseException(e.getMessage(), null, e);
        }
    }

    private String executeGoToJsonProcess(String content) throws IOException, InterruptedException {
        String output;
        Process process = this.processBuilder.start();
        this.errorConsumer.consumeStream(process.getErrorStream(), arg_0 -> ((Logger)LOG).debug(arg_0));
        try (OutputStream out = process.getOutputStream();){
            out.write(content.getBytes(StandardCharsets.UTF_8));
        }
        try (InputStream in = process.getInputStream();){
            output = GoConverter.readAsString(in);
        }
        boolean exited = process.waitFor(5000L, TimeUnit.MILLISECONDS);
        if (exited && process.exitValue() != 0) {
            throw new ParseException("Go parser external process returned non-zero exit value: " + process.exitValue());
        }
        if (process.isAlive()) {
            process.destroyForcibly();
            throw new ParseException("Go parser external process took too long. External process killed forcibly");
        }
        return output;
    }

    private static String readAsString(InputStream in) throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        GoConverter.copy(in, outputStream);
        return new String(outputStream.toByteArray(), StandardCharsets.UTF_8);
    }

    private static void copy(InputStream in, OutputStream out) throws IOException {
        int read;
        byte[] buffer = new byte[8192];
        while ((read = in.read(buffer)) >= 0) {
            out.write(buffer, 0, read);
        }
    }

    static class DefaultCommand
    implements Command {
        private final String command;

        DefaultCommand(File workDir) {
            try {
                this.command = DefaultCommand.extract(workDir);
            }
            catch (IOException e) {
                throw new IllegalStateException(e.getMessage(), e);
            }
        }

        @Override
        public List<String> getCommand() {
            return Arrays.asList(this.command, "-");
        }

        private static String extract(File workDir) throws IOException {
            byte[] executableData;
            String executable = DefaultCommand.getExecutableForCurrentOS(System.getProperty("os.name"), System.getProperty("os.arch"));
            File dest = new File(workDir, executable);
            if (!DefaultCommand.fileMatch(dest, executableData = DefaultCommand.getBytesFromResource(executable))) {
                Files.write(dest.toPath(), executableData, new OpenOption[0]);
                dest.setExecutable(true);
            }
            return dest.getAbsolutePath();
        }

        static boolean fileMatch(File dest, byte[] expectedContent) throws IOException {
            if (!dest.exists()) {
                return false;
            }
            byte[] actualContent = Files.readAllBytes(dest.toPath());
            return Arrays.equals(actualContent, expectedContent);
        }

        static byte[] getBytesFromResource(String executable) throws IOException {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            try (InputStream in = GoConverter.class.getClassLoader().getResourceAsStream(executable);){
                if (in == null) {
                    throw new IllegalStateException(executable + " binary not found on class path");
                }
                GoConverter.copy(in, out);
            }
            return out.toByteArray();
        }

        static String getExecutableForCurrentOS(String osName, String arch) {
            String os = osName.toLowerCase(Locale.ROOT);
            if (os.contains("win")) {
                return "sonar-go-to-slang-windows-amd64.exe";
            }
            if (os.contains("mac")) {
                if (arch.equals("aarch64")) {
                    return "sonar-go-to-slang-darwin-arm64";
                }
                return "sonar-go-to-slang-darwin-amd64";
            }
            return "sonar-go-to-slang-linux-amd64";
        }
    }

    static interface Command {
        public List<String> getCommand();
    }
}

