/*
 * 
 *  This file is part of Sourceprojects.org Lycia Copyright 2010 noctarius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */
package org.sourceprojects.xmlparser;

import java.io.File;
import java.io.InputStream;
import java.io.Reader;

import org.sourceprojects.lycia.LyciaParser;
import org.sourceprojects.xmlparser.exceptions.XmlParserParsingException;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;

/**
 * Interface definition of XmlParser.<br>
 * Standard implementation of this interface returned by
 * {@link XmlParserFactory} implements a standard behaviour to redirect all
 * registered tags to the corresponding {@link TagDefinitionParser}s.
 * 
 * @author noctarius
 * @since 0.0.1
 * 
 * @param <B>
 *            Type of ContextObject to use
 * @deprecated Use {@link LyciaParser} instead
 */
@Deprecated
public interface XmlParser<B> {

	/**
	 * Adds a {@link TagDefinitionParser} implementation to the instance of
	 * XmlParser
	 * 
	 * @param definitionParser
	 *            TagDefinitionParser implementation to add
	 */
	void addTagDefinitionParser(TagDefinitionParser<B> definitionParser);

	/**
	 * Adds a bunch of {@link TagDefinitionParser} implementations to the
	 * instance of XmlParser
	 * 
	 * @param definitionParsers
	 *            Array of TagDefinitionParser implementations to add
	 */
	void addTagDefinitionParsers(TagDefinitionParser<B>[] definitionParsers);

	/**
	 * Removes a {@link TagDefinitionParser} implementation from the instance of
	 * XmlParser
	 * 
	 * @param definitionParser
	 *            TagDefinitionParser implementation to remove
	 */
	void removeTagDefinitionParser(TagDefinitionParser<B> definitionParser);

	/**
	 * Removes a bunch of {@link TagDefinitionParser} implementations from the
	 * instance of XmlParser
	 * 
	 * @param definitionParsers
	 *            Array of TagDefinitionParser implementations to remove
	 */
	void removeTagDefinitionParsers(TagDefinitionParser<B>[] definitionParsers);

	/**
	 * Starts parsing Xml data from given {@link Reader} implementation
	 * 
	 * @param reader
	 *            Reader to read Xml data from
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	void parse(Reader reader) throws XmlParserParsingException;

	/**
	 * Starts parsing Xml data from given String
	 * 
	 * @param xml
	 *            A String filled with Xml data
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	void parse(String xml) throws XmlParserParsingException;

	/**
	 * Starts parsing Xml data from given {@link InputStream} implementation
	 * 
	 * @param inputStream
	 *            InputStream to read Xml data from
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	void parse(InputStream inputStream) throws XmlParserParsingException;

	/**
	 * Starts parsing Xml data from given {@link File}
	 * 
	 * @param file
	 *            File to read Xml data from
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	void parse(File file) throws XmlParserParsingException;

	/**
	 * Starts transforming Xml data from given {@link Reader} to new
	 * {@link Document} instance
	 * 
	 * @param reader
	 *            Reader to read Xml data from
	 * @return A new {@link Document} instance with transformed data
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	Document transform(Reader reader) throws XmlParserParsingException;

	/**
	 * Starts transforming Xml data from given String to new {@link Document}
	 * instance
	 * 
	 * @param xml
	 *            A String filled with Xml data
	 * @return A new {@link Document} instance with transformed data
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	Document transform(String xml) throws XmlParserParsingException;

	/**
	 * Starts transforming Xml data from given {@link InputStream} to new
	 * {@link Document} instance
	 * 
	 * @param inputStream
	 *            InputStream to read Xml data from
	 * @return A new {@link Document} instance with transformed data
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	Document transform(InputStream inputStream)
			throws XmlParserParsingException;

	/**
	 * Starts transforming Xml data from given {@link File} to new
	 * {@link Document} instance
	 * 
	 * @param file
	 *            File to read Xml data from
	 * @return A new {@link Document} instance with transformed data
	 * @throws XmlParserParsingException
	 *             Is thrown when some error occurs. Implementation specific
	 *             exceptions are wrapped into XmlParserParsingException
	 */
	Document transform(File file) throws XmlParserParsingException;

	/**
	 * Sets an instance of {@link ParserContext}
	 * 
	 * @param parserContext
	 *            ParserContext instance to set
	 */
	void setParserContext(ParserContext<B> parserContext);

	/**
	 * Returns an instance of {@link ParserContext}
	 * 
	 * @return ParserContext instance
	 */
	ParserContext<B> getParserContext();

	/**
	 * Sets an instance of {@link ResourceResolver}
	 * 
	 * @param resourceResolver
	 *            ResourceResolver instance to set
	 */
	void setResourceResolver(ResourceResolver resourceResolver);

	/**
	 * Returns an instance of {@link ResourceResolver}
	 * 
	 * @return ResourceResolver instance
	 */
	ResourceResolver getResourceResolver();

	/**
	 * Sets an instance of of {@link ErrorHandler}
	 * 
	 * @param errorHandler
	 *            ErrorHandler instance
	 */
	void setErrorHandler(ErrorHandler errorHandler);

	/**
	 * Returns an instance of {@link ErrorHandler}
	 * 
	 * @return ErrorHandler instance
	 */
	ErrorHandler getErrorHandler();

	/**
	 * Returns if XmlParser implementation should be aware of schema errors on
	 * reading Xml documents<br>
	 * Not all SAX parser implementations are able to do schema validation. If
	 * SAX parser does not support schema validation a warning is shown but
	 * parsing will work as normal (without validation).

	 * 
	 * @return XmlParser should be aware of schema errors
	 */
	boolean isValidatingSchema();

	/**
	 * Sets if XmlParser implementation should be aware of schema errors on
	 * reading Xml documents.<br>
	 * Not all SAX parser implementations are able to do schema validation. If
	 * SAX parser does not support schema validation a warning is shown but
	 * parsing will work as normal (without validation).
	 * 
	 * @param validatingSchema
	 *            XmlParser should be aware of schema errors
	 */
	void setValidatingSchema(boolean validatingSchema);
	
	/**
	 * Adds a {@link ProgressListener} instance to the notify list
	 * 
	 * @param listener ProgressListener to add
	 * @since 0.0.7
	 */
	void addProgressListener(ProgressListener listener);

	/**
	 * Removes a {@link ProgressListener} instance from the notify list
	 * 
	 * @param listener ProgressListener to remove
	 * @since 0.0.7
	 */
	void removeProgressListener(ProgressListener listener);
}
