/*
 * 
 *  This file is part of Sourceprojects.org Lycia Copyright 2010 noctarius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */
package org.sourceprojects.xmlparser.internal.dom;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.logging.Logger;

import org.osgi.framework.Bundle;
import org.sourceprojects.xmlparser.NamespaceResolver;
import org.sourceprojects.xmlparser.ResourceResolver;
import org.sourceprojects.xmlparser.exceptions.XmlParserInitializationException;
import org.sourceprojects.xmlparser.internal.resolver.OsgiBasicNamespaceResolver;
import org.sourceprojects.xmlparser.internal.resolver.BasicNamespaceResolver.NamespaceResolverHandlerContainer;
import org.w3c.dom.ls.LSInput;

/**
 * 
 * @author noctarius
 * @since 0.0.5
 */
public class OsgiResourceResolverImpl implements ResourceResolver {
	private static final Logger LOGGER = Logger.getLogger(OsgiResourceResolverImpl.class.getName());
	
	private final Map<String, NamespaceResolver> namespaceResolvers = new HashMap<String, NamespaceResolver>();
	private final OsgiBasicNamespaceResolver osgiBasicNamespaceResolver;
	private boolean showWarning = true;

	@SuppressWarnings("unchecked")
	public OsgiResourceResolverImpl(Bundle bundle) {
		try {
			final List<NamespaceResolverHandlerContainer> containers = new ArrayList<NamespaceResolverHandlerContainer>();

			final Enumeration<URL> resources = bundle
					.getResources("META-INF/xmlparser/schema.properties");

			while (resources.hasMoreElements()) {
				URL url = resources.nextElement();

				LOGGER.info("Found schema.properties in '" + url.toString()
						+ "'");

				final Properties properties = new Properties();
				properties.load(url.openStream());

				final Enumeration<Object> enumeration = properties.keys();
				while (enumeration.hasMoreElements()) {
					final String key = (String) enumeration.nextElement();
					final String value = properties.getProperty(key);

					containers.add(new NamespaceResolverHandlerContainer(key,
							value));
				}
			}
			osgiBasicNamespaceResolver = new OsgiBasicNamespaceResolver(
					containers
							.toArray(new NamespaceResolverHandlerContainer[containers
									.size()]), bundle);
		} catch (final IOException e) {
			throw new XmlParserInitializationException(
					"Failed internal initialization", e);
		}
	}

	@Override
	public boolean isShowWarning() {
		return showWarning;
	}

	@Override
	public void setShowWarning(boolean showWarning) {
		this.showWarning = showWarning;
	}

	@Override
	public void registerNamespaceResolver(NamespaceResolver namespaceResolver) {
		String namespaceURI = namespaceResolver.getNamespaceURI();

		if (namespaceResolvers.get(namespaceURI) != null && isShowWarning()) {
			StringBuilder sb = new StringBuilder();
			sb.append("Namespace ").append(namespaceURI).append(
					" is already defined and will be overridden.");
			LOGGER.warning(sb.toString());
		}

		namespaceResolvers.put(namespaceURI, namespaceResolver);
	}

	@Override
	public void unregisterNamespaceResolver(NamespaceResolver namespaceResolver) {
		unregisterNamespaceResolver(namespaceResolver.getNamespaceURI());
	}

	@Override
	public void unregisterNamespaceResolver(String namespaceURI) {
		namespaceResolvers.remove(namespaceURI);
	}

	@Override
	public LSInput resolveResource(String type, String namespaceURI,
			String publicId, String systemId, String baseURI) {

		NamespaceResolver resolver = namespaceResolvers.get(namespaceURI);
		if (resolver != null) {
			return resolver.getInput(type, namespaceURI, publicId, systemId,
					baseURI);
		} else {
			return osgiBasicNamespaceResolver.getInput(type, namespaceURI,
					publicId, systemId, baseURI);
		}
	}

}
