/**
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2021 Source Auditor Inc.
 */
package org.spdx.spdx_to_osv.osvmodel;

import java.util.List;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.annotations.SerializedName;

/**
 * OSV Vulnerability object as described at https://docs.google.com/document/d/1sylBGNooKtf220RHQn1I8pZRmqXZQADDQ_TOABrKTpA/edit
 * 
 * @author Gary O'Neall
 *
 */
public class OsvVulnerability {
    
    /**
     * The id field is a unique identifier for the vulnerability entry. 
     * It is a string of the format <DB>-<ENTRYID>, where DB names the database and 
     * ENTRYID is in the format used by the database. For example: “OSV-2020-111”, 
     * “CVE-2021-3114”, or “GHSA-vp9c-fpxx-744v”.
     */
    String id;
    
    /**
     * The RFC3339 timestamp indicating when this entry was published.
     */
    String published;
    
    /**
     *  The RFC3339 timestamp indicating when this entry was last modified.
     */
    String modified;
    
    /**
     * Optional. The RFC3339 timestamp indicating when this entry is considered to be withdrawn.
     */
    String withdrawn;
    
    /**
     * Optional. IDs for the same vulnerability in other databases.
     */
    List<String> aliases;
    
    /**
     * Optional. List of IDs of closely related vulnerabilities, 
     * such as the same problem in alternate ecosystems.
     */
    List<String> related;
    
    /**
     * Required. Package information.
     */
    @SerializedName(value="package")
    OsvPackage osvPackage;
    
    /**
     * Required. One line human readable summary for the vulnerability. 
     * It is recommended to keep this under 120 characters.
     */
    String summary;
    
    /**
     * Required. Any additional human readable details for the vulnerability.
     */
    String details;
    
    /**
     * Required. Affected commit ranges and versions. Deprecated.
     */
    @Deprecated
    OsvAffects affects;
    
    /**
     * Required. Affected commit ranges and versions.
     */
    List<OsvAffected> affected;
    
    /**
     * Optional. URLs to more information/advisories (including the scheme e.g "https://").
     */
    List<OsvReference> references;
    
    /**
     * Enum: "NONE" "LOW" "MEDIUM" "HIGH" "CRITICAL" Deprecated and will be removed in the future.
     */
    @Deprecated
    JsonArray severity;
    
    /**
     * Optional. JSON object holding additional information about the vulnerability 
     * as defined by the database for which the record applies.
     */
    JsonObject database_specific;
    
    /**
     * Optional. JSON object holding additional information about the vulnerability 
     * as defined by the ecosystem for which the record applies.
     */
    JsonObject ecosystem_specific;
    
    /**
     * required empty constructor
     */
    public OsvVulnerability() {
        // required empty constructor
    }

    /**
     * @return the id
     */
    public String getId() {
        return id;
    }

    /**
     * @param id the id to set
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the published
     */
    public String getPublished() {
        return published;
    }

    /**
     * @param published the published to set
     */
    public void setPublished(String published) {
        this.published = published;
    }

    /**
     * @return the modified
     */
    public String getModified() {
        return modified;
    }

    /**
     * @param modified the modified to set
     */
    public void setModified(String modified) {
        this.modified = modified;
    }

    /**
     * @return the withdrawn
     */
    public String getWithdrawn() {
        return withdrawn;
    }

    /**
     * @param withdrawn the withdrawn to set
     */
    public void setWithdrawn(String withdrawn) {
        this.withdrawn = withdrawn;
    }

    /**
     * @return the aliases
     */
    public List<String> getAliases() {
        return aliases;
    }

    /**
     * @param aliases the aliases to set
     */
    public void setAliases(List<String> aliases) {
        this.aliases = aliases;
    }

    /**
     * @return the related
     */
    public List<String> getRelated() {
        return related;
    }

    /**
     * @param related the related to set
     */
    public void setRelated(List<String> related) {
        this.related = related;
    }

    /**
     * @return the osvPackage
     */
    public OsvPackage getOsvPackage() {
        return osvPackage;
    }

    /**
     * @param osvPackage the osvPackage to set
     */
    public void setOsvPackage(OsvPackage osvPackage) {
        this.osvPackage = osvPackage;
    }

    /**
     * @return the summary
     */
    public String getSummary() {
        return summary;
    }

    /**
     * @param summary the summary to set
     */
    public void setSummary(String summary) {
        this.summary = summary;
    }

    /**
     * @return the details
     */
    public String getDetails() {
        return details;
    }

    /**
     * @param details the details to set
     */
    public void setDetails(String details) {
        this.details = details;
    }

    /**
     * @return the affects
     */
    @SuppressWarnings("deprecation")
    public OsvAffects getAffects() {
        return affects;
    }

    /**
     * @param affects the affects to set
     */
    @SuppressWarnings("deprecation")
    public void setAffects(OsvAffects affects) {
        this.affects = affects;
    }

    /**
     * @return the affected
     */
    public List<OsvAffected> getAffected() {
        return affected;
    }

    /**
     * @param affected the affected to set
     */
    public void setAffected(List<OsvAffected> affected) {
        this.affected = affected;
    }

    /**
     * @return the references
     */
    public List<OsvReference> getReferences() {
        return references;
    }

    /**
     * @param references the references to set
     */
    public void setReferences(List<OsvReference> references) {
        this.references = references;
    }

    /**
     * @return the severity
     */
    public JsonArray getSeverity() {
        return severity;
    }

    /**
     * @param severity the severity to set
     */
    public void setSeverity(JsonArray severity) {
        this.severity = severity;
    }

    /**
     * @return the database_specific
     */
    public JsonObject getDatabase_specific() {
        return database_specific;
    }

    /**
     * @param database_specific the database_specific to set
     */
    public void setDatabase_specific(JsonObject database_specific) {
        this.database_specific = database_specific;
    }

    /**
     * @return the ecosystem_specific
     */
    public JsonObject getEcosystem_specific() {
        return ecosystem_specific;
    }

    /**
     * @param ecosystem_specific the ecosystem_specific to set
     */
    public void setEcosystem_specific(JsonObject ecosystem_specific) {
        this.ecosystem_specific = ecosystem_specific;
    }
    
}
