/**
 * Copyright by Michael Weiss, weiss.michael@gmx.ch
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package org.spectrumauctions.sats.core.model.mrvm;

import org.spectrumauctions.sats.core.model.BidderSetup;
import org.spectrumauctions.sats.core.util.random.DoubleInterval;
import org.spectrumauctions.sats.core.util.random.UniformDistributionRNG;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.HashMap;
import java.util.Map;

/**
 * @author Michael Weiss
 *
 */
public abstract class MRVMBidderSetup extends BidderSetup {

    private final DoubleInterval alphaInterval;
    private final DoubleInterval betaInterval;
    private final DoubleInterval zLowInterval;
    private final DoubleInterval zHighInterval;

    private static final BigDecimal NONZERO_INCREMENT = BigDecimal.valueOf(0.01);


    protected MRVMBidderSetup(Builder builder) {
        super(builder);
        this.alphaInterval = builder.alphaInterval;
        this.betaInterval = builder.betaInterval;
        this.zLowInterval = builder.zLowInterval;
        this.zHighInterval = builder.zHighInterval;
    }

    /**
     * Draws the Alpha-Parameter uniformly at random.<br>
     * Alpha is a parameter defining an expected profit per served customer, if quality of service and regional discount are ignored.<br>
     * It can be understood as a relative bidder strength parameter.
     * @return a BigDecimal in [0,1]
     */
    public BigDecimal drawAlpha(UniformDistributionRNG rng) {
        return rng.nextBigDecimal(alphaInterval);
    }

    /**
     * Draws the Beta-Parameter uniformly at random.<br> 
     * Beta is a parameter defining the target market share this bidder intends to cover. <br>
     * The bidders value for a bundle increases heavily as soon as the capacity share he has in a region gets close to beta.
     * @return a BigDecimal in [0,1]
     */
    public BigDecimal drawBeta(MRVMRegionsMap.Region r, UniformDistributionRNG rng) {
        return rng.nextBigDecimal(betaInterval);
    }


    public HashMap<Integer, BigDecimal> drawZLow(Map<Integer, BigDecimal> betas, MRVMWorld world, UniformDistributionRNG rng) {
        HashMap<Integer, BigDecimal> result = new HashMap<>();
        for (Map.Entry<Integer, BigDecimal> beta : betas.entrySet()) {
            BigDecimal minTerm = beta.getValue().subtract(BigDecimal.valueOf(0.3));
            if (minTerm.compareTo(BigDecimal.ZERO) < 0) {
                minTerm = NONZERO_INCREMENT;
            }
            BigDecimal dividend = minTerm.multiply(world.getMaximumRegionalCapacity());
            MRVMRegionsMap.Region region = world.getRegionsMap().getRegion(beta.getKey());
            BigDecimal divisor = BigDecimal.valueOf(region.getPopulation()).multiply(beta.getValue());
            BigDecimal zvalue = dividend.divide(divisor, new MathContext(10, RoundingMode.UP));
            result.put(beta.getKey(), zvalue);
        }
        return result;
    }


    public HashMap<Integer, BigDecimal> drawZHigh(Map<Integer, BigDecimal> betas, MRVMWorld world, UniformDistributionRNG rng) {
        HashMap<Integer, BigDecimal> result = new HashMap<>();
        for (Map.Entry<Integer, BigDecimal> beta : betas.entrySet()) {
            BigDecimal maxTerm = beta.getValue().add(BigDecimal.valueOf(0.3));
            if (maxTerm.compareTo(BigDecimal.ONE) > 0) {
                maxTerm = BigDecimal.ONE;
            }
            BigDecimal dividend = maxTerm.multiply(world.getMaximumRegionalCapacity());
            MRVMRegionsMap.Region region = world.getRegionsMap().getRegion(beta.getKey());
            BigDecimal divisor = BigDecimal.valueOf(region.getPopulation()).multiply(beta.getValue());
            BigDecimal zvalue = dividend.divide(divisor, BigDecimal.ROUND_HALF_UP);
            result.put(beta.getKey(), zvalue);
        }
        return result;
    }


    public static abstract class Builder extends BidderSetup.Builder {

        private DoubleInterval alphaInterval;
        private DoubleInterval betaInterval;
        private DoubleInterval zLowInterval;
        private DoubleInterval zHighInterval;

        protected Builder(String setupName, int numberOfBidders, DoubleInterval alphaInterval, DoubleInterval betaInterval) {
            super(setupName, numberOfBidders);
            this.alphaInterval = alphaInterval;
            this.betaInterval = betaInterval;
        }


        /**
         * @return the interval from which the alpha value will be drawn. <br>
         * See {@link MRVMBidderSetup#alphaInterval} for explanation of alpha-parameter
         */
        public DoubleInterval getAlphaInterval() {
            return alphaInterval;
        }


        /**
         * Sets the interval from which the alpha value will be drawn. <br>
         * See {@link MRVMBidderSetup#alphaInterval} for explanation of alpha-parameter
         */
        public void setAlphaInterval(DoubleInterval alphaInterval) {
            this.alphaInterval = alphaInterval;
        }


        /**
         * @return the interval from which the beta value will be drawn. <br>
         * See {@link MRVMBidderSetup#betaInterval} for explanation of beta-parameter
         */
        public DoubleInterval getBetaInterval() {
            return betaInterval;
        }


        /**
         * Sets the interval from which the beta value will be drawn. <br>
         * See {@link MRVMBidderSetup#betaInterval} for explanation of beta-parameter
         */
        public void setBetaInterval(DoubleInterval betaInterval) {
            this.betaInterval = betaInterval;
        }


        public DoubleInterval getzLowInterval() {
            return zLowInterval;
        }


        public DoubleInterval getzHighInterval() {
            return zHighInterval;
        }


        public void setzLowInterval(DoubleInterval zLowInterval) {
            this.zLowInterval = zLowInterval;
        }


        public void setzHighInterval(DoubleInterval zHighInterval) {
            this.zHighInterval = zHighInterval;
        }


        @Override
        public abstract MRVMBidderSetup build();
    }


}
