package org.spincast.core.xml;

import java.io.InputStream;
import java.lang.reflect.Type;

import org.spincast.core.json.IJsonArray;
import org.spincast.core.json.IJsonObject;

/**
 * Provides methods to work with <code>XML</code> strings and objects.
 */
public interface IXmlManager {

    /**
     * Converts an object to XML.
     * 
     * If the object to convert is a <code>IJsonObject</code>, its elements
     * of type "IJsonArray" will have a "isArray='true'" attribute
     * added. This way, the XML can be deserialized back to a 
     * <code>IJsonObject</code> correctly.
     */
    public String toXml(Object obj);

    /**
     * Converts an object to XML.
     * @param pretty If <code>true</code>, the generated XML will
     * be formatted.
     */
    public String toXml(Object obj, boolean pretty);

    /**
     * Deserializes a XML to an <code>IJsonObject</code>. This
     * will correctly manage the XML generated by
     * toXml(), arrays included.
     */
    public IJsonObject fromXml(String xml);

    /**
     * Deserializes a XML to an <code>IJsonArray</code>. This
     * will correctly manage the XML generated by
     * toXml(), arrays included.
     */
    public IJsonArray fromXmlToJsonArray(String xml);

    /**
     * Deserializes a XML to the given Class.
     * Be aware that if you use a default Type like
     * Map&lt;String, Object&gt;, the arrays will probably
     * won't be deserialized correctly. Use the
     * version returning a <code>IJsonObject</code> to get the
     * arrays to work out of the box!
     */
    public <T> T fromXml(String xml, Class<T> clazz);

    /**
     * Deserializes a XML to the given Type.
     * Be aware that if you use a default Type like
     * Map&lt;String, Object&gt;, the arrays will probably
     * won't be deserialized correctly. Use the
     * version returning a <code>IJsonObject</code> to get the
     * arrays to work out of the box!
     */
    public <T> T fromXmlToType(String xml, Type type);

    /**
     * Deserializes a XML inputstream to the given Type.
     * Be aware that if you use a default Type like
     * Map&lt;String, Object&gt;, the arrays will probably
     * won't be deserialized correctly. Use the
     * version returning a <code>IJsonObject</code> to get the
     * arrays to work out of the box!
     */
    public <T> T fromXmlInputStream(InputStream inputStream, Class<T> clazz);

}
