package org.sterling.cli;

import static java.lang.Math.max;
import static java.util.Arrays.asList;

import java.io.PrintStream;
import java.util.*;
import java.util.Map.Entry;

public final class MainRunner {

    private static final String HELP = "help";
    private static final String RUN = "run";

    public static void main(String[] args)  {
        new MainRunner().run(new ArrayList<>(asList(args)));
    }

    private final Iterable<CommandLineRunner> runners;
    private final PrintStream out;
    private final CommandLineRunner helpRunner;

    public MainRunner() {
        this.out = System.out;
        this.runners = ServiceLoader.load(CommandLineRunner.class);
        this.helpRunner = new HelpRunner();
    }

    public MainRunner(PrintStream out, Collection<CommandLineRunner> runners) {
        this.out = out;
        this.runners = new ArrayList<>(runners);
        this.helpRunner = new HelpRunner();
    }

    public void run(List<String> args) {
        if (args.isEmpty()) {
            requireCommand();
            args.add(HELP);
        }
        runCommand(args);
    }

    private CommandLineRunner getRunner(String command) {
        for (CommandLineRunner runner : runners) {
            if (command.equals(runner.getCommand())) {
                return runner;
            }
        }
        return null;
    }

    private void requireCommand() {
        out.println("No command specified.");
        out.println();
    }

    private void runCommand(List<String> args) {
        String command = args.get(0);
        CommandLineRunner runner = getRunner(command);
        if (runner == null) {
            runner = getRunner(RUN);
            if (runner == null) {
                out.println("No handler for command '" + command + "'");
                runner = helpRunner;
            }
            runner.run(args);
        } else {
            runner.run(args.subList(1, args.size()));
        }
    }

    private final class HelpRunner implements CommandLineRunner {

        @Override
        public String getCommand() {
            return HELP;
        }

        @Override
        public String getHelpText() {
            return "Displays this help message";
        }

        @Override
        public void run(List<String> args) {
            out.println("Available commands:");
            out.println();
            Map<String, String> helps = new HashMap<>();
            int length = processHelps(helps);
            for (Entry<String, String> entry : helps.entrySet()) {
                String command = pad(entry.getKey() + ":", length + 1);
                String helpText = entry.getValue();
                out.println("\t" + command + " \t" + helpText);
            }
            out.println();
        }

        private String pad(String text, int length) {
            return String.format("%1$-" + length + "s", text);
        }

        private int processHelps(Map<String, String> helps) {
            int length = 0;
            for (CommandLineRunner runner : runners) {
                String command = runner.getCommand();
                length = max(length, command.length());
                helps.put(command, runner.getHelpText());
            }
            return length;
        }
    }
}
