package org.sterling.runtime;

import static java.io.File.pathSeparator;
import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.regex.Pattern.quote;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.sterling.SterlingException;
import org.sterling.cli.CommandLineRunner;
import org.sterling.cli.RunnerException;

public class SterlingRunner implements CommandLineRunner {

    private final ClassLoader delegateClassLoader;

    public SterlingRunner() {
        delegateClassLoader = currentThread().getContextClassLoader();
    }

    @Override
    public String getCommand() {
        return "run";
    }

    @Override
    public String getHelpText() {
        return "Runs the 'main' expression from a specified module";
    }

    @Override
    public void run(List<String> args) {
        try {
            ClassLoader classLoader = setModulePath(args);
            GlobalModule globalModule = new GlobalModule(classLoader, new Compiler());
            globalModule.load(getMain(args)).evaluate();
        } catch (SterlingException | IOException exception) {
            throw new RunnerException(exception);
        }
    }

    private String getMain(List<String> args) {
        if (args.size() < 1) {
            throw new RunnerException("No module specified");
        } else {
            int offset = (args.size() > 1) ? 1 : 0;
            return args.get(offset).replace('.', '/') + "/main";
        }
    }

    private List<String> listPaths(String paths) {
        List<String> pathList = new ArrayList<>(asList(paths.split(quote(pathSeparator))));
        if (!pathList.contains(".")) {
            pathList.add(".");
        }
        return pathList;
    }

    private ClassLoader setModulePath(List<String> args) throws IOException {
        Iterator<String> iterator = args.iterator();
        while (iterator.hasNext()) {
            if ("-mp".equals(iterator.next())) {
                iterator.remove();
                ClassLoader classLoader = setModulePath(iterator.next());
                iterator.remove();
                return classLoader;
            }
        }
        return setModulePath(".");
    }

    private ClassLoader setModulePath(String paths) throws IOException {
        List<String> pathList = listPaths(paths);
        List<URL> urls = new ArrayList<>();
        for (String path : pathList) {
            urls.add(toUrl(path));
        }
        return new URLClassLoader(urls.toArray(new URL[urls.size()]), delegateClassLoader);
    }

    private URL toUrl(String path) throws IOException {
        return new File(path).toURI().toURL();
    }
}
