package org.sterling.runtime.expression;

import java.io.PrintStream;
import org.sterling.SterlingException;
import org.sterling.util.PrinterState;

public class ExpressionPrinter implements ExpressionVisitor<Void, PrinterState> {

    private final int initialIndent;

    public ExpressionPrinter(int initialIndent) {
        this.initialIndent = initialIndent;
    }

    public void print(Expression expression, PrintStream out) throws SterlingException {
        visit(expression, new PrinterState(out, initialIndent));
    }

    @Override
    public Void visit(Expression expression, PrinterState state) throws SterlingException {
        expression.accept(this, state);
        return null;
    }

    @Override
    public Void visitAccessExpression(AccessedExpression expression, PrinterState state) throws SterlingException {
        state.begin(expression);
        visit(expression.getExpression(), state);
        visit(expression.getMember(), state);
        state.end();
        return null;
    }

    @Override
    public Void visitApplyExpression(AppliedExpression expression, PrinterState state) throws SterlingException {
        state.begin(expression);
        visit(expression.getExpression(), state);
        visit(expression.getArgument(), state);
        state.end();
        return null;
    }

    @Override
    public Void visitArgumentExpression(Argument expression, PrinterState state) throws SterlingException {
        state.begin(expression);
        visit(expression.getExpression(), state);
        state.end();
        return null;
    }

    @Override
    public Void visitBindExpression(BoundExpression expression, PrinterState state) throws SterlingException {
        state.begin(expression);
        visit(expression.getExpression(), state);
        visit(expression.getArgument(), state);
        state.end();
        return null;
    }

    @Override
    public Void visitLambda(Lambda lambda, PrinterState state) throws SterlingException {
        state.begin(lambda);
        visit(lambda.getVariable(), state);
        visit(lambda.getExpression(), state);
        state.end();
        return null;
    }

    @Override
    public Void visitObjectExpression(ObjectExpression object, PrinterState state) throws SterlingException {
        state.begin(object);
        for (DeclaredExpression declaration : object.getMembers()) {
            state.println("Member <<" + declaration.getIdentifier() + ">>");
            visit(declaration.getExpression(), state);
        }
        state.end();
        return null;
    }

    @Override
    public Void visitPrimaryExpression(Expression expression, PrinterState state) throws SterlingException {
        state.println(expression.toString());
        return null;
    }

    @Override
    public Void visitReference(Reference reference, PrinterState state) throws SterlingException {
        state.println("Reference <<" + reference.getSymbol() + ">>");
        return null;
    }
}
