package org.sterling.source.parser;

import static org.sterling.util.BufferUtil.buffer;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import org.sterling.SterlingException;
import org.sterling.cli.CommandLineRunner;
import org.sterling.cli.RunnerException;
import org.sterling.source.scanner.InputReader;
import org.sterling.source.scanner.ScannerFactory;
import org.sterling.source.syntax.SourcePrinter;

public class ParserRunner implements CommandLineRunner {

    private final BufferedReader input;
    private final PrintStream output;
    private final Parser parser;
    private final ScannerFactory scannerFactory;
    private final SourcePrinter printer;

    public ParserRunner() {
        this(new ParserRunnerProperties()
            .setInput(buffer(System.in))
            .setOutput(System.out)
            .setParser(new Parser())
            .setPrinter(new SourcePrinter())
            .setScannerFactory(new ScannerFactory())
        );
    }

    public ParserRunner(ParserRunnerProperties properties) {
        this.input = properties.getInput();
        this.output = properties.getOutput();
        this.parser = properties.getParser();
        this.scannerFactory = properties.getScannerFactory();
        this.printer = properties.getPrinter();
    }

    @Override
    public String getCommand() {
        return "parse";
    }

    @Override
    public String getHelpText() {
        return "Parses input and displays the resultant concrete syntax tree";
    }

    public void run() {
        run(new ArrayList<String>());
    }

    @Override
    public void run(List<String> args) {
        String input;
        output.println("Type any input to see how it parses:");
        output.print(">>> ");
        try {
            while (null != (input = this.input.readLine())) {
                if ("quit".equals(input)) {
                    break;
                } else if (!"".equals(input.trim())) {
                    try {
                        printer.print(
                            parser.parse(
                                scannerFactory.getScanner(
                                    new InputReader("Command Line", new ByteArrayInputStream(input.getBytes("UTF-8")))
                                )
                            ),
                            output
                        );
                    } catch (SterlingException exception) {
                        exception.printStackTrace(output);
                    }
                }
                output.print(">>> ");
            }
        } catch (IOException exception) {
            throw new RunnerException(exception);
        }
        output.println("Goodbye!");
    }
}
