package org.sterling.source.scanner;

import static java.lang.Character.MAX_VALUE;
import static java.lang.Character.toChars;
import static java.lang.Integer.parseInt;

public final class EscapeUtil {

    @SuppressWarnings({"PMD.CyclomaticComplexity", "PMD.NPathComplexity"})
    public static String unescapeString(String escapedString) {
        StringBuilder unescapedString = new StringBuilder(escapedString.length());
        boolean backslash = false;
        for (int position = 0; position < escapedString.length(); position++) {
            int codePoint = escapedString.codePointAt(position);
            if (escapedString.codePointAt(position) > MAX_VALUE) {
                position++;
            }
            if (!backslash) {
                if (codePoint == '\\') {
                    backslash = true;
                } else {
                    unescapedString.append(toChars(codePoint));
                }
                continue;
            }
            if (codePoint == '\\') {
                backslash = false;
                unescapedString.append('\\');
                unescapedString.append('\\');
                continue;
            }
            switch (codePoint) {
                case 'r': unescapedString.append('\r'); break;
                case 'n': unescapedString.append('\n'); break;
                case 'f': unescapedString.append('\f'); break;
                case 'b': unescapedString.append('\b'); break;
                case 't': unescapedString.append('\t'); break;
                case '8':
                case '9': throw new IllegalArgumentException("Illegal octal escape in string at position " + position);
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7': position--;
                case '0': {
                    if (position == escapedString.length() - position) {
                        unescapedString.append(toChars(0));
                        break;
                    }
                    position++;
                    int offset = 0;
                    for (int j = 0; j <= 2; j++) {
                        if (position == escapedString.length() - j) {
                            break;
                        }
                        int point = escapedString.charAt(position + j);
                        if (point < '0' || point > '7') {
                            break;
                        }
                        offset++;
                    }
                    if (offset == 0) {
                        position--;
                        unescapedString.append('\0');
                        break;
                    }
                    unescapedString.append(toChars(parseInt(escapedString.substring(position, position + offset), 8)));
                    position += offset - 1;
                    break;
                }
                case 'u': {
                    if (position + 4 > escapedString.length()) {
                        throw new IllegalArgumentException("Illegal length of \\u escape in string at position " + position);
                    }
                    int offset;
                    position++;
                    for (offset = 0; offset < 4; offset++) {
                        if (escapedString.charAt(position + offset) > 127) {
                            throw new IllegalArgumentException("Illegal non-ASCII hex digit in \\u escape at position " + position);
                        }
                    }
                    unescapedString.append(toChars(parseInt(escapedString.substring(position, position + offset), 16)));
                    position += offset - 1;
                    break;
                }
                default:
                    unescapedString.append('\\');
                    unescapedString.append(toChars(codePoint));
            }
            backslash = false;
        }
        if (backslash) {
            unescapedString.append('\\');
        }
        return unescapedString.toString();
    }

    private EscapeUtil() {
        // intentionally empty
    }
}
