package org.sterling.source.scanner;

import static java.util.Collections.unmodifiableMap;
import static org.sterling.source.syntax.NodeKind.*;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.sterling.source.syntax.NodeKind;

public class ScannerFactory {

    public Scanner getScanner(InputReader inputReader) {
        return new DelegatingScanner(inputReader, Holder.delegates);
    }

    private static final class Holder {

        private static final Map<NodeKind, ScannerDelegate> delegates;

        static {
            ScannerDelegateFactory factory = new ScannerDelegateFactory();
            Map<NodeKind, ScannerDelegate> map = new HashMap<>();
            map.put(IDENTIFIER, factory.getIdentifierScanner());
            map.put(INTEGER, factory.getIntegerScanner());
            map.put(DOUBLE, factory.getDoubleScanner());
            map.put(ADD, factory.getAddScanner());
            map.put(SUBTRACT, factory.getSubtractScanner());
            map.put(MULTIPLY, factory.getMultiplyScanner());
            map.put(DIVIDE, factory.getDivideScanner());
            map.put(MODULO, factory.getModuloScanner());
            map.put(POSITIVE, factory.getPositiveScanner());
            map.put(NEGATIVE, factory.getNegativeScanner());
            map.put(SHIFT_LEFT, factory.getShiftLeftScanner());
            map.put(SHIFT_RIGHT, factory.getShiftRightScanner());
            map.put(SIGNED_SHIFT_RIGHT, factory.getSignedShiftRightScanner());
            map.put(ACCESSOR, factory.getAccessorScanner());
            map.put(COALESCE, factory.getCoalesceScanner());
            map.put(BITWISE_AND, factory.getBitwiseAndScanner());
            map.put(BITWISE_OR, factory.getBitwiseOrScanner());
            map.put(BITWISE_XOR, factory.getBitwiseXorScanner());
            map.put(LOGICAL_AND, factory.getLogicalAndScanner());
            map.put(LOGICAL_OR, factory.getLogicalOrScanner());
            map.put(LOGICAL_NOT, factory.getLogicalNotScanner());
            map.put(GREATER_THAN, factory.getGreaterThanScanner());
            map.put(LESS_THAN, factory.getLessThanScanner());
            map.put(GREATER_THAN_EQUALS, factory.getGreaterThanEqualsScanner());
            map.put(LESS_THAN_EQUALS, factory.getLessThanEqualsScanner());
            map.put(EQUALS, factory.getEqualsScanner());
            map.put(NOT_EQUALS, factory.getNotEqualsScanner());
            map.put(ASSIGN, factory.getAssignScanner());
            map.put(BITWISE_NOT, factory.getBitwiseNotScanner());
            map.put(GROUP_OPEN, factory.getGroupOpenScanner());
            map.put(GROUP_CLOSE, factory.getGroupCloseScanner());
            map.put(INDEXER_OPEN, factory.getIndexerOpenScanner());
            map.put(INDEXER_CLOSE, factory.getIndexerCloseScanner());
            map.put(TERNARY_TRUE, factory.getTernaryTrueScanner());
            map.put(TERNARY_FALSE, factory.getTernaryFalseScanner());
            map.put(BOOLEAN, factory.getBooleanScanner());
            map.put(STRING, factory.getStringScanner());
            map.put(NOTHING, factory.getNullScanner());
            map.put(CHARACTER, factory.getCharacterScanner());
            map.put(TERMINATOR, factory.getTerminatorScanner());
            map.put(BLOCK_OPEN, factory.getBlockOpenScanner());
            map.put(BLOCK_CLOSE, factory.getBlockCloseScanner());
            map.put(END_OF_INPUT, factory.getEndOfInputScanner());
            map.put(ARROW, factory.getArrowScanner());
            map.put(KEYWORD_OBJECT, factory.getKeywordObjectScanner());
            map.put(KEYWORD_POSITIVE, factory.getKeywordPositiveScanner());
            map.put(KEYWORD_NEGATIVE, factory.getKeywordNegativeScanner());
            map.put(INDEXER_OPERATOR, factory.getIndexerOperatorScanner());
            map.put(SEPARATOR, factory.getSeparatorScanner());
            map.put(KEYWORD_IF, factory.getKeywordIfScanner());
            map.put(APPLY, factory.getApplyScanner());
            map.put(KEYWORD_MODULE, factory.getKeywordModuleScanner());
            map.put(KEYWORD_IMPORT, factory.getKeywordImportScanner());
            map.put(WILDCARD, factory.getWildcardScanner());
            map.put(KEYWORD_FROM, factory.getKeywordFromScanner());
            map.put(KEYWORD_AS, factory.getKeywordAsScanner());
            map.put(KEYWORD_JAVA, factory.getKeywordJavaScanner());
            delegates = unmodifiableMap(new ConcurrentHashMap<>(map));
        }

        private Holder() {
            // intentionally empty
        }
    }
}
