package org.sterling.source.scanner;

import static java.util.regex.Pattern.compile;
import static org.sterling.source.scanner.EscapeUtil.unescapeString;
import static org.sterling.source.scanner.ScannerUtil.invalidEscapeSequence;
import static org.sterling.source.scanner.ScannerUtil.unexpectedInput;
import static org.sterling.source.scanner.ScannerUtil.unterminatedString;

import java.util.regex.Pattern;
import org.sterling.SterlingException;
import org.sterling.source.syntax.NodeKind;
import org.sterling.source.syntax.StringToken;
import org.sterling.source.syntax.Token;

public class StringScanner implements ScannerDelegate {

    private final Pattern asciiEscape;
    private final Pattern hexEscape;
    private final Pattern octalEscape;

    public StringScanner() {
        asciiEscape = compile("^\\\\[btnfr\"'\\\\]");
        hexEscape = compile("^\\\\u[a-fA-F0-9]{4}");
        octalEscape = compile("^\\\\([0-3][0-7]{2}|[0-7]{1,2})");
    }

    @Override
    public boolean expect(NodeKind kind, InputReader reader) {
        return reader.expect('"');
    }

    @Override
    public Token require(NodeKind kind, InputReader reader) throws SterlingException {
        if (!expect(kind, reader)) {
            throw unexpectedInput(reader, kind);
        }
        reader.store();
        while (!reader.expect('"')) {
            storeInput(kind, reader);
        }
        reader.store();
        Token token = reader.accept(kind);
        return new StringToken(token, unescapeString(token.getValue().substring(1, token.getValue().length() - 1)));
    }

    private void storeInput(NodeKind kind, InputReader reader) throws SterlingException {
        if (!reader.store(asciiEscape, hexEscape, octalEscape)) {
            if (reader.expect('\\')) {
                throw invalidEscapeSequence(reader, kind);
            } else if (reader.expect('\n')) {
                throw unterminatedString(reader, kind);
            } else {
                reader.store();
            }
        }
    }
}
