package org.sterling.source.translator;

import static java.lang.ClassLoader.getSystemClassLoader;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.List;
import org.sterling.SterlingException;
import org.sterling.cli.CommandLineRunner;
import org.sterling.cli.RunnerException;
import org.sterling.runtime.Compiler;
import org.sterling.runtime.GlobalModule;
import org.sterling.runtime.expression.DeclaredExpression;
import org.sterling.runtime.expression.ExpressionPrinter;
import org.sterling.source.parser.Parser;
import org.sterling.source.scanner.InputReader;
import org.sterling.source.scanner.Scanner;
import org.sterling.source.scanner.ScannerFactory;

public class TranslatorRunner implements CommandLineRunner {

    private final Parser parser;
    private final ScannerFactory scannerFactory;
    private final Translator translator;
    private final ExpressionPrinter printer;
    private final PrintStream out;

    public TranslatorRunner() {
        parser = new Parser();
        scannerFactory = new ScannerFactory();
        translator = new Translator();
        printer = new ExpressionPrinter(1);
        out = System.out;
    }

    @Override
    public String getCommand() {
        return "translate";
    }

    @Override
    public String getHelpText() {
        return "Translates a file and displays the resultant abstract syntax tree";
    }

    @Override
    public void run(List<String> args) {
        File file = getFile(args);
        try (InputReader inputReader = new InputReader(file.getPath(), new FileInputStream(file))) {
            Scanner scanner = scannerFactory.getScanner(inputReader);
            GlobalModule globalModule = new GlobalModule(getSystemClassLoader(), new Compiler());
            for (DeclaredExpression declaration : translator.translate(parser.parse(scanner), globalModule)) {
                out.println();
                out.println("  <<" + declaration.getSymbol() + ">>");
                printer.print(declaration.getExpression(), out);
                out.println();
            }
            out.println();
        } catch (IOException | SterlingException exception) {
            throw new RunnerException(exception);
        }
    }

    private File getFile(List<String> args) {
        if (args.size() < 1) {
            throw new RunnerException("No file specified");
        } else {
            File file = new File(args.get(0));
            if (!file.exists()) {
                throw new RunnerException("File '" + file + "' does not exist");
            } else if (!file.isFile()) {
                throw new RunnerException("Path '" + file + "' is not a file");
            } else {
                return file;
            }
        }
    }
}
