// Copyright 2015-2020 Swim inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package swim.recon;

import swim.codec.Diagnostic;
import swim.codec.Input;
import swim.codec.Parser;
import swim.util.Builder;

final class ComparisonOperatorParser<I, V> extends Parser<V> {

  final ReconParser<I, V> recon;
  final Builder<I, V> builder;
  final Parser<V> lhsParser;
  final String operator;
  final Parser<V> rhsParser;
  final int step;

  ComparisonOperatorParser(ReconParser<I, V> recon, Builder<I, V> builder, Parser<V> lhsParser,
                           String operator, Parser<V> rhsParser, int step) {
    this.recon = recon;
    this.builder = builder;
    this.lhsParser = lhsParser;
    this.operator = operator;
    this.rhsParser = rhsParser;
    this.step = step;
  }

  static <I, V> Parser<V> parse(Input input, ReconParser<I, V> recon, Builder<I, V> builder,
                                Parser<V> lhsParser, String operator, Parser<V> rhsParser, int step) {
    int c = 0;
    if (step == 1) {
      if (lhsParser == null) {
        lhsParser = recon.parseAttrExpression(input, builder);
      }
      while (lhsParser.isCont() && !input.isEmpty()) {
        lhsParser = lhsParser.feed(input);
      }
      if (lhsParser.isDone()) {
        step = 2;
      } else if (lhsParser.isError()) {
        return lhsParser.asError();
      }
    }
    if (step == 2) {
      while (input.isCont()) {
        c = input.head();
        if (Recon.isSpace(c)) {
          input = input.step();
        } else {
          break;
        }
      }
      if (input.isCont()) {
        if (c == '!') {
          input = input.step();
          step = 3;
        } else if (c == '<') {
          input = input.step();
          step = 4;
        } else if (c == '>') {
          input = input.step();
          step = 5;
        } else if (c == '=') {
          input = input.step();
          step = 6;
        } else {
          return lhsParser;
        }
      } else if (input.isDone()) {
        return lhsParser;
      }
    }
    if (step == 3) {
      if (input.isCont()) {
        c = input.head();
        if (c == '=') {
          input = input.step();
          operator = "!=";
          step = 7;
        } else {
          operator = "!";
          step = 7;
        }
      } else if (input.isDone()) {
        return error(Diagnostic.unexpected(input));
      }
    }
    if (step == 4) {
      if (input.isCont()) {
        c = input.head();
        if (c == '=') {
          input = input.step();
          operator = "<=";
          step = 7;
        } else {
          operator = "<";
          step = 7;
        }
      } else if (input.isDone()) {
        return error(Diagnostic.unexpected(input));
      }
    }
    if (step == 5) {
      if (input.isCont()) {
        c = input.head();
        if (c == '=') {
          input = input.step();
          operator = ">=";
          step = 7;
        } else {
          operator = ">";
          step = 7;
        }
      } else if (input.isDone()) {
        return error(Diagnostic.unexpected(input));
      }
    }
    if (step == 6) {
      if (input.isCont()) {
        c = input.head();
        if (c == '=') {
          input = input.step();
          operator = "==";
          step = 7;
        } else if (c == '>') {
          return lhsParser;
        } else {
          operator = "=";
          step = 7;
        }
      } else if (input.isDone()) {
        return error(Diagnostic.unexpected(input));
      }
    }
    if (step == 7) {
      if (rhsParser == null) {
        rhsParser = recon.parseAttrExpression(input, builder);
      }
      while (rhsParser.isCont() && !input.isEmpty()) {
        rhsParser = rhsParser.feed(input);
      }
      if (rhsParser.isDone()) {
        final V lhs = lhsParser.bind();
        final V rhs = rhsParser.bind();
        if ("<".equals(operator)) {
          return done(recon.lt(lhs, rhs));
        } else if ("<=".equals(operator)) {
          return done(recon.le(lhs, rhs));
        } else if ("==".equals(operator)) {
          return done(recon.eq(lhs, rhs));
        } else if ("!=".equals(operator)) {
          return done(recon.ne(lhs, rhs));
        } else if (">=".equals(operator)) {
          return done(recon.ge(lhs, rhs));
        } else if (">".equals(operator)) {
          return done(recon.gt(lhs, rhs));
        } else {
          return error(Diagnostic.message(operator, input));
        }
      } else if (rhsParser.isError()) {
        return rhsParser.asError();
      }
    }
    if (input.isError()) {
      return error(input.trap());
    }
    return new ComparisonOperatorParser<I, V>(recon, builder, lhsParser, operator, rhsParser, step);
  }

  static <I, V> Parser<V> parse(Input input, ReconParser<I, V> recon, Builder<I, V> builder) {
    return parse(input, recon, builder, null, null, null, 1);
  }

  @Override
  public Parser<V> feed(Input input) {
    return parse(input, this.recon, this.builder, this.lhsParser, this.operator,
        this.rhsParser, this.step);
  }

}
