// Copyright 2015-2024 Nstream, inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package swim.structure;

import swim.codec.Output;
import swim.structure.operator.InvokeOperator;
import swim.structure.selector.ChildrenSelector;
import swim.structure.selector.DescendantsSelector;
import swim.structure.selector.FilterSelector;
import swim.structure.selector.GetAttrSelector;
import swim.structure.selector.GetItemSelector;
import swim.structure.selector.GetSelector;
import swim.structure.selector.IdentitySelector;
import swim.structure.selector.KeysSelector;
import swim.structure.selector.LiteralSelector;
import swim.structure.selector.ValuesSelector;

/**
 * A composable {@link Expression} that returns references to {@code Items} when
 * it is {@link #evaluate evaluated}, providing a foundation on top of which
 * expression languages may be built.
 */
public abstract class Selector extends Expression {

  public Selector() {
    // nop
  }

  @Override
  public boolean isConstant() {
    return false;
  }

  /**
   * Returns the {@code Selector} that this {@code Selector} uses to match
   * sub-selections.
   */
  public abstract Selector then();

  /**
   * Evaluates {@link Selectee#selected callback.selected} against the
   * {@code Items} that match this {@code Selector's} selection criteria. That
   * is, it pushes such {@code Items} to {@code interpreter}, then invokes
   * {@code callback} against it. To support chained {@code Selectors}, this is
   * a recursive procedure that invokes {@code forSelected} through
   * {@code this.then} wherever it exists (which it always does outside of
   * {@link IdentitySelector}); we define "subselection" to be such an
   * invocation.
   *
   * @return the result of executing {@code callback} from the context of the
   * last {@code Selector} in the chain formed by {@code Selector then} fields.
   */
  public abstract <T> T forSelected(Interpreter interpreter, Selectee<T> callback);

  public abstract Item mapSelected(Interpreter interpreter, Selectee<Item> transform);

  /**
   * Evaluates this {@code Selector} against some {@link Interpreter}. This is
   * accomplished by creating a new {@link SelecteeBuilder} and populating
   * its internal {@link Record} with (recursive) calls to {@link #forSelected}.
   */
  @Override
  public final Item evaluate(Interpreter interpreter) {
    final Record selected = Record.create();
    final Selectee<Object> callback = new SelecteeBuilder(selected);
    this.forSelected(interpreter, callback);
    return selected.isEmpty() ? Item.absent() : selected.flattened();
  }

  /**
   * The means to chain {@code Selectors}.
   *
   * <p>By design, this is not a strict functional composition. For two {@code
   * Selectors} {@code s1} and {@code s2}, {@code s1.andThen(s2)} <i>does not
   * necessarily</i> return a new {@code Selector} {@code s3} such that {@code
   * s3.evaluate(args)} is equivalent to {@code s2.evaluate(s1.evaluate(args))}.
   *
   * <p> The reason for this is that for <i>result set</i> yielding {@code
   * Selectors} (e.g. {@code ChildrenSelector}), we wish to invoke the {@code
   * then} against every result. Under strict functional rules, {@code
   * ChildrenSelector#andThen(getSelector).evaluate(args)} would instead
   * return at most one defined value regardless of the number of children.
   */
  public abstract Selector andThen(Selector then);

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link GetSelector}.
   *
   * @param key the {@code key} field in the composing {@code GetSelector}.
   */
  @Override
  public Selector get(Value key) {
    return this.andThen(new GetSelector(key, Selector.identity()));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link GetSelector}.
   *
   * @param key the {@code key} field in the composing {@code GetSelector}.
   */
  @Override
  public Selector get(String key) {
    return this.get(Text.from(key));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link
   * GetAttrSelector}.
   *
   * @param key the {@code key} field in the composing {@code GetAttrSelector}.
   */
  @Override
  public Selector getAttr(Text key) {
    return this.andThen(new GetAttrSelector(key, Selector.identity()));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link
   * GetAttrSelector}.
   *
   * @param key the {@code key} field in the composing {@code GetAttrSelector}.
   */
  @Override
  public Selector getAttr(String key) {
    return this.getAttr(Text.from(key));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link
   * GetItemSelector}.
   *
   * @param index the {@code index} field in the composing {@code
   *              GetItemSelector}.
   */
  public Selector getItem(Num index) {
    return this.andThen(new GetItemSelector(index, Selector.identity()));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link
   * GetItemSelector}.
   *
   * @param index the {@code index} field in the composing {@code
   *              GetItemSelector}.
   */
  @Override
  public Selector getItem(int index) {
    return this.getItem(Num.from(index));
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is the {@link
   * KeysSelector}.
   */
  public Selector keys() {
    return this.andThen(Selector.identity().keys());
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is the {@link
   * ValuesSelector}.
   */
  public Selector values() {
    return this.andThen(Selector.identity().values());
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is the {@link
   * ChildrenSelector}.
   */
  public Selector children() {
    return this.andThen(Selector.identity().children());
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is the {@link
   * DescendantsSelector}.
   */
  public Selector descendants() {
    return this.andThen(Selector.identity().descendants());
  }

  /**
   * Returns a new {@link FilterSelector} with {@code this} as the {@code
   * predicate}.
   */
  @Override
  public FilterSelector filter() {
    return new FilterSelector(this, Selector.identity());
  }

  /**
   * Shorthand to {@link #andThen} where {@code then} is a {@link
   * FilterSelector}.
   */
  @Override
  public Selector filter(Item predicate) {
    return this.andThen(predicate.filter());
  }

  /**
   * Creates, but does not evaluate, an {@link InvokeOperator} where this {@code
   * Selector} evaluates to the operator.
   */
  @Override
  public Operator invoke(Value args) {
    return new InvokeOperator(this, args);
  }

  @Override
  public int precedence() {
    return 11;
  }

  @Override
  public <T> Output<T> debug(Output<T> output) {
    output = output.write("Selector").write('.').write("identity").write('(').write(')');
    output = this.debugThen(output);
    return output;
  }

  public abstract <T> Output<T> debugThen(Output<T> output);

  @Override
  public int compareTo(Item other) {
    if (other instanceof Selector) {
      return this.compareTo((Selector) other);
    }
    return Integer.compare(this.typeOrder(), other.typeOrder());
  }

  protected abstract int compareTo(Selector that);

  public static Selector identity() {
    return IdentitySelector.identity();
  }

  /**
   * Lifts {@code item} into a {@link LiteralSelector} if it is not already a
   * {@code Selector}.
   */
  public static Selector literal(Item item) {
    if (item instanceof Selector) {
      return (Selector) item;
    }
    return new LiteralSelector(item, Selector.identity());
  }

}

/**
 * {@link Selectee} implementation that accumulates {@link Item Items} in a
 * mutable {@link Record}.
 */
final class SelecteeBuilder implements Selectee<Object> {

  final Record selected;

  SelecteeBuilder(Record selected) {
    this.selected = selected;
  }

  /**
   * Adds the top of the {@code interpreter}'s scope stack to the {@code
   * selected} record. Always returns {@code null} because {@code
   * SelecteeBuilder} never terminates early; it accumulates all visited items
   * into an internal, mutable {@code Record}.
   */
  @Override
  public Object selected(Interpreter interpreter) {
    final Item scope = interpreter.peekScope();
    if (scope != null) {
      this.selected.add(scope);
    }
    return null;
  }

}
