/*
 * Decompiled with CFR 0.152.
 */
package org.swisspush.gateleen.cache;

import com.google.common.base.Splitter;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.swisspush.gateleen.cache.fetch.CacheDataFetcher;
import org.swisspush.gateleen.cache.storage.CacheStorage;
import org.swisspush.gateleen.core.util.ResponseStatusCodeLogUtil;
import org.swisspush.gateleen.core.util.Result;
import org.swisspush.gateleen.core.util.StatusCode;
import org.swisspush.gateleen.core.util.StringUtils;

public class CacheHandler {
    public static final String CONTENT_TYPE_HEADER = "Content-Type";
    public static final String CONTENT_TYPE_JSON = "application/json";
    public static final String DEFAULT_CACHE_CONTROL_HEADER = "Cache-Control";
    private static final String NO_CACHE = "no-cache";
    private static final String MAX_AGE = "max-age=";
    private static final String MAX_AGE_ZERO = "max-age=0";
    private static final int TIMEOUT_MS = 30000;
    private final Logger log = LoggerFactory.getLogger(CacheHandler.class);
    private final CacheDataFetcher dataFetcher;
    private final CacheStorage cacheStorage;
    private final String cacheAdminUri;
    private final String cacheControlHeader;

    public CacheHandler(CacheDataFetcher dataFetcher, CacheStorage cacheStorage, String cacheAdminUri) {
        this(dataFetcher, cacheStorage, cacheAdminUri, DEFAULT_CACHE_CONTROL_HEADER);
    }

    public CacheHandler(CacheDataFetcher dataFetcher, CacheStorage cacheStorage, String cacheAdminUri, String customCacheControlHeader) {
        this.dataFetcher = dataFetcher;
        this.cacheStorage = cacheStorage;
        this.cacheAdminUri = cacheAdminUri;
        this.cacheControlHeader = customCacheControlHeader;
    }

    public boolean handle(HttpServerRequest request) {
        if (request.uri().startsWith(this.cacheAdminUri)) {
            if (HttpMethod.POST == request.method() && request.uri().equals(this.cacheAdminUri + "/clear")) {
                this.handleClearCache(request);
            } else if (HttpMethod.GET == request.method() && request.uri().equals(this.cacheAdminUri + "/count")) {
                this.handleCacheCount(request);
            } else if (HttpMethod.GET == request.method() && request.uri().equals(this.cacheAdminUri + "/entries")) {
                this.handleCacheEntries(request);
            } else {
                this.respondWith(StatusCode.METHOD_NOT_ALLOWED, request);
            }
            return true;
        }
        if (HttpMethod.GET != request.method() || !this.containsCacheHeaders(request)) {
            return false;
        }
        request.pause();
        this.log.debug("Got a request which may be be cached");
        Optional<Long> expireMs = this.extractExpireMs(request);
        if (expireMs.isEmpty()) {
            this.log.warn("Could not extract max-age value from Cache-Control request header");
            this.respondWith(StatusCode.BAD_REQUEST, request);
            return true;
        }
        String cacheIdentifier = request.uri();
        this.cacheStorage.cachedRequest(cacheIdentifier).setHandler(event -> {
            if (event.failed()) {
                this.log.warn("Failed to get cached request from storage", event.cause());
                this.respondWith(StatusCode.INTERNAL_SERVER_ERROR, request);
                return;
            }
            Optional cachedRequest = (Optional)event.result();
            if (cachedRequest.isPresent()) {
                this.log.debug("Request to {} found in cache storage", (Object)request.uri());
                this.respondWithPayload(request, (Buffer)cachedRequest.get());
            } else {
                this.updateCacheAndRespond(request, cacheIdentifier, (Long)expireMs.get());
            }
        });
        return true;
    }

    private void updateCacheAndRespond(HttpServerRequest request, String cacheIdentifier, Long expireMs) {
        this.log.debug("Request to {} not found in cache storage, going to fetch it.", (Object)request.uri());
        this.dataFetcher.fetchData(request.uri(), request.headers(), 30000L).setHandler(event -> {
            if (event.failed()) {
                this.log.warn("Failed to fetch data from request", event.cause());
                this.respondWith(StatusCode.INTERNAL_SERVER_ERROR, request);
                return;
            }
            Result result = (Result)event.result();
            if (result.isErr()) {
                this.respondWith((StatusCode)result.err(), request);
                return;
            }
            Buffer fetchedData = (Buffer)result.ok();
            this.cacheStorage.cacheRequest(cacheIdentifier, fetchedData, Duration.ofMillis(expireMs)).setHandler(event1 -> {
                if (event1.failed()) {
                    this.log.warn("Failed to store request to cache", event1.cause());
                }
                this.respondWithPayload(request, fetchedData);
            });
        });
    }

    private boolean containsCacheHeaders(HttpServerRequest request) {
        List cacheControlHeaderValues = request.headers().getAll(this.cacheControlHeader);
        for (String cacheControlHeaderValue : cacheControlHeaderValues) {
            if (NO_CACHE.equalsIgnoreCase(cacheControlHeaderValue) || cacheControlHeaderValue.toLowerCase().contains(MAX_AGE_ZERO)) {
                return false;
            }
            if (!cacheControlHeaderValue.toLowerCase().contains(MAX_AGE)) continue;
            return true;
        }
        return false;
    }

    private Optional<Long> extractExpireMs(HttpServerRequest request) {
        String cacheControlHeaderValue = request.headers().get(this.cacheControlHeader);
        if (cacheControlHeaderValue == null || !cacheControlHeaderValue.toLowerCase().contains(MAX_AGE)) {
            return Optional.empty();
        }
        cacheControlHeaderValue = StringUtils.trim((String)cacheControlHeaderValue).toLowerCase();
        List headerValues = Splitter.on((String)MAX_AGE).omitEmptyStrings().splitToList((CharSequence)cacheControlHeaderValue);
        if (headerValues.size() != 1) {
            return Optional.empty();
        }
        String headerValue = (String)headerValues.get(0);
        try {
            long expireSeconds = Long.parseLong(headerValue);
            return Optional.of(expireSeconds * 1000L);
        }
        catch (NumberFormatException ex) {
            this.log.warn("Value of {} max-age header is not a number: {}", (Object)this.cacheControlHeader, (Object)headerValue);
            return Optional.empty();
        }
    }

    private void handleClearCache(HttpServerRequest request) {
        this.log.debug("About to clear all cached entries manually");
        this.cacheStorage.clearCache().setHandler(event -> {
            if (event.failed()) {
                this.log.warn("Error while clearing cache", event.cause());
                this.respondWith(StatusCode.INTERNAL_SERVER_ERROR, request);
            }
            Long clearedCount = (Long)event.result();
            this.log.debug("Cleared {} cache entries", (Object)clearedCount);
            JsonObject clearedObj = new JsonObject().put("cleared", clearedCount);
            this.respondWithPayload(request, Buffer.buffer((String)clearedObj.encode()));
        });
    }

    private void handleCacheEntries(HttpServerRequest request) {
        this.log.debug("About to get cached entries list");
        this.cacheStorage.cacheEntries().setHandler(event -> {
            if (event.failed()) {
                this.log.warn("Error while getting cached entries list", event.cause());
                this.respondWith(StatusCode.INTERNAL_SERVER_ERROR, request);
            }
            Set cachedEntries = (Set)event.result();
            this.log.debug("{} entries in cache", (Object)cachedEntries.size());
            JsonArray entriesArray = new JsonArray();
            for (String cachedEntry : cachedEntries) {
                entriesArray.add(cachedEntry);
            }
            JsonObject entriesObj = new JsonObject().put("entries", entriesArray);
            this.respondWithPayload(request, Buffer.buffer((String)entriesObj.encode()));
        });
    }

    private void handleCacheCount(HttpServerRequest request) {
        this.log.debug("About to get cached entries count");
        this.cacheStorage.cacheEntriesCount().setHandler(event -> {
            if (event.failed()) {
                this.log.warn("Error while getting cached entries count", event.cause());
                this.respondWith(StatusCode.INTERNAL_SERVER_ERROR, request);
            }
            Long count = (Long)event.result();
            this.log.debug("{} entries in cache", (Object)count);
            JsonObject clearedObj = new JsonObject().put("count", count);
            this.respondWithPayload(request, Buffer.buffer((String)clearedObj.encode()));
        });
    }

    private void respondWith(StatusCode statusCode, HttpServerRequest request) {
        ResponseStatusCodeLogUtil.info((HttpServerRequest)request, (StatusCode)statusCode, CacheHandler.class);
        request.response().setStatusCode(statusCode.getStatusCode());
        request.response().setStatusMessage(statusCode.getStatusMessage());
        request.response().end();
        request.resume();
    }

    private void respondWithPayload(HttpServerRequest request, Buffer cachedRequestPayload) {
        ResponseStatusCodeLogUtil.info((HttpServerRequest)request, (StatusCode)StatusCode.OK, CacheHandler.class);
        request.response().setStatusCode(StatusCode.OK.getStatusCode());
        request.response().setStatusMessage(StatusCode.OK.getStatusMessage());
        request.response().headers().add(CONTENT_TYPE_HEADER, CONTENT_TYPE_JSON);
        request.response().end(cachedRequestPayload);
        request.resume();
    }
}

