/*
 * Decompiled with CFR 0.152.
 */
package org.swisspush.gateleen.delegate;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.Message;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.JsonObject;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.swisspush.gateleen.core.storage.ResourceStorage;
import org.swisspush.gateleen.core.util.ResourcesUtils;
import org.swisspush.gateleen.core.util.StatusCode;
import org.swisspush.gateleen.delegate.Delegate;
import org.swisspush.gateleen.delegate.DelegateFactory;
import org.swisspush.gateleen.monitoring.MonitoringHandler;
import org.swisspush.gateleen.validation.ValidationException;

public class DelegateHandler {
    private static final String DEFINITION_RESOURCE = "definition";
    private static final String EXECUTION_RESOURCE = "execution";
    private static final String SAVE_DELEGATE_ADDRESS = "gateleen.delegate-insert";
    private static final String REMOVE_DELEGATE_ADDRESS = "gateleen.delegate-remove";
    private static final Logger LOG = LoggerFactory.getLogger(DelegateHandler.class);
    private static final int NAME_GROUP_INDEX = 1;
    private static final int MESSAGE_NAME = 0;
    private static final int MESSAGE_URL = 1;
    private final Vertx vertx;
    private final HttpClient selfClient;
    private final ResourceStorage storage;
    private final String delegatesUri;
    private final DelegateFactory delegateFactory;
    private final Pattern delegateNamePattern;
    private final Map<String, Delegate> delegateMap;
    private boolean initialized;

    public DelegateHandler(Vertx vertx, HttpClient selfClient, ResourceStorage storage, MonitoringHandler monitoringHandler, String delegatesUri, Map<String, Object> properties) {
        this.vertx = vertx;
        this.selfClient = selfClient;
        this.storage = storage;
        this.delegatesUri = delegatesUri;
        String delegatesSchema = ResourcesUtils.loadResource((String)"gateleen_delegate_schema_delegates", (boolean)true);
        this.delegateFactory = new DelegateFactory(monitoringHandler, selfClient, properties, delegatesSchema);
        this.delegateNamePattern = Pattern.compile(delegatesUri + "([^/]+)(/" + DEFINITION_RESOURCE + "|/" + EXECUTION_RESOURCE + ".*" + "|/?)");
        this.delegateMap = new HashMap<String, Delegate>();
        this.initialized = false;
    }

    public void init() {
        if (!this.initialized) {
            this.registerDelegateRegistrationHandler();
            this.loadStoredDelegates();
        }
        this.initialized = true;
    }

    private void loadStoredDelegates() {
        HttpClientRequest selfRequest = this.selfClient.request(HttpMethod.GET, this.delegatesUri + "?expand=2", response -> {
            if (response.statusCode() == StatusCode.OK.getStatusCode()) {
                response.bodyHandler(event -> {
                    this.delegateMap.clear();
                    JsonObject responseObject = new JsonObject(event.toString());
                    if (responseObject.getValue("delegates") instanceof JsonObject) {
                        JsonObject delegates = responseObject.getJsonObject("delegates");
                        for (String delegateName : delegates.fieldNames()) {
                            JsonObject delegateContent = delegates.getJsonObject(delegateName);
                            JsonObject delegateDefinition = delegateContent.getJsonObject(DEFINITION_RESOURCE);
                            LOG.info("Loading delegate: {}", (Object)delegateName);
                            this.registerDelegate(Buffer.buffer((String)delegateDefinition.toString()), delegateName);
                        }
                    } else {
                        LOG.info("Currently are no delegates stored!");
                    }
                });
            } else if (response.statusCode() == StatusCode.NOT_FOUND.getStatusCode()) {
                LOG.debug("No delegates previously stored");
            } else {
                LOG.error("Delegates could not be loaded.");
            }
        });
        selfRequest.setTimeout(120000L);
        selfRequest.end();
    }

    private void registerDelegateRegistrationHandler() {
        if (LOG.isTraceEnabled()) {
            LOG.trace("registerDelegateRegistrationHandler");
        }
        this.vertx.eventBus().consumer(SAVE_DELEGATE_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> delegateEvent) {
                String[] messages = ((String)delegateEvent.body()).split(";");
                if (messages != null) {
                    DelegateHandler.this.storage.get(messages[1], buffer -> {
                        if (buffer != null) {
                            DelegateHandler.this.registerDelegate(buffer, messages[0]);
                        } else {
                            LOG.warn("Could not get URL '" + messages[1] + "' (getting delegate).");
                        }
                    });
                } else {
                    LOG.warn("Could not get Delegate, empty delegateEvent.");
                }
            }
        });
        this.vertx.eventBus().consumer(REMOVE_DELEGATE_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> delegateName) {
                DelegateHandler.this.unregisterDelegate((String)delegateName.body());
            }
        });
    }

    private void unregisterDelegate(String delegateName) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("unregisterDelegate: {}", (Object)delegateName);
        }
        this.delegateMap.remove(delegateName);
    }

    private void registerDelegate(Buffer buffer, String delegateName) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("registerDelegate: {}", (Object)delegateName);
        }
        try {
            Delegate delegate = this.delegateFactory.parseDelegate(delegateName, buffer);
            this.delegateMap.put(delegateName, delegate);
        }
        catch (ValidationException validationException) {
            LOG.error("Could not parse delegate: {}", (Object)validationException.toString());
        }
    }

    private void handleDelegateRegistration(HttpServerRequest request) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("handleDelegateRegistration: {}", (Object)request.uri());
        }
        request.bodyHandler(buffer -> {
            String delegateName = this.getDelegateName(request.uri());
            try {
                this.delegateFactory.parseDelegate(delegateName, (Buffer)buffer);
            }
            catch (ValidationException validationException) {
                LOG.warn("Could not parse delegate {}: {}", (Object)delegateName, (Object)validationException.toString());
                request.response().setStatusCode(StatusCode.BAD_REQUEST.getStatusCode());
                request.response().setStatusMessage(StatusCode.BAD_REQUEST.getStatusMessage() + " " + validationException.getMessage());
                if (validationException.getValidationDetails() != null) {
                    request.response().headers().add("Content-Type", "application/json");
                    request.response().end(validationException.getValidationDetails().encode());
                } else {
                    request.response().end(validationException.getMessage());
                }
                return;
            }
            this.storage.put(request.uri(), request.headers(), buffer, status -> {
                if (status.intValue() == StatusCode.OK.getStatusCode()) {
                    this.vertx.eventBus().publish(SAVE_DELEGATE_ADDRESS, (Object)(delegateName + ";" + request.uri()));
                } else {
                    request.response().setStatusCode(status.intValue());
                }
                request.response().end();
            });
        });
    }

    private void handleDelegateUnregistration(HttpServerRequest request) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("handleDelegateUnregistration: {}", (Object)request.uri());
        }
        String delegateName = this.getDelegateName(request.uri());
        this.storage.delete(this.delegatesUri + delegateName, status -> {
            this.vertx.eventBus().publish(REMOVE_DELEGATE_ADDRESS, (Object)delegateName);
            request.response().end();
        });
    }

    protected String getDelegateName(String uri) {
        Matcher nameMatcher = this.delegateNamePattern.matcher(uri);
        if (nameMatcher.matches()) {
            return nameMatcher.group(1);
        }
        return null;
    }

    public boolean handle(HttpServerRequest request) {
        String delegateName = this.getDelegateName(request.uri());
        if (delegateName != null) {
            if (request.method() == HttpMethod.PUT && request.uri().endsWith(DEFINITION_RESOURCE)) {
                LOG.debug("registering delegate");
                this.handleDelegateRegistration(request);
                return true;
            }
            if (this.delegateMap.containsKey(delegateName)) {
                if (request.uri().contains(EXECUTION_RESOURCE)) {
                    LOG.debug("executing delegate");
                    this.handleDelegateExecution(request);
                    return true;
                }
                if (request.method() == HttpMethod.DELETE) {
                    LOG.debug("unregister delegate");
                    this.handleDelegateUnregistration(request);
                    return true;
                }
            } else {
                LOG.warn("No delegate with the name [{}] registered. DelegateHandler will not process the given request!", (Object)delegateName);
            }
        }
        return false;
    }

    private void handleDelegateExecution(HttpServerRequest request) {
        String delegateName = this.getDelegateName(request.uri());
        Delegate delegate = this.delegateMap.get(delegateName);
        delegate.handle(request);
    }
}

