/*
 * Decompiled with CFR 0.152.
 */
package org.swisspush.gateleen.hook;

import io.vertx.core.Handler;
import io.vertx.core.MultiMap;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.Message;
import io.vertx.core.http.CaseInsensitiveHeaders;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import org.joda.time.LocalDateTime;
import org.joda.time.ReadablePartial;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.swisspush.gateleen.core.http.HttpRequest;
import org.swisspush.gateleen.core.storage.ResourceStorage;
import org.swisspush.gateleen.core.util.StatusCode;
import org.swisspush.gateleen.hook.HttpHook;
import org.swisspush.gateleen.hook.Listener;
import org.swisspush.gateleen.hook.ListenerRepository;
import org.swisspush.gateleen.hook.LocalListenerRepository;
import org.swisspush.gateleen.hook.LocalRouteRepository;
import org.swisspush.gateleen.hook.Route;
import org.swisspush.gateleen.hook.RouteRepository;
import org.swisspush.gateleen.logging.LoggingResourceManager;
import org.swisspush.gateleen.monitoring.MonitoringHandler;
import org.swisspush.gateleen.queue.expiry.ExpiryCheckHandler;
import org.swisspush.gateleen.queue.queuing.QueueClient;

public class HookHandler {
    public static final String HOOKED_HEADER = "x-hooked";
    public static final String HOOKS_LISTENERS_URI_PART = "/_hooks/listeners/";
    private static final String LISTENER_QUEUE_PREFIX = "listener-hook";
    private static final String LISTENER_HOOK_TARGET_PATH = "listeners/";
    public static final String HOOKS_ROUTE_URI_PART = "/_hooks/route";
    private static final String HOOK_STORAGE_PATH = "registrations/";
    private static final String HOOK_LISTENER_STORAGE_PATH = "registrations/listeners/";
    private static final String HOOK_ROUTE_STORAGE_PATH = "registrations/routes/";
    private static final String SAVE_LISTENER_ADDRESS = "gateleen.hook-listener-insert";
    private static final String REMOVE_LISTENER_ADDRESS = "gateleen.hook-listener-remove";
    private static final String SAVE_ROUTE_ADDRESS = "gateleen.hook-route-insert";
    private static final String REMOVE_ROUTE_ADDRESS = "gateleen.hook-route-remove";
    private static final int DEFAULT_HOOK_STORAGE_EXPIRE_AFTER_TIME = 3600;
    private static final int DEFAULT_HOOK_LISTENERS_EXPIRE_AFTER_TIME = 30;
    private static final int DEFAULT_CLEANUP_TIME = 15000;
    public static final String REQUESTURL = "requesturl";
    public static final String EXPIRATION_TIME = "expirationTime";
    public static final String HOOK = "hook";
    public static final String EXPIRE_AFTER = "expireAfter";
    public static final String FULL_URL = "fullUrl";
    private Logger log = LoggerFactory.getLogger(HookHandler.class);
    private Vertx vertx;
    private final ResourceStorage storage;
    private MonitoringHandler monitoringHandler;
    private LoggingResourceManager loggingResourceManager;
    private final HttpClient selfClient;
    private String userProfilePath;
    private String hookRootUri;
    private ListenerRepository listenerRepository;
    private RouteRepository routeRepository;
    private QueueClient queueClient;

    public HookHandler(Vertx vertx, HttpClient selfClient, ResourceStorage storage, LoggingResourceManager loggingResourceManager, MonitoringHandler monitoringHandler, String userProfilePath, String hookRootUri) {
        this.log.debug("Creating HookHandler ...");
        this.vertx = vertx;
        this.selfClient = selfClient;
        this.storage = storage;
        this.loggingResourceManager = loggingResourceManager;
        this.monitoringHandler = monitoringHandler;
        this.userProfilePath = userProfilePath;
        this.hookRootUri = hookRootUri;
        this.queueClient = new QueueClient(vertx, monitoringHandler);
        this.listenerRepository = new LocalListenerRepository();
        this.routeRepository = new LocalRouteRepository();
    }

    public void init() {
        this.registerListenerRegistrationHandler();
        this.registerRouteRegistrationHandler();
        this.loadStoredListeners();
        this.loadStoredRoutes();
        this.registerCleanupHandler();
    }

    private void registerCleanupHandler() {
        this.vertx.setPeriodic(15000L, (Handler)new Handler<Long>(){

            public void handle(Long timerID) {
                HookHandler.this.log.trace("Running hook cleanup ...");
                LocalDateTime nowAsTime = ExpiryCheckHandler.getActualTime();
                for (Listener listener : HookHandler.this.listenerRepository.getListeners()) {
                    if (!listener.getHook().getExpirationTime().isBefore((ReadablePartial)nowAsTime)) continue;
                    HookHandler.this.log.debug("Listener " + listener.getListenerId() + " expired at " + listener.getHook().getExpirationTime() + " and actual time is " + nowAsTime);
                    HookHandler.this.listenerRepository.removeListener(listener.getListenerId());
                    HookHandler.this.routeRepository.removeRoute(HookHandler.this.hookRootUri + HookHandler.LISTENER_HOOK_TARGET_PATH + listener.getListenerId());
                }
                Map<String, Route> routes = HookHandler.this.routeRepository.getRoutes();
                for (String key : routes.keySet()) {
                    Route route = routes.get(key);
                    if (!route.getHook().getExpirationTime().isBefore((ReadablePartial)nowAsTime)) continue;
                    HookHandler.this.routeRepository.removeRoute(key);
                }
                HookHandler.this.log.trace("done");
            }
        });
    }

    private void loadStoredRoutes() {
        this.log.debug("loadStoredRoutes");
        HttpClientRequest selfRequest = this.selfClient.request(HttpMethod.GET, this.hookRootUri + HOOK_ROUTE_STORAGE_PATH + "?expand=1", response -> {
            if (response.statusCode() == StatusCode.OK.getStatusCode()) {
                this.makeResponse((HttpClientResponse)response);
            } else if (response.statusCode() == StatusCode.NOT_FOUND.getStatusCode()) {
                this.log.debug("No route previously stored");
            } else {
                this.log.error("Routes could not be loaded.");
            }
        });
        selfRequest.setTimeout(120000L);
        selfRequest.end();
    }

    private void makeResponse(HttpClientResponse response) {
        response.bodyHandler(event -> {
            JsonObject responseObject = new JsonObject(event.toString());
            if (responseObject.getValue("routes") instanceof JsonObject) {
                JsonObject routes = responseObject.getJsonObject("routes");
                for (String routeStorageId : routes.fieldNames()) {
                    this.log.info("Loading route with storage id: " + routeStorageId);
                    JsonObject storageObject = routes.getJsonObject(routeStorageId);
                    this.registerRoute(Buffer.buffer((String)storageObject.toString()));
                }
            } else {
                this.log.info("Currently are no routes stored!");
            }
        });
    }

    private void loadStoredListeners() {
        this.log.debug("loadStoredListeners");
        HttpClientRequest selfRequest = this.selfClient.request(HttpMethod.GET, this.hookRootUri + HOOK_LISTENER_STORAGE_PATH + "?expand=1", (Handler)new Handler<HttpClientResponse>(){

            public void handle(HttpClientResponse response) {
                if (response.statusCode() == StatusCode.OK.getStatusCode()) {
                    response.bodyHandler((Handler)new Handler<Buffer>(){

                        public void handle(Buffer event) {
                            JsonObject responseObject = new JsonObject(event.toString());
                            if (responseObject.getValue("listeners") instanceof JsonObject) {
                                JsonObject listeners = responseObject.getJsonObject("listeners");
                                for (String listenerStorageId : listeners.fieldNames()) {
                                    HookHandler.this.log.info("Loading listener with storage id: " + listenerStorageId);
                                    JsonObject storageObject = listeners.getJsonObject(listenerStorageId);
                                    HookHandler.this.registerListener(Buffer.buffer((String)storageObject.toString()));
                                }
                            } else {
                                HookHandler.this.log.info("Currently are no listeners stored!");
                            }
                        }
                    });
                } else if (response.statusCode() == StatusCode.NOT_FOUND.getStatusCode()) {
                    HookHandler.this.log.debug("No listener previously stored");
                } else {
                    HookHandler.this.log.error("Listeners could not be loaded.");
                }
            }
        });
        selfRequest.setTimeout(120000L);
        selfRequest.end();
    }

    private void registerRouteRegistrationHandler() {
        this.vertx.eventBus().consumer(SAVE_ROUTE_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> event) {
                HookHandler.this.storage.get((String)event.body(), buffer -> {
                    if (buffer != null) {
                        HookHandler.this.registerRoute(buffer);
                    } else {
                        HookHandler.this.log.warn("Could not get URL '" + (event.body() == null ? "<null>" : (String)event.body()) + "' (getting hook route).");
                    }
                });
            }
        });
        this.vertx.eventBus().consumer(REMOVE_ROUTE_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> event) {
                HookHandler.this.unregisterRoute((String)event.body());
            }
        });
    }

    private void registerListenerRegistrationHandler() {
        this.vertx.eventBus().consumer(SAVE_LISTENER_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> event) {
                HookHandler.this.storage.get((String)event.body(), buffer -> {
                    if (buffer != null) {
                        HookHandler.this.registerListener(buffer);
                    } else {
                        HookHandler.this.log.warn("Could not get URL '" + (event.body() == null ? "<null>" : (String)event.body()) + "' (getting hook listener).");
                    }
                });
            }
        });
        this.vertx.eventBus().consumer(REMOVE_LISTENER_ADDRESS, (Handler)new Handler<Message<String>>(){

            public void handle(Message<String> event) {
                HookHandler.this.unregisterListener((String)event.body());
            }
        });
    }

    public boolean handle(HttpServerRequest request) {
        boolean consumed = false;
        if (this.isHookListenerRegistration(request)) {
            this.handleListenerRegistration(request);
            return true;
        }
        if (this.isHookListenerUnregistration(request)) {
            this.handleListenerUnregistration(request);
            return true;
        }
        if (this.isHookRouteRegistration(request)) {
            this.handleRouteRegistration(request);
            return true;
        }
        if (this.isHookRouteUnregistration(request)) {
            this.handleRouteUnregistration(request);
            return true;
        }
        List<Listener> listeners = this.listenerRepository.findListeners(request.uri(), request.method().name());
        if (!listeners.isEmpty() && !this.isRequestAlreadyHooked(request)) {
            this.installBodyHandler(request, listeners);
            consumed = true;
        }
        if (!consumed) {
            return this.routeRequestIfNeeded(request);
        }
        return true;
    }

    private boolean routeRequestIfNeeded(HttpServerRequest request) {
        Route route = this.routeRepository.getRoute(request.uri());
        if (route != null && (route.getHook().getMethods().isEmpty() || route.getHook().getMethods().contains(request.method().name()))) {
            this.log.debug("Forward request " + request.uri());
            route.forward(request);
            return true;
        }
        return false;
    }

    private void installBodyHandler(final HttpServerRequest request, final List<Listener> listeners) {
        request.bodyHandler(buffer -> {
            Handler<Void> doneHandler = new Handler<Void>(){
                private AtomicInteger currentCount = new AtomicInteger(0);
                private boolean sent = false;

                public void handle(Void event) {
                    if (this.currentCount.incrementAndGet() == listeners.size() && !this.sent) {
                        this.sent = true;
                        Route route = HookHandler.this.routeRepository.getRoute(request.uri());
                        if (route != null && (route.getHook().getMethods().isEmpty() || route.getHook().getMethods().contains(request.method().name()))) {
                            HookHandler.this.log.debug("Forward request (consumed) " + request.uri());
                            route.forward(request, buffer);
                        } else {
                            request.headers().add(HookHandler.HOOKED_HEADER, "true");
                            HookHandler.this.createSelfRequest(request, buffer);
                        }
                    }
                }
            };
            for (Listener listener : listeners) {
                String targetUri;
                this.log.debug("Enqueue request matching " + request.method() + " " + listener.getMonitoredUrl() + " with listener " + listener.getListener());
                String path = request.uri();
                if (!listener.getHook().isFullUrl()) {
                    path = request.uri().replace(listener.getMonitoredUrl(), "");
                }
                if (listener.getHook().getDestination().startsWith("/")) {
                    targetUri = listener.getListener() + path;
                    this.log.debug(" > internal target: " + targetUri);
                } else {
                    targetUri = this.hookRootUri + LISTENER_HOOK_TARGET_PATH + listener.getListener() + path;
                    this.log.debug(" > external target: " + targetUri);
                }
                String queue = "listener-hook-" + listener.getListenerId();
                CaseInsensitiveHeaders queueHeaders = new CaseInsensitiveHeaders();
                queueHeaders.addAll(request.headers());
                if (ExpiryCheckHandler.getExpireAfter((MultiMap)queueHeaders) == null) {
                    ExpiryCheckHandler.setExpireAfter((MultiMap)queueHeaders, (int)listener.getHook().getExpireAfter());
                }
                queueHeaders.add("x-translate-status-4xx", "200");
                this.queueClient.enqueue(new HttpRequest(request.method(), targetUri, (MultiMap)queueHeaders, buffer.getBytes()), queue, (Handler)doneHandler);
            }
        });
    }

    private void handleRouteUnregistration(HttpServerRequest request) {
        this.log.debug("handleRouteUnregistration > " + request.uri());
        String routeStorageUri = this.hookRootUri + HOOK_ROUTE_STORAGE_PATH + this.getStorageIdentifier(request.uri());
        this.storage.delete(routeStorageUri, status -> {
            this.vertx.eventBus().publish(REMOVE_ROUTE_ADDRESS, (Object)request.uri());
            request.response().end();
        });
    }

    private void handleRouteRegistration(HttpServerRequest request) {
        this.log.debug("handleRouteRegistration > " + request.uri());
        request.bodyHandler(hookData -> {
            String routeStorageUri = this.hookRootUri + HOOK_ROUTE_STORAGE_PATH + this.getStorageIdentifier(request.uri());
            Integer expireAfter = ExpiryCheckHandler.getExpireAfter((MultiMap)request.headers());
            if (expireAfter == null) {
                expireAfter = 3600;
            }
            ExpiryCheckHandler.setExpireAfter((HttpServerRequest)request, (int)expireAfter);
            LocalDateTime expirationTime = ExpiryCheckHandler.getExpirationTime((int)expireAfter);
            JsonObject storageObject = new JsonObject();
            storageObject.put(REQUESTURL, request.uri());
            storageObject.put(EXPIRATION_TIME, ExpiryCheckHandler.printDateTime((LocalDateTime)expirationTime));
            storageObject.put(HOOK, new JsonObject(hookData.toString()));
            this.storage.put(routeStorageUri, request.headers(), Buffer.buffer((String)storageObject.toString()), status -> {
                if (status.intValue() == StatusCode.OK.getStatusCode()) {
                    this.vertx.eventBus().publish(SAVE_ROUTE_ADDRESS, (Object)routeStorageUri);
                } else {
                    request.response().setStatusCode(status.intValue());
                }
                request.response().end();
            });
        });
    }

    private String getStorageIdentifier(String url) {
        return url.replace("/", "+");
    }

    private void handleListenerUnregistration(HttpServerRequest request) {
        this.log.debug("handleListenerUnregistration > " + request.uri());
        String listenerStorageUri = this.hookRootUri + HOOK_LISTENER_STORAGE_PATH + this.getUniqueListenerId(request.uri());
        this.storage.delete(listenerStorageUri, status -> {
            this.vertx.eventBus().publish(REMOVE_LISTENER_ADDRESS, (Object)request.uri());
            request.response().end();
        });
    }

    private void handleListenerRegistration(HttpServerRequest request) {
        this.log.debug("handleListenerRegistration > " + request.uri());
        request.bodyHandler(hookData -> {
            String listenerStorageUri = this.hookRootUri + HOOK_LISTENER_STORAGE_PATH + this.getUniqueListenerId(request.uri());
            Integer expireAfter = ExpiryCheckHandler.getExpireAfter((MultiMap)request.headers());
            if (expireAfter == null) {
                expireAfter = 3600;
            }
            ExpiryCheckHandler.setExpireAfter((HttpServerRequest)request, (int)expireAfter);
            LocalDateTime expirationTime = ExpiryCheckHandler.getExpirationTime((int)expireAfter);
            JsonObject storageObject = new JsonObject();
            storageObject.put(REQUESTURL, request.uri());
            storageObject.put(EXPIRATION_TIME, ExpiryCheckHandler.printDateTime((LocalDateTime)expirationTime));
            storageObject.put(HOOK, new JsonObject(hookData.toString()));
            this.storage.put(listenerStorageUri, request.headers(), Buffer.buffer((String)storageObject.toString()), status -> {
                if (status.intValue() == StatusCode.OK.getStatusCode()) {
                    this.vertx.eventBus().publish(SAVE_LISTENER_ADDRESS, (Object)listenerStorageUri);
                } else {
                    request.response().setStatusCode(status.intValue());
                }
                request.response().end();
            });
        });
    }

    private void createSelfRequest(final HttpServerRequest request, Buffer requestBody) {
        this.log.debug("Create self request for " + request.uri());
        HttpClientRequest selfRequest = this.selfClient.request(request.method(), request.uri(), (Handler)new Handler<HttpClientResponse>(){

            public void handle(HttpClientResponse response) {
                request.response().setStatusCode(response.statusCode());
                request.response().setStatusMessage(response.statusMessage());
                request.response().setChunked(true);
                request.response().headers().addAll(response.headers());
                request.response().headers().remove("Content-Length");
                response.handler(data -> request.response().write(data));
                response.endHandler(v -> request.response().end());
            }
        });
        if (request.headers() != null && !request.headers().isEmpty()) {
            selfRequest.headers().setAll(request.headers());
        }
        selfRequest.exceptionHandler(exception -> this.log.warn("HookHandler HOOK_ERROR: Failed self request to " + request.uri() + ": " + exception.getMessage()));
        selfRequest.setTimeout(120000L);
        if (requestBody != null) {
            selfRequest.end(requestBody);
        } else {
            selfRequest.end();
        }
    }

    public boolean isRequestAlreadyHooked(HttpServerRequest request) {
        String hooked = request.headers().get(HOOKED_HEADER);
        return hooked != null ? hooked.equals("true") : false;
    }

    private void unregisterRoute(String requestUrl) {
        String routedUrl = this.getRoutedUrlSegment(requestUrl);
        this.log.debug("Unregister route " + routedUrl);
        this.routeRepository.removeRoute(routedUrl);
    }

    private void unregisterListener(String requestUrl) {
        String listenerId = this.getUniqueListenerId(requestUrl);
        this.log.debug("Unregister listener " + listenerId);
        this.routeRepository.removeRoute(this.hookRootUri + LISTENER_HOOK_TARGET_PATH + this.getListenerUrlSegment(requestUrl));
        this.listenerRepository.removeListener(listenerId);
    }

    private void registerListener(Buffer buffer) {
        JsonObject storageObject = new JsonObject(buffer.toString());
        String requestUrl = storageObject.getString(REQUESTURL);
        if (this.log.isTraceEnabled()) {
            this.log.trace("Request URL: " + requestUrl);
        }
        String target = this.getListenerUrlSegment(requestUrl);
        String listenerId = this.getUniqueListenerId(requestUrl);
        if (this.log.isTraceEnabled()) {
            this.log.trace("Target (1st): " + target);
        }
        JsonObject jsonHook = storageObject.getJsonObject(HOOK);
        JsonArray jsonMethods = jsonHook.getJsonArray("methods");
        HttpHook hook = new HttpHook(jsonHook.getString("destination"));
        if (jsonMethods != null) {
            hook.setMethods(jsonMethods.getList());
        }
        if (jsonHook.containsKey("filter")) {
            hook.setFilter(jsonHook.getString("filter"));
        }
        if (jsonHook.containsKey("filter")) {
            hook.setFilter(jsonHook.getString("filter"));
        }
        if (jsonHook.getInteger(EXPIRE_AFTER) != null) {
            hook.setExpireAfter(jsonHook.getInteger(EXPIRE_AFTER));
        } else {
            hook.setExpireAfter(30);
        }
        String expirationTimeExpression = storageObject.getString(EXPIRATION_TIME);
        LocalDateTime expirationTime = null;
        if (expirationTimeExpression != null) {
            try {
                expirationTime = ExpiryCheckHandler.parseDateTime((String)expirationTimeExpression);
            }
            catch (Exception e) {
                this.log.warn("Listener " + listenerId + " for target " + target + " has an invalid expiration time " + expirationTimeExpression + " and will not be registred!", (Throwable)e);
                return;
            }
        } else {
            this.log.warn("Listener " + listenerId + " for target " + target + " has no expiration time and will not be registred!");
            return;
        }
        this.log.debug("Register listener and  route " + target + " with expiration at " + expirationTime);
        hook.setExpirationTime(expirationTime);
        boolean fullUrl = jsonHook.getBoolean(FULL_URL, Boolean.valueOf(false));
        hook.setFullUrl(fullUrl);
        if (hook.getDestination().startsWith("/")) {
            if (this.log.isTraceEnabled()) {
                this.log.trace("internal target, switching target!");
            }
            target = hook.getDestination();
        } else {
            String urlPattern = this.hookRootUri + LISTENER_HOOK_TARGET_PATH + target;
            this.routeRepository.addRoute(urlPattern, this.createRoute(urlPattern, hook));
            if (this.log.isTraceEnabled()) {
                this.log.trace("external target, add route for urlPattern: " + urlPattern);
            }
        }
        if (this.log.isTraceEnabled()) {
            this.log.trace("Target (2nd): " + target);
        }
        this.listenerRepository.addListener(new Listener(listenerId, this.getMonitoredUrlSegment(requestUrl), target, hook));
    }

    private String getUniqueListenerId(String requestUrl) {
        StringBuffer listenerId = new StringBuffer();
        listenerId.append(this.convertToStoragePattern(this.getListenerUrlSegment(requestUrl)));
        listenerId.append(this.convertToStoragePattern(this.getMonitoredUrlSegment(requestUrl)));
        return listenerId.toString();
    }

    private String convertToStoragePattern(String urlSegment) {
        return urlSegment.replace("/", "+").replace(".", "+").replace(":", "+");
    }

    private void registerRoute(Buffer buffer) {
        JsonObject storageObject = new JsonObject(buffer.toString());
        String requestUrl = storageObject.getString(REQUESTURL);
        String routedUrl = this.getRoutedUrlSegment(requestUrl);
        this.log.debug("Register route to  " + routedUrl);
        JsonObject jsonHook = storageObject.getJsonObject(HOOK);
        JsonArray jsonMethods = jsonHook.getJsonArray("methods");
        HttpHook hook = new HttpHook(jsonHook.getString("destination"));
        if (jsonMethods != null) {
            hook.setMethods(jsonMethods.getList());
        }
        if (jsonHook.getInteger(EXPIRE_AFTER) != null) {
            hook.setExpireAfter(jsonHook.getInteger(EXPIRE_AFTER));
        } else {
            hook.setExpireAfter(30);
        }
        String expirationTimeExpression = storageObject.getString(EXPIRATION_TIME);
        if (expirationTimeExpression != null) {
            try {
                hook.setExpirationTime(ExpiryCheckHandler.parseDateTime((String)expirationTimeExpression));
            }
            catch (Exception e) {
                this.log.warn("Route " + routedUrl + " has an invalid expiration time " + expirationTimeExpression + " and will not be registred!");
                return;
            }
        } else {
            this.log.warn("Route " + routedUrl + " has no expiration time and will not be registred!");
            return;
        }
        boolean fullUrl = storageObject.getBoolean(FULL_URL, Boolean.valueOf(false));
        hook.setFullUrl(fullUrl);
        this.routeRepository.addRoute(routedUrl, this.createRoute(routedUrl, hook));
    }

    private Route createRoute(String urlPattern, HttpHook hook) {
        return new Route(this.vertx, this.storage, this.loggingResourceManager, this.monitoringHandler, this.userProfilePath, hook, urlPattern);
    }

    private String getRoutedUrlSegment(String requestUrl) {
        return requestUrl.substring(0, requestUrl.indexOf(HOOKS_ROUTE_URI_PART));
    }

    private String getMonitoredUrlSegment(String requestUrl) {
        return requestUrl.substring(0, requestUrl.indexOf(HOOKS_LISTENERS_URI_PART));
    }

    private String getListenerUrlSegment(String requestUrl) {
        String segment = requestUrl.substring(requestUrl.indexOf(HOOKS_LISTENERS_URI_PART));
        segment = segment.replace(HOOKS_LISTENERS_URI_PART, "");
        return segment;
    }

    private boolean isHookListenerUnregistration(HttpServerRequest request) {
        return request.uri().contains(HOOKS_LISTENERS_URI_PART) && HttpMethod.DELETE == request.method();
    }

    private boolean isHookListenerRegistration(HttpServerRequest request) {
        return request.uri().contains(HOOKS_LISTENERS_URI_PART) && HttpMethod.PUT == request.method();
    }

    private boolean isHookRouteRegistration(HttpServerRequest request) {
        return request.uri().contains(HOOKS_ROUTE_URI_PART) && HttpMethod.PUT == request.method();
    }

    private boolean isHookRouteUnregistration(HttpServerRequest request) {
        return request.uri().contains(HOOKS_ROUTE_URI_PART) && HttpMethod.DELETE == request.method();
    }
}

