/*
 * Decompiled with CFR 0.152.
 */
package org.swisspush.gateleen.validation;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Joiner;
import com.networknt.schema.JsonSchema;
import com.networknt.schema.JsonSchemaFactory;
import com.networknt.schema.SpecVersion;
import com.networknt.schema.ValidationMessage;
import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.Json;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Optional;
import java.util.Set;
import org.slf4j.Logger;
import org.swisspush.gateleen.core.http.RequestLoggerFactory;
import org.swisspush.gateleen.core.json.JsonUtil;
import org.swisspush.gateleen.core.storage.ResourceStorage;
import org.swisspush.gateleen.core.util.StringUtils;
import org.swisspush.gateleen.core.validation.ValidationResult;
import org.swisspush.gateleen.core.validation.ValidationStatus;
import org.swisspush.gateleen.validation.SchemaLocation;
import org.swisspush.gateleen.validation.ValidationSchemaProvider;

public class Validator {
    private static final String SCHEMA_DECLARATION = "http://json-schema.org/draft-04/schema#";
    private String schemaRoot;
    private ResourceStorage storage;
    private ValidationSchemaProvider schemaProvider;

    public Validator(ResourceStorage storage, String schemaRoot, ValidationSchemaProvider schemaProvider) {
        this.storage = storage;
        this.schemaRoot = schemaRoot;
        this.schemaProvider = schemaProvider;
    }

    public void validate(HttpServerRequest req, String type, Buffer jsonBuffer, SchemaLocation schemaLocation, Handler<ValidationResult> callback) {
        if (schemaLocation == null) {
            this.validate(req, type, jsonBuffer, callback);
        } else {
            Logger log = RequestLoggerFactory.getLogger(Validator.class, (HttpServerRequest)req);
            if (!req.path().startsWith(this.schemaRoot)) {
                log.debug("Validating request");
                this.schemaProvider.schemaFromLocation(schemaLocation).setHandler(event -> {
                    if (event.failed()) {
                        callback.handle((Object)new ValidationResult(ValidationStatus.COULD_NOT_VALIDATE, "Error while getting schema. Cause: " + event.cause().getMessage()));
                        return;
                    }
                    Optional schemaOptional = (Optional)event.result();
                    if (schemaOptional.isEmpty()) {
                        callback.handle((Object)new ValidationResult(ValidationStatus.COULD_NOT_VALIDATE, "No schema found in location " + schemaLocation.schemaLocation()));
                        return;
                    }
                    JsonSchema jsonSchema = (JsonSchema)schemaOptional.get();
                    Validator.performValidation(jsonSchema, log, schemaLocation.schemaLocation(), jsonBuffer, type, req.path(), callback);
                });
            }
        }
    }

    public void validate(HttpServerRequest req, String type, Buffer jsonBuffer, Handler<ValidationResult> callback) {
        Logger log = RequestLoggerFactory.getLogger(Validator.class, (HttpServerRequest)req);
        if (!req.path().startsWith(this.schemaRoot)) {
            log.debug("Validating request");
            this.doValidate(jsonBuffer, req.path(), this.schemaRoot, type, (req.path().replaceFirst("^/", "") + "/" + type).split("/"), log, callback);
        }
    }

    private void doValidate(Buffer jsonBuffer, String path, String base, String type, String[] segments, Logger log, Handler<ValidationResult> callback) {
        this.storage.get(base, buffer -> {
            if (buffer != null) {
                String[] newSegments;
                String dataString = buffer.toString();
                JsonObject data = new JsonObject(dataString);
                String[] stringArray = newSegments = segments.length > 0 && segments[0].equals("") ? Arrays.copyOfRange(segments, 1, segments.length) : segments;
                if (newSegments.length == 0) {
                    Validator.performValidation(dataString, data, log, base, jsonBuffer, type, path, callback);
                } else {
                    this.validateRecursively(data, newSegments, base, jsonBuffer, path, type, log, callback);
                }
            } else {
                log.warn("Could not get path {}", (Object)base);
                if (base.replaceFirst("/$", "").endsWith(path)) {
                    log.info("try again with lowdash instead of last segment (variableId)");
                    String baseWithLowDash = base.replaceFirst("[^/]*/$", "") + "_/";
                    this.doValidate(jsonBuffer, path, baseWithLowDash, type, segments, log, callback);
                } else {
                    callback.handle((Object)new ValidationResult(ValidationStatus.COULD_NOT_VALIDATE, "Could not get path " + base + " (_ was used to try for schemas with variable ids)"));
                }
            }
        });
    }

    private void validateRecursively(JsonObject data, String[] newSegments, String base, Buffer jsonBuffer, String path, String type, Logger log, Handler<ValidationResult> callback) {
        Object newBase = base;
        String baseWithoutTrailingSlash = base.replaceFirst("/$", "");
        String arrayName = baseWithoutTrailingSlash.substring(baseWithoutTrailingSlash.lastIndexOf(47) + 1);
        JsonArray array = data.getJsonArray(arrayName);
        if (array != null && array.size() > 0) {
            if (array.contains((Object)(newSegments[0] + "/")) || array.contains((Object)newSegments[0])) {
                newBase = (String)newBase + newSegments[0] + (newSegments.length > 1 ? "/" : "");
                this.doValidate(jsonBuffer, path, (String)newBase, type, Arrays.copyOfRange(newSegments, 1, newSegments.length), log, callback);
            } else if (array.contains((Object)"_/") || array.contains((Object)"_")) {
                newBase = (String)newBase + newSegments[0] + "/";
                this.doValidate(jsonBuffer, path, (String)newBase, type, Arrays.copyOfRange(newSegments, 1, newSegments.length), log, callback);
            } else {
                String message = "No schema for " + path + " (" + type + ") [2]";
                log.warn(message);
                callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message));
            }
        } else {
            String message = "No schema for " + path + " (" + type + ") [3]";
            log.warn(message);
            callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message));
        }
    }

    public static ValidationResult validateStatic(Buffer dataToBeValidated, String schemaAsString, Logger log) {
        if (!JsonUtil.isValidJson((Buffer)dataToBeValidated)) {
            return new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, "Unable to parse json");
        }
        JsonObject schemaObject = new JsonObject(schemaAsString);
        if (SCHEMA_DECLARATION.equals(schemaObject.getString("$schema"))) {
            JsonSchema schema;
            try {
                schema = JsonSchemaFactory.getInstance((SpecVersion.VersionFlag)SpecVersion.VersionFlag.V4).getSchema(schemaAsString);
            }
            catch (RuntimeException e) {
                String message = "Cannot load schema";
                log.warn(message, (Throwable)e);
                return new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message);
            }
            try {
                JsonNode jsonNode = new ObjectMapper().readTree(dataToBeValidated.getBytes());
                Set valMsgs = schema.validate(jsonNode);
                if (valMsgs.isEmpty()) {
                    return new ValidationResult(ValidationStatus.VALIDATED_POSITIV);
                }
                JsonArray validationDetails = Validator.extractMessagesAsJson(valMsgs, log);
                return new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, "Validation failed", validationDetails);
            }
            catch (IOException e) {
                String message = "Cannot read JSON";
                log.warn(message, (Object)e.getMessage());
                return new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message);
            }
        }
        String message = "Invalid schema: Expected property '$schema' with content 'http://json-schema.org/draft-04/schema#'";
        log.warn(message);
        return new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message);
    }

    private static void performValidation(JsonSchema schema, Logger log, String base, Buffer jsonBuffer, String type, String path, Handler<ValidationResult> callback) {
        try {
            JsonNode jsonNode = new ObjectMapper().readTree(jsonBuffer.getBytes());
            if (jsonNode == null) {
                throw new IOException("no vaild JSON object: " + jsonBuffer.toString());
            }
            Set valMsgs = schema.validate(jsonNode);
            if (valMsgs.isEmpty()) {
                log.debug("Valid ({})", (Object)type);
                log.debug("Used schema: {}", (Object)base);
                callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_POSITIV));
            } else {
                JsonArray validationDetails = Validator.extractMessagesAsJson(valMsgs, log);
                String messages = StringUtils.getStringOrEmpty((String)Validator.extractMessages(valMsgs));
                StringBuilder msgBuilder = new StringBuilder();
                msgBuilder.append("Invalid JSON for ").append(path).append(" (").append(type).append("). Messages: ").append(messages).append(" | Report: ").append(Validator.getReportAsString(valMsgs));
                if (log.isDebugEnabled()) {
                    msgBuilder.append(" | Validated JSON: ").append(jsonBuffer.toString());
                }
                log.warn(msgBuilder.toString());
                callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, msgBuilder.toString(), validationDetails));
                log.warn("Used schema: {}", (Object)base);
            }
        }
        catch (IOException e) {
            String message = "Cannot read JSON  (" + type + ")";
            log.warn(message, (Object)e.getMessage());
            callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message));
        }
    }

    private static void performValidation(String dataString, JsonObject data, Logger log, String base, Buffer jsonBuffer, String type, String path, Handler<ValidationResult> callback) {
        if (SCHEMA_DECLARATION.equals(data.getString("$schema"))) {
            JsonSchema schema;
            try {
                schema = JsonSchemaFactory.getInstance((SpecVersion.VersionFlag)SpecVersion.VersionFlag.V4).getSchema(dataString);
            }
            catch (RuntimeException e) {
                String message = "Cannot load schema " + base;
                log.warn(message, (Throwable)e);
                callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message));
                return;
            }
            Validator.performValidation(schema, log, base, jsonBuffer, type, path, callback);
        } else {
            String message = "No schema for " + path + " (" + type + ") [1]";
            log.warn(message);
            callback.handle((Object)new ValidationResult(ValidationStatus.VALIDATED_NEGATIV, message));
        }
    }

    private static JsonArray extractMessagesAsJson(Set<ValidationMessage> valMsgs, Logger log) {
        JsonArray resultArray = new JsonArray();
        valMsgs.forEach(msg -> {
            if (log != null) {
                log.warn(msg.toString());
            }
            resultArray.add(JsonObject.mapFrom((Object)msg));
        });
        return resultArray;
    }

    private static String extractMessages(Set<ValidationMessage> valMsgs) {
        ArrayList messages = new ArrayList();
        valMsgs.forEach(msg -> {
            if (StringUtils.isNotEmpty((CharSequence)msg.getMessage())) {
                messages.add(msg.getMessage());
            }
        });
        if (messages.isEmpty()) {
            return null;
        }
        Joiner joiner = Joiner.on((String)"; ").skipNulls();
        return joiner.join(messages);
    }

    private static String getReportAsString(Set<ValidationMessage> valMsgs) {
        ArrayList messages = new ArrayList();
        valMsgs.forEach(msg -> {
            if (StringUtils.isNotEmpty((CharSequence)msg.getMessage())) {
                messages.add(Json.encodePrettily((Object)msg));
            }
        });
        if (messages.isEmpty()) {
            return "no report available";
        }
        Joiner joiner = Joiner.on((String)"; ").skipNulls();
        return joiner.join(messages);
    }
}

