package org.swisspush.kobuka.client.base;

import java.lang.Boolean;
import java.lang.Class;
import java.lang.Double;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Short;
import java.lang.String;
import java.util.List;
import org.apache.kafka.common.config.types.Password;

public interface ProducerConfigFields<T extends ProducerConfigFields<T>> {
  /**
   * <b>bootstrap.servers</b><p>
   * A list of host/port pairs to use for establishing the initial connection to the Kafka cluster.<p>The client will make use of all servers irrespective of which servers are specified here for bootstrapping&mdash;this list only impacts the initial hosts used to discover the full set of servers.<p>This list should be in the form <code>host1:port1,host2:port2,...</code>.<p>Since these servers are just used for the initial connection to discover the full cluster membership (which may change dynamically), this list need not contain the full set of servers (you may want more than one, though, in case a server is down).
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> high
   */
  T bootstrapServers(List<String> value);

  /**
   * <b>bootstrap.servers</b><p>
   * A list of host/port pairs to use for establishing the initial connection to the Kafka cluster.<p>The client will make use of all servers irrespective of which servers are specified here for bootstrapping&mdash;this list only impacts the initial hosts used to discover the full set of servers.<p>This list should be in the form <code>host1:port1,host2:port2,...</code>.<p>Since these servers are just used for the initial connection to discover the full cluster membership (which may change dynamically), this list need not contain the full set of servers (you may want more than one, though, in case a server is down).
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> high
   */
  T bootstrapServers(String value);

  /**
   * <b>client.dns.lookup</b><p>
   * Controls how the client uses DNS lookups.<p>If set to <code>use_all_dns_ips</code>, connect to each returned IP address in sequence until a successful connection is established.<p>After a disconnection, the next IP is used.<p>Once all IPs have been used once, the client resolves the IP(s) from the hostname again (both the JVM and the OS cache DNS name lookups, however).<p>If set to <code>resolve_canonical_bootstrap_servers_only</code>, resolve each bootstrap address into a list of canonical names.<p>After the bootstrap phase, this behaves the same as <code>use_all_dns_ips</code>.<p>If set to <code>default</code> (deprecated), attempt to connect to the first IP address returned by the lookup, even if the lookup returns multiple IP addresses.
   * <p><b>Default:</b> use_all_dns_ips
   * <p><b>Valid Values:</b> [default, use_all_dns_ips, resolve_canonical_bootstrap_servers_only]
   * <p><b>Importance:</b> medium
   */
  T clientDnsLookup(String value);

  /**
   * <b>buffer.memory</b><p>
   * The total bytes of memory the producer can use to buffer records waiting to be sent to the server.<p>If records are sent faster than they can be delivered to the server the producer will block for <code>max.block.ms</code> after which it will throw an exception.<p>This setting should correspond roughly to the total memory the producer will use, but is not a hard bound since not all memory the producer uses is used for buffering.<p>Some additional memory will be used for compression (if compression is enabled) as well as for maintaining in-flight requests.
   * <p><b>Default:</b> 33554432
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> high
   */
  T bufferMemory(Long value);

  /**
   * <b>retries</b><p>
   * Setting a value greater than zero will cause the client to resend any record whose send fails with a potentially transient error.<p>Note that this retry is no different than if the client resent the record upon receiving the error.<p>Allowing retries without setting <code>max.in.flight.requests.per.connection</code> to 1 will potentially change the ordering of records because if two batches are sent to a single partition, and the first fails and is retried but the second succeeds, then the records in the second batch may appear first.<p>Note additionally that produce requests will be failed before the number of retries has been exhausted if the timeout configured by <code>delivery.timeout.ms</code> expires first before successful acknowledgement.<p>Users should generally prefer to leave this config unset and instead use <code>delivery.timeout.ms</code> to control retry behavior.
   * <p><b>Default:</b> 2147483647
   * <p><b>Valid Values:</b> [0,...,2147483647]
   * <p><b>Importance:</b> high
   */
  T retries(Integer value);

  /**
   * <b>acks</b><p>
   * The number of acknowledgments the producer requires the leader to have received before considering a request complete.<p>This controls the  durability of records that are sent.<p>The following settings are allowed:  <ul> <li><code>acks=0</code> If set to zero then the producer will not wait for any acknowledgment from the server at all.<p>The record will be immediately added to the socket buffer and considered sent.<p>No guarantee can be made that the server has received the record in this case, and the <code>retries</code> configuration will not take effect (as the client won't generally know of any failures).<p>The offset given back for each record will always be set to <code>-1</code>.<p><li><code>acks=1</code> This will mean the leader will write the record to its local log but will respond without awaiting full acknowledgement from all followers.<p>In this case should the leader fail immediately after acknowledging the record but before the followers have replicated it then the record will be lost.<p><li><code>acks=all</code> This means the leader will wait for the full set of in-sync replicas to acknowledge the record.<p>This guarantees that the record will not be lost as long as at least one in-sync replica remains alive.<p>This is the strongest available guarantee.<p>This is equivalent to the acks=-1 setting.</ul>
   * <p><b>Default:</b> 1
   * <p><b>Valid Values:</b> [all, -1, 0, 1]
   * <p><b>Importance:</b> high
   */
  T acks(String value);

  /**
   * <b>compression.type</b><p>
   * The compression type for all data generated by the producer.<p>The default is none (i.e.<p>no compression).<p>Valid  values are <code>none</code>, <code>gzip</code>, <code>snappy</code>, <code>lz4</code>, or <code>zstd</code>.<p>Compression is of full batches of data, so the efficacy of batching will also impact the compression ratio (more batching means better compression).
   * <p><b>Default:</b> none
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T compressionType(String value);

  /**
   * <b>batch.size</b><p>
   * The producer will attempt to batch records together into fewer requests whenever multiple records are being sent to the same partition.<p>This helps performance on both the client and the server.<p>This configuration controls the default batch size in bytes.<p><p>No attempt will be made to batch records larger than this size.<p><p>Requests sent to brokers will contain multiple batches, one for each partition with data available to be sent.<p><p>A small batch size will make batching less common and may reduce throughput (a batch size of zero will disable batching entirely).<p>A very large batch size may use memory a bit more wastefully as we will always allocate a buffer of the specified batch size in anticipation of additional records.
   * <p><b>Default:</b> 16384
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T batchSize(Integer value);

  /**
   * <b>linger.ms</b><p>
   * The producer groups together any records that arrive in between request transmissions into a single batched request.<p>Normally this occurs only under load when records arrive faster than they can be sent out.<p>However in some circumstances the client may want to reduce the number of requests even under moderate load.<p>This setting accomplishes this by adding a small amount of artificial delay&mdash;that is, rather than immediately sending out a record the producer will wait for up to the given delay to allow other records to be sent so that the sends can be batched together.<p>This can be thought of as analogous to Nagle's algorithm in TCP.<p>This setting gives the upper bound on the delay for batching: once we get <code>batch.size</code> worth of records for a partition it will be sent immediately regardless of this setting, however if we have fewer than this many bytes accumulated for this partition we will 'linger' for the specified time waiting for more records to show up.<p>This setting defaults to 0 (i.e.<p>no delay).<p>Setting <code>linger.ms=5</code>, for example, would have the effect of reducing the number of requests sent but would add up to 5ms of latency to records sent in the absence of load.
   * <p><b>Default:</b> 0
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T lingerMs(Long value);

  /**
   * <b>delivery.timeout.ms</b><p>
   * An upper bound on the time to report success or failure after a call to <code>send()</code> returns.<p>This limits the total time that a record will be delayed prior to sending, the time to await acknowledgement from the broker (if expected), and the time allowed for retriable send failures.<p>The producer may report failure to send a record earlier than this config if either an unrecoverable error is encountered, the retries have been exhausted, or the record is added to a batch which reached an earlier delivery expiration deadline.<p>The value of this config should be greater than or equal to the sum of <code>request.timeout.ms</code> and <code>linger.ms</code>.
   * <p><b>Default:</b> 120000 (2 minutes)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T deliveryTimeoutMs(Integer value);

  /**
   * <b>client.id</b><p>
   * An id string to pass to the server when making requests.<p>The purpose of this is to be able to track the source of requests beyond just ip/port by allowing a logical application name to be included in server-side request logging.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T clientId(String value);

  /**
   * <b>send.buffer.bytes</b><p>
   * The size of the TCP send buffer (SO_SNDBUF) to use when sending data.<p>If the value is -1, the OS default will be used.
   * <p><b>Default:</b> 131072 (128 kibibytes)
   * <p><b>Valid Values:</b> [-1,...]
   * <p><b>Importance:</b> medium
   */
  T sendBufferBytes(Integer value);

  /**
   * <b>receive.buffer.bytes</b><p>
   * The size of the TCP receive buffer (SO_RCVBUF) to use when reading data.<p>If the value is -1, the OS default will be used.
   * <p><b>Default:</b> 32768 (32 kibibytes)
   * <p><b>Valid Values:</b> [-1,...]
   * <p><b>Importance:</b> medium
   */
  T receiveBufferBytes(Integer value);

  /**
   * <b>max.request.size</b><p>
   * The maximum size of a request in bytes.<p>This setting will limit the number of record batches the producer will send in a single request to avoid sending huge requests.<p>This is also effectively a cap on the maximum uncompressed record batch size.<p>Note that the server has its own cap on the record batch size (after compression if compression is enabled) which may be different from this.
   * <p><b>Default:</b> 1048576
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T maxRequestSize(Integer value);

  /**
   * <b>reconnect.backoff.ms</b><p>
   * The base amount of time to wait before attempting to reconnect to a given host.<p>This avoids repeatedly connecting to a host in a tight loop.<p>This backoff applies to all connection attempts by the client to a broker.
   * <p><b>Default:</b> 50
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T reconnectBackoffMs(Long value);

  /**
   * <b>reconnect.backoff.max.ms</b><p>
   * The maximum amount of time in milliseconds to wait when reconnecting to a broker that has repeatedly failed to connect.<p>If provided, the backoff per host will increase exponentially for each consecutive connection failure, up to this maximum.<p>After calculating the backoff increase, 20% random jitter is added to avoid connection storms.
   * <p><b>Default:</b> 1000 (1 second)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T reconnectBackoffMaxMs(Long value);

  /**
   * <b>retry.backoff.ms</b><p>
   * The amount of time to wait before attempting to retry a failed request to a given topic partition.<p>This avoids repeatedly sending requests in a tight loop under some failure scenarios.
   * <p><b>Default:</b> 100
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T retryBackoffMs(Long value);

  /**
   * <b>max.block.ms</b><p>
   * The configuration controls how long the <code>KafkaProducer</code>'s <code>send()</code>, <code>partitionsFor()</code>, <code>initTransactions()</code>, <code>sendOffsetsToTransaction()</code>, <code>commitTransaction()</code> and <code>abortTransaction()</code> methods will block.<p>For <code>send()</code> this timeout bounds the total time waiting for both metadata fetch and buffer allocation (blocking in the user-supplied serializers or partitioner is not counted against this timeout).<p>For <code>partitionsFor()</code> this timeout bounds the time spent waiting for metadata if it is unavailable.<p>The transaction-related methods always block, but may timeout if the transaction coordinator could not be discovered or did not respond within the timeout.
   * <p><b>Default:</b> 60000 (1 minute)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T maxBlockMs(Long value);

  /**
   * <b>request.timeout.ms</b><p>
   * The configuration controls the maximum amount of time the client will wait for the response of a request.<p>If the response is not received before the timeout elapses the client will resend the request if necessary or fail the request if retries are exhausted.<p>This should be larger than <code>replica.lag.time.max.ms</code> (a broker configuration) to reduce the possibility of message duplication due to unnecessary producer retries.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T requestTimeoutMs(Integer value);

  /**
   * <b>metadata.max.age.ms</b><p>
   * The period of time in milliseconds after which we force a refresh of metadata even if we haven't seen any partition leadership changes to proactively discover any new brokers or partitions.
   * <p><b>Default:</b> 300000 (5 minutes)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T metadataMaxAgeMs(Long value);

  /**
   * <b>metadata.max.idle.ms</b><p>
   * Controls how long the producer will cache metadata for a topic that's idle.<p>If the elapsed time since a topic was last produced to exceeds the metadata idle duration, then the topic's metadata is forgotten and the next access to it will force a metadata fetch request.
   * <p><b>Default:</b> 300000 (5 minutes)
   * <p><b>Valid Values:</b> [5000,...]
   * <p><b>Importance:</b> low
   */
  T metadataMaxIdleMs(Long value);

  /**
   * <b>metrics.sample.window.ms</b><p>
   * The window of time a metrics sample is computed over.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T metricsSampleWindowMs(Long value);

  /**
   * <b>metrics.num.samples</b><p>
   * The number of samples maintained to compute metrics.
   * <p><b>Default:</b> 2
   * <p><b>Valid Values:</b> [1,...]
   * <p><b>Importance:</b> low
   */
  T metricsNumSamples(Integer value);

  /**
   * <b>metrics.recording.level</b><p>
   * The highest recording level for metrics.
   * <p><b>Default:</b> INFO
   * <p><b>Valid Values:</b> [INFO, DEBUG, TRACE]
   * <p><b>Importance:</b> low
   */
  T metricsRecordingLevel(String value);

  /**
   * <b>metric.reporters</b><p>
   * A list of classes to use as metrics reporters.<p>Implementing the <code>org.apache.kafka.common.metrics.MetricsReporter</code> interface allows plugging in classes that will be notified of new metric creation.<p>The JmxReporter is always included to register JMX statistics.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T metricReporters(List<String> value);

  /**
   * <b>metric.reporters</b><p>
   * A list of classes to use as metrics reporters.<p>Implementing the <code>org.apache.kafka.common.metrics.MetricsReporter</code> interface allows plugging in classes that will be notified of new metric creation.<p>The JmxReporter is always included to register JMX statistics.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T metricReporters(String value);

  /**
   * <b>max.in.flight.requests.per.connection</b><p>
   * The maximum number of unacknowledged requests the client will send on a single connection before blocking.<p>Note that if this setting is set to be greater than 1 and there are failed sends, there is a risk of message re-ordering due to retries (i.e., if retries are enabled).
   * <p><b>Default:</b> 5
   * <p><b>Valid Values:</b> [1,...]
   * <p><b>Importance:</b> low
   */
  T maxInFlightRequestsPerConnection(Integer value);

  /**
   * <b>key.serializer</b><p>
   * Serializer class for key that implements the <code>org.apache.kafka.common.serialization.Serializer</code> interface.
   * <p><b>Default:</b> 
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T keySerializer(Class value);

  /**
   * <b>value.serializer</b><p>
   * Serializer class for value that implements the <code>org.apache.kafka.common.serialization.Serializer</code> interface.
   * <p><b>Default:</b> 
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T valueSerializer(Class value);

  /**
   * <b>socket.connection.setup.timeout.ms</b><p>
   * The amount of time the client will wait for the socket connection to be established.<p>If the connection is not built before the timeout elapses, clients will close the socket channel.
   * <p><b>Default:</b> 10000 (10 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T socketConnectionSetupTimeoutMs(Long value);

  /**
   * <b>socket.connection.setup.timeout.max.ms</b><p>
   * The maximum amount of time the client will wait for the socket connection to be established.<p>The connection setup timeout will increase exponentially for each consecutive connection failure up to this maximum.<p>To avoid connection storms, a randomization factor of 0.2 will be applied to the timeout resulting in a random range between 20% below and 20% above the computed value.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T socketConnectionSetupTimeoutMaxMs(Long value);

  /**
   * <b>connections.max.idle.ms</b><p>
   * Close idle connections after the number of milliseconds specified by this config.
   * <p><b>Default:</b> 540000 (9 minutes)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T connectionsMaxIdleMs(Long value);

  /**
   * <b>partitioner.class</b><p>
   * Partitioner class that implements the <code>org.apache.kafka.clients.producer.Partitioner</code> interface.
   * <p><b>Default:</b> org.apache.kafka.clients.producer.internals.DefaultPartitioner
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T partitionerClass(Class value);

  /**
   * <b>interceptor.classes</b><p>
   * A list of classes to use as interceptors.<p>Implementing the <code>org.apache.kafka.clients.producer.ProducerInterceptor</code> interface allows you to intercept (and possibly mutate) the records received by the producer before they are published to the Kafka cluster.<p>By default, there are no interceptors.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T interceptorClasses(List<String> value);

  /**
   * <b>interceptor.classes</b><p>
   * A list of classes to use as interceptors.<p>Implementing the <code>org.apache.kafka.clients.producer.ProducerInterceptor</code> interface allows you to intercept (and possibly mutate) the records received by the producer before they are published to the Kafka cluster.<p>By default, there are no interceptors.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T interceptorClasses(String value);

  /**
   * <b>security.protocol</b><p>
   * Protocol used to communicate with brokers.<p>Valid values are: PLAINTEXT, SSL, SASL_PLAINTEXT, SASL_SSL.
   * <p><b>Default:</b> PLAINTEXT
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T securityProtocol(String value);

  /**
   * <b>security.providers</b><p>
   * A list of configurable creator classes each returning a provider implementing security algorithms.<p>These classes should implement the <code>org.apache.kafka.common.security.auth.SecurityProviderCreator</code> interface.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T securityProviders(String value);

  /**
   * <b>ssl.protocol</b><p>
   * The SSL protocol used to generate the SSLContext.<p>The default is 'TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>This value should be fine for most use cases.<p>Allowed values in recent JVMs are 'TLSv1.2' and 'TLSv1.3'.<p>'TLS', 'TLSv1.1', 'SSL', 'SSLv2' and 'SSLv3' may be supported in older JVMs, but their usage is discouraged due to known security vulnerabilities.<p>With the default value for this config and 'ssl.enabled.protocols', clients will downgrade to 'TLSv1.2' if the server does not support 'TLSv1.3'.<p>If this config is set to 'TLSv1.2', clients will not use 'TLSv1.3' even if it is one of the values in ssl.enabled.protocols and the server only supports 'TLSv1.3'.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslProtocol(String value);

  /**
   * <b>ssl.provider</b><p>
   * The name of the security provider used for SSL connections.<p>Default value is the default security provider of the JVM.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslProvider(String value);

  /**
   * <b>ssl.cipher.suites</b><p>
   * A list of cipher suites.<p>This is a named combination of authentication, encryption, MAC and key exchange algorithm used to negotiate the security settings for a network connection using TLS or SSL network protocol.<p>By default all the available cipher suites are supported.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslCipherSuites(List<String> value);

  /**
   * <b>ssl.cipher.suites</b><p>
   * A list of cipher suites.<p>This is a named combination of authentication, encryption, MAC and key exchange algorithm used to negotiate the security settings for a network connection using TLS or SSL network protocol.<p>By default all the available cipher suites are supported.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslCipherSuites(String value);

  /**
   * <b>ssl.enabled.protocols</b><p>
   * The list of protocols enabled for SSL connections.<p>The default is 'TLSv1.2,TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>With the default value for Java 11, clients and servers will prefer TLSv1.3 if both support it and fallback to TLSv1.2 otherwise (assuming both support at least TLSv1.2).<p>This default should be fine for most cases.<p>Also see the config documentation for `ssl.protocol`.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslEnabledProtocols(List<String> value);

  /**
   * <b>ssl.enabled.protocols</b><p>
   * The list of protocols enabled for SSL connections.<p>The default is 'TLSv1.2,TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>With the default value for Java 11, clients and servers will prefer TLSv1.3 if both support it and fallback to TLSv1.2 otherwise (assuming both support at least TLSv1.2).<p>This default should be fine for most cases.<p>Also see the config documentation for `ssl.protocol`.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslEnabledProtocols(String value);

  /**
   * <b>ssl.keystore.type</b><p>
   * The file format of the key store file.<p>This is optional for client.
   * <p><b>Default:</b> JKS
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslKeystoreType(String value);

  /**
   * <b>ssl.keystore.location</b><p>
   * The location of the key store file.<p>This is optional for client and can be used for two-way authentication for client.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreLocation(String value);

  /**
   * <b>ssl.keystore.password</b><p>
   * The store password for the key store file.<p>This is optional for client and only needed if 'ssl.keystore.location' is configured.<p> Key store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystorePassword(Password value);

  /**
   * <b>ssl.keystore.password</b><p>
   * The store password for the key store file.<p>This is optional for client and only needed if 'ssl.keystore.location' is configured.<p> Key store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystorePassword(String value);

  /**
   * <b>ssl.key.password</b><p>
   * The password of the private key in the key store file orthe PEM key specified in `ssl.keystore.key'.<p>This is required for clients only if two-way authentication is configured.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeyPassword(Password value);

  /**
   * <b>ssl.key.password</b><p>
   * The password of the private key in the key store file orthe PEM key specified in `ssl.keystore.key'.<p>This is required for clients only if two-way authentication is configured.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeyPassword(String value);

  /**
   * <b>ssl.keystore.key</b><p>
   * Private key in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with PKCS#8 keys.<p>If the key is encrypted, key password must be specified using 'ssl.key.password'
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreKey(Password value);

  /**
   * <b>ssl.keystore.key</b><p>
   * Private key in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with PKCS#8 keys.<p>If the key is encrypted, key password must be specified using 'ssl.key.password'
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreKey(String value);

  /**
   * <b>ssl.keystore.certificate.chain</b><p>
   * Certificate chain in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with a list of X.509 certificates
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreCertificateChain(Password value);

  /**
   * <b>ssl.keystore.certificate.chain</b><p>
   * Certificate chain in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with a list of X.509 certificates
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreCertificateChain(String value);

  /**
   * <b>ssl.truststore.certificates</b><p>
   * Trusted certificates in the format specified by 'ssl.truststore.type'.<p>Default SSL engine factory supports only PEM format with X.509 certificates.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreCertificates(Password value);

  /**
   * <b>ssl.truststore.certificates</b><p>
   * Trusted certificates in the format specified by 'ssl.truststore.type'.<p>Default SSL engine factory supports only PEM format with X.509 certificates.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreCertificates(String value);

  /**
   * <b>ssl.truststore.type</b><p>
   * The file format of the trust store file.
   * <p><b>Default:</b> JKS
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslTruststoreType(String value);

  /**
   * <b>ssl.truststore.location</b><p>
   * The location of the trust store file.<p>
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreLocation(String value);

  /**
   * <b>ssl.truststore.password</b><p>
   * The password for the trust store file.<p>If a password is not set, trust store file configured will still be used, but integrity checking is disabled.<p>Trust store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststorePassword(Password value);

  /**
   * <b>ssl.truststore.password</b><p>
   * The password for the trust store file.<p>If a password is not set, trust store file configured will still be used, but integrity checking is disabled.<p>Trust store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststorePassword(String value);

  /**
   * <b>ssl.keymanager.algorithm</b><p>
   * The algorithm used by key manager factory for SSL connections.<p>Default value is the key manager factory algorithm configured for the Java Virtual Machine.
   * <p><b>Default:</b> SunX509
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslKeymanagerAlgorithm(String value);

  /**
   * <b>ssl.trustmanager.algorithm</b><p>
   * The algorithm used by trust manager factory for SSL connections.<p>Default value is the trust manager factory algorithm configured for the Java Virtual Machine.
   * <p><b>Default:</b> PKIX
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslTrustmanagerAlgorithm(String value);

  /**
   * <b>ssl.endpoint.identification.algorithm</b><p>
   * The endpoint identification algorithm to validate server hostname using server certificate.<p>
   * <p><b>Default:</b> https
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslEndpointIdentificationAlgorithm(String value);

  /**
   * <b>ssl.secure.random.implementation</b><p>
   * The SecureRandom PRNG implementation to use for SSL cryptography operations.<p>
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslSecureRandomImplementation(String value);

  /**
   * <b>ssl.engine.factory.class</b><p>
   * The class of type org.apache.kafka.common.security.auth.SslEngineFactory to provide SSLEngine objects.<p>Default value is org.apache.kafka.common.security.ssl.DefaultSslEngineFactory
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslEngineFactoryClass(Class value);

  /**
   * <b>sasl.kerberos.service.name</b><p>
   * The Kerberos principal name that Kafka runs as.<p>This can be defined either in Kafka's JAAS config or in Kafka's config.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslKerberosServiceName(String value);

  /**
   * <b>sasl.kerberos.kinit.cmd</b><p>
   * Kerberos kinit command path.
   * <p><b>Default:</b> /usr/bin/kinit
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosKinitCmd(String value);

  /**
   * <b>sasl.kerberos.ticket.renew.window.factor</b><p>
   * Login thread will sleep until the specified window factor of time from last refresh to ticket's expiry has been reached, at which time it will try to renew the ticket.
   * <p><b>Default:</b> 0.8
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosTicketRenewWindowFactor(Double value);

  /**
   * <b>sasl.kerberos.ticket.renew.jitter</b><p>
   * Percentage of random jitter added to the renewal time.
   * <p><b>Default:</b> 0.05
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosTicketRenewJitter(Double value);

  /**
   * <b>sasl.kerberos.min.time.before.relogin</b><p>
   * Login thread sleep time between refresh attempts.
   * <p><b>Default:</b> 60000
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosMinTimeBeforeRelogin(Long value);

  /**
   * <b>sasl.login.refresh.window.factor</b><p>
   * Login refresh thread will sleep until the specified window factor relative to the credential's lifetime has been reached, at which time it will try to refresh the credential.<p>Legal values are between 0.5 (50%) and 1.0 (100%) inclusive; a default value of 0.8 (80%) is used if no value is specified.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 0.8
   * <p><b>Valid Values:</b> [0.5,...,1.0]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshWindowFactor(Double value);

  /**
   * <b>sasl.login.refresh.window.jitter</b><p>
   * The maximum amount of random jitter relative to the credential's lifetime that is added to the login refresh thread's sleep time.<p>Legal values are between 0 and 0.25 (25%) inclusive; a default value of 0.05 (5%) is used if no value is specified.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 0.05
   * <p><b>Valid Values:</b> [0.0,...,0.25]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshWindowJitter(Double value);

  /**
   * <b>sasl.login.refresh.min.period.seconds</b><p>
   * The desired minimum time for the login refresh thread to wait before refreshing a credential, in seconds.<p>Legal values are between 0 and 900 (15 minutes); a default value of 60 (1 minute) is used if no value is specified.<p> This value and  sasl.login.refresh.buffer.seconds are both ignored if their sum exceeds the remaining lifetime of a credential.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 60
   * <p><b>Valid Values:</b> [0,...,900]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshMinPeriodSeconds(Short value);

  /**
   * <b>sasl.login.refresh.buffer.seconds</b><p>
   * The amount of buffer time before credential expiration to maintain when refreshing a credential, in seconds.<p>If a refresh would otherwise occur closer to expiration than the number of buffer seconds then the refresh will be moved up to maintain as much of the buffer time as possible.<p>Legal values are between 0 and 3600 (1 hour); a default value of  300 (5 minutes) is used if no value is specified.<p>This value and sasl.login.refresh.min.period.seconds are both ignored if their sum exceeds the remaining lifetime of a credential.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 300
   * <p><b>Valid Values:</b> [0,...,3600]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshBufferSeconds(Short value);

  /**
   * <b>sasl.mechanism</b><p>
   * SASL mechanism used for client connections.<p>This may be any mechanism for which a security provider is available.<p>GSSAPI is the default mechanism.
   * <p><b>Default:</b> GSSAPI
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslMechanism(String value);

  /**
   * <b>sasl.jaas.config</b><p>
   * JAAS login context parameters for SASL connections in the format used by JAAS configuration files.<p>JAAS configuration file format is described <a href="http://docs.oracle.com/javase/8/docs/technotes/guides/security/jgss/tutorials/LoginConfigFile.html">here</a>.<p>The format for the value is: <code>loginModuleClass controlFlag (optionName=optionValue)*;</code>.<p>For brokers, the config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.jaas.config=com.example.ScramLoginModule required;
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslJaasConfig(Password value);

  /**
   * <b>sasl.jaas.config</b><p>
   * JAAS login context parameters for SASL connections in the format used by JAAS configuration files.<p>JAAS configuration file format is described <a href="http://docs.oracle.com/javase/8/docs/technotes/guides/security/jgss/tutorials/LoginConfigFile.html">here</a>.<p>The format for the value is: <code>loginModuleClass controlFlag (optionName=optionValue)*;</code>.<p>For brokers, the config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.jaas.config=com.example.ScramLoginModule required;
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslJaasConfig(String value);

  /**
   * <b>sasl.client.callback.handler.class</b><p>
   * The fully qualified name of a SASL client callback handler class that implements the AuthenticateCallbackHandler interface.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslClientCallbackHandlerClass(Class value);

  /**
   * <b>sasl.login.callback.handler.class</b><p>
   * The fully qualified name of a SASL login callback handler class that implements the AuthenticateCallbackHandler interface.<p>For brokers, login callback handler config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.login.callback.handler.class=com.example.CustomScramLoginCallbackHandler
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslLoginCallbackHandlerClass(Class value);

  /**
   * <b>sasl.login.class</b><p>
   * The fully qualified name of a class that implements the Login interface.<p>For brokers, login config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.login.class=com.example.CustomScramLogin
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslLoginClass(Class value);

  /**
   * <b>enable.idempotence</b><p>
   * When set to 'true', the producer will ensure that exactly one copy of each message is written in the stream.<p>If 'false', producer retries due to broker failures, etc., may write duplicates of the retried message in the stream.<p>Note that enabling idempotence requires <code>max.in.flight.requests.per.connection</code> to be less than or equal to 5, <code>retries</code> to be greater than 0 and <code>acks</code> must be 'all'.<p>If these values are not explicitly set by the user, suitable values will be chosen.<p>If incompatible values are set, a <code>ConfigException</code> will be thrown.
   * <p><b>Default:</b> false
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T enableIdempotence(Boolean value);

  /**
   * <b>transaction.timeout.ms</b><p>
   * The maximum amount of time in ms that the transaction coordinator will wait for a transaction status update from the producer before proactively aborting the ongoing transaction.If this value is larger than the transaction.max.timeout.ms setting in the broker, the request will fail with a <code>InvalidTxnTimeoutException</code> error.
   * <p><b>Default:</b> 60000 (1 minute)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T transactionTimeoutMs(Integer value);

  /**
   * <b>transactional.id</b><p>
   * The TransactionalId to use for transactional delivery.<p>This enables reliability semantics which span multiple producer sessions since it allows the client to guarantee that transactions using the same TransactionalId have been completed prior to starting any new transactions.<p>If no TransactionalId is provided, then the producer is limited to idempotent delivery.<p>If a TransactionalId is configured, <code>enable.idempotence</code> is implied.<p>By default the TransactionId is not configured, which means transactions cannot be used.<p>Note that, by default, transactions require a cluster of at least three brokers which is the recommended setting for production; for development you can change this, by adjusting broker setting <code>transaction.state.log.replication.factor</code>.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> non-empty string
   * <p><b>Importance:</b> low
   */
  T transactionalId(String value);

  default T self() {
    return (T)this;
  }
}
