package org.swisspush.kobuka.client.base;

import java.lang.Boolean;
import java.lang.Class;
import java.lang.Double;
import java.lang.Integer;
import java.lang.Long;
import java.lang.Short;
import java.lang.String;
import java.util.List;
import org.apache.kafka.common.config.types.Password;

public interface ConsumerConfigFields<T extends ConsumerConfigFields<T>> {
  /**
   * <b>bootstrap.servers</b><p>
   * A list of host/port pairs to use for establishing the initial connection to the Kafka cluster.<p>The client will make use of all servers irrespective of which servers are specified here for bootstrapping&mdash;this list only impacts the initial hosts used to discover the full set of servers.<p>This list should be in the form <code>host1:port1,host2:port2,...</code>.<p>Since these servers are just used for the initial connection to discover the full cluster membership (which may change dynamically), this list need not contain the full set of servers (you may want more than one, though, in case a server is down).
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> high
   */
  T bootstrapServers(List<String> value);

  /**
   * <b>bootstrap.servers</b><p>
   * A list of host/port pairs to use for establishing the initial connection to the Kafka cluster.<p>The client will make use of all servers irrespective of which servers are specified here for bootstrapping&mdash;this list only impacts the initial hosts used to discover the full set of servers.<p>This list should be in the form <code>host1:port1,host2:port2,...</code>.<p>Since these servers are just used for the initial connection to discover the full cluster membership (which may change dynamically), this list need not contain the full set of servers (you may want more than one, though, in case a server is down).
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> high
   */
  T bootstrapServers(String value);

  /**
   * <b>client.dns.lookup</b><p>
   * Controls how the client uses DNS lookups.<p>If set to <code>use_all_dns_ips</code>, connect to each returned IP address in sequence until a successful connection is established.<p>After a disconnection, the next IP is used.<p>Once all IPs have been used once, the client resolves the IP(s) from the hostname again (both the JVM and the OS cache DNS name lookups, however).<p>If set to <code>resolve_canonical_bootstrap_servers_only</code>, resolve each bootstrap address into a list of canonical names.<p>After the bootstrap phase, this behaves the same as <code>use_all_dns_ips</code>.
   * <p><b>Default:</b> use_all_dns_ips
   * <p><b>Valid Values:</b> [use_all_dns_ips, resolve_canonical_bootstrap_servers_only]
   * <p><b>Importance:</b> medium
   */
  T clientDnsLookup(String value);

  /**
   * <b>group.id</b><p>
   * A unique string that identifies the consumer group this consumer belongs to.<p>This property is required if the consumer uses either the group management functionality by using <code>subscribe(topic)</code> or the Kafka-based offset management strategy.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T groupId(String value);

  /**
   * <b>group.instance.id</b><p>
   * A unique identifier of the consumer instance provided by the end user.<p>Only non-empty strings are permitted.<p>If set, the consumer is treated as a static member, which means that only one instance with this ID is allowed in the consumer group at any time.<p>This can be used in combination with a larger session timeout to avoid group rebalances caused by transient unavailability (e.g.<p>process restarts).<p>If not set, the consumer will join the group as a dynamic member, which is the traditional behavior.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T groupInstanceId(String value);

  /**
   * <b>session.timeout.ms</b><p>
   * The timeout used to detect client failures when using Kafka's group management facility.<p>The client sends periodic heartbeats to indicate its liveness to the broker.<p>If no heartbeats are received by the broker before the expiration of this session timeout, then the broker will remove this client from the group and initiate a rebalance.<p>Note that the value must be in the allowable range as configured in the broker configuration by <code>group.min.session.timeout.ms</code> and <code>group.max.session.timeout.ms</code>.
   * <p><b>Default:</b> 45000 (45 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sessionTimeoutMs(Integer value);

  /**
   * <b>heartbeat.interval.ms</b><p>
   * The expected time between heartbeats to the consumer coordinator when using Kafka's group management facilities.<p>Heartbeats are used to ensure that the consumer's session stays active and to facilitate rebalancing when new consumers join or leave the group.<p>The value must be set lower than <code>session.timeout.ms</code>, but typically should be set no higher than 1/3 of that value.<p>It can be adjusted even lower to control the expected time for normal rebalances.
   * <p><b>Default:</b> 3000 (3 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T heartbeatIntervalMs(Integer value);

  /**
   * <b>partition.assignment.strategy</b><p>
   * A list of class names or class types, ordered by preference, of supported partition assignment strategies that the client will use to distribute partition ownership amongst consumer instances when group management is used.<p>Available options are:<ul><li><code>org.apache.kafka.clients.consumer.RangeAssignor</code>: Assigns partitions on a per-topic basis.</li><li><code>org.apache.kafka.clients.consumer.RoundRobinAssignor</code>: Assigns partitions to consumers in a round-robin fashion.</li><li><code>org.apache.kafka.clients.consumer.StickyAssignor</code>: Guarantees an assignment that is maximally balanced while preserving as many existing partition assignments as possible.</li><li><code>org.apache.kafka.clients.consumer.CooperativeStickyAssignor</code>: Follows the same StickyAssignor logic, but allows for cooperative rebalancing.</li></ul><p>The default assignor is [RangeAssignor, CooperativeStickyAssignor], which will use the RangeAssignor by default, but allows upgrading to the CooperativeStickyAssignor with just a single rolling bounce that removes the RangeAssignor from the list.</p><p>Implementing the <code>org.apache.kafka.clients.consumer.ConsumerPartitionAssignor</code> interface allows you to plug in a custom assignment strategy.</p>
   * <p><b>Default:</b> class org.apache.kafka.clients.consumer.RangeAssignor,class org.apache.kafka.clients.consumer.CooperativeStickyAssignor
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> medium
   */
  T partitionAssignmentStrategy(List<String> value);

  /**
   * <b>partition.assignment.strategy</b><p>
   * A list of class names or class types, ordered by preference, of supported partition assignment strategies that the client will use to distribute partition ownership amongst consumer instances when group management is used.<p>Available options are:<ul><li><code>org.apache.kafka.clients.consumer.RangeAssignor</code>: Assigns partitions on a per-topic basis.</li><li><code>org.apache.kafka.clients.consumer.RoundRobinAssignor</code>: Assigns partitions to consumers in a round-robin fashion.</li><li><code>org.apache.kafka.clients.consumer.StickyAssignor</code>: Guarantees an assignment that is maximally balanced while preserving as many existing partition assignments as possible.</li><li><code>org.apache.kafka.clients.consumer.CooperativeStickyAssignor</code>: Follows the same StickyAssignor logic, but allows for cooperative rebalancing.</li></ul><p>The default assignor is [RangeAssignor, CooperativeStickyAssignor], which will use the RangeAssignor by default, but allows upgrading to the CooperativeStickyAssignor with just a single rolling bounce that removes the RangeAssignor from the list.</p><p>Implementing the <code>org.apache.kafka.clients.consumer.ConsumerPartitionAssignor</code> interface allows you to plug in a custom assignment strategy.</p>
   * <p><b>Default:</b> class org.apache.kafka.clients.consumer.RangeAssignor,class org.apache.kafka.clients.consumer.CooperativeStickyAssignor
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> medium
   */
  T partitionAssignmentStrategy(String value);

  /**
   * <b>metadata.max.age.ms</b><p>
   * The period of time in milliseconds after which we force a refresh of metadata even if we haven't seen any partition leadership changes to proactively discover any new brokers or partitions.
   * <p><b>Default:</b> 300000 (5 minutes)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T metadataMaxAgeMs(Long value);

  /**
   * <b>enable.auto.commit</b><p>
   * If true the consumer's offset will be periodically committed in the background.
   * <p><b>Default:</b> true
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T enableAutoCommit(Boolean value);

  /**
   * <b>auto.commit.interval.ms</b><p>
   * The frequency in milliseconds that the consumer offsets are auto-committed to Kafka if <code>enable.auto.commit</code> is set to <code>true</code>.
   * <p><b>Default:</b> 5000 (5 seconds)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T autoCommitIntervalMs(Integer value);

  /**
   * <b>client.id</b><p>
   * An id string to pass to the server when making requests.<p>The purpose of this is to be able to track the source of requests beyond just ip/port by allowing a logical application name to be included in server-side request logging.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T clientId(String value);

  /**
   * <b>client.rack</b><p>
   * A rack identifier for this client.<p>This can be any string value which indicates where this client is physically located.<p>It corresponds with the broker config 'broker.rack'
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T clientRack(String value);

  /**
   * <b>max.partition.fetch.bytes</b><p>
   * The maximum amount of data per-partition the server will return.<p>Records are fetched in batches by the consumer.<p>If the first record batch in the first non-empty partition of the fetch is larger than this limit, the batch will still be returned to ensure that the consumer can make progress.<p>The maximum record batch size accepted by the broker is defined via <code>message.max.bytes</code> (broker config) or <code>max.message.bytes</code> (topic config).<p>See fetch.max.bytes for limiting the consumer request size.
   * <p><b>Default:</b> 1048576 (1 mebibyte)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> high
   */
  T maxPartitionFetchBytes(Integer value);

  /**
   * <b>send.buffer.bytes</b><p>
   * The size of the TCP send buffer (SO_SNDBUF) to use when sending data.<p>If the value is -1, the OS default will be used.
   * <p><b>Default:</b> 131072 (128 kibibytes)
   * <p><b>Valid Values:</b> [-1,...]
   * <p><b>Importance:</b> medium
   */
  T sendBufferBytes(Integer value);

  /**
   * <b>receive.buffer.bytes</b><p>
   * The size of the TCP receive buffer (SO_RCVBUF) to use when reading data.<p>If the value is -1, the OS default will be used.
   * <p><b>Default:</b> 65536 (64 kibibytes)
   * <p><b>Valid Values:</b> [-1,...]
   * <p><b>Importance:</b> medium
   */
  T receiveBufferBytes(Integer value);

  /**
   * <b>fetch.min.bytes</b><p>
   * The minimum amount of data the server should return for a fetch request.<p>If insufficient data is available the request will wait for that much data to accumulate before answering the request.<p>The default setting of 1 byte means that fetch requests are answered as soon as a single byte of data is available or the fetch request times out waiting for data to arrive.<p>Setting this to something greater than 1 will cause the server to wait for larger amounts of data to accumulate which can improve server throughput a bit at the cost of some additional latency.
   * <p><b>Default:</b> 1
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> high
   */
  T fetchMinBytes(Integer value);

  /**
   * <b>fetch.max.bytes</b><p>
   * The maximum amount of data the server should return for a fetch request.<p>Records are fetched in batches by the consumer, and if the first record batch in the first non-empty partition of the fetch is larger than this value, the record batch will still be returned to ensure that the consumer can make progress.<p>As such, this is not a absolute maximum.<p>The maximum record batch size accepted by the broker is defined via <code>message.max.bytes</code> (broker config) or <code>max.message.bytes</code> (topic config).<p>Note that the consumer performs multiple fetches in parallel.
   * <p><b>Default:</b> 52428800 (50 mebibytes)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T fetchMaxBytes(Integer value);

  /**
   * <b>fetch.max.wait.ms</b><p>
   * The maximum amount of time the server will block before answering the fetch request if there isn't sufficient data to immediately satisfy the requirement given by fetch.min.bytes.
   * <p><b>Default:</b> 500
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T fetchMaxWaitMs(Integer value);

  /**
   * <b>reconnect.backoff.ms</b><p>
   * The base amount of time to wait before attempting to reconnect to a given host.<p>This avoids repeatedly connecting to a host in a tight loop.<p>This backoff applies to all connection attempts by the client to a broker.
   * <p><b>Default:</b> 50
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T reconnectBackoffMs(Long value);

  /**
   * <b>reconnect.backoff.max.ms</b><p>
   * The maximum amount of time in milliseconds to wait when reconnecting to a broker that has repeatedly failed to connect.<p>If provided, the backoff per host will increase exponentially for each consecutive connection failure, up to this maximum.<p>After calculating the backoff increase, 20% random jitter is added to avoid connection storms.
   * <p><b>Default:</b> 1000 (1 second)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T reconnectBackoffMaxMs(Long value);

  /**
   * <b>retry.backoff.ms</b><p>
   * The amount of time to wait before attempting to retry a failed request to a given topic partition.<p>This avoids repeatedly sending requests in a tight loop under some failure scenarios.
   * <p><b>Default:</b> 100
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T retryBackoffMs(Long value);

  /**
   * <b>auto.offset.reset</b><p>
   * What to do when there is no initial offset in Kafka or if the current offset does not exist any more on the server (e.g.<p>because that data has been deleted): <ul><li>earliest: automatically reset the offset to the earliest offset<li>latest: automatically reset the offset to the latest offset</li><li>none: throw exception to the consumer if no previous offset is found for the consumer's group</li><li>anything else: throw exception to the consumer.</li></ul>
   * <p><b>Default:</b> latest
   * <p><b>Valid Values:</b> [latest, earliest, none]
   * <p><b>Importance:</b> medium
   */
  T autoOffsetReset(String value);

  /**
   * <b>check.crcs</b><p>
   * Automatically check the CRC32 of the records consumed.<p>This ensures no on-the-wire or on-disk corruption to the messages occurred.<p>This check adds some overhead, so it may be disabled in cases seeking extreme performance.
   * <p><b>Default:</b> true
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T checkCrcs(Boolean value);

  /**
   * <b>metrics.sample.window.ms</b><p>
   * The window of time a metrics sample is computed over.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> low
   */
  T metricsSampleWindowMs(Long value);

  /**
   * <b>metrics.num.samples</b><p>
   * The number of samples maintained to compute metrics.
   * <p><b>Default:</b> 2
   * <p><b>Valid Values:</b> [1,...]
   * <p><b>Importance:</b> low
   */
  T metricsNumSamples(Integer value);

  /**
   * <b>metrics.recording.level</b><p>
   * The highest recording level for metrics.
   * <p><b>Default:</b> INFO
   * <p><b>Valid Values:</b> [INFO, DEBUG, TRACE]
   * <p><b>Importance:</b> low
   */
  T metricsRecordingLevel(String value);

  /**
   * <b>metric.reporters</b><p>
   * A list of classes to use as metrics reporters.<p>Implementing the <code>org.apache.kafka.common.metrics.MetricsReporter</code> interface allows plugging in classes that will be notified of new metric creation.<p>The JmxReporter is always included to register JMX statistics.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T metricReporters(List<String> value);

  /**
   * <b>metric.reporters</b><p>
   * A list of classes to use as metrics reporters.<p>Implementing the <code>org.apache.kafka.common.metrics.MetricsReporter</code> interface allows plugging in classes that will be notified of new metric creation.<p>The JmxReporter is always included to register JMX statistics.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T metricReporters(String value);

  /**
   * <b>key.deserializer</b><p>
   * Deserializer class for key that implements the <code>org.apache.kafka.common.serialization.Deserializer</code> interface.
   * <p><b>Default:</b> 
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T keyDeserializer(Class value);

  /**
   * <b>value.deserializer</b><p>
   * Deserializer class for value that implements the <code>org.apache.kafka.common.serialization.Deserializer</code> interface.
   * <p><b>Default:</b> 
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T valueDeserializer(Class value);

  /**
   * <b>request.timeout.ms</b><p>
   * The configuration controls the maximum amount of time the client will wait for the response of a request.<p>If the response is not received before the timeout elapses the client will resend the request if necessary or fail the request if retries are exhausted.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T requestTimeoutMs(Integer value);

  /**
   * <b>default.api.timeout.ms</b><p>
   * Specifies the timeout (in milliseconds) for client APIs.<p>This configuration is used as the default timeout for all client operations that do not specify a <code>timeout</code> parameter.
   * <p><b>Default:</b> 60000 (1 minute)
   * <p><b>Valid Values:</b> [0,...]
   * <p><b>Importance:</b> medium
   */
  T defaultApiTimeoutMs(Integer value);

  /**
   * <b>socket.connection.setup.timeout.ms</b><p>
   * The amount of time the client will wait for the socket connection to be established.<p>If the connection is not built before the timeout elapses, clients will close the socket channel.
   * <p><b>Default:</b> 10000 (10 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T socketConnectionSetupTimeoutMs(Long value);

  /**
   * <b>socket.connection.setup.timeout.max.ms</b><p>
   * The maximum amount of time the client will wait for the socket connection to be established.<p>The connection setup timeout will increase exponentially for each consecutive connection failure up to this maximum.<p>To avoid connection storms, a randomization factor of 0.2 will be applied to the timeout resulting in a random range between 20% below and 20% above the computed value.
   * <p><b>Default:</b> 30000 (30 seconds)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T socketConnectionSetupTimeoutMaxMs(Long value);

  /**
   * <b>connections.max.idle.ms</b><p>
   * Close idle connections after the number of milliseconds specified by this config.
   * <p><b>Default:</b> 540000 (9 minutes)
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T connectionsMaxIdleMs(Long value);

  /**
   * <b>interceptor.classes</b><p>
   * A list of classes to use as interceptors.<p>Implementing the <code>org.apache.kafka.clients.consumer.ConsumerInterceptor</code> interface allows you to intercept (and possibly mutate) records received by the consumer.<p>By default, there are no interceptors.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T interceptorClasses(List<String> value);

  /**
   * <b>interceptor.classes</b><p>
   * A list of classes to use as interceptors.<p>Implementing the <code>org.apache.kafka.clients.consumer.ConsumerInterceptor</code> interface allows you to intercept (and possibly mutate) records received by the consumer.<p>By default, there are no interceptors.
   * <p><b>Default:</b> ""
   * <p><b>Valid Values:</b> non-null string
   * <p><b>Importance:</b> low
   */
  T interceptorClasses(String value);

  /**
   * <b>max.poll.records</b><p>
   * The maximum number of records returned in a single call to poll().<p>Note, that <code>max.poll.records</code> does not impact the underlying fetching behavior.<p>The consumer will cache the records from each fetch request and returns them incrementally from each poll.
   * <p><b>Default:</b> 500
   * <p><b>Valid Values:</b> [1,...]
   * <p><b>Importance:</b> medium
   */
  T maxPollRecords(Integer value);

  /**
   * <b>max.poll.interval.ms</b><p>
   * The maximum delay between invocations of poll() when using consumer group management.<p>This places an upper bound on the amount of time that the consumer can be idle before fetching more records.<p>If poll() is not called before expiration of this timeout, then the consumer is considered failed and the group will rebalance in order to reassign the partitions to another member.<p>For consumers using a non-null <code>group.instance.id</code> which reach this timeout, partitions will not be immediately reassigned.<p>Instead, the consumer will stop sending heartbeats and partitions will be reassigned after expiration of <code>session.timeout.ms</code>.<p>This mirrors the behavior of a static consumer which has shutdown.
   * <p><b>Default:</b> 300000 (5 minutes)
   * <p><b>Valid Values:</b> [1,...]
   * <p><b>Importance:</b> medium
   */
  T maxPollIntervalMs(Integer value);

  /**
   * <b>exclude.internal.topics</b><p>
   * Whether internal topics matching a subscribed pattern should be excluded from the subscription.<p>It is always possible to explicitly subscribe to an internal topic.
   * <p><b>Default:</b> true
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T excludeInternalTopics(Boolean value);

  /**
   * <b>isolation.level</b><p>
   * Controls how to read messages written transactionally.<p>If set to <code>read_committed</code>, consumer.poll() will only return transactional messages which have been committed.<p>If set to <code>read_uncommitted</code> (the default), consumer.poll() will return all messages, even transactional messages which have been aborted.<p>Non-transactional messages will be returned unconditionally in either mode.<p><p>Messages will always be returned in offset order.<p>Hence, in  <code>read_committed</code> mode, consumer.poll() will only return messages up to the last stable offset (LSO), which is the one less than the offset of the first open transaction.<p>In particular any messages appearing after messages belonging to ongoing transactions will be withheld until the relevant transaction has been completed.<p>As a result, <code>read_committed</code> consumers will not be able to read up to the high watermark when there are in flight transactions.</p><p> Further, when in <code>read_committed</code> the seekToEnd method will return the LSO
   * <p><b>Default:</b> read_uncommitted
   * <p><b>Valid Values:</b> [read_committed, read_uncommitted]
   * <p><b>Importance:</b> medium
   */
  T isolationLevel(String value);

  /**
   * <b>allow.auto.create.topics</b><p>
   * Allow automatic topic creation on the broker when subscribing to or assigning a topic.<p>A topic being subscribed to will be automatically created only if the broker allows for it using `auto.create.topics.enable` broker configuration.<p>This configuration must be set to `false` when using brokers older than 0.11.0
   * <p><b>Default:</b> true
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T allowAutoCreateTopics(Boolean value);

  /**
   * <b>security.providers</b><p>
   * A list of configurable creator classes each returning a provider implementing security algorithms.<p>These classes should implement the <code>org.apache.kafka.common.security.auth.SecurityProviderCreator</code> interface.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T securityProviders(String value);

  /**
   * <b>security.protocol</b><p>
   * Protocol used to communicate with brokers.<p>Valid values are: PLAINTEXT, SSL, SASL_PLAINTEXT, SASL_SSL.
   * <p><b>Default:</b> PLAINTEXT
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T securityProtocol(String value);

  /**
   * <b>ssl.protocol</b><p>
   * The SSL protocol used to generate the SSLContext.<p>The default is 'TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>This value should be fine for most use cases.<p>Allowed values in recent JVMs are 'TLSv1.2' and 'TLSv1.3'.<p>'TLS', 'TLSv1.1', 'SSL', 'SSLv2' and 'SSLv3' may be supported in older JVMs, but their usage is discouraged due to known security vulnerabilities.<p>With the default value for this config and 'ssl.enabled.protocols', clients will downgrade to 'TLSv1.2' if the server does not support 'TLSv1.3'.<p>If this config is set to 'TLSv1.2', clients will not use 'TLSv1.3' even if it is one of the values in ssl.enabled.protocols and the server only supports 'TLSv1.3'.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslProtocol(String value);

  /**
   * <b>ssl.provider</b><p>
   * The name of the security provider used for SSL connections.<p>Default value is the default security provider of the JVM.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslProvider(String value);

  /**
   * <b>ssl.cipher.suites</b><p>
   * A list of cipher suites.<p>This is a named combination of authentication, encryption, MAC and key exchange algorithm used to negotiate the security settings for a network connection using TLS or SSL network protocol.<p>By default all the available cipher suites are supported.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslCipherSuites(List<String> value);

  /**
   * <b>ssl.cipher.suites</b><p>
   * A list of cipher suites.<p>This is a named combination of authentication, encryption, MAC and key exchange algorithm used to negotiate the security settings for a network connection using TLS or SSL network protocol.<p>By default all the available cipher suites are supported.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslCipherSuites(String value);

  /**
   * <b>ssl.enabled.protocols</b><p>
   * The list of protocols enabled for SSL connections.<p>The default is 'TLSv1.2,TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>With the default value for Java 11, clients and servers will prefer TLSv1.3 if both support it and fallback to TLSv1.2 otherwise (assuming both support at least TLSv1.2).<p>This default should be fine for most cases.<p>Also see the config documentation for `ssl.protocol`.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslEnabledProtocols(List<String> value);

  /**
   * <b>ssl.enabled.protocols</b><p>
   * The list of protocols enabled for SSL connections.<p>The default is 'TLSv1.2,TLSv1.3' when running with Java 11 or newer, 'TLSv1.2' otherwise.<p>With the default value for Java 11, clients and servers will prefer TLSv1.3 if both support it and fallback to TLSv1.2 otherwise (assuming both support at least TLSv1.2).<p>This default should be fine for most cases.<p>Also see the config documentation for `ssl.protocol`.
   * <p><b>Default:</b> TLSv1.2
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslEnabledProtocols(String value);

  /**
   * <b>ssl.keystore.type</b><p>
   * The file format of the key store file.<p>This is optional for client.
   * <p><b>Default:</b> JKS
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslKeystoreType(String value);

  /**
   * <b>ssl.keystore.location</b><p>
   * The location of the key store file.<p>This is optional for client and can be used for two-way authentication for client.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreLocation(String value);

  /**
   * <b>ssl.keystore.password</b><p>
   * The store password for the key store file.<p>This is optional for client and only needed if 'ssl.keystore.location' is configured.<p> Key store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystorePassword(Password value);

  /**
   * <b>ssl.keystore.password</b><p>
   * The store password for the key store file.<p>This is optional for client and only needed if 'ssl.keystore.location' is configured.<p> Key store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystorePassword(String value);

  /**
   * <b>ssl.key.password</b><p>
   * The password of the private key in the key store file orthe PEM key specified in `ssl.keystore.key'.<p>This is required for clients only if two-way authentication is configured.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeyPassword(Password value);

  /**
   * <b>ssl.key.password</b><p>
   * The password of the private key in the key store file orthe PEM key specified in `ssl.keystore.key'.<p>This is required for clients only if two-way authentication is configured.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeyPassword(String value);

  /**
   * <b>ssl.keystore.key</b><p>
   * Private key in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with PKCS#8 keys.<p>If the key is encrypted, key password must be specified using 'ssl.key.password'
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreKey(Password value);

  /**
   * <b>ssl.keystore.key</b><p>
   * Private key in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with PKCS#8 keys.<p>If the key is encrypted, key password must be specified using 'ssl.key.password'
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreKey(String value);

  /**
   * <b>ssl.keystore.certificate.chain</b><p>
   * Certificate chain in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with a list of X.509 certificates
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreCertificateChain(Password value);

  /**
   * <b>ssl.keystore.certificate.chain</b><p>
   * Certificate chain in the format specified by 'ssl.keystore.type'.<p>Default SSL engine factory supports only PEM format with a list of X.509 certificates
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslKeystoreCertificateChain(String value);

  /**
   * <b>ssl.truststore.certificates</b><p>
   * Trusted certificates in the format specified by 'ssl.truststore.type'.<p>Default SSL engine factory supports only PEM format with X.509 certificates.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreCertificates(Password value);

  /**
   * <b>ssl.truststore.certificates</b><p>
   * Trusted certificates in the format specified by 'ssl.truststore.type'.<p>Default SSL engine factory supports only PEM format with X.509 certificates.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreCertificates(String value);

  /**
   * <b>ssl.truststore.type</b><p>
   * The file format of the trust store file.
   * <p><b>Default:</b> JKS
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T sslTruststoreType(String value);

  /**
   * <b>ssl.truststore.location</b><p>
   * The location of the trust store file.<p>
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststoreLocation(String value);

  /**
   * <b>ssl.truststore.password</b><p>
   * The password for the trust store file.<p>If a password is not set, trust store file configured will still be used, but integrity checking is disabled.<p>Trust store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststorePassword(Password value);

  /**
   * <b>ssl.truststore.password</b><p>
   * The password for the trust store file.<p>If a password is not set, trust store file configured will still be used, but integrity checking is disabled.<p>Trust store password is not supported for PEM format.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> high
   */
  T sslTruststorePassword(String value);

  /**
   * <b>ssl.keymanager.algorithm</b><p>
   * The algorithm used by key manager factory for SSL connections.<p>Default value is the key manager factory algorithm configured for the Java Virtual Machine.
   * <p><b>Default:</b> SunX509
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslKeymanagerAlgorithm(String value);

  /**
   * <b>ssl.trustmanager.algorithm</b><p>
   * The algorithm used by trust manager factory for SSL connections.<p>Default value is the trust manager factory algorithm configured for the Java Virtual Machine.
   * <p><b>Default:</b> PKIX
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslTrustmanagerAlgorithm(String value);

  /**
   * <b>ssl.endpoint.identification.algorithm</b><p>
   * The endpoint identification algorithm to validate server hostname using server certificate.<p>
   * <p><b>Default:</b> https
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslEndpointIdentificationAlgorithm(String value);

  /**
   * <b>ssl.secure.random.implementation</b><p>
   * The SecureRandom PRNG implementation to use for SSL cryptography operations.<p>
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslSecureRandomImplementation(String value);

  /**
   * <b>ssl.engine.factory.class</b><p>
   * The class of type org.apache.kafka.common.security.auth.SslEngineFactory to provide SSLEngine objects.<p>Default value is org.apache.kafka.common.security.ssl.DefaultSslEngineFactory
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T sslEngineFactoryClass(Class value);

  /**
   * <b>sasl.kerberos.service.name</b><p>
   * The Kerberos principal name that Kafka runs as.<p>This can be defined either in Kafka's JAAS config or in Kafka's config.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslKerberosServiceName(String value);

  /**
   * <b>sasl.kerberos.kinit.cmd</b><p>
   * Kerberos kinit command path.
   * <p><b>Default:</b> /usr/bin/kinit
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosKinitCmd(String value);

  /**
   * <b>sasl.kerberos.ticket.renew.window.factor</b><p>
   * Login thread will sleep until the specified window factor of time from last refresh to ticket's expiry has been reached, at which time it will try to renew the ticket.
   * <p><b>Default:</b> 0.8
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosTicketRenewWindowFactor(Double value);

  /**
   * <b>sasl.kerberos.ticket.renew.jitter</b><p>
   * Percentage of random jitter added to the renewal time.
   * <p><b>Default:</b> 0.05
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosTicketRenewJitter(Double value);

  /**
   * <b>sasl.kerberos.min.time.before.relogin</b><p>
   * Login thread sleep time between refresh attempts.
   * <p><b>Default:</b> 60000
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> low
   */
  T saslKerberosMinTimeBeforeRelogin(Long value);

  /**
   * <b>sasl.login.refresh.window.factor</b><p>
   * Login refresh thread will sleep until the specified window factor relative to the credential's lifetime has been reached, at which time it will try to refresh the credential.<p>Legal values are between 0.5 (50%) and 1.0 (100%) inclusive; a default value of 0.8 (80%) is used if no value is specified.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 0.8
   * <p><b>Valid Values:</b> [0.5,...,1.0]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshWindowFactor(Double value);

  /**
   * <b>sasl.login.refresh.window.jitter</b><p>
   * The maximum amount of random jitter relative to the credential's lifetime that is added to the login refresh thread's sleep time.<p>Legal values are between 0 and 0.25 (25%) inclusive; a default value of 0.05 (5%) is used if no value is specified.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 0.05
   * <p><b>Valid Values:</b> [0.0,...,0.25]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshWindowJitter(Double value);

  /**
   * <b>sasl.login.refresh.min.period.seconds</b><p>
   * The desired minimum time for the login refresh thread to wait before refreshing a credential, in seconds.<p>Legal values are between 0 and 900 (15 minutes); a default value of 60 (1 minute) is used if no value is specified.<p> This value and  sasl.login.refresh.buffer.seconds are both ignored if their sum exceeds the remaining lifetime of a credential.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 60
   * <p><b>Valid Values:</b> [0,...,900]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshMinPeriodSeconds(Short value);

  /**
   * <b>sasl.login.refresh.buffer.seconds</b><p>
   * The amount of buffer time before credential expiration to maintain when refreshing a credential, in seconds.<p>If a refresh would otherwise occur closer to expiration than the number of buffer seconds then the refresh will be moved up to maintain as much of the buffer time as possible.<p>Legal values are between 0 and 3600 (1 hour); a default value of  300 (5 minutes) is used if no value is specified.<p>This value and sasl.login.refresh.min.period.seconds are both ignored if their sum exceeds the remaining lifetime of a credential.<p>Currently applies only to OAUTHBEARER.
   * <p><b>Default:</b> 300
   * <p><b>Valid Values:</b> [0,...,3600]
   * <p><b>Importance:</b> low
   */
  T saslLoginRefreshBufferSeconds(Short value);

  /**
   * <b>sasl.mechanism</b><p>
   * SASL mechanism used for client connections.<p>This may be any mechanism for which a security provider is available.<p>GSSAPI is the default mechanism.
   * <p><b>Default:</b> GSSAPI
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslMechanism(String value);

  /**
   * <b>sasl.jaas.config</b><p>
   * JAAS login context parameters for SASL connections in the format used by JAAS configuration files.<p>JAAS configuration file format is described <a href="http://docs.oracle.com/javase/8/docs/technotes/guides/security/jgss/tutorials/LoginConfigFile.html">here</a>.<p>The format for the value is: <code>loginModuleClass controlFlag (optionName=optionValue)*;</code>.<p>For brokers, the config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.jaas.config=com.example.ScramLoginModule required;
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslJaasConfig(Password value);

  /**
   * <b>sasl.jaas.config</b><p>
   * JAAS login context parameters for SASL connections in the format used by JAAS configuration files.<p>JAAS configuration file format is described <a href="http://docs.oracle.com/javase/8/docs/technotes/guides/security/jgss/tutorials/LoginConfigFile.html">here</a>.<p>The format for the value is: <code>loginModuleClass controlFlag (optionName=optionValue)*;</code>.<p>For brokers, the config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.jaas.config=com.example.ScramLoginModule required;
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslJaasConfig(String value);

  /**
   * <b>sasl.client.callback.handler.class</b><p>
   * The fully qualified name of a SASL client callback handler class that implements the AuthenticateCallbackHandler interface.
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslClientCallbackHandlerClass(Class value);

  /**
   * <b>sasl.login.callback.handler.class</b><p>
   * The fully qualified name of a SASL login callback handler class that implements the AuthenticateCallbackHandler interface.<p>For brokers, login callback handler config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.login.callback.handler.class=com.example.CustomScramLoginCallbackHandler
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslLoginCallbackHandlerClass(Class value);

  /**
   * <b>sasl.login.class</b><p>
   * The fully qualified name of a class that implements the Login interface.<p>For brokers, login config must be prefixed with listener prefix and SASL mechanism name in lower-case.<p>For example, listener.name.sasl_ssl.scram-sha-256.sasl.login.class=com.example.CustomScramLogin
   * <p><b>Default:</b> null
   * <p><b>Valid Values:</b> 
   * <p><b>Importance:</b> medium
   */
  T saslLoginClass(Class value);

  default T self() {
    return (T)this;
  }
}
