/*
 * Copyright 2008-2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.synyx.hades.dao.config;

import static org.springframework.beans.factory.support.BeanDefinitionBuilder.*;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.parsing.BeanComponentDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.RootBeanDefinition;
import org.springframework.beans.factory.xml.BeanDefinitionParser;
import org.springframework.beans.factory.xml.ParserContext;
import org.springframework.util.StringUtils;
import org.w3c.dom.Element;


/**
 * {@link BeanDefinitionParser} for the Hades {@code auditing} element.
 * 
 * @author Oliver Gierke
 */
public class AuditingBeanDefinitionParser implements BeanDefinitionParser {

    private static final String AUDITING_ENTITY_LISTENER_CLASS_NAME =
            "org.synyx.hades.domain.auditing.support.AuditingEntityListener";
    private static final String AUDITING_BFPP_CLASS_NAME =
            "org.synyx.hades.domain.auditing.support.AuditingBeanFactoryPostProcessor";


    /*
     * (non-Javadoc)
     * 
     * @see
     * org.springframework.beans.factory.xml.BeanDefinitionParser#parse(org.
     * w3c.dom.Element, org.springframework.beans.factory.xml.ParserContext)
     */
    public BeanDefinition parse(Element element, ParserContext parserContext) {

        new SpringConfiguredBeanDefinitionParser()
                .parse(element, parserContext);

        BeanDefinitionBuilder builder =
                rootBeanDefinition(AUDITING_ENTITY_LISTENER_CLASS_NAME);

        String auditorAwareRef = element.getAttribute("auditor-aware-ref");

        if (StringUtils.hasText(auditorAwareRef)) {
            builder.addPropertyReference("auditorAware", auditorAwareRef);
        }

        registerInfrastructureBeanWithId(builder.getRawBeanDefinition(),
                AUDITING_ENTITY_LISTENER_CLASS_NAME, parserContext, element);

        RootBeanDefinition def = new RootBeanDefinition();
        def.setBeanClassName(AUDITING_BFPP_CLASS_NAME);
        registerInfrastructureBeanWithId(def, AUDITING_BFPP_CLASS_NAME,
                parserContext, element);

        return null;
    }


    private void registerInfrastructureBeanWithId(AbstractBeanDefinition def,
            String id, ParserContext context, Element element) {

        // Mark as infrastructure bean and attach source location.
        def.setRole(BeanDefinition.ROLE_INFRASTRUCTURE);
        def.setSource(context.extractSource(element));
        context.registerBeanComponent(new BeanComponentDefinition(def, id));
    }

    /**
     * Copied code of SpringConfiguredBeanDefinitionParser until this class gets
     * public.
     * 
     * @see http://jira.springframework.org/browse/SPR-7340
     * @author Juergen Hoeller
     */
    private static class SpringConfiguredBeanDefinitionParser implements
            BeanDefinitionParser {

        /**
         * The bean name of the internally managed bean configurer aspect.
         */
        private static final String BEAN_CONFIGURER_ASPECT_BEAN_NAME =
                "org.springframework.context.config.internalBeanConfigurerAspect";

        private static final String BEAN_CONFIGURER_ASPECT_CLASS_NAME =
                "org.springframework.beans.factory.aspectj.AnnotationBeanConfigurerAspect";


        public BeanDefinition parse(Element element, ParserContext parserContext) {

            if (!parserContext.getRegistry().containsBeanDefinition(
                    BEAN_CONFIGURER_ASPECT_BEAN_NAME)) {
                RootBeanDefinition def = new RootBeanDefinition();
                def.setBeanClassName(BEAN_CONFIGURER_ASPECT_CLASS_NAME);
                def.setFactoryMethodName("aspectOf");

                // Mark as infrastructure bean and attach source location.
                def.setRole(BeanDefinition.ROLE_INFRASTRUCTURE);
                def.setSource(parserContext.extractSource(element));
                parserContext
                        .registerBeanComponent(new BeanComponentDefinition(def,
                                BEAN_CONFIGURER_ASPECT_BEAN_NAME));
            }
            return null;
        }
    }
}
