/*
 *  Copyright (C) :	2002,2003,2004,2005,2006,2007,2008,2009
 *			European Synchrotron Radiation Facility
 *			BP 220, Grenoble 38043
 *			FRANCE
 * 
 *  This file is part of Tango.
 * 
 *  Tango is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  Tango is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *  
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Tango.  If not, see <http://www.gnu.org/licenses/>.
 */
 
// File:          StatusViewer.java
// Created:       2005-02-14 18:15:00, poncet
// By:            <poncet@esrf.fr>
//
// $Id$
//
// Description:

package fr.esrf.tangoatk.widget.attribute;

import fr.esrf.tangoatk.core.*;
import fr.esrf.tangoatk.core.ErrorEvent;
import fr.esrf.tangoatk.widget.util.ATKConstant;
import fr.esrf.tangoatk.widget.util.jdraw.JDrawable;

import javax.swing.*;
import javax.swing.border.TitledBorder;
import java.awt.*;

/**
 *
 * @author  root
 */
public class StatusViewer extends JPanel implements IStringScalarListener, JDrawable
{

    private JScrollPane scrollPane;
    private JTextArea status;
    private IStringScalar model;
    private boolean useDeviceAlias = true;

    static String[] exts = {"verticalScrollBar","horizontalScrollBar"};

    /** Creates new form StatusViewer */

    public StatusViewer()
    {
      model = null;
      initComponents();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {
      scrollPane = new JScrollPane();
      status = new JTextArea();

      setLayout(new BorderLayout());

      setBorder(new TitledBorder("Status"));
      scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
      status.setLineWrap(true);
      status.setEditable(false);
      status.setColumns(50);
      status.setRows(4);
      status.setText("Unknown");
      status.setBackground(new java.awt.Color(204, 204, 204));
      scrollPane.setViewportView(status);

      add(scrollPane, java.awt.BorderLayout.CENTER);
    }

  public void setModel(IStringScalar statusAtt)
  {
    if (model != null)
      model.removeStringScalarListener(this);

    status.setText("Unknown");

    model = statusAtt;
    if( model!=null )
    {
      model.addStringScalarListener(this);
      //model.refresh();
      setStatus(model.getStringDeviceValue());
      if( getBorder() instanceof TitledBorder ) {
        if (useDeviceAlias) {
          if (model.getDevice().getAlias() != null)
            setBorder(new TitledBorder(model.getDevice().getAlias()));
          else
            setBorder(new TitledBorder(model.getDevice().getName()));
        } else
          setBorder(new TitledBorder(model.getDevice().getName()));
      }
    }
  }

  public void clearModel() {
    setModel(null);
  }

  /**
   * <code>getModel</code> gets the model of this statusViewer.
   *
   * @return a <code>IStringScalar</code> value
   */
  public IStringScalar getModel()
  {
    return model;
  }


  public int getRows()
  {
    return status.getRows();
  }

  public void setRows(int rows)
  {
    status.setRows(rows);
  }

  public int getColumns()
  {
    return status.getColumns();
  }

  public void setColumns(int columns)
  {
    status.setColumns(columns);
  }

  /**
   * <code>getUseDeviceAlias</code> returns true if the device alias is displayed instead of device name
   *
   * @return a <code>boolean</code> value
   */
  public boolean getUseDeviceAlias() {
    return useDeviceAlias;
  }

  /**
   * <code>setUseDeviceAlias</code> use or not use device alias
   *
   * @param b True to enable device alias usage.
   */
  public void setUseDeviceAlias(boolean b) {
    useDeviceAlias=b;
  }


  public JTextArea getText()
  {
    return status;
  }

  public void stringScalarChange(StringScalarEvent evt)
  {
    if (!status.isEnabled())
      status.setEnabled(true);

    setStatus(evt.getValue());
  }

  public void errorChange(ErrorEvent evt)
  {
    setStatus(IDevice.UNKNOWN);
    status.setEnabled(false);
  }

  public void stateChange(AttributeStateEvent attributeStateEvent)
  {
    attStateChange(attributeStateEvent.getState());
  }

  private void setStatus(String s)
  {
    if (s == null)
    {
      if ( "".equals(status.getText()) || status.getText() == null )
        return;
      else
        status.setText("");
    }
    else if (s.equals(status.getText()))
      return;
    else {
      status.setText(s);
      status.setCaretPosition(0);
    }
  }

  private void attStateChange(String state)
  {
    if ("VALID".equals(state))
    {
      status.setBackground(getBackground());
      return;
    }
    status.setBackground(ATKConstant.getColor4Quality(state));
  }

  public void setFont(Font font) {
    if(status!=null)
      status.setFont(font);
  }

  public void setBackground(Color color) {
    super.setBackground(color);
    if(status!=null)
      status.setBackground(color);
  }

  @Override
  public void initForEditing() {
    status.setText("This is a multiline\nsample text");
  }

  @Override
  public JComponent getComponent() {
    return this;
  }

  @Override
  public String[] getExtensionList() {
    return exts;
  }

  @Override
  public boolean setExtendedParam(String name, String value, boolean popupAllowed) {

    if(name.equals("verticalScrollBar")) {

      if(value.equalsIgnoreCase("ASNEEDED")) {
        scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        return true;
      } else if (value.equalsIgnoreCase("ALWAYS")) {
        scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        return true;
      } else if (value.equalsIgnoreCase("NEVER")) {
        scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);
        return true;
      } else {
        showJdrawError(popupAllowed,"verticalScrollBar","Wrong syntax: 'ASNEEDED','ALWAYS'or 'NEVER' expected.");
        return false;
      }

    } else if(name.equals("horizontalScrollBar")) {

      if(value.equalsIgnoreCase("ASNEEDED")) {
        scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        return true;
      } else if (value.equalsIgnoreCase("ALWAYS")) {
        scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        return true;
      } else if (value.equalsIgnoreCase("NEVER")) {
        scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        return true;
      } else {
        showJdrawError(popupAllowed,"horizontalScrollBar","Wrong syntax: 'ASNEEDED','ALWAYS'or 'NEVER' expected.");
        return false;
      }

    }
    return false;
  }

  @Override
  public String getExtendedParam(String name) {

    if(name.equals("verticalScrollBar")) {
      return getScrollBarPolicy(scrollPane.getVerticalScrollBarPolicy());
    } else if(name.equals("horizontalScrollBar")) {
      return getScrollBarPolicy(scrollPane.getHorizontalScrollBarPolicy());
    }

    return "";
  }

  @Override
  public String getDescription(String extName) {
    if(extName.equals("verticalScrollBar")) {
      return "Vertical scrollBar policy.\n" +
              "Possible values are: ASNEEDED,ALWAYS,NEVER";
    } else if(extName.equals("horizontalScrollBar")) {
      return "Horizontal scrollBar policy.\n" +
              "Possible values are: ASNEEDED,ALWAYS,NEVER";
    } else {
      return "";
    }
  }

  private String getScrollBarPolicy(int policy) {
    switch (policy) {
      case JScrollPane.VERTICAL_SCROLLBAR_ALWAYS:
      case JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS:
        return "ALWAYS";
      case JScrollPane.VERTICAL_SCROLLBAR_NEVER:
      case JScrollPane.HORIZONTAL_SCROLLBAR_NEVER:
        return "NEVER";
      case JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED:
      case JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED:
        return "ASNEEDED";
    }
    return Integer.toString(policy);
  }

  private void showJdrawError(boolean popup,String paramName,String message) {
    if(popup)
      JOptionPane.showMessageDialog(null, "StatusViewer: "+paramName+" incorrect.\n" + message,
              "Error",JOptionPane.ERROR_MESSAGE);
  }

  public static void main(String[] args)
  {
    final fr.esrf.tangoatk.core.AttributeList  attList = new fr.esrf.tangoatk.core.AttributeList();
    IStringScalar                              attStatus;
    JFrame                                     mainFrame;

    StatusViewer                                stsv = new StatusViewer();


    try
    {
      attStatus = (IStringScalar) attList.add("dev/test/10/Status");
      stsv.setModel(attStatus);
    }
    catch (Exception ex)
    {
      System.out.println("caught exception : "+ ex.getMessage());
      System.exit(-1);
    }

    mainFrame = new JFrame();

    mainFrame.addWindowListener(
            new java.awt.event.WindowAdapter()
            {
              public void windowActivated(java.awt.event.WindowEvent evt)
              {
                // To be sure that the refresher (an independente thread)
                // will begin when the the layout manager has finished
                // to size and position all the components of the window
                attList.startRefresher();
              }
            }
    );


    mainFrame.setContentPane(stsv);
    mainFrame.pack();

    mainFrame.setVisible(true);

  } // end of main ()

}
