/*
 *  Copyright (C) :	2002,2003,2004,2005,2006,2007,2008,2009
 *			European Synchrotron Radiation Facility
 *			BP 220, Grenoble 38043
 *			FRANCE
 * 
 *  This file is part of Tango.
 * 
 *  Tango is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  Tango is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *  
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Tango.  If not, see <http://www.gnu.org/licenses/>.
 */
 

package fr.esrf.tangoatk.widget.device;

import fr.esrf.tangoatk.core.Device;
import fr.esrf.tangoatk.core.IDevice;
import fr.esrf.tangoatk.core.IDeviceApplication;
import fr.esrf.tangoatk.core.StateEvent;
import fr.esrf.tangoatk.core.ErrorEvent;

import fr.esrf.tangoatk.widget.util.*;


/**
 * <code>State</code>
 * State is a viewer to surveil the state of a
 * {@link fr.esrf.tangoatk.core.Device}. Normally one connects the
 * device with the viewer like this:<br>
 * <code>
 * Device device = getDevice(); // some method to obtain the device.
 * State = new State();
 * state.setModel(device);
 * </code>
 * @author <a href="mailto:assum@esrf.fr">Erik Assum</a>
 * @version $Revision$
 */
public class StateViewer extends javax.swing.JPanel
        implements fr.esrf.tangoatk.core.IStateListener {

  IDevice device;
  String state = IDevice.UNKNOWN;
  boolean externalSetText = false;
  boolean stateClickable = true;
  IDeviceApplication application;
  
  private boolean          stateInTooltip = false;

  public StateViewer() {
    initComponents();
    UIManagerHelper.setAll("StateViewer.Label", textLabel);
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  private void initComponents() {//GEN-BEGIN:initComponents
    textLabel = new javax.swing.JLabel();
    valueLabel = new javax.swing.JLabel();

    setLayout(new java.awt.GridBagLayout());
    java.awt.GridBagConstraints gridBagConstraints1;

    textLabel.setText("Not Connected");
    textLabel.setFont(new java.awt.Font("Dialog", 0, 12));
    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 1;
    gridBagConstraints1.gridy = 0;
    gridBagConstraints1.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints1.insets = new java.awt.Insets(0, 4, 0, 3);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints1.weightx = 0.1;
    gridBagConstraints1.weighty = 0.1;
    add(textLabel, gridBagConstraints1);

    valueLabel.setBackground(java.awt.Color.red);
    valueLabel.setFont(new java.awt.Font("Dialog", 0, 12));
    valueLabel.setPreferredSize(new java.awt.Dimension(40, 14));
    valueLabel.setOpaque(true);
    valueLabel.addMouseListener(new java.awt.event.MouseAdapter() {
      public void mouseClicked(java.awt.event.MouseEvent evt) {
        valueLabelMouseClicked(evt);
      }
    });

    gridBagConstraints1 = new java.awt.GridBagConstraints();
    gridBagConstraints1.gridx = 0;
    gridBagConstraints1.gridy = 0;
    gridBagConstraints1.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints1.insets = new java.awt.Insets(0, 3, 0, 4);
    gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints1.weightx = 0.2;
    gridBagConstraints1.weighty = 0.2;
    add(valueLabel, gridBagConstraints1);

  }//GEN-END:initComponents

  protected void valueLabelMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_valueLabelMouseClicked
    // Add your handling code here:
    if ((evt.getModifiers() &
            java.awt.event.InputEvent.BUTTON3_MASK) != 0) {
      if (application == null) return;
      application.setModel(device);
      application.run();
      return;
    }

    if (stateClickable && device != null) {
      popUp.setModel(device);
      popUp.setVisible(true);
    }

  }//GEN-LAST:event_valueLabelMouseClicked

  IDevicePopUp popUp = SingletonStatusViewer.getInstance();

  /**
   * Get the value of popUp.
   * @return value of popUp.
   */
  public IDevicePopUp getPopUp() {
    return popUp;
  }

  /**
   * Set the value of popUp.
   * @param v  Value to assign to popUp.
   */
  public void setPopUp(IDevicePopUp v) {
    this.popUp = v;
  }

  /**
   * <code>setModel</code> sets the model of this viewer.
   * If the textLabel property is not set, the name of the device is
   * shown on the textLabel.
   * @param devModel a <code>Device</code> to survey
   */
  public void setModel(IDevice devModel) {
    clearModel();
    
    if (devModel == null)
       return;
       
    this.device = devModel;
    if (this.device instanceof Device)
    {
          Device dev = (Device) this.device;
          if (!dev.areDevPropertiesLoaded())
          {
              dev.loadDevProperties();
          }
    }
    device.addStateListener(this);
    setState(device.getState());
    if (!externalSetText)
      textLabel.setText(device.getName());

    valueLabel.setToolTipText(device.getName());
  }
  
  /**
   * <code>clearModel</code> clear the model of this viewer.  
   */
  public void clearModel() {
     if(device != null)
     {
         device.removeStateListener(this);
         textLabel.setText("Not Connected");
         state = IDevice.UNKNOWN;
         device = null;
     }
  }

public javax.swing.JLabel getTextLabel() {
    return textLabel;
}
public javax.swing.JLabel getValueLabel() {
    return valueLabel;
}
  /**
   * <code>getModel</code> gets the model of this stateviewer.
   *
   * @return a <code>Device</code> value
   */
  public IDevice getModel() {
    return device;
  }

  /**
   * <code>setState</code>
   *
   * @param state a <code>String</code> value
   */
  private void setState(String state) {
    this.state = state;
    valueLabel.setBackground(ATKConstant.getColor4State(state, device.getInvertedOpenClose(), device.getInvertedInsertExtract()));
    if (stateInTooltip)
       if (device != null)
	  valueLabel.setToolTipText(device.getName() + " : " + state);
  }

  /**
   * <code>getState</code>
   *
   * @return a <code>String</code> value presenting the state of the device
   */
  public String getState() {
    return state;
  }

  /**
   * <code>setLabel</code> set the text of the label. The
   * default value is to show the name of the device.
   * @param label a <code>String</code> value
   */
  public void setLabel(String label) {
    externalSetText = true;
    textLabel.setText(label);
  }

  public String getLabel() {
    return textLabel.getText();
  }

  /**
   * <code>setLabelVisisble</code> makes the label visible or not.
   *
   * @param visible a <code>boolean</code> value
   */
  public void setLabelVisible(boolean visible) {
    textLabel.setVisible(visible);
  }

  /**
   * <code>isLabelVisible</code> returns the visibility of the label
   *
   * @return a <code>boolean</code> value
   */
  public boolean isLabelVisible() {
    return textLabel.isVisible();
  }

  /**
   * <code>setStateVisible</code> makes the state visible or not.
   *
   * @param visible a <code>boolean</code> value
   */
  public void setStateVisible(boolean visible) {
    valueLabel.setVisible(visible);
  }

  /**
   * <code>isStateVisible</code> returns the visibility of the
   * state
   * @return a <code>boolean</code> value
   */
  public boolean isStateVisible() {
    return valueLabel.isVisible();
  }


  /**
   * <code>stateChange</code> inherited from IStateListener
   *
   * @param evt a <code>StateEvent</code> value
   */
  public void stateChange(StateEvent evt) {
    setState(evt.getState());
  }

  public void errorChange(ErrorEvent evt) {
    setState("UNKNOWN");

  }

  /**
   * <code>setStateText</code> sets the text on the colored state box
   *
   * @param text a <code>String</code> value
   */
  public void setStateText(String text) {
    valueLabel.setText(text);
  }

  /**
   * <code>getStateText</code> gets the text that is on the
   * colored state box
   * @return a <code>String</code> value
   */
  public String getStateText() {
    return valueLabel.getText();
  }


  public void setStateFont(java.awt.Font font) {
    valueLabel.setFont(font);
  }

  public void setFont(java.awt.Font font) {
    if (valueLabel != null) {
      valueLabel.setFont(font);
    }
    if (textLabel != null) {
      textLabel.setFont(font);
    }

    super.setFont(font);
  }

  public java.awt.Font getStateFont() {
    return valueLabel.getFont();
  }

  public void setStateForeground(java.awt.Color color) {
    valueLabel.setForeground(color);
  }


  public java.awt.Color getStateForeground() {
    return valueLabel.getForeground();
  }


  public void setForeground(java.awt.Color color) {
    if (valueLabel != null) {
      valueLabel.setForeground(color);
    }
    if (textLabel != null) {
      textLabel.setForeground(color);
    }
    super.setForeground(color);
  }


  /**
   * <code>setStateHorizontalAlignement</code>
   * @see javax.swing.SwingConstants
   * @param i an <code>int</code> value
   */
  public void setStateHorizontalAlignment(int i) {
    valueLabel.setHorizontalAlignment(i);
  }

  public int getStateHorizontalAlignment() {
    return valueLabel.getHorizontalAlignment();
  }

  public void setStatePreferredSize(java.awt.Dimension dimension) {
    valueLabel.setPreferredSize(dimension);
  }

  public java.awt.Dimension getStatePreferredSize() {
    return valueLabel.getPreferredSize();
  }

  public void setStateBorder(javax.swing.border.Border border) {
    if (valueLabel == null) return;

    valueLabel.setBorder(border);
  }

  public javax.swing.border.Border getStateBorder() {
    if (valueLabel == null) return null;

    return valueLabel.getBorder();
  }
  
  /**
   * <code>getStateInTooltip</code> returns true if the device state is displayed inside the viewer's tooltip
   *
   * @return a <code>boolean</code> value
   */
  public boolean getStateInTooltip() {
    return stateInTooltip;
  }

  /**
   * <code>setStateInTooltip</code> display or not the device state inside the tooltip
   *
   * @param b If True the device state will be displayed inside the tooltip.
   */
  public void setStateInTooltip(boolean b) {
    if (stateInTooltip != b)
    {
       if (b == false)
          if (device != null)
              valueLabel.setToolTipText(device.getName());
       stateInTooltip=b;
    }
  }

  /**
   * <code>setStateClickable</code> will the state be clickable?
   *
   * @param clickable a <code>boolean</code> value
   */
  public void setStateClickable(boolean clickable) {
    stateClickable = clickable;
  }

  /**
   * <code>isStateClickable</code> returns if the state is clickable or not.
   *
   * @return a <code>boolean</code> value
   */
  public boolean isStateClickable() {
    return stateClickable;
  }

  public void setApplication(IDeviceApplication runnable) {
    application = runnable;
  }

  public IDeviceApplication getApplication() {
    return application;
  }

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JLabel textLabel;
  private javax.swing.JLabel valueLabel;
  // End of variables declaration//GEN-END:variables

}
