/*-
 * ========================LICENSE_START=================================
 * TeamApps Antu Icon Provider
 * ---
 * Copyright (C) 2014 - 2020 TeamApps.org
 * ---
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.teamapps.icon.antu;

import org.teamapps.icons.Icon;
import org.teamapps.icons.spi.IconLibrary;

import java.util.*;
import java.util.stream.Collectors;

@IconLibrary(
        name = "antu",
        encoder = AntuIconEncoder.class,
        decoder = AntuIconDecoder.class,
        loader = AntuIconLoader.class,
        defaultStyleSupplier = AntuIconDefaultIconSupplier.class
)
public class AntuIcon implements Icon<AntuIcon, AntuIconStyle> {

    private static final Map<String, AntuIcon> ICONS_BY_ID_LIGHT = new HashMap<>();
    private static final Map<String, AntuIcon> ICONS_BY_ID_DARK = new HashMap<>();
    private final String iconId;
    private final AntuIconStyle style;
    private final String iconPath;

    private AntuIcon(String iconId, String iconPath, AntuIconStyle style) {
        this.iconId = iconId;
        this.iconPath = iconPath;
        this.style = Objects.requireNonNullElse(style, AntuIconStyle.LIGHT);
    }

    public static AntuIcon forId(String iconId) {
        return forId(iconId,AntuIconStyle.LIGHT);
    }
    public static AntuIcon forId(String id, AntuIconStyle style) {
        if (style == null || AntuIconStyle.LIGHT.equals(style)) {
            return ICONS_BY_ID_LIGHT.get(id);
        } else {
            return ICONS_BY_ID_DARK.get(id);
        }
    }

    public static List<AntuIcon> getIcons() {
        return ICONS_BY_ID_LIGHT.values().stream()
                .sorted(Comparator.comparing(AntuIcon::getIconId))
                .collect(Collectors.toList());
    }

    @Override
    public AntuIcon withStyle(AntuIconStyle antuIconStyle) {
        return forId(iconId, antuIconStyle);
    }

    @Override
    public AntuIconStyle getStyle() {
        return style;
    }

    public String getIconId() {
        return iconId;
    }
    public String getIconPath() {
        return iconPath;
    }

    private static AntuIcon create(String id, String path) {
        AntuIcon icon = new AntuIcon(id, path, null);
        ICONS_BY_ID_LIGHT.put(id, icon);
        AntuIcon darkIcon = new AntuIcon(id, path, AntuIconStyle.DARK);
        ICONS_BY_ID_DARK.put(id, darkIcon);
        return icon;
    }


    // APPLETS
    public static final AntuIcon APPLET_ACTIVITYBAR_256 = create("APPLET_ACTIVITYBAR_256", "applets/256/org.kde.plasma.activitybar");
    public static final AntuIcon APPLET_ACTIVITYPAGER_256 = create("APPLET_ACTIVITYPAGER_256", "applets/256/org.kde.plasma.activitypager");
    public static final AntuIcon APPLET_ANALOGCLOCK_256 = create("APPLET_ANALOGCLOCK_256", "applets/256/org.kde.plasma.analogclock");
    public static final AntuIcon APPLET_APPLETS_TEMPLATE_256 = create("APPLET_APPLETS_TEMPLATE_256", "applets/256/applets-template");
    public static final AntuIcon APPLET_AWESOMEWIDGET_256 = create("APPLET_AWESOMEWIDGET_256", "applets/256/org.kde.plasma.awesomewidget");
    public static final AntuIcon APPLET_BATTERY_256 = create("APPLET_BATTERY_256", "applets/256/org.kde.plasma.battery");
    public static final AntuIcon APPLET_BINARYCLOCK_256 = create("APPLET_BINARYCLOCK_256", "applets/256/org.kde.plasma.binaryclock");
    public static final AntuIcon APPLET_BUMBLEBEE_INDICATOR_256 = create("APPLET_BUMBLEBEE_INDICATOR_256", "applets/256/com.bxabi.bumblebee-indicator");
    public static final AntuIcon APPLET_CALCULATOR_256 = create("APPLET_CALCULATOR_256", "applets/256/org.kde.plasma.calculator");
    public static final AntuIcon APPLET_CALENDAR_256 = create("APPLET_CALENDAR_256", "applets/256/org.kde.plasma.calendar");
    public static final AntuIcon APPLET_CLIPBOARD_256 = create("APPLET_CLIPBOARD_256", "applets/256/org.kde.plasma.clipboard");
    public static final AntuIcon APPLET_COLORPICKER_256 = create("APPLET_COLORPICKER_256", "applets/256/org.kde.plasma.colorpicker");
    public static final AntuIcon APPLET_COMIC_256 = create("APPLET_COMIC_256", "applets/256/org.kde.plasma.comic");
    public static final AntuIcon APPLET_DEVICENOTIFIER_256 = create("APPLET_DEVICENOTIFIER_256", "applets/256/org.kde.plasma.devicenotifier");
    public static final AntuIcon APPLET_DIGITALCLOCK_256 = create("APPLET_DIGITALCLOCK_256", "applets/256/org.kde.plasma.digitalclock");
    public static final AntuIcon APPLET_DISKQUOTA_256 = create("APPLET_DISKQUOTA_256", "applets/256/org.kde.plasma.diskquota");
    public static final AntuIcon APPLET_FIFTEENPUZZLE_256 = create("APPLET_FIFTEENPUZZLE_256", "applets/256/org.kde.plasma.fifteenpuzzle");
    public static final AntuIcon APPLET_FOLDER_256 = create("APPLET_FOLDER_256", "applets/256/org.kde.plasma.folder");
    public static final AntuIcon APPLET_FRAME_256 = create("APPLET_FRAME_256", "applets/256/org.kde.plasma.frame");
    public static final AntuIcon APPLET_FUZZYCLOCK_256 = create("APPLET_FUZZYCLOCK_256", "applets/256/org.kde.plasma.fuzzyclock");
    public static final AntuIcon APPLET_GMAILFEED_256 = create("APPLET_GMAILFEED_256", "applets/256/org.kde.plasma.gmailfeed");
    public static final AntuIcon APPLET_ICONTASKS_256 = create("APPLET_ICONTASKS_256", "applets/256/org.kde.plasma.icontasks");
    public static final AntuIcon APPLET_KEYSTATE_PLASMOID_256 = create("APPLET_KEYSTATE_PLASMOID_256", "applets/256/eu.blumenstingl.martin.keystateplasmoid");
    public static final AntuIcon APPLET_KICKER_256 = create("APPLET_KICKER_256", "applets/256/org.kde.plasma.kicker");
    public static final AntuIcon APPLET_KICKERDASH_256 = create("APPLET_KICKERDASH_256", "applets/256/org.kde.plasma.kickerdash");
    public static final AntuIcon APPLET_KICKOFF_256 = create("APPLET_KICKOFF_256", "applets/256/org.kde.plasma.kickoff");
    public static final AntuIcon APPLET_KIMPANEL_256 = create("APPLET_KIMPANEL_256", "applets/256/org.kde.plasma.kimpanel");
    public static final AntuIcon APPLET_KTPCONTACTLIST_256 = create("APPLET_KTPCONTACTLIST_256", "applets/256/org.kde.ktpcontactlist");
    public static final AntuIcon APPLET_MEDIACONTROLLER_256 = create("APPLET_MEDIACONTROLLER_256", "applets/256/org.kde.plasma.mediacontroller");
    public static final AntuIcon APPLET_MEDIACONTROLLERCOMPACT_256 = create("APPLET_MEDIACONTROLLERCOMPACT_256", "applets/256/org.kde.plasma.mediacontrollercompact");
    public static final AntuIcon APPLET_MULTIMEDIACONTROLS_256 = create("APPLET_MULTIMEDIACONTROLS_256", "applets/256/org.kde.plasma.multimediacontrols");
    public static final AntuIcon APPLET_MUONNOTIFIER_256 = create("APPLET_MUONNOTIFIER_256", "applets/256/org.kde.muonnotifier");
    public static final AntuIcon APPLET_NETWORKMANAGEMENT_256 = create("APPLET_NETWORKMANAGEMENT_256", "applets/256/org.kde.plasma.networkmanagement");
    public static final AntuIcon APPLET_NETWORKMONITOR_256 = create("APPLET_NETWORKMONITOR_256", "applets/256/org.kde.networkMonitor");
    public static final AntuIcon APPLET_NOTES_256 = create("APPLET_NOTES_256", "applets/256/org.kde.plasma.notes");
    public static final AntuIcon APPLET_PAGER_256 = create("APPLET_PAGER_256", "applets/256/org.kde.plasma.pager");
    public static final AntuIcon APPLET_QUICKLAUNCH_256 = create("APPLET_QUICKLAUNCH_256", "applets/256/org.kde.plasma.quicklaunch");
    public static final AntuIcon APPLET_REDSHIFTCONTROL_256 = create("APPLET_REDSHIFTCONTROL_256", "applets/256/org.kde.redshiftControl");
    public static final AntuIcon APPLET_RSS_FEEDS_256 = create("APPLET_RSS_FEEDS_256", "applets/256/RSS_feeds");
    public static final AntuIcon APPLET_SHOWACTIVITYMANAGER_256 = create("APPLET_SHOWACTIVITYMANAGER_256", "applets/256/org.kde.plasma.showActivityManager");
    public static final AntuIcon APPLET_SIMPLEDATEANDTIME_256 = create("APPLET_SIMPLEDATEANDTIME_256", "applets/256/org.kde.contrib.simpledateandtime");
    public static final AntuIcon APPLET_SYSTEMLOADVIEWER_256 = create("APPLET_SYSTEMLOADVIEWER_256", "applets/256/org.kde.plasma.systemloadviewer");
    public static final AntuIcon APPLET_SYSTEMMONITOR_CPU_256 = create("APPLET_SYSTEMMONITOR_CPU_256", "applets/256/org.kde.plasma.systemmonitor.cpu");
    public static final AntuIcon APPLET_SYSTEMMONITOR_DISKACTIVITY_256 = create("APPLET_SYSTEMMONITOR_DISKACTIVITY_256", "applets/256/org.kde.plasma.systemmonitor.diskactivity");
    public static final AntuIcon APPLET_SYSTEMMONITOR_DISKUSAGE_256 = create("APPLET_SYSTEMMONITOR_DISKUSAGE_256", "applets/256/org.kde.plasma.systemmonitor.diskusage");
    public static final AntuIcon APPLET_SYSTEMMONITOR_MEMORY_256 = create("APPLET_SYSTEMMONITOR_MEMORY_256", "applets/256/org.kde.plasma.systemmonitor.memory");
    public static final AntuIcon APPLET_SYSTEMMONITOR_NET_256 = create("APPLET_SYSTEMMONITOR_NET_256", "applets/256/org.kde.plasma.systemmonitor.net");
    public static final AntuIcon APPLET_SYSTEMTRAY_256 = create("APPLET_SYSTEMTRAY_256", "applets/256/org.kde.plasma.systemtray");
    public static final AntuIcon APPLET_TAKEABREAK_256 = create("APPLET_TAKEABREAK_256", "applets/256/org.kde.contrib.takeabreak");
    public static final AntuIcon APPLET_TASKMANAGER_256 = create("APPLET_TASKMANAGER_256", "applets/256/org.kde.plasma.taskmanager");
    public static final AntuIcon APPLET_THERMALMONITOR_256 = create("APPLET_THERMALMONITOR_256", "applets/256/org.kde.thermalMonitor");
    public static final AntuIcon APPLET_TIMER_256 = create("APPLET_TIMER_256", "applets/256/org.kde.plasma.timer");
    public static final AntuIcon APPLET_UPTIMEWIDGET_256 = create("APPLET_UPTIMEWIDGET_256", "applets/256/org.kde.uptimeWidget");
    public static final AntuIcon APPLET_USERSWITCHER_256 = create("APPLET_USERSWITCHER_256", "applets/256/org.kde.plasma.userswitcher");
    public static final AntuIcon APPLET_VOLUME_256 = create("APPLET_VOLUME_256", "applets/256/org.kde.plasma.volume");
    public static final AntuIcon APPLET_WEATHERWIDGET_256 = create("APPLET_WEATHERWIDGET_256", "applets/256/org.kde.weatherWidget");
    public static final AntuIcon APPLET_WEBSLICE_256 = create("APPLET_WEBSLICE_256", "applets/256/cqcb.plasma.webslice");
    public static final AntuIcon APPLET_WINDOWLIST_256 = create("APPLET_WINDOWLIST_256", "applets/256/org.kde.plasma.windowlist");
    public static final AntuIcon APPLET_YAHOOWEATHER_256 = create("APPLET_YAHOOWEATHER_256", "applets/256/com.librehat.yahooweather");

    // APPS
    public static final AntuIcon APP_0AD_48 = create("APP_0AD_48", "apps/48/0ad");
    public static final AntuIcon APP_1PASSWORD_48 = create("APP_1PASSWORD_48", "apps/48/1password");
    public static final AntuIcon APP_2048_48 = create("APP_2048_48", "apps/48/2048");
    public static final AntuIcon APP_2048_QT_48 = create("APP_2048_QT_48", "apps/48/2048-qt");
    public static final AntuIcon APP_4KVIDEODOWNLOADER_48 = create("APP_4KVIDEODOWNLOADER_48", "apps/48/4kvideodownloader");
    public static final AntuIcon APP_67EF_ADDONINSTALLER_0_48 = create("APP_67EF_ADDONINSTALLER_0_48", "apps/48/67EF_addoninstaller.0");
    public static final AntuIcon APP_7Z_48 = create("APP_7Z_48", "apps/48/7z");
    public static final AntuIcon APP_7ZIP_48 = create("APP_7ZIP_48", "apps/48/7zip");
    public static final AntuIcon APP_8_BALL_POOL_48 = create("APP_8_BALL_POOL_48", "apps/48/8-ball-pool");
    public static final AntuIcon APP_ABRT_48 = create("APP_ABRT_48", "apps/48/abrt");
    public static final AntuIcon APP_ACCESS_48 = create("APP_ACCESS_48", "apps/48/access");
    public static final AntuIcon APP_ACCESSIBILITY_48 = create("APP_ACCESSIBILITY_48", "apps/48/accessibility");
    public static final AntuIcon APP_ACCESSIBILITY_DIRECTORY_48 = create("APP_ACCESSIBILITY_DIRECTORY_48", "apps/48/accessibility-directory");
    public static final AntuIcon APP_ACCESSIBILITY_SECTION_48 = create("APP_ACCESSIBILITY_SECTION_48", "apps/48/accessibility_section");
    public static final AntuIcon APP_ACCESSORIES_ARCHIVER_48 = create("APP_ACCESSORIES_ARCHIVER_48", "apps/48/accessories-archiver");
    public static final AntuIcon APP_ACCESSORIES_CALCULATOR_48 = create("APP_ACCESSORIES_CALCULATOR_48", "apps/48/accessories-calculator");
    public static final AntuIcon APP_ACCESSORIES_CAMERA_48 = create("APP_ACCESSORIES_CAMERA_48", "apps/48/accessories-camera");
    public static final AntuIcon APP_ACCESSORIES_CHARACTER_MAP_48 = create("APP_ACCESSORIES_CHARACTER_MAP_48", "apps/48/accessories-character-map");
    public static final AntuIcon APP_ACCESSORIES_CLOCK_48 = create("APP_ACCESSORIES_CLOCK_48", "apps/48/accessories-clock");
    public static final AntuIcon APP_ACCESSORIES_DICTIONARY_48 = create("APP_ACCESSORIES_DICTIONARY_48", "apps/48/accessories-dictionary");
    public static final AntuIcon APP_ACCESSORIES_DOCUMENT_VIEWER_48 = create("APP_ACCESSORIES_DOCUMENT_VIEWER_48", "apps/48/accessories-document-viewer");
    public static final AntuIcon APP_ACCESSORIES_EBOOK_READER_48 = create("APP_ACCESSORIES_EBOOK_READER_48", "apps/48/accessories-ebook-reader");
    public static final AntuIcon APP_ACCESSORIES_NOTES_48 = create("APP_ACCESSORIES_NOTES_48", "apps/48/accessories-notes");
    public static final AntuIcon APP_ACCESSORIES_TEXT_EDITOR_48 = create("APP_ACCESSORIES_TEXT_EDITOR_48", "apps/48/accessories-text-editor");
    public static final AntuIcon APP_ACETINO2_48 = create("APP_ACETINO2_48", "apps/48/Acetino2");
    public static final AntuIcon APP_ACETONEISO_48 = create("APP_ACETONEISO_48", "apps/48/acetoneiso");
    public static final AntuIcon APP_ACROREAD_48 = create("APP_ACROREAD_48", "apps/48/acroread");
    public static final AntuIcon APP_ACTIVITY_LOG_MANAGER_48 = create("APP_ACTIVITY_LOG_MANAGER_48", "apps/48/activity-log-manager");
    public static final AntuIcon APP_ADDRESSBOOK_48 = create("APP_ADDRESSBOOK_48", "apps/48/addressbook");
    public static final AntuIcon APP_ADJUST_COLORS_48 = create("APP_ADJUST_COLORS_48", "apps/48/adjust-colors");
    public static final AntuIcon APP_ADMINISTRATION_48 = create("APP_ADMINISTRATION_48", "apps/48/administration");
    public static final AntuIcon APP_ADOBE_AIR_48 = create("APP_ADOBE_AIR_48", "apps/48/adobe-air");
    public static final AntuIcon APP_ADOBE_FLASHPLAYER_48 = create("APP_ADOBE_FLASHPLAYER_48", "apps/48/adobe-flashplayer");
    public static final AntuIcon APP_ADOBEAFTEREFFECT_48 = create("APP_ADOBEAFTEREFFECT_48", "apps/48/AdobeAfterEffect");
    public static final AntuIcon APP_ADOBEAUDITION_48 = create("APP_ADOBEAUDITION_48", "apps/48/AdobeAudition");
    public static final AntuIcon APP_ADOBEBRIDGE_48 = create("APP_ADOBEBRIDGE_48", "apps/48/AdobeBridge");
    public static final AntuIcon APP_ADOBEDREAMWEAVER_48 = create("APP_ADOBEDREAMWEAVER_48", "apps/48/AdobeDreamweaver");
    public static final AntuIcon APP_ADOBEENCORE_48 = create("APP_ADOBEENCORE_48", "apps/48/AdobeEncore");
    public static final AntuIcon APP_ADOBEFIREWORK_48 = create("APP_ADOBEFIREWORK_48", "apps/48/AdobeFirework");
    public static final AntuIcon APP_ADOBEFLASH_48 = create("APP_ADOBEFLASH_48", "apps/48/AdobeFlash");
    public static final AntuIcon APP_ADOBEFLASHBUILDER_48 = create("APP_ADOBEFLASHBUILDER_48", "apps/48/AdobeFlashBuilder");
    public static final AntuIcon APP_ADOBEFLASHPLUGIN_48 = create("APP_ADOBEFLASHPLUGIN_48", "apps/48/adobeflashplugin");
    public static final AntuIcon APP_ADOBEILLUSTRATOR_48 = create("APP_ADOBEILLUSTRATOR_48", "apps/48/Adobeillustrator");
    public static final AntuIcon APP_ADOBEINDESIGN_48 = create("APP_ADOBEINDESIGN_48", "apps/48/AdobeIndesign");
    public static final AntuIcon APP_ADOBELIGHTROOM_48 = create("APP_ADOBELIGHTROOM_48", "apps/48/AdobeLightroom");
    public static final AntuIcon APP_ADOBEPHOTOSHOP_48 = create("APP_ADOBEPHOTOSHOP_48", "apps/48/AdobePhotoshop");
    public static final AntuIcon APP_ADOBEPRELUDE_48 = create("APP_ADOBEPRELUDE_48", "apps/48/AdobePrelude");
    public static final AntuIcon APP_ADOBEPREMIEREPRO_48 = create("APP_ADOBEPREMIEREPRO_48", "apps/48/AdobePremierePro");
    public static final AntuIcon APP_ADOBESPEEDGRADE_48 = create("APP_ADOBESPEEDGRADE_48", "apps/48/AdobeSpeedgrade");
    public static final AntuIcon APP_ADOBEUPDATE_48 = create("APP_ADOBEUPDATE_48", "apps/48/AdobeUpdate");
    public static final AntuIcon APP_ADOBEWIDGETBROWSER_48 = create("APP_ADOBEWIDGETBROWSER_48", "apps/48/AdobeWidgetBrowser");
    public static final AntuIcon APP_AIM_48 = create("APP_AIM_48", "apps/48/aim");
    public static final AntuIcon APP_AIM_PROTOCOL_48 = create("APP_AIM_PROTOCOL_48", "apps/48/aim_protocol");
    public static final AntuIcon APP_AIRPORTUTILITY_48 = create("APP_AIRPORTUTILITY_48", "apps/48/AirPortUtility");
    public static final AntuIcon APP_AIRWAVE_MANAGER_48 = create("APP_AIRWAVE_MANAGER_48", "apps/48/airwave-manager");
    public static final AntuIcon APP_AKONADI_48 = create("APP_AKONADI_48", "apps/48/akonadi");
    public static final AntuIcon APP_AKONADICONSOLE_48 = create("APP_AKONADICONSOLE_48", "apps/48/akonadiconsole");
    public static final AntuIcon APP_AKONADITRAY_48 = create("APP_AKONADITRAY_48", "apps/48/akonaditray");
    public static final AntuIcon APP_AKREGATOR_22 = create("APP_AKREGATOR_22", "apps/22/akregator");
    public static final AntuIcon APP_AKREGATOR_48 = create("APP_AKREGATOR_48", "apps/48/akregator");
    public static final AntuIcon APP_AKREGATOR_EMPTY_48 = create("APP_AKREGATOR_EMPTY_48", "apps/48/akregator_empty");
    public static final AntuIcon APP_ALACARTE_48 = create("APP_ALACARTE_48", "apps/48/alacarte");
    public static final AntuIcon APP_ALARM_CLOCK_48 = create("APP_ALARM_CLOCK_48", "apps/48/alarm-clock");
    public static final AntuIcon APP_ALCHEMY_48 = create("APP_ALCHEMY_48", "apps/48/alchemy");
    public static final AntuIcon APP_ALEXANDRA_48 = create("APP_ALEXANDRA_48", "apps/48/alexandra");
    public static final AntuIcon APP_ALIEN_ARENA_48 = create("APP_ALIEN_ARENA_48", "apps/48/alien-arena");
    public static final AntuIcon APP_ALIENARENA_48 = create("APP_ALIENARENA_48", "apps/48/alienarena");
    public static final AntuIcon APP_ALSAMIXERGUI_48 = create("APP_ALSAMIXERGUI_48", "apps/48/alsamixergui");
    public static final AntuIcon APP_AMARK_48 = create("APP_AMARK_48", "apps/48/amark");
    public static final AntuIcon APP_AMAROK_48 = create("APP_AMAROK_48", "apps/48/amarok");
    public static final AntuIcon APP_AMAROK_CHANGE_LANGUAGE_32 = create("APP_AMAROK_CHANGE_LANGUAGE_32", "apps/32/amarok_change_language");
    public static final AntuIcon APP_AMAROK_CHANGE_LANGUAGE_48 = create("APP_AMAROK_CHANGE_LANGUAGE_48", "apps/48/amarok_change_language");
    public static final AntuIcon APP_AMAROK_ICON_48 = create("APP_AMAROK_ICON_48", "apps/48/amarok-icon");
    public static final AntuIcon APP_AMAROK2_48 = create("APP_AMAROK2_48", "apps/48/amarok2");
    public static final AntuIcon APP_AMAZON_MP3_STORE_SOURCE_48 = create("APP_AMAZON_MP3_STORE_SOURCE_48", "apps/48/amazon-mp3-store-source");
    public static final AntuIcon APP_AMAZON_STORE_48 = create("APP_AMAZON_STORE_48", "apps/48/amazon-store");
    public static final AntuIcon APP_AMD_48 = create("APP_AMD_48", "apps/48/amd");
    public static final AntuIcon APP_AMD_ATI_48 = create("APP_AMD_ATI_48", "apps/48/amd-ati");
    public static final AntuIcon APP_AMULE_48 = create("APP_AMULE_48", "apps/48/amule");
    public static final AntuIcon APP_AMZDOWNLOADER_48 = create("APP_AMZDOWNLOADER_48", "apps/48/amzdownloader");
    public static final AntuIcon APP_ANACONDA_48 = create("APP_ANACONDA_48", "apps/48/anaconda");
    public static final AntuIcon APP_ANDROID_FILE_TRANSFER_48 = create("APP_ANDROID_FILE_TRANSFER_48", "apps/48/android-file-transfer");
    public static final AntuIcon APP_ANDROID_SDK_48 = create("APP_ANDROID_SDK_48", "apps/48/android-sdk");
    public static final AntuIcon APP_ANDROID_STUDIO_48 = create("APP_ANDROID_STUDIO_48", "apps/48/android-studio");
    public static final AntuIcon APP_ANDROIDSTUDIO_48 = create("APP_ANDROIDSTUDIO_48", "apps/48/androidstudio");
    public static final AntuIcon APP_ANJUTA_48 = create("APP_ANJUTA_48", "apps/48/anjuta");
    public static final AntuIcon APP_ANKI_48 = create("APP_ANKI_48", "apps/48/anki");
    public static final AntuIcon APP_ANY_DO_48 = create("APP_ANY_DO_48", "apps/48/any_do");
    public static final AntuIcon APP_APP_LAUNCHER_48 = create("APP_APP_LAUNCHER_48", "apps/48/app-launcher");
    public static final AntuIcon APP_APPARMOR_VIEW_PROFILE_48 = create("APP_APPARMOR_VIEW_PROFILE_48", "apps/48/apparmor_view_profile");
    public static final AntuIcon APP_APPGRID_48 = create("APP_APPGRID_48", "apps/48/appgrid");
    public static final AntuIcon APP_APPLETS_SCREENSHOOTER_48 = create("APP_APPLETS_SCREENSHOOTER_48", "apps/48/applets-screenshooter");
    public static final AntuIcon APP_APPLICATION_COMMUNITY_48 = create("APP_APPLICATION_COMMUNITY_48", "apps/48/application-community");
    public static final AntuIcon APP_APPLICATION_DEFAULT_ICON_48 = create("APP_APPLICATION_DEFAULT_ICON_48", "apps/48/application-default-icon");
    public static final AntuIcon APP_APPLICATION_SXW_48 = create("APP_APPLICATION_SXW_48", "apps/48/application-sxw");
    public static final AntuIcon APP_APPLICATION_X_CLEMENTINE_48 = create("APP_APPLICATION_X_CLEMENTINE_48", "apps/48/application-x-clementine");
    public static final AntuIcon APP_APPLICATION_X_REMOTE_CONNECTION_48 = create("APP_APPLICATION_X_REMOTE_CONNECTION_48", "apps/48/application-x-remote-connection");
    public static final AntuIcon APP_APPLICATION_X_VNC_48 = create("APP_APPLICATION_X_VNC_48", "apps/48/application-x-vnc");
    public static final AntuIcon APP_APPLICATION_X_YAROCK_48 = create("APP_APPLICATION_X_YAROCK_48", "apps/48/application-x-yarock");
    public static final AntuIcon APP_APPLICATIONS_ACCESSORIES_48 = create("APP_APPLICATIONS_ACCESSORIES_48", "apps/48/applications-accessories");
    public static final AntuIcon APP_APPLICATIONS_DEVELOPMENT_48 = create("APP_APPLICATIONS_DEVELOPMENT_48", "apps/48/applications-development");
    public static final AntuIcon APP_APPLICATIONS_DEVELOPMENT_TRANSLATION_48 = create("APP_APPLICATIONS_DEVELOPMENT_TRANSLATION_48", "apps/48/applications-development-translation");
    public static final AntuIcon APP_APPLICATIONS_DEVELOPMENT_WEB_48 = create("APP_APPLICATIONS_DEVELOPMENT_WEB_48", "apps/48/applications-development-web");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_48 = create("APP_APPLICATIONS_EDUCATION_48", "apps/48/applications-education");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_LANGUAGE_48 = create("APP_APPLICATIONS_EDUCATION_LANGUAGE_48", "apps/48/applications-education-language");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_MATHEMATICS_48 = create("APP_APPLICATIONS_EDUCATION_MATHEMATICS_48", "apps/48/applications-education-mathematics");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_MISCELLANEOUS_48 = create("APP_APPLICATIONS_EDUCATION_MISCELLANEOUS_48", "apps/48/applications-education-miscellaneous");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_PRESCHOOL_48 = create("APP_APPLICATIONS_EDUCATION_PRESCHOOL_48", "apps/48/applications-education-preschool");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_SCHOOL_48 = create("APP_APPLICATIONS_EDUCATION_SCHOOL_48", "apps/48/applications-education-school");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_SCIENCE_48 = create("APP_APPLICATIONS_EDUCATION_SCIENCE_48", "apps/48/applications-education-science");
    public static final AntuIcon APP_APPLICATIONS_EDUCATION_UNIVERSITY_48 = create("APP_APPLICATIONS_EDUCATION_UNIVERSITY_48", "apps/48/applications-education-university");
    public static final AntuIcon APP_APPLICATIONS_ENGINEERING_48 = create("APP_APPLICATIONS_ENGINEERING_48", "apps/48/applications-engineering");
    public static final AntuIcon APP_APPLICATIONS_GAMES_48 = create("APP_APPLICATIONS_GAMES_48", "apps/48/applications-games");
    public static final AntuIcon APP_APPLICATIONS_GRAPHICS_48 = create("APP_APPLICATIONS_GRAPHICS_48", "apps/48/applications-graphics");
    public static final AntuIcon APP_APPLICATIONS_INTERNET_48 = create("APP_APPLICATIONS_INTERNET_48", "apps/48/applications-internet");
    public static final AntuIcon APP_APPLICATIONS_MULTIMEDIA_48 = create("APP_APPLICATIONS_MULTIMEDIA_48", "apps/48/applications-multimedia");
    public static final AntuIcon APP_APPLICATIONS_OFFICE_48 = create("APP_APPLICATIONS_OFFICE_48", "apps/48/applications-office");
    public static final AntuIcon APP_APPLICATIONS_OTHER_48 = create("APP_APPLICATIONS_OTHER_48", "apps/48/applications-other");
    public static final AntuIcon APP_APPLICATIONS_SCIENCE_48 = create("APP_APPLICATIONS_SCIENCE_48", "apps/48/applications-science");
    public static final AntuIcon APP_APPLICATIONS_SYSTEM_48 = create("APP_APPLICATIONS_SYSTEM_48", "apps/48/applications-system");
    public static final AntuIcon APP_APPLICATIONS_SYSTEM_ORANGE_48 = create("APP_APPLICATIONS_SYSTEM_ORANGE_48", "apps/48/applications-system-orange");
    public static final AntuIcon APP_APPLICATIONS_TOYS_48 = create("APP_APPLICATIONS_TOYS_48", "apps/48/applications-toys");
    public static final AntuIcon APP_APPLICATIONS_UTILITIES_48 = create("APP_APPLICATIONS_UTILITIES_48", "apps/48/applications-utilities");
    public static final AntuIcon APP_APPORT_48 = create("APP_APPORT_48", "apps/48/apport");
    public static final AntuIcon APP_APPORT_GTK2_48 = create("APP_APPORT_GTK2_48", "apps/48/apport-gtk2");
    public static final AntuIcon APP_APTDAEMON_DOWNLOAD_48 = create("APP_APTDAEMON_DOWNLOAD_48", "apps/48/aptdaemon-download");
    public static final AntuIcon APP_APTDAEMON_UPDATE_CACHE_48 = create("APP_APTDAEMON_UPDATE_CACHE_48", "apps/48/aptdaemon-update-cache");
    public static final AntuIcon APP_APTDAEMON_UPGRADE_48 = create("APP_APTDAEMON_UPGRADE_48", "apps/48/aptdaemon-upgrade");
    public static final AntuIcon APP_APTDAEMON_WORKING_48 = create("APP_APTDAEMON_WORKING_48", "apps/48/aptdaemon-working");
    public static final AntuIcon APP_APTIK_48 = create("APP_APTIK_48", "apps/48/aptik");
    public static final AntuIcon APP_ARCHIVE_MANAGER_48 = create("APP_ARCHIVE_MANAGER_48", "apps/48/archive-manager");
    public static final AntuIcon APP_ARCHIVEMANAGER_48 = create("APP_ARCHIVEMANAGER_48", "apps/48/archivemanager");
    public static final AntuIcon APP_ARDOUR_48 = create("APP_ARDOUR_48", "apps/48/ardour");
    public static final AntuIcon APP_ARDUINO_48 = create("APP_ARDUINO_48", "apps/48/arduino");
    public static final AntuIcon APP_ARDUINO_ICON_SMALL_48 = create("APP_ARDUINO_ICON_SMALL_48", "apps/48/arduino-icon-small");
    public static final AntuIcon APP_ARDUINO_IDE_48 = create("APP_ARDUINO_IDE_48", "apps/48/arduino-ide");
    public static final AntuIcon APP_ARIO_48 = create("APP_ARIO_48", "apps/48/ario");
    public static final AntuIcon APP_ARK_48 = create("APP_ARK_48", "apps/48/ark");
    public static final AntuIcon APP_ARRONAX_48 = create("APP_ARRONAX_48", "apps/48/arronax");
    public static final AntuIcon APP_ARTHA_48 = create("APP_ARTHA_48", "apps/48/artha");
    public static final AntuIcon APP_ARTIKULATE_48 = create("APP_ARTIKULATE_48", "apps/48/artikulate");
    public static final AntuIcon APP_ARTS_48 = create("APP_ARTS_48", "apps/48/arts");
    public static final AntuIcon APP_ASOUNDER_48 = create("APP_ASOUNDER_48", "apps/48/asounder");
    public static final AntuIcon APP_ASSAULTCUBE_48 = create("APP_ASSAULTCUBE_48", "apps/48/assaultcube");
    public static final AntuIcon APP_ASSISTANT_48 = create("APP_ASSISTANT_48", "apps/48/assistant");
    public static final AntuIcon APP_ASSISTANT_QT4_48 = create("APP_ASSISTANT_QT4_48", "apps/48/assistant-qt4");
    public static final AntuIcon APP_ASSISTANT_QT5_48 = create("APP_ASSISTANT_QT5_48", "apps/48/assistant-qt5");
    public static final AntuIcon APP_ASSISTANT5_48 = create("APP_ASSISTANT5_48", "apps/48/assistant5");
    public static final AntuIcon APP_ASTAH_COMMUNITY_48 = create("APP_ASTAH_COMMUNITY_48", "apps/48/astah-community");
    public static final AntuIcon APP_ASUNDER_48 = create("APP_ASUNDER_48", "apps/48/asunder");
    public static final AntuIcon APP_ATOM_48 = create("APP_ATOM_48", "apps/48/atom");
    public static final AntuIcon APP_ATOM_TEXT_EDITOR_48 = create("APP_ATOM_TEXT_EDITOR_48", "apps/48/atom-text-editor");
    public static final AntuIcon APP_ATOMIX_48 = create("APP_ATOMIX_48", "apps/48/atomix");
    public static final AntuIcon APP_ATOMIX_ICON_48 = create("APP_ATOMIX_ICON_48", "apps/48/atomix-icon");
    public static final AntuIcon APP_ATUNES_48 = create("APP_ATUNES_48", "apps/48/atunes");
    public static final AntuIcon APP_AUDACIOUS_48 = create("APP_AUDACIOUS_48", "apps/48/audacious");
    public static final AntuIcon APP_AUDACITY_48 = create("APP_AUDACITY_48", "apps/48/audacity");
    public static final AntuIcon APP_AUDEX_48 = create("APP_AUDEX_48", "apps/48/audex");
    public static final AntuIcon APP_AUDIENCE_48 = create("APP_AUDIENCE_48", "apps/48/audience");
    public static final AntuIcon APP_AUDIO_EDITOR_48 = create("APP_AUDIO_EDITOR_48", "apps/48/audio-editor");
    public static final AntuIcon APP_AUDIO_PLAYER_48 = create("APP_AUDIO_PLAYER_48", "apps/48/audio-player");
    public static final AntuIcon APP_AUDIO_TAG_TOOL_48 = create("APP_AUDIO_TAG_TOOL_48", "apps/48/audio-tag-tool");
    public static final AntuIcon APP_AUDIOBOOK_48 = create("APP_AUDIOBOOK_48", "apps/48/audiobook");
    public static final AntuIcon APP_AUTHOR_48 = create("APP_AUTHOR_48", "apps/48/author");
    public static final AntuIcon APP_AUTOPLUS_48 = create("APP_AUTOPLUS_48", "apps/48/autoplus");
    public static final AntuIcon APP_AVASTGUI_48 = create("APP_AVASTGUI_48", "apps/48/avastgui");
    public static final AntuIcon APP_AVIDEMUX_48 = create("APP_AVIDEMUX_48", "apps/48/avidemux");
    public static final AntuIcon APP_AWF_48 = create("APP_AWF_48", "apps/48/awf");
    public static final AntuIcon APP_AWN_APPLET_48 = create("APP_AWN_APPLET_48", "apps/48/awn-applet");
    public static final AntuIcon APP_AWN_APPLET_DIGITAL_CLOCK_48 = create("APP_AWN_APPLET_DIGITAL_CLOCK_48", "apps/48/awn-applet-digital-clock");
    public static final AntuIcon APP_AX_APPLET_48 = create("APP_AX_APPLET_48", "apps/48/ax-applet");
    public static final AntuIcon APP_AZUREUS_48 = create("APP_AZUREUS_48", "apps/48/azureus");
    public static final AntuIcon APP_BACKGROUND_48 = create("APP_BACKGROUND_48", "apps/48/background");
    public static final AntuIcon APP_BALOO_32 = create("APP_BALOO_32", "apps/32/baloo");
    public static final AntuIcon APP_BALOO_48 = create("APP_BALOO_48", "apps/48/baloo");
    public static final AntuIcon APP_BANGARANG_48 = create("APP_BANGARANG_48", "apps/48/bangarang");
    public static final AntuIcon APP_BANSHEE_1_48 = create("APP_BANSHEE_1_48", "apps/48/banshee-1");
    public static final AntuIcon APP_BANSHEE_48 = create("APP_BANSHEE_48", "apps/48/banshee");
    public static final AntuIcon APP_BAOBAB_48 = create("APP_BAOBAB_48", "apps/48/baobab");
    public static final AntuIcon APP_BASH_48 = create("APP_BASH_48", "apps/48/bash");
    public static final AntuIcon APP_BASKET_48 = create("APP_BASKET_48", "apps/48/basket");
    public static final AntuIcon APP_BATTLEFORWESNOTH_48 = create("APP_BATTLEFORWESNOTH_48", "apps/48/battleforwesnoth");
    public static final AntuIcon APP_BBALL_48 = create("APP_BBALL_48", "apps/48/bball");
    public static final AntuIcon APP_BEATBOX_48 = create("APP_BEATBOX_48", "apps/48/beatbox");
    public static final AntuIcon APP_BERYL_48 = create("APP_BERYL_48", "apps/48/beryl");
    public static final AntuIcon APP_BERYL_SETTINGS_48 = create("APP_BERYL_SETTINGS_48", "apps/48/beryl-settings");
    public static final AntuIcon APP_BIJIBEN_48 = create("APP_BIJIBEN_48", "apps/48/bijiben");
    public static final AntuIcon APP_BIRDIE_48 = create("APP_BIRDIE_48", "apps/48/birdie");
    public static final AntuIcon APP_BITCOIN_QT_48 = create("APP_BITCOIN_QT_48", "apps/48/bitcoin-qt");
    public static final AntuIcon APP_BITDEFENDER_48 = create("APP_BITDEFENDER_48", "apps/48/bitdefender");
    public static final AntuIcon APP_BITMAP2COMPONENT_48 = create("APP_BITMAP2COMPONENT_48", "apps/48/bitmap2component");
    public static final AntuIcon APP_BITTORENT_48 = create("APP_BITTORENT_48", "apps/48/bittorent");
    public static final AntuIcon APP_BITTORRENT_48 = create("APP_BITTORRENT_48", "apps/48/bittorrent");
    public static final AntuIcon APP_BITWIG_STUDIO_48 = create("APP_BITWIG_STUDIO_48", "apps/48/bitwig-studio");
    public static final AntuIcon APP_BLEACHBIT_48 = create("APP_BLEACHBIT_48", "apps/48/bleachbit");
    public static final AntuIcon APP_BLENDER_48 = create("APP_BLENDER_48", "apps/48/blender");
    public static final AntuIcon APP_BLESS_48X48_48 = create("APP_BLESS_48X48_48", "apps/48/bless-48x48");
    public static final AntuIcon APP_BLOGGER_48 = create("APP_BLOGGER_48", "apps/48/blogger");
    public static final AntuIcon APP_BLOGILO_48 = create("APP_BLOGILO_48", "apps/48/blogilo");
    public static final AntuIcon APP_BLUEJ_48 = create("APP_BLUEJ_48", "apps/48/bluej");
    public static final AntuIcon APP_BLUETOOTH_48 = create("APP_BLUETOOTH_48", "apps/48/bluetooth");
    public static final AntuIcon APP_BOINC_MANAGER_48 = create("APP_BOINC_MANAGER_48", "apps/48/boinc-manager");
    public static final AntuIcon APP_BOMI_48 = create("APP_BOMI_48", "apps/48/bomi");
    public static final AntuIcon APP_BOOT_REPAIR_48 = create("APP_BOOT_REPAIR_48", "apps/48/boot-repair");
    public static final AntuIcon APP_BRACKETS_48 = create("APP_BRACKETS_48", "apps/48/brackets");
    public static final AntuIcon APP_BRAINDUMP_48 = create("APP_BRAINDUMP_48", "apps/48/braindump");
    public static final AntuIcon APP_BRASERO_48 = create("APP_BRASERO_48", "apps/48/brasero");
    public static final AntuIcon APP_BREEZE_SETTINGS_48 = create("APP_BREEZE_SETTINGS_48", "apps/48/breeze-settings");
    public static final AntuIcon APP_BROKENSWORD_48 = create("APP_BROKENSWORD_48", "apps/48/BrokenSword");
    public static final AntuIcon APP_BTSYNC_USER_48 = create("APP_BTSYNC_USER_48", "apps/48/btsync-user");
    public static final AntuIcon APP_BUG_BUDDY_48 = create("APP_BUG_BUDDY_48", "apps/48/bug-buddy");
    public static final AntuIcon APP_BZR_ICON_64_48 = create("APP_BZR_ICON_64_48", "apps/48/bzr-icon-64");
    public static final AntuIcon APP_CAFFEINE_48 = create("APP_CAFFEINE_48", "apps/48/caffeine");
    public static final AntuIcon APP_CALAMARES_48 = create("APP_CALAMARES_48", "apps/48/calamares");
    public static final AntuIcon APP_CALENDAR_48 = create("APP_CALENDAR_48", "apps/48/calendar");
    public static final AntuIcon APP_CALIBRE_48 = create("APP_CALIBRE_48", "apps/48/calibre");
    public static final AntuIcon APP_CALIBRE_EBOOK_EDIT_48 = create("APP_CALIBRE_EBOOK_EDIT_48", "apps/48/calibre-ebook-edit");
    public static final AntuIcon APP_CALIBRE_GUI_48 = create("APP_CALIBRE_GUI_48", "apps/48/calibre-gui");
    public static final AntuIcon APP_CALIBRE_VIEWER_48 = create("APP_CALIBRE_VIEWER_48", "apps/48/calibre-viewer");
    public static final AntuIcon APP_CALIBRE2_48 = create("APP_CALIBRE2_48", "apps/48/calibre2");
    public static final AntuIcon APP_CALLIGRA_EXTRAS_48 = create("APP_CALLIGRA_EXTRAS_48", "apps/48/calligra-extras");
    public static final AntuIcon APP_CALLIGRAAUTHOR_48 = create("APP_CALLIGRAAUTHOR_48", "apps/48/calligraauthor");
    public static final AntuIcon APP_CALLIGRADEVTOOLS_48 = create("APP_CALLIGRADEVTOOLS_48", "apps/48/calligradevtools");
    public static final AntuIcon APP_CALLIGRAFILTERS_48 = create("APP_CALLIGRAFILTERS_48", "apps/48/calligrafilters");
    public static final AntuIcon APP_CALLIGRAFLOW_48 = create("APP_CALLIGRAFLOW_48", "apps/48/calligraflow");
    public static final AntuIcon APP_CALLIGRAGEMINI_48 = create("APP_CALLIGRAGEMINI_48", "apps/48/calligragemini");
    public static final AntuIcon APP_CALLIGRAHANDBOOK_48 = create("APP_CALLIGRAHANDBOOK_48", "apps/48/calligrahandbook");
    public static final AntuIcon APP_CALLIGRAKARBON_48 = create("APP_CALLIGRAKARBON_48", "apps/48/calligrakarbon");
    public static final AntuIcon APP_CALLIGRAKEXI_48 = create("APP_CALLIGRAKEXI_48", "apps/48/calligrakexi");
    public static final AntuIcon APP_CALLIGRAKRITA_48 = create("APP_CALLIGRAKRITA_48", "apps/48/calligrakrita");
    public static final AntuIcon APP_CALLIGRAKRITA_L_48 = create("APP_CALLIGRAKRITA_L_48", "apps/48/calligrakrita_l");
    public static final AntuIcon APP_CALLIGRAPLAN_48 = create("APP_CALLIGRAPLAN_48", "apps/48/calligraplan");
    public static final AntuIcon APP_CALLIGRAPLANWORK_48 = create("APP_CALLIGRAPLANWORK_48", "apps/48/calligraplanwork");
    public static final AntuIcon APP_CALLIGRAPLUGINS_48 = create("APP_CALLIGRAPLUGINS_48", "apps/48/calligraplugins");
    public static final AntuIcon APP_CALLIGRASHEETS_48 = create("APP_CALLIGRASHEETS_48", "apps/48/calligrasheets");
    public static final AntuIcon APP_CALLIGRASTAGE_48 = create("APP_CALLIGRASTAGE_48", "apps/48/calligrastage");
    public static final AntuIcon APP_CALLIGRAWORDS_48 = create("APP_CALLIGRAWORDS_48", "apps/48/calligrawords");
    public static final AntuIcon APP_CAMERA_WEB_48 = create("APP_CAMERA_WEB_48", "apps/48/camera-web");
    public static final AntuIcon APP_CANOPY_48 = create("APP_CANOPY_48", "apps/48/canopy");
    public static final AntuIcon APP_CANTATA_48 = create("APP_CANTATA_48", "apps/48/cantata");
    public static final AntuIcon APP_CANTOR_48 = create("APP_CANTOR_48", "apps/48/cantor");
    public static final AntuIcon APP_CCC_LARGE_48 = create("APP_CCC_LARGE_48", "apps/48/ccc_large");
    public static final AntuIcon APP_CELESTIA_48 = create("APP_CELESTIA_48", "apps/48/celestia");
    public static final AntuIcon APP_CHAKRA_BACKUP_48 = create("APP_CHAKRA_BACKUP_48", "apps/48/chakra-backup");
    public static final AntuIcon APP_CHAKRA_FLAT_48 = create("APP_CHAKRA_FLAT_48", "apps/48/chakra-flat");
    public static final AntuIcon APP_CHAKRA_SHIELD_48 = create("APP_CHAKRA_SHIELD_48", "apps/48/chakra-shield");
    public static final AntuIcon APP_CHAKRA_SHINY_48 = create("APP_CHAKRA_SHINY_48", "apps/48/chakra-shiny");
    public static final AntuIcon APP_CHAKRA_SMOOTH_48 = create("APP_CHAKRA_SMOOTH_48", "apps/48/chakra-smooth");
    public static final AntuIcon APP_CHARM_48 = create("APP_CHARM_48", "apps/48/Charm");
    public static final AntuIcon APP_CHECKBOX_48 = create("APP_CHECKBOX_48", "apps/48/checkbox");
    public static final AntuIcon APP_CHECKBOX_QT_48 = create("APP_CHECKBOX_QT_48", "apps/48/checkbox-qt");
    public static final AntuIcon APP_CHEESE_48 = create("APP_CHEESE_48", "apps/48/cheese");
    public static final AntuIcon APP_CHERRYTREE_48 = create("APP_CHERRYTREE_48", "apps/48/cherrytree");
    public static final AntuIcon APP_CHESS_48 = create("APP_CHESS_48", "apps/48/chess");
    public static final AntuIcon APP_CHINESECALENDAR_64_48 = create("APP_CHINESECALENDAR_64_48", "apps/48/chinesecalendar-64");
    public static final AntuIcon APP_CHOQOK_48 = create("APP_CHOQOK_48", "apps/48/choqok");
    public static final AntuIcon APP_CHROME_48 = create("APP_CHROME_48", "apps/48/chrome");
    public static final AntuIcon APP_CHROME_AHFGEIENLIHCKOGMOHJHADLKJGOCPLEB_DEFAULT_48 = create("APP_CHROME_AHFGEIENLIHCKOGMOHJHADLKJGOCPLEB_DEFAULT_48", "apps/48/chrome-ahfgeienlihckogmohjhadlkjgocpleb-Default");
    public static final AntuIcon APP_CHROME_APP_LIST_48 = create("APP_CHROME_APP_LIST_48", "apps/48/chrome-app-list");
    public static final AntuIcon APP_CHROME_CMNPFFGFPCOHHPODDJANKJANOLCEKBNI_DEFAULT_48 = create("APP_CHROME_CMNPFFGFPCOHHPODDJANKJANOLCEKBNI_DEFAULT_48", "apps/48/chrome-cmnpffgfpcohhpoddjankjanolcekbni-Default");
    public static final AntuIcon APP_CHROME_DKFHFAPHFKOPDGPBFKEBJFCBLCAFCMPI_DEFAULT_48 = create("APP_CHROME_DKFHFAPHFKOPDGPBFKEBJFCBLCAFCMPI_DEFAULT_48", "apps/48/chrome-dkfhfaphfkopdgpbfkebjfcblcafcmpi-Default");
    public static final AntuIcon APP_CHROME_FHBJGBIFLINJBDGGEHCDDCBNCDDDOMOP_DEFAULT_48 = create("APP_CHROME_FHBJGBIFLINJBDGGEHCDDCBNCDDDOMOP_DEFAULT_48", "apps/48/chrome-fhbjgbiflinjbdggehcddcbncdddomop-Default");
    public static final AntuIcon APP_CHROME_FNIBMBCDEEPAAHJMDDIIHOHJANLIMLMJ_DEFAULT_48 = create("APP_CHROME_FNIBMBCDEEPAAHJMDDIIHOHJANLIMLMJ_DEFAULT_48", "apps/48/chrome-fnibmbcdeepaahjmddiihohjanlimlmj-Default");
    public static final AntuIcon APP_CHROME_GIGHMMPIOBKLFEPJOCNAMGKKBIGLIDOM_DEFAULT_48 = create("APP_CHROME_GIGHMMPIOBKLFEPJOCNAMGKKBIGLIDOM_DEFAULT_48", "apps/48/chrome-gighmmpiobklfepjocnamgkkbiglidom-Default");
    public static final AntuIcon APP_CHROME_IGGPFPNAHKGPNINDFKDNCKNOLDGNCCDG_DEFAULT_48 = create("APP_CHROME_IGGPFPNAHKGPNINDFKDNCKNOLDGNCCDG_DEFAULT_48", "apps/48/chrome-iggpfpnahkgpnindfkdncknoldgnccdg-Default");
    public static final AntuIcon APP_CHROME_IJFECBILADPINDDBJFODAAIAHGGOMHAF_DEFAULT_48 = create("APP_CHROME_IJFECBILADPINDDBJFODAAIAHGGOMHAF_DEFAULT_48", "apps/48/chrome-ijfecbiladpinddbjfodaaiahggomhaf-Default");
    public static final AntuIcon APP_CHROME_JFBADLNDCMINBKFOJHLIMNKGAACKJMDO_DEFAULT_48 = create("APP_CHROME_JFBADLNDCMINBKFOJHLIMNKGAACKJMDO_DEFAULT_48", "apps/48/chrome-jfbadlndcminbkfojhlimnkgaackjmdo-Default");
    public static final AntuIcon APP_CHROME_PKBFAKCPHDOKDNBDHLNDLPCEHEBMPLAB_DEFAULT_48 = create("APP_CHROME_PKBFAKCPHDOKDNBDHLNDLPCEHEBMPLAB_DEFAULT_48", "apps/48/chrome-pkbfakcphdokdnbdhlndlpcehebmplab-Default");
    public static final AntuIcon APP_CHROMIUM_48 = create("APP_CHROMIUM_48", "apps/48/chromium");
    public static final AntuIcon APP_CHROMIUM_APP_LIST_48 = create("APP_CHROMIUM_APP_LIST_48", "apps/48/chromium-app-list");
    public static final AntuIcon APP_CHROMIUM_BROWSER_48 = create("APP_CHROMIUM_BROWSER_48", "apps/48/chromium-browser");
    public static final AntuIcon APP_CHROMIUM_BSU_48 = create("APP_CHROMIUM_BSU_48", "apps/48/chromium-bsu");
    public static final AntuIcon APP_CISCO_PT_48 = create("APP_CISCO_PT_48", "apps/48/cisco-pt");
    public static final AntuIcon APP_CLAMAV_48 = create("APP_CLAMAV_48", "apps/48/clamav");
    public static final AntuIcon APP_CLASSICMENU_INDICATOR_DARK_48 = create("APP_CLASSICMENU_INDICATOR_DARK_48", "apps/48/classicmenu-indicator-dark");
    public static final AntuIcon APP_CLEMENTINE_48 = create("APP_CLEMENTINE_48", "apps/48/clementine");
    public static final AntuIcon APP_CLEMENTINE_PANEL_48 = create("APP_CLEMENTINE_PANEL_48", "apps/48/clementine-panel");
    public static final AntuIcon APP_CLEMENTINE_PANEL_GREY_48 = create("APP_CLEMENTINE_PANEL_GREY_48", "apps/48/clementine-panel-grey");
    public static final AntuIcon APP_CLEMENTINE2_48 = create("APP_CLEMENTINE2_48", "apps/48/clementine2");
    public static final AntuIcon APP_CLIPGRAB_48 = create("APP_CLIPGRAB_48", "apps/48/clipgrab");
    public static final AntuIcon APP_CLOCK_48 = create("APP_CLOCK_48", "apps/48/clock");
    public static final AntuIcon APP_CMAKE_48 = create("APP_CMAKE_48", "apps/48/cmake");
    public static final AntuIcon APP_CMAKESETUP_48 = create("APP_CMAKESETUP_48", "apps/48/cmakesetup");
    public static final AntuIcon APP_CMAKESETUP32_48 = create("APP_CMAKESETUP32_48", "apps/48/CMakeSetup32");
    public static final AntuIcon APP_CMPLAYER_48 = create("APP_CMPLAYER_48", "apps/48/cmplayer");
    public static final AntuIcon APP_CODEBLOCKS_48 = create("APP_CODEBLOCKS_48", "apps/48/codeblocks");
    public static final AntuIcon APP_CODELITE_48 = create("APP_CODELITE_48", "apps/48/codelite");
    public static final AntuIcon APP_COG_48 = create("APP_COG_48", "apps/48/cog");
    public static final AntuIcon APP_COLORHUG_48 = create("APP_COLORHUG_48", "apps/48/colorhug");
    public static final AntuIcon APP_COLORHUG_CCMX_48 = create("APP_COLORHUG_CCMX_48", "apps/48/colorhug-ccmx");
    public static final AntuIcon APP_COLORHUG_FLASH_48 = create("APP_COLORHUG_FLASH_48", "apps/48/colorhug-flash");
    public static final AntuIcon APP_COLORS_48 = create("APP_COLORS_48", "apps/48/colors");
    public static final AntuIcon APP_COMIX_48 = create("APP_COMIX_48", "apps/48/comix");
    public static final AntuIcon APP_COMPUTER_48 = create("APP_COMPUTER_48", "apps/48/computer");
    public static final AntuIcon APP_CONFIG_USERS_48 = create("APP_CONFIG_USERS_48", "apps/48/config-users");
    public static final AntuIcon APP_CONKY_48 = create("APP_CONKY_48", "apps/48/conky");
    public static final AntuIcon APP_CONKY_MANAGER_48 = create("APP_CONKY_MANAGER_48", "apps/48/conky-manager");
    public static final AntuIcon APP_CONVERSEEN_48 = create("APP_CONVERSEEN_48", "apps/48/converseen");
    public static final AntuIcon APP_COPY_48 = create("APP_COPY_48", "apps/48/copy");
    public static final AntuIcon APP_COPYQ_48 = create("APP_COPYQ_48", "apps/48/copyq");
    public static final AntuIcon APP_CRASHPLAN_48 = create("APP_CRASHPLAN_48", "apps/48/crashplan");
    public static final AntuIcon APP_CS_PRINTER_48 = create("APP_CS_PRINTER_48", "apps/48/cs-printer");
    public static final AntuIcon APP_CS_TABLET_48 = create("APP_CS_TABLET_48", "apps/48/cs-tablet");
    public static final AntuIcon APP_CUPS_48 = create("APP_CUPS_48", "apps/48/cups");
    public static final AntuIcon APP_CUPS_LAUNCHER_48 = create("APP_CUPS_LAUNCHER_48", "apps/48/cups-launcher");
    public static final AntuIcon APP_CUTEGRAM_48 = create("APP_CUTEGRAM_48", "apps/48/cutegram");
    public static final AntuIcon APP_CUTTLEFISH_48 = create("APP_CUTTLEFISH_48", "apps/48/cuttlefish");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CROSSOVER_48 = create("APP_CXMENU_CXOFFICE_0_CROSSOVER_48", "apps/48/cxmenu-cxoffice-0-crossover");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CXDOC_48 = create("APP_CXMENU_CXOFFICE_0_CXDOC_48", "apps/48/cxmenu-cxoffice-0-cxdoc");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CXREGISTER_48 = create("APP_CXMENU_CXOFFICE_0_CXREGISTER_48", "apps/48/cxmenu-cxoffice-0-cxregister");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CXRESET_48 = create("APP_CXMENU_CXOFFICE_0_CXRESET_48", "apps/48/cxmenu-cxoffice-0-cxreset");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CXRUN_48 = create("APP_CXMENU_CXOFFICE_0_CXRUN_48", "apps/48/cxmenu-cxoffice-0-cxrun");
    public static final AntuIcon APP_CXMENU_CXOFFICE_0_CXUNINSTALL_48 = create("APP_CXMENU_CXOFFICE_0_CXUNINSTALL_48", "apps/48/cxmenu-cxoffice-0-cxuninstall");
    public static final AntuIcon APP_CYCLE_48 = create("APP_CYCLE_48", "apps/48/cycle");
    public static final AntuIcon APP_DAEMON_48 = create("APP_DAEMON_48", "apps/48/daemon");
    public static final AntuIcon APP_DARKTABLE_48 = create("APP_DARKTABLE_48", "apps/48/darktable");
    public static final AntuIcon APP_DART_48 = create("APP_DART_48", "apps/48/dart");
    public static final AntuIcon APP_DARWINIA_48 = create("APP_DARWINIA_48", "apps/48/Darwinia");
    public static final AntuIcon APP_DASSAULT_SYSTEMES_DRAFTSIGHT_48 = create("APP_DASSAULT_SYSTEMES_DRAFTSIGHT_48", "apps/48/dassault-systemes.draftsight");
    public static final AntuIcon APP_DASSAULT_SYSTEMS_DRAFTSIGHT_48 = create("APP_DASSAULT_SYSTEMS_DRAFTSIGHT_48", "apps/48/dassault-systems.draftsight");
    public static final AntuIcon APP_DCPP_48 = create("APP_DCPP_48", "apps/48/dc++");
    public static final AntuIcon APP_DECIBEL_48 = create("APP_DECIBEL_48", "apps/48/decibel");
    public static final AntuIcon APP_DEFCON_48 = create("APP_DEFCON_48", "apps/48/defcon");
    public static final AntuIcon APP_DEJA_DUP_48 = create("APP_DEJA_DUP_48", "apps/48/deja-dup");
    public static final AntuIcon APP_DESIGNER_48 = create("APP_DESIGNER_48", "apps/48/designer");
    public static final AntuIcon APP_DESIGNER_QT4_48 = create("APP_DESIGNER_QT4_48", "apps/48/designer-qt4");
    public static final AntuIcon APP_DESIGNER_QT5_48 = create("APP_DESIGNER_QT5_48", "apps/48/designer-qt5");
    public static final AntuIcon APP_DESKTOP_EFFECTS_48 = create("APP_DESKTOP_EFFECTS_48", "apps/48/desktop-effects");
    public static final AntuIcon APP_DESKTOP_ENVIROMENT_KDE_48 = create("APP_DESKTOP_ENVIROMENT_KDE_48", "apps/48/desktop-enviroment-kde");
    public static final AntuIcon APP_DESMUME_48 = create("APP_DESMUME_48", "apps/48/DeSmuME");
    public static final AntuIcon APP_DESURA_48 = create("APP_DESURA_48", "apps/48/desura");
    public static final AntuIcon APP_DEVDOCS_48 = create("APP_DEVDOCS_48", "apps/48/DevDocs");
    public static final AntuIcon APP_DEVHELP_48 = create("APP_DEVHELP_48", "apps/48/devhelp");
    public static final AntuIcon APP_DEVIANTART_48 = create("APP_DEVIANTART_48", "apps/48/deviantart");
    public static final AntuIcon APP_DEVICE_NOTIFIER_48 = create("APP_DEVICE_NOTIFIER_48", "apps/48/device-notifier");
    public static final AntuIcon APP_DEWDROP_48 = create("APP_DEWDROP_48", "apps/48/dewdrop");
    public static final AntuIcon APP_DIA_48 = create("APP_DIA_48", "apps/48/dia");
    public static final AntuIcon APP_DIANARA_48 = create("APP_DIANARA_48", "apps/48/dianara");
    public static final AntuIcon APP_DIFFUSE_48 = create("APP_DIFFUSE_48", "apps/48/diffuse");
    public static final AntuIcon APP_DIGIKAM_48 = create("APP_DIGIKAM_48", "apps/48/digikam");
    public static final AntuIcon APP_DIODON_48 = create("APP_DIODON_48", "apps/48/diodon");
    public static final AntuIcon APP_DOCKBARX_48 = create("APP_DOCKBARX_48", "apps/48/dockbarx");
    public static final AntuIcon APP_DOCKY_48 = create("APP_DOCKY_48", "apps/48/docky");
    public static final AntuIcon APP_DOCUMENT_PRINT_PREVIEW_48 = create("APP_DOCUMENT_PRINT_PREVIEW_48", "apps/48/document-print-preview");
    public static final AntuIcon APP_DOCUMENTATION_48 = create("APP_DOCUMENTATION_48", "apps/48/documentation");
    public static final AntuIcon APP_DOLPHIN_48 = create("APP_DOLPHIN_48", "apps/48/dolphin");
    public static final AntuIcon APP_DOLPHIN_EMU_48 = create("APP_DOLPHIN_EMU_48", "apps/48/dolphin-emu");
    public static final AntuIcon APP_DOLPHIN_ROOT_48 = create("APP_DOLPHIN_ROOT_48", "apps/48/dolphin-root");
    public static final AntuIcon APP_DOSBOX_48 = create("APP_DOSBOX_48", "apps/48/dosbox");
    public static final AntuIcon APP_DOTA_48 = create("APP_DOTA_48", "apps/48/dota");
    public static final AntuIcon APP_DOUBLECMD_48 = create("APP_DOUBLECMD_48", "apps/48/doublecmd");
    public static final AntuIcon APP_DRAGONPLAYER_48 = create("APP_DRAGONPLAYER_48", "apps/48/dragonplayer");
    public static final AntuIcon APP_DRIVE_REMOVABLE_MEDIA_32 = create("APP_DRIVE_REMOVABLE_MEDIA_32", "apps/32/drive-removable-media");
    public static final AntuIcon APP_DRIVE_REMOVABLE_MEDIA_48 = create("APP_DRIVE_REMOVABLE_MEDIA_48", "apps/48/drive-removable-media");
    public static final AntuIcon APP_DROPBOX_48 = create("APP_DROPBOX_48", "apps/48/dropbox");
    public static final AntuIcon APP_DRRACKET_48 = create("APP_DRRACKET_48", "apps/48/drracket");
    public static final AntuIcon APP_DUKTO_48 = create("APP_DUKTO_48", "apps/48/dukto");
    public static final AntuIcon APP_EAGLE_48 = create("APP_EAGLE_48", "apps/48/eagle");
    public static final AntuIcon APP_EASY_TAG_48 = create("APP_EASY_TAG_48", "apps/48/easy-tag");
    public static final AntuIcon APP_EASYMP3GAIN_48 = create("APP_EASYMP3GAIN_48", "apps/48/easyMP3Gain");
    public static final AntuIcon APP_EASYSTROKE_48 = create("APP_EASYSTROKE_48", "apps/48/easystroke");
    public static final AntuIcon APP_EBAY_48 = create("APP_EBAY_48", "apps/48/ebay");
    public static final AntuIcon APP_ECLIPSE_48 = create("APP_ECLIPSE_48", "apps/48/eclipse");
    public static final AntuIcon APP_EMACS_48 = create("APP_EMACS_48", "apps/48/emacs");
    public static final AntuIcon APP_EMERALD_THEME_MANAGER_ICON_48 = create("APP_EMERALD_THEME_MANAGER_ICON_48", "apps/48/emerald-theme-manager-icon");
    public static final AntuIcon APP_EMESENE_48 = create("APP_EMESENE_48", "apps/48/emesene");
    public static final AntuIcon APP_EMPATHY_48 = create("APP_EMPATHY_48", "apps/48/empathy");
    public static final AntuIcon APP_EOG_48 = create("APP_EOG_48", "apps/48/eog");
    public static final AntuIcon APP_EPOPTES_48 = create("APP_EPOPTES_48", "apps/48/epoptes");
    public static final AntuIcon APP_ERADIO_48 = create("APP_ERADIO_48", "apps/48/eradio");
    public static final AntuIcon APP_ERIC_48 = create("APP_ERIC_48", "apps/48/eric");
    public static final AntuIcon APP_ESD_48 = create("APP_ESD_48", "apps/48/esd");
    public static final AntuIcon APP_ESPEAK_GUI_48 = create("APP_ESPEAK_GUI_48", "apps/48/espeak-gui");
    public static final AntuIcon APP_ETR_48 = create("APP_ETR_48", "apps/48/etr");
    public static final AntuIcon APP_EVERNOTE_48 = create("APP_EVERNOTE_48", "apps/48/evernote");
    public static final AntuIcon APP_EVOLUTION_48 = create("APP_EVOLUTION_48", "apps/48/evolution");
    public static final AntuIcon APP_EVOLUTION_CALENDAR_48 = create("APP_EVOLUTION_CALENDAR_48", "apps/48/evolution-calendar");
    public static final AntuIcon APP_EVOLUTION_TASKS_48 = create("APP_EVOLUTION_TASKS_48", "apps/48/evolution-tasks");
    public static final AntuIcon APP_EXFALSO_48 = create("APP_EXFALSO_48", "apps/48/exfalso");
    public static final AntuIcon APP_EXPOBLENDING_48 = create("APP_EXPOBLENDING_48", "apps/48/expoblending");
    public static final AntuIcon APP_EXTENSIONS_48 = create("APP_EXTENSIONS_48", "apps/48/extensions");
    public static final AntuIcon APP_EYES_48 = create("APP_EYES_48", "apps/48/eyes");
    public static final AntuIcon APP_FACE_SMILE_32 = create("APP_FACE_SMILE_32", "apps/32/face-smile");
    public static final AntuIcon APP_FACE_SMILE_48 = create("APP_FACE_SMILE_48", "apps/48/face-smile");
    public static final AntuIcon APP_FACEBOOK_48 = create("APP_FACEBOOK_48", "apps/48/facebook");
    public static final AntuIcon APP_FADEIN_48 = create("APP_FADEIN_48", "apps/48/fadein");
    public static final AntuIcon APP_FBMESSENGER_48 = create("APP_FBMESSENGER_48", "apps/48/fbmessenger");
    public static final AntuIcon APP_FCITX_48 = create("APP_FCITX_48", "apps/48/fcitx");
    public static final AntuIcon APP_FCITX_SOGOUPINYIN_48 = create("APP_FCITX_SOGOUPINYIN_48", "apps/48/fcitx-sogoupinyin");
    public static final AntuIcon APP_FEEDLY_48 = create("APP_FEEDLY_48", "apps/48/feedly");
    public static final AntuIcon APP_FFADO_48 = create("APP_FFADO_48", "apps/48/ffado");
    public static final AntuIcon APP_FIFTEENPUZZLE_48 = create("APP_FIFTEENPUZZLE_48", "apps/48/fifteenpuzzle");
    public static final AntuIcon APP_FILE_ROLLER_48 = create("APP_FILE_ROLLER_48", "apps/48/file-roller");
    public static final AntuIcon APP_FILEBOT_48 = create("APP_FILEBOT_48", "apps/48/filebot");
    public static final AntuIcon APP_FILELIGHT_48 = create("APP_FILELIGHT_48", "apps/48/filelight");
    public static final AntuIcon APP_FILEZILLA_48 = create("APP_FILEZILLA_48", "apps/48/filezilla");
    public static final AntuIcon APP_FINDER_W_48 = create("APP_FINDER_W_48", "apps/48/Finder_w");
    public static final AntuIcon APP_FIREFOX_48 = create("APP_FIREFOX_48", "apps/48/firefox");
    public static final AntuIcon APP_FIREFOX_DEVELOPER_48 = create("APP_FIREFOX_DEVELOPER_48", "apps/48/firefox-developer");
    public static final AntuIcon APP_FIREFOX_DEVELOPER_EDITION_48 = create("APP_FIREFOX_DEVELOPER_EDITION_48", "apps/48/firefox-developer-edition");
    public static final AntuIcon APP_FIREFOX_TRUNK_48 = create("APP_FIREFOX_TRUNK_48", "apps/48/firefox-trunk");
    public static final AntuIcon APP_FIRESTORM_ICON_48 = create("APP_FIRESTORM_ICON_48", "apps/48/firestorm_icon");
    public static final AntuIcon APP_FITYK_48 = create("APP_FITYK_48", "apps/48/fityk");
    public static final AntuIcon APP_FLACON_48 = create("APP_FLACON_48", "apps/48/flacon");
    public static final AntuIcon APP_FLARE_48 = create("APP_FLARE_48", "apps/48/flare");
    public static final AntuIcon APP_FLAREGET_48 = create("APP_FLAREGET_48", "apps/48/flareget");
    public static final AntuIcon APP_FLASH_48 = create("APP_FLASH_48", "apps/48/flash");
    public static final AntuIcon APP_FLICKR_48 = create("APP_FLICKR_48", "apps/48/flickr");
    public static final AntuIcon APP_FLIGHTGEAR_48 = create("APP_FLIGHTGEAR_48", "apps/48/flightgear");
    public static final AntuIcon APP_FLIPBOARD_48 = create("APP_FLIPBOARD_48", "apps/48/flipboard");
    public static final AntuIcon APP_FLOW_48 = create("APP_FLOW_48", "apps/48/flow");
    public static final AntuIcon APP_FLOWBLADE_48 = create("APP_FLOWBLADE_48", "apps/48/flowblade");
    public static final AntuIcon APP_FLUID_48 = create("APP_FLUID_48", "apps/48/fluid");
    public static final AntuIcon APP_FLUXGUI_48 = create("APP_FLUXGUI_48", "apps/48/fluxgui");
    public static final AntuIcon APP_FOGGER_48 = create("APP_FOGGER_48", "apps/48/fogger");
    public static final AntuIcon APP_FONTFORGE_48 = create("APP_FONTFORGE_48", "apps/48/fontforge");
    public static final AntuIcon APP_FOTO_48 = create("APP_FOTO_48", "apps/48/foto");
    public static final AntuIcon APP_FREECIV_CLIENT_48 = create("APP_FREECIV_CLIENT_48", "apps/48/freeciv-client");
    public static final AntuIcon APP_FREECIV_SERVER_48 = create("APP_FREECIV_SERVER_48", "apps/48/freeciv-server");
    public static final AntuIcon APP_FREEMIND_48 = create("APP_FREEMIND_48", "apps/48/freemind");
    public static final AntuIcon APP_FREEPLANE_48 = create("APP_FREEPLANE_48", "apps/48/freeplane");
    public static final AntuIcon APP_FRITZING_48 = create("APP_FRITZING_48", "apps/48/fritzing");
    public static final AntuIcon APP_FROSTWIRE_48 = create("APP_FROSTWIRE_48", "apps/48/frostwire");
    public static final AntuIcon APP_FROZENSYNAPSE_48 = create("APP_FROZENSYNAPSE_48", "apps/48/frozensynapse");
    public static final AntuIcon APP_FSLINT_48 = create("APP_FSLINT_48", "apps/48/fslint");
    public static final AntuIcon APP_FTL_48 = create("APP_FTL_48", "apps/48/FTL");
    public static final AntuIcon APP_FURIUSISOMOUNT_48 = create("APP_FURIUSISOMOUNT_48", "apps/48/furiusisomount");
    public static final AntuIcon APP_FUSION_ICON_48 = create("APP_FUSION_ICON_48", "apps/48/fusion-icon");
    public static final AntuIcon APP_GADU_PROTOCOL_48 = create("APP_GADU_PROTOCOL_48", "apps/48/gadu_protocol");
    public static final AntuIcon APP_GCONF_EDITOR_48 = create("APP_GCONF_EDITOR_48", "apps/48/gconf-editor");
    public static final AntuIcon APP_GCSTAR_48 = create("APP_GCSTAR_48", "apps/48/gcstar");
    public static final AntuIcon APP_GDEBI_48 = create("APP_GDEBI_48", "apps/48/gdebi");
    public static final AntuIcon APP_GEANY_48 = create("APP_GEANY_48", "apps/48/geany");
    public static final AntuIcon APP_GELEMENTAL_48 = create("APP_GELEMENTAL_48", "apps/48/gelemental");
    public static final AntuIcon APP_GENSGS_48 = create("APP_GENSGS_48", "apps/48/gensgs");
    public static final AntuIcon APP_GENYMOTION_48 = create("APP_GENYMOTION_48", "apps/48/genymotion");
    public static final AntuIcon APP_GEOCLUE_48 = create("APP_GEOCLUE_48", "apps/48/geoclue");
    public static final AntuIcon APP_GEOGEBRA_48 = create("APP_GEOGEBRA_48", "apps/48/geogebra");
    public static final AntuIcon APP_GESPEAKER_48 = create("APP_GESPEAKER_48", "apps/48/gespeaker");
    public static final AntuIcon APP_GFTP_48 = create("APP_GFTP_48", "apps/48/gFTP");
    public static final AntuIcon APP_GHEX_48 = create("APP_GHEX_48", "apps/48/ghex");
    public static final AntuIcon APP_GIGGLE_48 = create("APP_GIGGLE_48", "apps/48/giggle");
    public static final AntuIcon APP_GIMP_48 = create("APP_GIMP_48", "apps/48/gimp");
    public static final AntuIcon APP_GIT_48 = create("APP_GIT_48", "apps/48/git");
    public static final AntuIcon APP_GIT_COLA_48 = create("APP_GIT_COLA_48", "apps/48/git-cola");
    public static final AntuIcon APP_GIT_DAG_48 = create("APP_GIT_DAG_48", "apps/48/git-dag");
    public static final AntuIcon APP_GKSU_48 = create("APP_GKSU_48", "apps/48/gksu");
    public static final AntuIcon APP_GLOOBUS_48 = create("APP_GLOOBUS_48", "apps/48/gloobus");
    public static final AntuIcon APP_GLOOBUS_PREVIEW_48 = create("APP_GLOOBUS_PREVIEW_48", "apps/48/gloobus-preview");
    public static final AntuIcon APP_GMAIL_48 = create("APP_GMAIL_48", "apps/48/gmail");
    public static final AntuIcon APP_GMAIL_OFFLINE_48 = create("APP_GMAIL_OFFLINE_48", "apps/48/gmail-offline");
    public static final AntuIcon APP_GMSH_48 = create("APP_GMSH_48", "apps/48/gmsh");
    public static final AntuIcon APP_GNAC_48 = create("APP_GNAC_48", "apps/48/gnac");
    public static final AntuIcon APP_GNASH_48 = create("APP_GNASH_48", "apps/48/gnash");
    public static final AntuIcon APP_GNOME_AISLERIOT_48 = create("APP_GNOME_AISLERIOT_48", "apps/48/gnome-aisleriot");
    public static final AntuIcon APP_GNOME_CONTACTS_48 = create("APP_GNOME_CONTACTS_48", "apps/48/gnome-contacts");
    public static final AntuIcon APP_GNOME_DISKS_48 = create("APP_GNOME_DISKS_48", "apps/48/gnome-disks");
    public static final AntuIcon APP_GNOME_MPLAYER_48 = create("APP_GNOME_MPLAYER_48", "apps/48/gnome-mplayer");
    public static final AntuIcon APP_GNOME_PDF_48 = create("APP_GNOME_PDF_48", "apps/48/gnome-pdf");
    public static final AntuIcon APP_GNOME_SEARCH_TOOL_48 = create("APP_GNOME_SEARCH_TOOL_48", "apps/48/gnome-search-tool");
    public static final AntuIcon APP_GNOME_SOUND_RECORDER_48 = create("APP_GNOME_SOUND_RECORDER_48", "apps/48/gnome-sound-recorder");
    public static final AntuIcon APP_GNOME_STICKY_NOTES_APPLET_48 = create("APP_GNOME_STICKY_NOTES_APPLET_48", "apps/48/gnome-sticky-notes-applet");
    public static final AntuIcon APP_GNOME_SUBTITLES_48 = create("APP_GNOME_SUBTITLES_48", "apps/48/gnome-subtitles");
    public static final AntuIcon APP_GOA_ACCOUNT_48 = create("APP_GOA_ACCOUNT_48", "apps/48/goa-account");
    public static final AntuIcon APP_GOA_ACCOUNT_KERBEROS_48 = create("APP_GOA_ACCOUNT_KERBEROS_48", "apps/48/goa-account-kerberos");
    public static final AntuIcon APP_GOA_ACCOUNT_OWNCLOUD_48 = create("APP_GOA_ACCOUNT_OWNCLOUD_48", "apps/48/goa-account-owncloud");
    public static final AntuIcon APP_GOOGLE_48 = create("APP_GOOGLE_48", "apps/48/google");
    public static final AntuIcon APP_GOOGLE_CALCULATOR_48 = create("APP_GOOGLE_CALCULATOR_48", "apps/48/google-calculator");
    public static final AntuIcon APP_GOOGLE_CALENDAR_48 = create("APP_GOOGLE_CALENDAR_48", "apps/48/google-calendar");
    public static final AntuIcon APP_GOOGLE_CHROME_48 = create("APP_GOOGLE_CHROME_48", "apps/48/google-chrome");
    public static final AntuIcon APP_GOOGLE_CHROME_BETA_48 = create("APP_GOOGLE_CHROME_BETA_48", "apps/48/google-chrome-beta");
    public static final AntuIcon APP_GOOGLE_CHROME_DEVELOPER_TOOL_48 = create("APP_GOOGLE_CHROME_DEVELOPER_TOOL_48", "apps/48/google-chrome-developer-tool");
    public static final AntuIcon APP_GOOGLE_CHROME_UNSTABLE_48 = create("APP_GOOGLE_CHROME_UNSTABLE_48", "apps/48/google-chrome-unstable");
    public static final AntuIcon APP_GOOGLE_DOCS_48 = create("APP_GOOGLE_DOCS_48", "apps/48/google-docs");
    public static final AntuIcon APP_GOOGLE_DRAWS_48 = create("APP_GOOGLE_DRAWS_48", "apps/48/google-draws");
    public static final AntuIcon APP_GOOGLE_DRIVE_48 = create("APP_GOOGLE_DRIVE_48", "apps/48/google-drive");
    public static final AntuIcon APP_GOOGLE_EARTH_48 = create("APP_GOOGLE_EARTH_48", "apps/48/google-earth");
    public static final AntuIcon APP_GOOGLE_FORMS_48 = create("APP_GOOGLE_FORMS_48", "apps/48/google-forms");
    public static final AntuIcon APP_GOOGLE_HANGOUTS_48 = create("APP_GOOGLE_HANGOUTS_48", "apps/48/google-hangouts");
    public static final AntuIcon APP_GOOGLE_KEEP_48 = create("APP_GOOGLE_KEEP_48", "apps/48/google-keep");
    public static final AntuIcon APP_GOOGLE_MAIL_48 = create("APP_GOOGLE_MAIL_48", "apps/48/google-mail");
    public static final AntuIcon APP_GOOGLE_MAPS_48 = create("APP_GOOGLE_MAPS_48", "apps/48/google-maps");
    public static final AntuIcon APP_GOOGLE_MOVIES_48 = create("APP_GOOGLE_MOVIES_48", "apps/48/google-movies");
    public static final AntuIcon APP_GOOGLE_MUSIC_48 = create("APP_GOOGLE_MUSIC_48", "apps/48/google-music");
    public static final AntuIcon APP_GOOGLE_PHOTOS_48 = create("APP_GOOGLE_PHOTOS_48", "apps/48/google-photos");
    public static final AntuIcon APP_GOOGLE_PLAY_BOOKS_48 = create("APP_GOOGLE_PLAY_BOOKS_48", "apps/48/google-play-books");
    public static final AntuIcon APP_GOOGLE_PLAY_MUSIC_48 = create("APP_GOOGLE_PLAY_MUSIC_48", "apps/48/google-play-music");
    public static final AntuIcon APP_GOOGLE_PLUS_PICTURES_48 = create("APP_GOOGLE_PLUS_PICTURES_48", "apps/48/google-plus-pictures");
    public static final AntuIcon APP_GOOGLE_SHEETS_48 = create("APP_GOOGLE_SHEETS_48", "apps/48/google-sheets");
    public static final AntuIcon APP_GOOGLE_SLIDES_48 = create("APP_GOOGLE_SLIDES_48", "apps/48/google-slides");
    public static final AntuIcon APP_GOOGLE_SPREADSHEETS_48 = create("APP_GOOGLE_SPREADSHEETS_48", "apps/48/google-spreadsheets");
    public static final AntuIcon APP_GOOGLE_TRANSLATE_48 = create("APP_GOOGLE_TRANSLATE_48", "apps/48/google-translate");
    public static final AntuIcon APP_GOOGLEEARTH_48 = create("APP_GOOGLEEARTH_48", "apps/48/googleearth");
    public static final AntuIcon APP_GOOGLEPLUS_48 = create("APP_GOOGLEPLUS_48", "apps/48/googleplus");
    public static final AntuIcon APP_GPARTED_48 = create("APP_GPARTED_48", "apps/48/gparted");
    public static final AntuIcon APP_GPDFTEXT_48 = create("APP_GPDFTEXT_48", "apps/48/gpdftext");
    public static final AntuIcon APP_GPICK_48 = create("APP_GPICK_48", "apps/48/gpick");
    public static final AntuIcon APP_GRAMPS_48 = create("APP_GRAMPS_48", "apps/48/gramps");
    public static final AntuIcon APP_GRAPHICS_VIEWER_DOCUMENT_48 = create("APP_GRAPHICS_VIEWER_DOCUMENT_48", "apps/48/graphics-viewer-document");
    public static final AntuIcon APP_GREEDCORP_48 = create("APP_GREEDCORP_48", "apps/48/GreedCorp");
    public static final AntuIcon APP_GRIP_48 = create("APP_GRIP_48", "apps/48/grip");
    public static final AntuIcon APP_GROMIT_48 = create("APP_GROMIT_48", "apps/48/gromit");
    public static final AntuIcon APP_GROUPWISE_PROTOCOL_48 = create("APP_GROUPWISE_PROTOCOL_48", "apps/48/groupwise_protocol");
    public static final AntuIcon APP_GRSYNC_48 = create("APP_GRSYNC_48", "apps/48/grsync");
    public static final AntuIcon APP_GRSYNC_RESTORE_48 = create("APP_GRSYNC_RESTORE_48", "apps/48/grsync-restore");
    public static final AntuIcon APP_GRUB_CUSTOMIZER_48 = create("APP_GRUB_CUSTOMIZER_48", "apps/48/grub-customizer");
    public static final AntuIcon APP_GSTREAMER_48 = create("APP_GSTREAMER_48", "apps/48/gstreamer");
    public static final AntuIcon APP_GTICK_48 = create("APP_GTICK_48", "apps/48/gtick");
    public static final AntuIcon APP_GTK_INFO_48 = create("APP_GTK_INFO_48", "apps/48/gtk-info");
    public static final AntuIcon APP_GTK3_DEMO_48 = create("APP_GTK3_DEMO_48", "apps/48/gtk3-demo");
    public static final AntuIcon APP_GTK3_WIDGET_FACTORY_48 = create("APP_GTK3_WIDGET_FACTORY_48", "apps/48/gtk3-widget-factory");
    public static final AntuIcon APP_GTKCONFIG_32 = create("APP_GTKCONFIG_32", "apps/32/gtkconfig");
    public static final AntuIcon APP_GTKCONFIG_48 = create("APP_GTKCONFIG_48", "apps/48/gtkconfig");
    public static final AntuIcon APP_GTKHASH_48 = create("APP_GTKHASH_48", "apps/48/gtkhash");
    public static final AntuIcon APP_GUAYADEQUE_48 = create("APP_GUAYADEQUE_48", "apps/48/guayadeque");
    public static final AntuIcon APP_GUFW_48 = create("APP_GUFW_48", "apps/48/gufw");
    public static final AntuIcon APP_GUMMI_48 = create("APP_GUMMI_48", "apps/48/gummi");
    public static final AntuIcon APP_GVIM_48 = create("APP_GVIM_48", "apps/48/gvim");
    public static final AntuIcon APP_GWENVIEW_48 = create("APP_GWENVIEW_48", "apps/48/gwenview");
    public static final AntuIcon APP_GWIBBER_48 = create("APP_GWIBBER_48", "apps/48/gwibber");
    public static final AntuIcon APP_H2_ICON_48 = create("APP_H2_ICON_48", "apps/48/h2-icon");
    public static final AntuIcon APP_HAGUICHI_48 = create("APP_HAGUICHI_48", "apps/48/haguichi");
    public static final AntuIcon APP_HAROOPAD_48 = create("APP_HAROOPAD_48", "apps/48/haroopad");
    public static final AntuIcon APP_HEDGEWARS_48 = create("APP_HEDGEWARS_48", "apps/48/hedgewars");
    public static final AntuIcon APP_HEIMDALL_FRONTEND_48 = create("APP_HEIMDALL_FRONTEND_48", "apps/48/heimdall-frontend");
    public static final AntuIcon APP_HELP_48 = create("APP_HELP_48", "apps/48/help");
    public static final AntuIcon APP_HELP_ABOUT_32 = create("APP_HELP_ABOUT_32", "apps/32/help-about");
    public static final AntuIcon APP_HELP_ABOUT_48 = create("APP_HELP_ABOUT_48", "apps/48/help-about");
    public static final AntuIcon APP_HELP_BROWSER_48 = create("APP_HELP_BROWSER_48", "apps/48/help-browser");
    public static final AntuIcon APP_HIGAN_48 = create("APP_HIGAN_48", "apps/48/higan");
    public static final AntuIcon APP_HIPCHAT_48 = create("APP_HIPCHAT_48", "apps/48/hipchat");
    public static final AntuIcon APP_HOMEBANK_48 = create("APP_HOMEBANK_48", "apps/48/homebank");
    public static final AntuIcon APP_HOMERUN_22 = create("APP_HOMERUN_22", "apps/22/homerun");
    public static final AntuIcon APP_HOMERUN_48 = create("APP_HOMERUN_48", "apps/48/homerun");
    public static final AntuIcon APP_HOTOT_48 = create("APP_HOTOT_48", "apps/48/hotot");
    public static final AntuIcon APP_HP_LOGO_48 = create("APP_HP_LOGO_48", "apps/48/hp_logo");
    public static final AntuIcon APP_HTOP_48 = create("APP_HTOP_48", "apps/48/htop");
    public static final AntuIcon APP_HUGIN_48 = create("APP_HUGIN_48", "apps/48/hugin");
    public static final AntuIcon APP_HWINFO_48 = create("APP_HWINFO_48", "apps/48/hwinfo");
    public static final AntuIcon APP_I2PROUTER_48 = create("APP_I2PROUTER_48", "apps/48/i2prouter");
    public static final AntuIcon APP_IBUS_SETUP_48 = create("APP_IBUS_SETUP_48", "apps/48/ibus-setup");
    public static final AntuIcon APP_ICEDOVE_48 = create("APP_ICEDOVE_48", "apps/48/icedove");
    public static final AntuIcon APP_ICEWEASEL_48 = create("APP_ICEWEASEL_48", "apps/48/iceweasel");
    public static final AntuIcon APP_ICON_48 = create("APP_ICON_48", "apps/48/icon");
    public static final AntuIcon APP_ICQ_PROTOCOL_48 = create("APP_ICQ_PROTOCOL_48", "apps/48/icq_protocol");
    public static final AntuIcon APP_IDEA_48 = create("APP_IDEA_48", "apps/48/idea");
    public static final AntuIcon APP_IM_AIM_48 = create("APP_IM_AIM_48", "apps/48/im-aim");
    public static final AntuIcon APP_IM_QQ_48 = create("APP_IM_QQ_48", "apps/48/im-qq");
    public static final AntuIcon APP_IMAGEWRITER_48 = create("APP_IMAGEWRITER_48", "apps/48/imagewriter");
    public static final AntuIcon APP_IMLOGVIEW_48 = create("APP_IMLOGVIEW_48", "apps/48/imlogview");
    public static final AntuIcon APP_INCREDIPEDE_48 = create("APP_INCREDIPEDE_48", "apps/48/Incredipede");
    public static final AntuIcon APP_INDICATOR_CHINA_WEATHER_48 = create("APP_INDICATOR_CHINA_WEATHER_48", "apps/48/indicator-china-weather");
    public static final AntuIcon APP_INDICATOR_REMINDOR_48 = create("APP_INDICATOR_REMINDOR_48", "apps/48/indicator-remindor");
    public static final AntuIcon APP_INDICATOR_SOUND_SWITCHER_48 = create("APP_INDICATOR_SOUND_SWITCHER_48", "apps/48/indicator-sound-switcher");
    public static final AntuIcon APP_INDICATOR_WEATHER_48 = create("APP_INDICATOR_WEATHER_48", "apps/48/indicator-weather");
    public static final AntuIcon APP_INFO_48 = create("APP_INFO_48", "apps/48/info");
    public static final AntuIcon APP_INKSCAPE_48 = create("APP_INKSCAPE_48", "apps/48/inkscape");
    public static final AntuIcon APP_INKSCAPE_ALT_48 = create("APP_INKSCAPE_ALT_48", "apps/48/inkscape-alt");
    public static final AntuIcon APP_INKSCAPE_LOGO_48 = create("APP_INKSCAPE_LOGO_48", "apps/48/inkscape-logo");
    public static final AntuIcon APP_INSTAGRAM_48 = create("APP_INSTAGRAM_48", "apps/48/instagram");
    public static final AntuIcon APP_INSYNC_48 = create("APP_INSYNC_48", "apps/48/insync");
    public static final AntuIcon APP_INTEL_INSTALLER_48 = create("APP_INTEL_INSTALLER_48", "apps/48/intel-installer");
    public static final AntuIcon APP_INTERNET_MAIL_22 = create("APP_INTERNET_MAIL_22", "apps/22/internet-mail");
    public static final AntuIcon APP_INTERNET_MAIL_48 = create("APP_INTERNET_MAIL_48", "apps/48/internet-mail");
    public static final AntuIcon APP_INTERNET_TELEPHONY_48 = create("APP_INTERNET_TELEPHONY_48", "apps/48/internet-telephony");
    public static final AntuIcon APP_INTERNET_WEB_BROWSER_48 = create("APP_INTERNET_WEB_BROWSER_48", "apps/48/internet-web-browser");
    public static final AntuIcon APP_IRC_48 = create("APP_IRC_48", "apps/48/irc");
    public static final AntuIcon APP_IRC_PROTOCOL_48 = create("APP_IRC_PROTOCOL_48", "apps/48/irc_protocol");
    public static final AntuIcon APP_IRSSI_48 = create("APP_IRSSI_48", "apps/48/irssi");
    public static final AntuIcon APP_ISTANBUL_48 = create("APP_ISTANBUL_48", "apps/48/istanbul");
    public static final AntuIcon APP_ITUNES_48 = create("APP_ITUNES_48", "apps/48/itunes");
    public static final AntuIcon APP_JABBER_48 = create("APP_JABBER_48", "apps/48/jabber");
    public static final AntuIcon APP_JABBER_PROTOCOL_48 = create("APP_JABBER_PROTOCOL_48", "apps/48/jabber_protocol");
    public static final AntuIcon APP_JABREF_48 = create("APP_JABREF_48", "apps/48/jabref");
    public static final AntuIcon APP_JAVA_48 = create("APP_JAVA_48", "apps/48/java");
    public static final AntuIcon APP_JITSI_48 = create("APP_JITSI_48", "apps/48/jitsi");
    public static final AntuIcon APP_JOCKEY_48 = create("APP_JOCKEY_48", "apps/48/jockey");
    public static final AntuIcon APP_JUFFED_48 = create("APP_JUFFED_48", "apps/48/juffed");
    public static final AntuIcon APP_JUK_48 = create("APP_JUK_48", "apps/48/juk");
    public static final AntuIcon APP_JUPITER_48 = create("APP_JUPITER_48", "apps/48/jupiter");
    public static final AntuIcon APP_K3B_48 = create("APP_K3B_48", "apps/48/k3b");
    public static final AntuIcon APP_K3D_48 = create("APP_K3D_48", "apps/48/k3d");
    public static final AntuIcon APP_KADDRESSBOOK_48 = create("APP_KADDRESSBOOK_48", "apps/48/kaddressbook");
    public static final AntuIcon APP_KADU_48 = create("APP_KADU_48", "apps/48/kadu");
    public static final AntuIcon APP_KAFFEINE_48 = create("APP_KAFFEINE_48", "apps/48/kaffeine");
    public static final AntuIcon APP_KALGEBRA_48 = create("APP_KALGEBRA_48", "apps/48/kalgebra");
    public static final AntuIcon APP_KAMERKA_48 = create("APP_KAMERKA_48", "apps/48/kamerka");
    public static final AntuIcon APP_KAMOSO_48 = create("APP_KAMOSO_48", "apps/48/kamoso");
    public static final AntuIcon APP_KARBON_48 = create("APP_KARBON_48", "apps/48/karbon");
    public static final AntuIcon APP_KATE_48 = create("APP_KATE_48", "apps/48/kate");
    public static final AntuIcon APP_KATOMIC_48 = create("APP_KATOMIC_48", "apps/48/katomic");
    public static final AntuIcon APP_KBIBTEX_48 = create("APP_KBIBTEX_48", "apps/48/KBibTex");
    public static final AntuIcon APP_KBLOGGER_48 = create("APP_KBLOGGER_48", "apps/48/kblogger");
    public static final AntuIcon APP_KBRUCH_48 = create("APP_KBRUCH_48", "apps/48/kbruch");
    public static final AntuIcon APP_KBUGBUSTER_48 = create("APP_KBUGBUSTER_48", "apps/48/kbugbuster");
    public static final AntuIcon APP_KCALC_48 = create("APP_KCALC_48", "apps/48/kcalc");
    public static final AntuIcon APP_KCHART_48 = create("APP_KCHART_48", "apps/48/kchart");
    public static final AntuIcon APP_KCM_SYSTEMD_48 = create("APP_KCM_SYSTEMD_48", "apps/48/kcm-systemd");
    public static final AntuIcon APP_KCOLORCHOOSER_48 = create("APP_KCOLORCHOOSER_48", "apps/48/kcolorchooser");
    public static final AntuIcon APP_KDCRAW_48 = create("APP_KDCRAW_48", "apps/48/kdcraw");
    public static final AntuIcon APP_KDE_22 = create("APP_KDE_22", "apps/22/kde");
    public static final AntuIcon APP_KDE_48 = create("APP_KDE_48", "apps/48/kde");
    public static final AntuIcon APP_KDE_GTK_CONFIG_32 = create("APP_KDE_GTK_CONFIG_32", "apps/32/kde-gtk-config");
    public static final AntuIcon APP_KDE_GTK_CONFIG_48 = create("APP_KDE_GTK_CONFIG_48", "apps/48/kde-gtk-config");
    public static final AntuIcon APP_KDE_SERVICES_48 = create("APP_KDE_SERVICES_48", "apps/48/kde-services");
    public static final AntuIcon APP_KDE_WINDOWS_48 = create("APP_KDE_WINDOWS_48", "apps/48/kde-windows");
    public static final AntuIcon APP_KDEAPP_48 = create("APP_KDEAPP_48", "apps/48/kdeapp");
    public static final AntuIcon APP_KDECONNECT_32 = create("APP_KDECONNECT_32", "apps/32/kdeconnect");
    public static final AntuIcon APP_KDECONNECT_48 = create("APP_KDECONNECT_48", "apps/48/kdeconnect");
    public static final AntuIcon APP_KDENLIVE_48 = create("APP_KDENLIVE_48", "apps/48/kdenlive");
    public static final AntuIcon APP_KDEPIM_DROPBOX_48 = create("APP_KDEPIM_DROPBOX_48", "apps/48/kdepim-dropbox");
    public static final AntuIcon APP_KDESVN_48 = create("APP_KDESVN_48", "apps/48/kdesvn");
    public static final AntuIcon APP_KDEVELOP_48 = create("APP_KDEVELOP_48", "apps/48/kdevelop");
    public static final AntuIcon APP_KDF_48 = create("APP_KDF_48", "apps/48/kdf");
    public static final AntuIcon APP_KDIFF3_48 = create("APP_KDIFF3_48", "apps/48/kdiff3");
    public static final AntuIcon APP_KDOTS_48 = create("APP_KDOTS_48", "apps/48/kdots");
    public static final AntuIcon APP_KEEPASS_48 = create("APP_KEEPASS_48", "apps/48/keepass");
    public static final AntuIcon APP_KEEPASSX_48 = create("APP_KEEPASSX_48", "apps/48/keepassx");
    public static final AntuIcon APP_KERNEL_48 = create("APP_KERNEL_48", "apps/48/kernel");
    public static final AntuIcon APP_KEXI_48 = create("APP_KEXI_48", "apps/48/kexi");
    public static final AntuIcon APP_KEYSYNC_48 = create("APP_KEYSYNC_48", "apps/48/keysync");
    public static final AntuIcon APP_KFILEREPLACE_48 = create("APP_KFILEREPLACE_48", "apps/48/kfilereplace");
    public static final AntuIcon APP_KFIND_48 = create("APP_KFIND_48", "apps/48/kfind");
    public static final AntuIcon APP_KFONTVIEW_48 = create("APP_KFONTVIEW_48", "apps/48/kfontview");
    public static final AntuIcon APP_KFORMULA_48 = create("APP_KFORMULA_48", "apps/48/kformula");
    public static final AntuIcon APP_KFTPGRABBER_48 = create("APP_KFTPGRABBER_48", "apps/48/kftpgrabber");
    public static final AntuIcon APP_KGEOGRAPHY_48 = create("APP_KGEOGRAPHY_48", "apps/48/kgeography");
    public static final AntuIcon APP_KGET_22 = create("APP_KGET_22", "apps/22/kget");
    public static final AntuIcon APP_KGET_48 = create("APP_KGET_48", "apps/48/kget");
    public static final AntuIcon APP_KGPG_48 = create("APP_KGPG_48", "apps/48/kgpg");
    public static final AntuIcon APP_KICAD_48 = create("APP_KICAD_48", "apps/48/kicad");
    public static final AntuIcon APP_KIG_48 = create("APP_KIG_48", "apps/48/kig");
    public static final AntuIcon APP_KILE_48 = create("APP_KILE_48", "apps/48/kile");
    public static final AntuIcon APP_KIMAGEMAPEDITOR_48 = create("APP_KIMAGEMAPEDITOR_48", "apps/48/kimagemapeditor");
    public static final AntuIcon APP_KINDLE_48 = create("APP_KINDLE_48", "apps/48/kindle");
    public static final AntuIcon APP_KIPI_48 = create("APP_KIPI_48", "apps/48/kipi");
    public static final AntuIcon APP_KIPI_DEBIANSCREENSHOTS_48 = create("APP_KIPI_DEBIANSCREENSHOTS_48", "apps/48/kipi-debianscreenshots");
    public static final AntuIcon APP_KIPI_DLNA_48 = create("APP_KIPI_DLNA_48", "apps/48/kipi-dlna");
    public static final AntuIcon APP_KIPI_DNGCONVERTER_48 = create("APP_KIPI_DNGCONVERTER_48", "apps/48/kipi-dngconverter");
    public static final AntuIcon APP_KIPI_DROPBOX_48 = create("APP_KIPI_DROPBOX_48", "apps/48/kipi-dropbox");
    public static final AntuIcon APP_KIPI_EXPOBLENDING_48 = create("APP_KIPI_EXPOBLENDING_48", "apps/48/kipi-expoblending");
    public static final AntuIcon APP_KIPI_FLASH_48 = create("APP_KIPI_FLASH_48", "apps/48/kipi-flash");
    public static final AntuIcon APP_KIPI_GOOGLEDRIVE_48 = create("APP_KIPI_GOOGLEDRIVE_48", "apps/48/kipi-googledrive");
    public static final AntuIcon APP_KIPI_GPSIMAGETAG_48 = create("APP_KIPI_GPSIMAGETAG_48", "apps/48/kipi-gpsimagetag");
    public static final AntuIcon APP_KIPI_HQ_48 = create("APP_KIPI_HQ_48", "apps/48/kipi-hq");
    public static final AntuIcon APP_KIPI_IMAGESHACK_48 = create("APP_KIPI_IMAGESHACK_48", "apps/48/kipi-imageshack");
    public static final AntuIcon APP_KIPI_IMGUR_48 = create("APP_KIPI_IMGUR_48", "apps/48/kipi-imgur");
    public static final AntuIcon APP_KIPI_JALBUM_48 = create("APP_KIPI_JALBUM_48", "apps/48/kipi-jalbum");
    public static final AntuIcon APP_KIPI_METADATAEDIT_48 = create("APP_KIPI_METADATAEDIT_48", "apps/48/kipi-metadataedit");
    public static final AntuIcon APP_KIPI_OGL_48 = create("APP_KIPI_OGL_48", "apps/48/kipi-ogl");
    public static final AntuIcon APP_KIPI_PANORAMA_48 = create("APP_KIPI_PANORAMA_48", "apps/48/kipi-panorama");
    public static final AntuIcon APP_KIPI_RAJCE_48 = create("APP_KIPI_RAJCE_48", "apps/48/kipi-rajce");
    public static final AntuIcon APP_KIPI_SHWUP_48 = create("APP_KIPI_SHWUP_48", "apps/48/kipi-shwup");
    public static final AntuIcon APP_KIPI_SLIDESHOW_48 = create("APP_KIPI_SLIDESHOW_48", "apps/48/kipi-slideshow");
    public static final AntuIcon APP_KIPI_SMUGMUG_48 = create("APP_KIPI_SMUGMUG_48", "apps/48/kipi-smugmug");
    public static final AntuIcon APP_KIPI_TIMEADJUST_48 = create("APP_KIPI_TIMEADJUST_48", "apps/48/kipi-timeadjust");
    public static final AntuIcon APP_KITEN_48 = create("APP_KITEN_48", "apps/48/kiten");
    public static final AntuIcon APP_KIVIO_48 = create("APP_KIVIO_48", "apps/48/kivio");
    public static final AntuIcon APP_KJOTS_48 = create("APP_KJOTS_48", "apps/48/kjots");
    public static final AntuIcon APP_KJOURNAL_48 = create("APP_KJOURNAL_48", "apps/48/kjournal");
    public static final AntuIcon APP_KLEOPATRA_48 = create("APP_KLEOPATRA_48", "apps/48/kleopatra");
    public static final AntuIcon APP_KLINKSTATUS_48 = create("APP_KLINKSTATUS_48", "apps/48/klinkstatus");
    public static final AntuIcon APP_KLIPPER_48 = create("APP_KLIPPER_48", "apps/48/klipper");
    public static final AntuIcon APP_KMAG_48 = create("APP_KMAG_48", "apps/48/kmag");
    public static final AntuIcon APP_KMAIL_48 = create("APP_KMAIL_48", "apps/48/kmail");
    public static final AntuIcon APP_KMAIL2_48 = create("APP_KMAIL2_48", "apps/48/kmail2");
    public static final AntuIcon APP_KMENUEDIT_48 = create("APP_KMENUEDIT_48", "apps/48/kmenuedit");
    public static final AntuIcon APP_KMIX_48 = create("APP_KMIX_48", "apps/48/kmix");
    public static final AntuIcon APP_KMLDONKEY_48 = create("APP_KMLDONKEY_48", "apps/48/kmldonkey");
    public static final AntuIcon APP_KMOUSETOOL_48 = create("APP_KMOUSETOOL_48", "apps/48/kmousetool");
    public static final AntuIcon APP_KMOUTH_48 = create("APP_KMOUTH_48", "apps/48/kmouth");
    public static final AntuIcon APP_KMPLAYER_48 = create("APP_KMPLAYER_48", "apps/48/kmplayer");
    public static final AntuIcon APP_KMPLOT_48 = create("APP_KMPLOT_48", "apps/48/kmplot");
    public static final AntuIcon APP_KMYMONEY_48 = create("APP_KMYMONEY_48", "apps/48/kmymoney");
    public static final AntuIcon APP_KNEMO_48 = create("APP_KNEMO_48", "apps/48/knemo");
    public static final AntuIcon APP_KNETATTACH_48 = create("APP_KNETATTACH_48", "apps/48/knetattach");
    public static final AntuIcon APP_KNEWSTICKER_48 = create("APP_KNEWSTICKER_48", "apps/48/knewsticker");
    public static final AntuIcon APP_KNODE_48 = create("APP_KNODE_48", "apps/48/knode");
    public static final AntuIcon APP_KNOTES_48 = create("APP_KNOTES_48", "apps/48/knotes");
    public static final AntuIcon APP_KODI_48 = create("APP_KODI_48", "apps/48/kodi");
    public static final AntuIcon APP_KOLAB_48 = create("APP_KOLAB_48", "apps/48/kolab");
    public static final AntuIcon APP_KOMODO_48 = create("APP_KOMODO_48", "apps/48/komodo");
    public static final AntuIcon APP_KOMPARE_48 = create("APP_KOMPARE_48", "apps/48/kompare");
    public static final AntuIcon APP_KOMPOZER_48 = create("APP_KOMPOZER_48", "apps/48/kompozer");
    public static final AntuIcon APP_KONQUEROR_48 = create("APP_KONQUEROR_48", "apps/48/konqueror");
    public static final AntuIcon APP_KONSTRUKTOR_48 = create("APP_KONSTRUKTOR_48", "apps/48/konstruktor");
    public static final AntuIcon APP_KONTACT_48 = create("APP_KONTACT_48", "apps/48/kontact");
    public static final AntuIcon APP_KONTACT_IMPORT_WIZARD_48 = create("APP_KONTACT_IMPORT_WIZARD_48", "apps/48/kontact-import-wizard");
    public static final AntuIcon APP_KONVERSATION_48 = create("APP_KONVERSATION_48", "apps/48/konversation");
    public static final AntuIcon APP_KOPETE_48 = create("APP_KOPETE_48", "apps/48/kopete");
    public static final AntuIcon APP_KOPETE_OFFLINE_48 = create("APP_KOPETE_OFFLINE_48", "apps/48/kopete-offline");
    public static final AntuIcon APP_KORG_JOURNAL_48 = create("APP_KORG_JOURNAL_48", "apps/48/korg-journal");
    public static final AntuIcon APP_KORG_TODO_22 = create("APP_KORG_TODO_22", "apps/22/korg-todo");
    public static final AntuIcon APP_KORG_TODO_48 = create("APP_KORG_TODO_48", "apps/48/korg-todo");
    public static final AntuIcon APP_KORGAC_48 = create("APP_KORGAC_48", "apps/48/korgac");
    public static final AntuIcon APP_KORGANIZER_48 = create("APP_KORGANIZER_48", "apps/48/korganizer");
    public static final AntuIcon APP_KPACKAGE_48 = create("APP_KPACKAGE_48", "apps/48/kpackage");
    public static final AntuIcon APP_KPAGER_48 = create("APP_KPAGER_48", "apps/48/kpager");
    public static final AntuIcon APP_KPAT_48 = create("APP_KPAT_48", "apps/48/kpat");
    public static final AntuIcon APP_KPLATO_48 = create("APP_KPLATO_48", "apps/48/kplato");
    public static final AntuIcon APP_KPRESENTER_48 = create("APP_KPRESENTER_48", "apps/48/kpresenter");
    public static final AntuIcon APP_KRAFT_48 = create("APP_KRAFT_48", "apps/48/kraft");
    public static final AntuIcon APP_KRDC_48 = create("APP_KRDC_48", "apps/48/krdc");
    public static final AntuIcon APP_KRECIPES_48 = create("APP_KRECIPES_48", "apps/48/krecipes");
    public static final AntuIcon APP_KRENAME_48 = create("APP_KRENAME_48", "apps/48/krename");
    public static final AntuIcon APP_KRFB_48 = create("APP_KRFB_48", "apps/48/krfb");
    public static final AntuIcon APP_KRITA_48 = create("APP_KRITA_48", "apps/48/krita");
    public static final AntuIcon APP_KRITAGEMINI_48 = create("APP_KRITAGEMINI_48", "apps/48/kritagemini");
    public static final AntuIcon APP_KRITASKETCH_48 = create("APP_KRITASKETCH_48", "apps/48/kritasketch");
    public static final AntuIcon APP_KRONOMETER_48 = create("APP_KRONOMETER_48", "apps/48/kronometer");
    public static final AntuIcon APP_KRULER_48 = create("APP_KRULER_48", "apps/48/kruler");
    public static final AntuIcon APP_KRUSADER_48 = create("APP_KRUSADER_48", "apps/48/krusader");
    public static final AntuIcon APP_KRUSADER_ROOT_2_48 = create("APP_KRUSADER_ROOT_2_48", "apps/48/krusader-root-2");
    public static final AntuIcon APP_KRUSADER_ROOT_48 = create("APP_KRUSADER_ROOT_48", "apps/48/krusader-root");
    public static final AntuIcon APP_KRUSADER_USER_48 = create("APP_KRUSADER_USER_48", "apps/48/krusader_user");
    public static final AntuIcon APP_KSCD_48 = create("APP_KSCD_48", "apps/48/kscd");
    public static final AntuIcon APP_KSNAPSHOT_48 = create("APP_KSNAPSHOT_48", "apps/48/ksnapshot");
    public static final AntuIcon APP_KSNIFFER_48 = create("APP_KSNIFFER_48", "apps/48/ksniffer");
    public static final AntuIcon APP_KSPREAD_48 = create("APP_KSPREAD_48", "apps/48/kspread");
    public static final AntuIcon APP_KSUDOKU_48 = create("APP_KSUDOKU_48", "apps/48/ksudoku");
    public static final AntuIcon APP_KSYSGUARDD_48 = create("APP_KSYSGUARDD_48", "apps/48/ksysguardd");
    public static final AntuIcon APP_KTEATIME_48 = create("APP_KTEATIME_48", "apps/48/kteatime");
    public static final AntuIcon APP_KTHESAURUS_48 = create("APP_KTHESAURUS_48", "apps/48/kthesaurus");
    public static final AntuIcon APP_KTIP_48 = create("APP_KTIP_48", "apps/48/ktip");
    public static final AntuIcon APP_KTORRENT_22 = create("APP_KTORRENT_22", "apps/22/ktorrent");
    public static final AntuIcon APP_KTORRENT_48 = create("APP_KTORRENT_48", "apps/48/ktorrent");
    public static final AntuIcon APP_KTP_CONTACTLIST_48 = create("APP_KTP_CONTACTLIST_48", "apps/48/ktp-contactlist");
    public static final AntuIcon APP_KTURTLE_48 = create("APP_KTURTLE_48", "apps/48/kturtle");
    public static final AntuIcon APP_KUAIPAN4UK_48 = create("APP_KUAIPAN4UK_48", "apps/48/kuaipan4uk");
    public static final AntuIcon APP_KUICKSHOW_48 = create("APP_KUICKSHOW_48", "apps/48/kuickshow");
    public static final AntuIcon APP_KUP_48 = create("APP_KUP_48", "apps/48/kup");
    public static final AntuIcon APP_KUPFER_48 = create("APP_KUPFER_48", "apps/48/kupfer");
    public static final AntuIcon APP_KVANTUMMANAGER_48 = create("APP_KVANTUMMANAGER_48", "apps/48/kvantummanager");
    public static final AntuIcon APP_KVERBOS_48 = create("APP_KVERBOS_48", "apps/48/kverbos");
    public static final AntuIcon APP_KWALLETMANAGER_48 = create("APP_KWALLETMANAGER_48", "apps/48/kwalletmanager");
    public static final AntuIcon APP_KWALLETMANAGER2_48 = create("APP_KWALLETMANAGER2_48", "apps/48/kwalletmanager2");
    public static final AntuIcon APP_KWIKDISK_48 = create("APP_KWIKDISK_48", "apps/48/kwikdisk");
    public static final AntuIcon APP_KWIN_48 = create("APP_KWIN_48", "apps/48/kwin");
    public static final AntuIcon APP_KWORD_48 = create("APP_KWORD_48", "apps/48/kword");
    public static final AntuIcon APP_KWPLAYER_48 = create("APP_KWPLAYER_48", "apps/48/kwplayer");
    public static final AntuIcon APP_KWRITE_48 = create("APP_KWRITE_48", "apps/48/kwrite");
    public static final AntuIcon APP_LABPLOT_48 = create("APP_LABPLOT_48", "apps/48/labplot");
    public static final AntuIcon APP_LANCELOT_48 = create("APP_LANCELOT_48", "apps/48/lancelot");
    public static final AntuIcon APP_LANCELOT_START_48 = create("APP_LANCELOT_START_48", "apps/48/lancelot-start");
    public static final AntuIcon APP_LASTFM_48 = create("APP_LASTFM_48", "apps/48/lastfm");
    public static final AntuIcon APP_LASTPASS_VAULT_48 = create("APP_LASTPASS_VAULT_48", "apps/48/lastpass-vault");
    public static final AntuIcon APP_LAUNCHPAD_LAUNCHPAD_NET_48 = create("APP_LAUNCHPAD_LAUNCHPAD_NET_48", "apps/48/Launchpad-launchpad.net");
    public static final AntuIcon APP_LAZARUS_48 = create("APP_LAZARUS_48", "apps/48/lazarus");
    public static final AntuIcon APP_LDLITE_48 = create("APP_LDLITE_48", "apps/48/ldlite");
    public static final AntuIcon APP_LDVIEW_48 = create("APP_LDVIEW_48", "apps/48/ldview");
    public static final AntuIcon APP_LEOCAD_48 = create("APP_LEOCAD_48", "apps/48/leocad");
    public static final AntuIcon APP_LIBRARY_INTERNET_RADIO_48 = create("APP_LIBRARY_INTERNET_RADIO_48", "apps/48/library-internet-radio");
    public static final AntuIcon APP_LIBRECAD_48 = create("APP_LIBRECAD_48", "apps/48/librecad");
    public static final AntuIcon APP_LIBREOFFICE_BASE_48 = create("APP_LIBREOFFICE_BASE_48", "apps/48/libreoffice-base");
    public static final AntuIcon APP_LIBREOFFICE_CALC_48 = create("APP_LIBREOFFICE_CALC_48", "apps/48/libreoffice-calc");
    public static final AntuIcon APP_LIBREOFFICE_DRAW_48 = create("APP_LIBREOFFICE_DRAW_48", "apps/48/libreoffice-draw");
    public static final AntuIcon APP_LIBREOFFICE_IMPRESS_48 = create("APP_LIBREOFFICE_IMPRESS_48", "apps/48/libreoffice-impress");
    public static final AntuIcon APP_LIBREOFFICE_MAIN_48 = create("APP_LIBREOFFICE_MAIN_48", "apps/48/libreoffice-main");
    public static final AntuIcon APP_LIBREOFFICE_MATH_48 = create("APP_LIBREOFFICE_MATH_48", "apps/48/libreoffice-math");
    public static final AntuIcon APP_LIBREOFFICE_PRINTERADMIN_48 = create("APP_LIBREOFFICE_PRINTERADMIN_48", "apps/48/libreoffice-printeradmin");
    public static final AntuIcon APP_LIBREOFFICE_STARCENTER_48 = create("APP_LIBREOFFICE_STARCENTER_48", "apps/48/libreoffice-starcenter");
    public static final AntuIcon APP_LIBREOFFICE_STARTCENTER_48 = create("APP_LIBREOFFICE_STARTCENTER_48", "apps/48/libreoffice-startcenter");
    public static final AntuIcon APP_LIBREOFFICE_WRITER_48 = create("APP_LIBREOFFICE_WRITER_48", "apps/48/libreoffice-writer");
    public static final AntuIcon APP_LIGHTSOFF_48 = create("APP_LIGHTSOFF_48", "apps/48/Lightsoff");
    public static final AntuIcon APP_LIGHTWORKS_48 = create("APP_LIGHTWORKS_48", "apps/48/lightworks");
    public static final AntuIcon APP_LIGHTZONE_48 = create("APP_LIGHTZONE_48", "apps/48/lightzone");
    public static final AntuIcon APP_LINGUIST_48 = create("APP_LINGUIST_48", "apps/48/linguist");
    public static final AntuIcon APP_LINGUIST_QT4_48 = create("APP_LINGUIST_QT4_48", "apps/48/linguist-qt4");
    public static final AntuIcon APP_LINGUIST_QT5_48 = create("APP_LINGUIST_QT5_48", "apps/48/linguist-qt5");
    public static final AntuIcon APP_LINGUISTIC_48 = create("APP_LINGUISTIC_48", "apps/48/linguistic");
    public static final AntuIcon APP_LINKEDIN_48 = create("APP_LINKEDIN_48", "apps/48/linkedin");
    public static final AntuIcon APP_LIVE_48 = create("APP_LIVE_48", "apps/48/live");
    public static final AntuIcon APP_LIVEINST_48 = create("APP_LIVEINST_48", "apps/48/liveinst");
    public static final AntuIcon APP_LIVES_48 = create("APP_LIVES_48", "apps/48/lives");
    public static final AntuIcon APP_LIVEUSB_CREATOR_48 = create("APP_LIVEUSB_CREATOR_48", "apps/48/liveusb-creator");
    public static final AntuIcon APP_LOGOQLE2_48 = create("APP_LOGOQLE2_48", "apps/48/logoqle2");
    public static final AntuIcon APP_LOGVIEWER_48 = create("APP_LOGVIEWER_48", "apps/48/logviewer");
    public static final AntuIcon APP_LOKALIZE_48 = create("APP_LOKALIZE_48", "apps/48/lokalize");
    public static final AntuIcon APP_LORDOFULTIMA_48 = create("APP_LORDOFULTIMA_48", "apps/48/lordofultima");
    public static final AntuIcon APP_LOVE_APP_48 = create("APP_LOVE_APP_48", "apps/48/love-app");
    public static final AntuIcon APP_LPUB_48 = create("APP_LPUB_48", "apps/48/lpub");
    public static final AntuIcon APP_LTICON_48 = create("APP_LTICON_48", "apps/48/lticon");
    public static final AntuIcon APP_LUNA_48 = create("APP_LUNA_48", "apps/48/luna");
    public static final AntuIcon APP_LYX_48 = create("APP_LYX_48", "apps/48/lyx");
    public static final AntuIcon APP_M64PY_48 = create("APP_M64PY_48", "apps/48/m64py");
    public static final AntuIcon APP_MAHJONGG_48 = create("APP_MAHJONGG_48", "apps/48/mahjongg");
    public static final AntuIcon APP_MAIL_CLIENT_48 = create("APP_MAIL_CLIENT_48", "apps/48/mail-client");
    public static final AntuIcon APP_MAILRU_MAIL_RU_48 = create("APP_MAILRU_MAIL_RU_48", "apps/48/mailru-mail.ru");
    public static final AntuIcon APP_MAKEHUMAN_48 = create("APP_MAKEHUMAN_48", "apps/48/makehuman");
    public static final AntuIcon APP_MAMEEXECUTOR_48 = create("APP_MAMEEXECUTOR_48", "apps/48/mameexecutor");
    public static final AntuIcon APP_MANSA_48 = create("APP_MANSA_48", "apps/48/mansa");
    public static final AntuIcon APP_MAPS_48 = create("APP_MAPS_48", "apps/48/maps");
    public static final AntuIcon APP_MARBLE_48 = create("APP_MARBLE_48", "apps/48/marble");
    public static final AntuIcon APP_MASTER_PDF_EDITOR_48 = create("APP_MASTER_PDF_EDITOR_48", "apps/48/master-pdf-editor");
    public static final AntuIcon APP_MASTERPDFEDITOR_48 = create("APP_MASTERPDFEDITOR_48", "apps/48/masterpdfeditor");
    public static final AntuIcon APP_MATHEMATICA_48 = create("APP_MATHEMATICA_48", "apps/48/mathematica");
    public static final AntuIcon APP_MATLAB_48 = create("APP_MATLAB_48", "apps/48/matlab");
    public static final AntuIcon APP_MAXTHON_BROWSER_48 = create("APP_MAXTHON_BROWSER_48", "apps/48/maxthon-browser");
    public static final AntuIcon APP_MCOMIX_48 = create("APP_MCOMIX_48", "apps/48/mcomix");
    public static final AntuIcon APP_MEDIA_PLAYER_BANSHEE_48 = create("APP_MEDIA_PLAYER_BANSHEE_48", "apps/48/media-player-banshee");
    public static final AntuIcon APP_MEDIAINFO_GUI_48 = create("APP_MEDIAINFO_GUI_48", "apps/48/mediainfo-gui");
    public static final AntuIcon APP_MEGASYNC_48 = create("APP_MEGASYNC_48", "apps/48/megasync");
    public static final AntuIcon APP_MELD_48 = create("APP_MELD_48", "apps/48/meld");
    public static final AntuIcon APP_MELLOWPLAYER_48 = create("APP_MELLOWPLAYER_48", "apps/48/mellowplayer");
    public static final AntuIcon APP_MENDELEY_48 = create("APP_MENDELEY_48", "apps/48/mendeley");
    public static final AntuIcon APP_MENU_EDITOR_48 = create("APP_MENU_EDITOR_48", "apps/48/menu-editor");
    public static final AntuIcon APP_MIDORI_48 = create("APP_MIDORI_48", "apps/48/midori");
    public static final AntuIcon APP_MINITUBE_48 = create("APP_MINITUBE_48", "apps/48/minitube");
    public static final AntuIcon APP_MIRROR_CHECK_48 = create("APP_MIRROR_CHECK_48", "apps/48/mirror-check");
    public static final AntuIcon APP_MIXXX_48 = create("APP_MIXXX_48", "apps/48/mixxx-icon");
    public static final AntuIcon APP_MKVMERGE_48 = create("APP_MKVMERGE_48", "apps/48/mkvmerge");
    public static final AntuIcon APP_MKVTOOLNIX_GUI_48 = create("APP_MKVTOOLNIX_GUI_48", "apps/48/mkvtoolnix-gui");
    public static final AntuIcon APP_MLCAD_48 = create("APP_MLCAD_48", "apps/48/mlcad");
    public static final AntuIcon APP_MODELIO_48 = create("APP_MODELIO_48", "apps/48/modelio");
    public static final AntuIcon APP_MODEMMANAGER_48 = create("APP_MODEMMANAGER_48", "apps/48/ModemManager");
    public static final AntuIcon APP_MOINMOIN_48 = create("APP_MOINMOIN_48", "apps/48/moinmoin");
    public static final AntuIcon APP_MONGODB_48 = create("APP_MONGODB_48", "apps/48/mongodb");
    public static final AntuIcon APP_MONODEVELOP_48 = create("APP_MONODEVELOP_48", "apps/48/monodevelop");
    public static final AntuIcon APP_MOVIE_EDITOR_48 = create("APP_MOVIE_EDITOR_48", "apps/48/movie-editor");
    public static final AntuIcon APP_MPLAYER_48 = create("APP_MPLAYER_48", "apps/48/mplayer");
    public static final AntuIcon APP_MPV_48 = create("APP_MPV_48", "apps/48/mpv");
    public static final AntuIcon APP_MS_EXCEL_48 = create("APP_MS_EXCEL_48", "apps/48/ms-excel");
    public static final AntuIcon APP_MS_ONENOTE_48 = create("APP_MS_ONENOTE_48", "apps/48/ms-onenote");
    public static final AntuIcon APP_MS_POWERPOINT_48 = create("APP_MS_POWERPOINT_48", "apps/48/ms-powerpoint");
    public static final AntuIcon APP_MS_WORD_48 = create("APP_MS_WORD_48", "apps/48/ms-word");
    public static final AntuIcon APP_MSN_PROTOCOL_48 = create("APP_MSN_PROTOCOL_48", "apps/48/msn_protocol");
    public static final AntuIcon APP_MULTIMEDIA_VOLUME_CONTROL_48 = create("APP_MULTIMEDIA_VOLUME_CONTROL_48", "apps/48/multimedia-volume-control");
    public static final AntuIcon APP_MULTISYSTEM_48 = create("APP_MULTISYSTEM_48", "apps/48/multisystem");
    public static final AntuIcon APP_MUMBLE_48 = create("APP_MUMBLE_48", "apps/48/mumble");
    public static final AntuIcon APP_MUON_48 = create("APP_MUON_48", "apps/48/muon");
    public static final AntuIcon APP_MUONDISCOVER_48 = create("APP_MUONDISCOVER_48", "apps/48/muondiscover");
    public static final AntuIcon APP_MUPEN64PLUS_48 = create("APP_MUPEN64PLUS_48", "apps/48/mupen64plus");
    public static final AntuIcon APP_MUSESCORE_48 = create("APP_MUSESCORE_48", "apps/48/musescore");
    public static final AntuIcon APP_MUSICBRAINZ_48 = create("APP_MUSICBRAINZ_48", "apps/48/musicbrainz");
    public static final AntuIcon APP_MUSICTUBE_48 = create("APP_MUSICTUBE_48", "apps/48/musictube");
    public static final AntuIcon APP_MUSIQUE_48 = create("APP_MUSIQUE_48", "apps/48/musique");
    public static final AntuIcon APP_MYSMS_48 = create("APP_MYSMS_48", "apps/48/mysms");
    public static final AntuIcon APP_MYSQL_WORKBENCH_48 = create("APP_MYSQL_WORKBENCH_48", "apps/48/mysql-workbench");
    public static final AntuIcon APP_NAVICAT_48 = create("APP_NAVICAT_48", "apps/48/navicat");
    public static final AntuIcon APP_NEMIVER_48 = create("APP_NEMIVER_48", "apps/48/nemiver");
    public static final AntuIcon APP_NEPOMUK_16 = create("APP_NEPOMUK_16", "apps/16/nepomuk");
    public static final AntuIcon APP_NEPOMUK_32 = create("APP_NEPOMUK_32", "apps/32/nepomuk");
    public static final AntuIcon APP_NEPOMUK_48 = create("APP_NEPOMUK_48", "apps/48/nepomuk");
    public static final AntuIcon APP_NETBEANS_48 = create("APP_NETBEANS_48", "apps/48/netbeans");
    public static final AntuIcon APP_NETBEANS_BETA_48 = create("APP_NETBEANS_BETA_48", "apps/48/netbeans-beta");
    public static final AntuIcon APP_NETBEANS_IDE_48 = create("APP_NETBEANS_IDE_48", "apps/48/netbeans-ide");
    public static final AntuIcon APP_NETFLIX_DESKTOP_48 = create("APP_NETFLIX_DESKTOP_48", "apps/48/netflix-desktop");
    public static final AntuIcon APP_NETWORK_MANAGER_48 = create("APP_NETWORK_MANAGER_48", "apps/48/network-manager");
    public static final AntuIcon APP_NEVERBALL_48 = create("APP_NEVERBALL_48", "apps/48/neverball");
    public static final AntuIcon APP_NEVERPUTT_48 = create("APP_NEVERPUTT_48", "apps/48/neverputt");
    public static final AntuIcon APP_NEWSBLUR_48 = create("APP_NEWSBLUR_48", "apps/48/newsblur");
    public static final AntuIcon APP_NEXUIZ_48 = create("APP_NEXUIZ_48", "apps/48/nexuiz");
    public static final AntuIcon APP_NINJA_IDE_48 = create("APP_NINJA_IDE_48", "apps/48/ninja-ide");
    public static final AntuIcon APP_NITRO_48 = create("APP_NITRO_48", "apps/48/nitro");
    public static final AntuIcon APP_NITROGEN_48 = create("APP_NITROGEN_48", "apps/48/nitrogen");
    public static final AntuIcon APP_NITROSHARE_48 = create("APP_NITROSHARE_48", "apps/48/nitroshare");
    public static final AntuIcon APP_NITROTASKS_48 = create("APP_NITROTASKS_48", "apps/48/nitrotasks");
    public static final AntuIcon APP_NM_DEVICE_WIRELESS_48 = create("APP_NM_DEVICE_WIRELESS_48", "apps/48/nm-device-wireless");
    public static final AntuIcon APP_NMAP_48 = create("APP_NMAP_48", "apps/48/nmap");
    public static final AntuIcon APP_NMAP_ICON_48 = create("APP_NMAP_ICON_48", "apps/48/nmap_icon");
    public static final AntuIcon APP_NOMACS_48 = create("APP_NOMACS_48", "apps/48/nomacs");
    public static final AntuIcon APP_NOTIFYOSDCONF_48 = create("APP_NOTIFYOSDCONF_48", "apps/48/notifyosdconf");
    public static final AntuIcon APP_NTFS_CONFIG_48 = create("APP_NTFS_CONFIG_48", "apps/48/ntfs-config");
    public static final AntuIcon APP_NUVOLAPLAYER_48 = create("APP_NUVOLAPLAYER_48", "apps/48/nuvolaplayer");
    public static final AntuIcon APP_NVCLOCK_48 = create("APP_NVCLOCK_48", "apps/48/nvclock");
    public static final AntuIcon APP_NVDOCK_48 = create("APP_NVDOCK_48", "apps/48/nvdock");
    public static final AntuIcon APP_NVIDIA_310_SETTINGS_48 = create("APP_NVIDIA_310_SETTINGS_48", "apps/48/nvidia-310-settings");
    public static final AntuIcon APP_NVIDIA_48 = create("APP_NVIDIA_48", "apps/48/nvidia");
    public static final AntuIcon APP_NVIDIA_CURRENT_SETTINGS_48 = create("APP_NVIDIA_CURRENT_SETTINGS_48", "apps/48/nvidia-current-settings");
    public static final AntuIcon APP_NVIDIA_CURRENT_UPDATES_SETTINGS_48 = create("APP_NVIDIA_CURRENT_UPDATES_SETTINGS_48", "apps/48/nvidia-current-updates-settings");
    public static final AntuIcon APP_NVIDIA_DRIVERS_SETTINGS_48 = create("APP_NVIDIA_DRIVERS_SETTINGS_48", "apps/48/nvidia-drivers-settings");
    public static final AntuIcon APP_NVIDIA_SETTINGS_48 = create("APP_NVIDIA_SETTINGS_48", "apps/48/nvidia-settings");
    public static final AntuIcon APP_NVIDIA_X_SERVER_SETTING_48 = create("APP_NVIDIA_X_SERVER_SETTING_48", "apps/48/nvidia-x-server-setting");
    public static final AntuIcon APP_NVIDIA_X_SERVER_SETTINGS_48 = create("APP_NVIDIA_X_SERVER_SETTINGS_48", "apps/48/nvidia-x-server-settings");
    public static final AntuIcon APP_NYLAS_48 = create("APP_NYLAS_48", "apps/48/nylas");
    public static final AntuIcon APP_OCENAUDIO_48 = create("APP_OCENAUDIO_48", "apps/48/ocenaudio");
    public static final AntuIcon APP_OCTAVE_48 = create("APP_OCTAVE_48", "apps/48/octave");
    public static final AntuIcon APP_OCTAVE_ICON_48 = create("APP_OCTAVE_ICON_48", "apps/48/octave-icon");
    public static final AntuIcon APP_OCTAVE_LOGO_48 = create("APP_OCTAVE_LOGO_48", "apps/48/octave-logo");
    public static final AntuIcon APP_OCTOPI_48 = create("APP_OCTOPI_48", "apps/48/octopi");
    public static final AntuIcon APP_OCTOPI_NOTIFIER_48 = create("APP_OCTOPI_NOTIFIER_48", "apps/48/octopi-notifier");
    public static final AntuIcon APP_OCTOPI_PANEL_48 = create("APP_OCTOPI_PANEL_48", "apps/48/octopi-panel");
    public static final AntuIcon APP_OCTOPI_RED_48 = create("APP_OCTOPI_RED_48", "apps/48/octopi_red");
    public static final AntuIcon APP_OCTOPI_YELLOW_48 = create("APP_OCTOPI_YELLOW_48", "apps/48/octopi_yellow");
    public static final AntuIcon APP_OCTOPI2_48 = create("APP_OCTOPI2_48", "apps/48/octopi2");
    public static final AntuIcon APP_OFFICE_ADDRESS_BOOK_48 = create("APP_OFFICE_ADDRESS_BOOK_48", "apps/48/office-address-book");
    public static final AntuIcon APP_OFFICE_CALENDAR_48 = create("APP_OFFICE_CALENDAR_48", "apps/48/office-calendar");
    public static final AntuIcon APP_OFFICE_PRINTER_48 = create("APP_OFFICE_PRINTER_48", "apps/48/office-printer");
    public static final AntuIcon APP_OKTETA_48 = create("APP_OKTETA_48", "apps/48/okteta");
    public static final AntuIcon APP_OKULAR_48 = create("APP_OKULAR_48", "apps/48/okular");
    public static final AntuIcon APP_OMEGAT_48 = create("APP_OMEGAT_48", "apps/48/omegat");
    public static final AntuIcon APP_ONBOARD_48 = create("APP_ONBOARD_48", "apps/48/onboard");
    public static final AntuIcon APP_ONBOARD_SETTINGS_48 = create("APP_ONBOARD_SETTINGS_48", "apps/48/onboard-settings");
    public static final AntuIcon APP_ONEDRIVE_48 = create("APP_ONEDRIVE_48", "apps/48/onedrive");
    public static final AntuIcon APP_OOO_48 = create("APP_OOO_48", "apps/48/ooo");
    public static final AntuIcon APP_OPENARENA_48 = create("APP_OPENARENA_48", "apps/48/openarena");
    public static final AntuIcon APP_OPENBRAVO_ERP_48 = create("APP_OPENBRAVO_ERP_48", "apps/48/openbravo-erp");
    public static final AntuIcon APP_OPENJDK_6_48 = create("APP_OPENJDK_6_48", "apps/48/openjdk-6");
    public static final AntuIcon APP_OPENJDK_7_48 = create("APP_OPENJDK_7_48", "apps/48/openjdk-7");
    public static final AntuIcon APP_OPENJDK_8_48 = create("APP_OPENJDK_8_48", "apps/48/openjdk-8");
    public static final AntuIcon APP_OPENSCAD_48 = create("APP_OPENSCAD_48", "apps/48/openscad");
    public static final AntuIcon APP_OPENSHOT_48 = create("APP_OPENSHOT_48", "apps/48/openshot");
    public static final AntuIcon APP_OPENTTD_48 = create("APP_OPENTTD_48", "apps/48/openttd");
    public static final AntuIcon APP_OPERA_48 = create("APP_OPERA_48", "apps/48/opera");
    public static final AntuIcon APP_OPERA_BETA_48 = create("APP_OPERA_BETA_48", "apps/48/opera-beta");
    public static final AntuIcon APP_OPERA_DEVELOPER_48 = create("APP_OPERA_DEVELOPER_48", "apps/48/opera-developer");
    public static final AntuIcon APP_OPERA_NEXT_BROWSER_48 = create("APP_OPERA_NEXT_BROWSER_48", "apps/48/opera-next-browser");
    public static final AntuIcon APP_OPERA_WIDGET_48 = create("APP_OPERA_WIDGET_48", "apps/48/opera-widget");
    public static final AntuIcon APP_OTHMAN_BROWSER_48 = create("APP_OTHMAN_BROWSER_48", "apps/48/othman-browser");
    public static final AntuIcon APP_OTTER_BROWSER_48 = create("APP_OTTER_BROWSER_48", "apps/48/otter-browser");
    public static final AntuIcon APP_OUTLOOK_48 = create("APP_OUTLOOK_48", "apps/48/outlook");
    public static final AntuIcon APP_OX_48 = create("APP_OX_48", "apps/48/ox");
    public static final AntuIcon APP_OXYGEN_48 = create("APP_OXYGEN_48", "apps/48/oxygen");
    public static final AntuIcon APP_PACKAGE_ADMINISTRATION_48 = create("APP_PACKAGE_ADMINISTRATION_48", "apps/48/package_administration");
    public static final AntuIcon APP_PACKAGE_DEVELOPMENT_DEBUGGER_48 = create("APP_PACKAGE_DEVELOPMENT_DEBUGGER_48", "apps/48/package_development_debugger");
    public static final AntuIcon APP_PACKAGE_DEVELOPMENT_TOOLS_48 = create("APP_PACKAGE_DEVELOPMENT_TOOLS_48", "apps/48/package_development_tools");
    public static final AntuIcon APP_PACKAGE_DEVELOPMENT_TRANSLATION_48 = create("APP_PACKAGE_DEVELOPMENT_TRANSLATION_48", "apps/48/package_development_translation");
    public static final AntuIcon APP_PACKAGE_EDUTAINMENT_ASTRONOMY_48 = create("APP_PACKAGE_EDUTAINMENT_ASTRONOMY_48", "apps/48/package_edutainment_astronomy");
    public static final AntuIcon APP_PACKAGE_EDUTAINMENT_CHEMICAL_48 = create("APP_PACKAGE_EDUTAINMENT_CHEMICAL_48", "apps/48/package_edutainment_chemical");
    public static final AntuIcon APP_PACKAGE_EDUTAINMENT_SCIENCE_48 = create("APP_PACKAGE_EDUTAINMENT_SCIENCE_48", "apps/48/package_edutainment_science");
    public static final AntuIcon APP_PACKAGE_MULTIMEDIA_48 = create("APP_PACKAGE_MULTIMEDIA_48", "apps/48/package_multimedia");
    public static final AntuIcon APP_PACKAGE_SETTINGS_48 = create("APP_PACKAGE_SETTINGS_48", "apps/48/package_settings");
    public static final AntuIcon APP_PALEMOON_48 = create("APP_PALEMOON_48", "apps/48/palemoon");
    public static final AntuIcon APP_PARAVIEW_48 = create("APP_PARAVIEW_48", "apps/48/paraview");
    public static final AntuIcon APP_PARTITIONMANAGER_48 = create("APP_PARTITIONMANAGER_48", "apps/48/partitionmanager");
    public static final AntuIcon APP_PASSWORD_48 = create("APP_PASSWORD_48", "apps/48/password");
    public static final AntuIcon APP_PASSWORD_MANAGER_48 = create("APP_PASSWORD_MANAGER_48", "apps/48/password-manager");
    public static final AntuIcon APP_PASSWORDS_48 = create("APP_PASSWORDS_48", "apps/48/passwords");
    public static final AntuIcon APP_PATCHAGE_48 = create("APP_PATCHAGE_48", "apps/48/patchage");
    public static final AntuIcon APP_PAVUMETER_48 = create("APP_PAVUMETER_48", "apps/48/pavumeter");
    public static final AntuIcon APP_PAYPAL_48 = create("APP_PAYPAL_48", "apps/48/paypal");
    public static final AntuIcon APP_PCSX_ICON_48 = create("APP_PCSX_ICON_48", "apps/48/pcsx-icon");
    public static final AntuIcon APP_PCSX2_48 = create("APP_PCSX2_48", "apps/48/pcsx2");
    public static final AntuIcon APP_PDFMOD_48 = create("APP_PDFMOD_48", "apps/48/pdfmod");
    public static final AntuIcon APP_PENCIL_48 = create("APP_PENCIL_48", "apps/48/pencil");
    public static final AntuIcon APP_PGADMIN_48 = create("APP_PGADMIN_48", "apps/48/pgadmin");
    public static final AntuIcon APP_PGLGUI_48 = create("APP_PGLGUI_48", "apps/48/pglgui");
    public static final AntuIcon APP_PHATCH_48 = create("APP_PHATCH_48", "apps/48/phatch");
    public static final AntuIcon APP_PHATCH_INSPECTOR_48 = create("APP_PHATCH_INSPECTOR_48", "apps/48/phatch-inspector");
    public static final AntuIcon APP_PHONON_GSTREAMER_48 = create("APP_PHONON_GSTREAMER_48", "apps/48/phonon-gstreamer");
    public static final AntuIcon APP_PHOTOFILMSTRIP_48 = create("APP_PHOTOFILMSTRIP_48", "apps/48/photofilmstrip");
    public static final AntuIcon APP_PHOTOLAYOUTSEDITOR_48 = create("APP_PHOTOLAYOUTSEDITOR_48", "apps/48/photolayoutseditor");
    public static final AntuIcon APP_PHOTOQT_48 = create("APP_PHOTOQT_48", "apps/48/photoqt");
    public static final AntuIcon APP_PHP_48 = create("APP_PHP_48", "apps/48/php");
    public static final AntuIcon APP_PHPMYADMIN_48 = create("APP_PHPMYADMIN_48", "apps/48/phpmyadmin");
    public static final AntuIcon APP_PICARD_48 = create("APP_PICARD_48", "apps/48/picard");
    public static final AntuIcon APP_PICASA_48 = create("APP_PICASA_48", "apps/48/picasa");
    public static final AntuIcon APP_PICMI_48 = create("APP_PICMI_48", "apps/48/picmi");
    public static final AntuIcon APP_PIDGIN_48 = create("APP_PIDGIN_48", "apps/48/pidgin");
    public static final AntuIcon APP_PIMSETTINGEXPORTER_48 = create("APP_PIMSETTINGEXPORTER_48", "apps/48/pimsettingexporter");
    public static final AntuIcon APP_PINO_48 = create("APP_PINO_48", "apps/48/pino");
    public static final AntuIcon APP_PLAN_48 = create("APP_PLAN_48", "apps/48/plan");
    public static final AntuIcon APP_PLANK_48 = create("APP_PLANK_48", "apps/48/plank");
    public static final AntuIcon APP_PLANK_CONFIG_48 = create("APP_PLANK_CONFIG_48", "apps/48/plank-config");
    public static final AntuIcon APP_PLANNER_48 = create("APP_PLANNER_48", "apps/48/planner");
    public static final AntuIcon APP_PLANWORK_48 = create("APP_PLANWORK_48", "apps/48/planwork");
    public static final AntuIcon APP_PLASMA_32 = create("APP_PLASMA_32", "apps/32/plasma");
    public static final AntuIcon APP_PLASMA_48 = create("APP_PLASMA_48", "apps/48/plasma");
    public static final AntuIcon APP_PLASMA_MEDIA_CENTER_48 = create("APP_PLASMA_MEDIA_CENTER_48", "apps/48/plasma-media-center");
    public static final AntuIcon APP_PLASMA_MOBILE_PHONE_48 = create("APP_PLASMA_MOBILE_PHONE_48", "apps/48/plasma-mobile-phone");
    public static final AntuIcon APP_PLASMA_SEARCH_32 = create("APP_PLASMA_SEARCH_32", "apps/32/plasma-search");
    public static final AntuIcon APP_PLASMA_SEARCH_48 = create("APP_PLASMA_SEARCH_48", "apps/48/plasma-search");
    public static final AntuIcon APP_PLASMAGIK_48 = create("APP_PLASMAGIK_48", "apps/48/plasmagik");
    public static final AntuIcon APP_PLAY_STORE_48 = create("APP_PLAY_STORE_48", "apps/48/play-store");
    public static final AntuIcon APP_PLEXHOMETHEATER_48 = create("APP_PLEXHOMETHEATER_48", "apps/48/plexhometheater");
    public static final AntuIcon APP_PLEXMEDIAMANAGER_48 = create("APP_PLEXMEDIAMANAGER_48", "apps/48/plexmediamanager");
    public static final AntuIcon APP_PLV_48 = create("APP_PLV_48", "apps/48/plv");
    public static final AntuIcon APP_POCKET_48 = create("APP_POCKET_48", "apps/48/pocket");
    public static final AntuIcon APP_PODCAST_48 = create("APP_PODCAST_48", "apps/48/podcast");
    public static final AntuIcon APP_PODCAST_AMAROK_32 = create("APP_PODCAST_AMAROK_32", "apps/32/podcast-amarok");
    public static final AntuIcon APP_PODCAST_AMAROK_48 = create("APP_PODCAST_AMAROK_48", "apps/48/podcast-amarok");
    public static final AntuIcon APP_POPCORN_TIME_48 = create("APP_POPCORN_TIME_48", "apps/48/popcorn-time");
    public static final AntuIcon APP_POPCORNTIME_48 = create("APP_POPCORNTIME_48", "apps/48/popcorntime");
    public static final AntuIcon APP_POSTMAN_IMAGE_UPLOADER_48 = create("APP_POSTMAN_IMAGE_UPLOADER_48", "apps/48/postman-image-uploader");
    public static final AntuIcon APP_POWDER_TOY_48 = create("APP_POWDER_TOY_48", "apps/48/powder-toy");
    public static final AntuIcon APP_PRAGHA_48 = create("APP_PRAGHA_48", "apps/48/pragha");
    public static final AntuIcon APP_PREFERENCES_CONTACT_LIST_48 = create("APP_PREFERENCES_CONTACT_LIST_48", "apps/48/preferences-contact-list");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_32 = create("APP_PREFERENCES_DESKTOP_32", "apps/32/preferences-desktop");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_48 = create("APP_PREFERENCES_DESKTOP_48", "apps/48/preferences-desktop");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_ACCESSIBILITY_32 = create("APP_PREFERENCES_DESKTOP_ACCESSIBILITY_32", "apps/32/preferences-desktop-accessibility");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_ACCESSIBILITY_48 = create("APP_PREFERENCES_DESKTOP_ACCESSIBILITY_48", "apps/48/preferences-desktop-accessibility");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_BALOO_32 = create("APP_PREFERENCES_DESKTOP_BALOO_32", "apps/32/preferences-desktop-baloo");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_BALOO_48 = create("APP_PREFERENCES_DESKTOP_BALOO_48", "apps/48/preferences-desktop-baloo");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_COLOR_32 = create("APP_PREFERENCES_DESKTOP_COLOR_32", "apps/32/preferences-desktop-color");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_COLOR_48 = create("APP_PREFERENCES_DESKTOP_COLOR_48", "apps/48/preferences-desktop-color");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_CRYPTOGRAPHY_32 = create("APP_PREFERENCES_DESKTOP_CRYPTOGRAPHY_32", "apps/32/preferences-desktop-cryptography");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_CRYPTOGRAPHY_48 = create("APP_PREFERENCES_DESKTOP_CRYPTOGRAPHY_48", "apps/48/preferences-desktop-cryptography");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DEFAULT_APPLICATIONS_32 = create("APP_PREFERENCES_DESKTOP_DEFAULT_APPLICATIONS_32", "apps/32/preferences-desktop-default-applications");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DEFAULT_APPLICATIONS_48 = create("APP_PREFERENCES_DESKTOP_DEFAULT_APPLICATIONS_48", "apps/48/preferences-desktop-default-applications");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DISPLAY_32 = create("APP_PREFERENCES_DESKTOP_DISPLAY_32", "apps/32/preferences-desktop-display");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DISPLAY_48 = create("APP_PREFERENCES_DESKTOP_DISPLAY_48", "apps/48/preferences-desktop-display");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DISPLAY_COLOR_32 = create("APP_PREFERENCES_DESKTOP_DISPLAY_COLOR_32", "apps/32/preferences-desktop-display-color");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_DISPLAY_COLOR_48 = create("APP_PREFERENCES_DESKTOP_DISPLAY_COLOR_48", "apps/48/preferences-desktop-display-color");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_EFFECTS_48 = create("APP_PREFERENCES_DESKTOP_EFFECTS_48", "apps/48/preferences-desktop-effects");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_EMOTICONS_32 = create("APP_PREFERENCES_DESKTOP_EMOTICONS_32", "apps/32/preferences-desktop-emoticons");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_EMOTICONS_48 = create("APP_PREFERENCES_DESKTOP_EMOTICONS_48", "apps/48/preferences-desktop-emoticons");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FILETYPE_ASSOCIATION_32 = create("APP_PREFERENCES_DESKTOP_FILETYPE_ASSOCIATION_32", "apps/32/preferences-desktop-filetype-association");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FILETYPE_ASSOCIATION_48 = create("APP_PREFERENCES_DESKTOP_FILETYPE_ASSOCIATION_48", "apps/48/preferences-desktop-filetype-association");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FONT_22 = create("APP_PREFERENCES_DESKTOP_FONT_22", "apps/22/preferences-desktop-font");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FONT_32 = create("APP_PREFERENCES_DESKTOP_FONT_32", "apps/32/preferences-desktop-font");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FONT_48 = create("APP_PREFERENCES_DESKTOP_FONT_48", "apps/48/preferences-desktop-font");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FONT_INSTALLER_32 = create("APP_PREFERENCES_DESKTOP_FONT_INSTALLER_32", "apps/32/preferences-desktop-font-installer");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_FONT_INSTALLER_48 = create("APP_PREFERENCES_DESKTOP_FONT_INSTALLER_48", "apps/48/preferences-desktop-font-installer");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_GAMING_32 = create("APP_PREFERENCES_DESKTOP_GAMING_32", "apps/32/preferences-desktop-gaming");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_GAMING_48 = create("APP_PREFERENCES_DESKTOP_GAMING_48", "apps/48/preferences-desktop-gaming");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_ICONS_32 = create("APP_PREFERENCES_DESKTOP_ICONS_32", "apps/32/preferences-desktop-icons");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_ICONS_48 = create("APP_PREFERENCES_DESKTOP_ICONS_48", "apps/48/preferences-desktop-icons");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_KEYBOARD_32 = create("APP_PREFERENCES_DESKTOP_KEYBOARD_32", "apps/32/preferences-desktop-keyboard");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_KEYBOARD_48 = create("APP_PREFERENCES_DESKTOP_KEYBOARD_48", "apps/48/preferences-desktop-keyboard");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_KEYBOARD_SHORTCUTS_48 = create("APP_PREFERENCES_DESKTOP_KEYBOARD_SHORTCUTS_48", "apps/48/preferences-desktop-keyboard-shortcuts");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_LAUNCH_FEEDBACK_32 = create("APP_PREFERENCES_DESKTOP_LAUNCH_FEEDBACK_32", "apps/32/preferences-desktop-launch-feedback");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_LAUNCH_FEEDBACK_48 = create("APP_PREFERENCES_DESKTOP_LAUNCH_FEEDBACK_48", "apps/48/preferences-desktop-launch-feedback");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_LOCALE_32 = create("APP_PREFERENCES_DESKTOP_LOCALE_32", "apps/32/preferences-desktop-locale");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_LOCALE_48 = create("APP_PREFERENCES_DESKTOP_LOCALE_48", "apps/48/preferences-desktop-locale");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_MOUSE_32 = create("APP_PREFERENCES_DESKTOP_MOUSE_32", "apps/32/preferences-desktop-mouse");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_MOUSE_48 = create("APP_PREFERENCES_DESKTOP_MOUSE_48", "apps/48/preferences-desktop-mouse");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_MULTIMEDIA_32 = create("APP_PREFERENCES_DESKTOP_MULTIMEDIA_32", "apps/32/preferences-desktop-multimedia");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_MULTIMEDIA_48 = create("APP_PREFERENCES_DESKTOP_MULTIMEDIA_48", "apps/48/preferences-desktop-multimedia");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_NOTIFICATION_32 = create("APP_PREFERENCES_DESKTOP_NOTIFICATION_32", "apps/32/preferences-desktop-notification");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_NOTIFICATION_48 = create("APP_PREFERENCES_DESKTOP_NOTIFICATION_48", "apps/48/preferences-desktop-notification");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_NOTIFICATION_BELL_32 = create("APP_PREFERENCES_DESKTOP_NOTIFICATION_BELL_32", "apps/32/preferences-desktop-notification-bell");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_NOTIFICATION_BELL_48 = create("APP_PREFERENCES_DESKTOP_NOTIFICATION_BELL_48", "apps/48/preferences-desktop-notification-bell");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_ONLINE_ACCOUNTS_48 = create("APP_PREFERENCES_DESKTOP_ONLINE_ACCOUNTS_48", "apps/48/preferences-desktop-online-accounts");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PERIPHERALS_32 = create("APP_PREFERENCES_DESKTOP_PERIPHERALS_32", "apps/32/preferences-desktop-peripherals");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PERIPHERALS_48 = create("APP_PREFERENCES_DESKTOP_PERIPHERALS_48", "apps/48/preferences-desktop-peripherals");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PERSONAL_48 = create("APP_PREFERENCES_DESKTOP_PERSONAL_48", "apps/48/preferences-desktop-personal");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PLASMA_32 = create("APP_PREFERENCES_DESKTOP_PLASMA_32", "apps/32/preferences-desktop-plasma");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PLASMA_48 = create("APP_PREFERENCES_DESKTOP_PLASMA_48", "apps/48/preferences-desktop-plasma");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_PRINTER_48 = create("APP_PREFERENCES_DESKTOP_PRINTER_48", "apps/48/preferences-desktop-printer");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_REMOTE_DESKTOP_48 = create("APP_PREFERENCES_DESKTOP_REMOTE_DESKTOP_48", "apps/48/preferences-desktop-remote-desktop");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_SCREENSAVER_32 = create("APP_PREFERENCES_DESKTOP_SCREENSAVER_32", "apps/32/preferences-desktop-screensaver");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_SCREENSAVER_48 = create("APP_PREFERENCES_DESKTOP_SCREENSAVER_48", "apps/48/preferences-desktop-screensaver");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_SEARCH_48 = create("APP_PREFERENCES_DESKTOP_SEARCH_48", "apps/48/preferences-desktop-search");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_SOUND_32 = create("APP_PREFERENCES_DESKTOP_SOUND_32", "apps/32/preferences-desktop-sound");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_SOUND_48 = create("APP_PREFERENCES_DESKTOP_SOUND_48", "apps/48/preferences-desktop-sound");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_TEXT_TO_SPEECH_32 = create("APP_PREFERENCES_DESKTOP_TEXT_TO_SPEECH_32", "apps/32/preferences-desktop-text-to-speech");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_TEXT_TO_SPEECH_48 = create("APP_PREFERENCES_DESKTOP_TEXT_TO_SPEECH_48", "apps/48/preferences-desktop-text-to-speech");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_THEME_32 = create("APP_PREFERENCES_DESKTOP_THEME_32", "apps/32/preferences-desktop-theme");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_THEME_48 = create("APP_PREFERENCES_DESKTOP_THEME_48", "apps/48/preferences-desktop-theme");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_USER_32 = create("APP_PREFERENCES_DESKTOP_USER_32", "apps/32/preferences-desktop-user");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_USER_48 = create("APP_PREFERENCES_DESKTOP_USER_48", "apps/48/preferences-desktop-user");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_USER_PASSWORD_32 = create("APP_PREFERENCES_DESKTOP_USER_PASSWORD_32", "apps/32/preferences-desktop-user-password");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_USER_PASSWORD_48 = create("APP_PREFERENCES_DESKTOP_USER_PASSWORD_48", "apps/48/preferences-desktop-user-password");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_WALLPAPER_32 = create("APP_PREFERENCES_DESKTOP_WALLPAPER_32", "apps/32/preferences-desktop-wallpaper");
    public static final AntuIcon APP_PREFERENCES_DESKTOP_WALLPAPER_48 = create("APP_PREFERENCES_DESKTOP_WALLPAPER_48", "apps/48/preferences-desktop-wallpaper");
    public static final AntuIcon APP_PREFERENCES_DRIVER_48 = create("APP_PREFERENCES_DRIVER_48", "apps/48/preferences-driver");
    public static final AntuIcon APP_PREFERENCES_GTK_CONFIG_32 = create("APP_PREFERENCES_GTK_CONFIG_32", "apps/32/preferences-gtk-config");
    public static final AntuIcon APP_PREFERENCES_GTK_CONFIG_48 = create("APP_PREFERENCES_GTK_CONFIG_48", "apps/48/preferences-gtk-config");
    public static final AntuIcon APP_PREFERENCES_KCALC_CONSTANTS_48 = create("APP_PREFERENCES_KCALC_CONSTANTS_48", "apps/48/preferences-kcalc-constants");
    public static final AntuIcon APP_PREFERENCES_KDE_CONNECT_32 = create("APP_PREFERENCES_KDE_CONNECT_32", "apps/32/preferences-kde-connect");
    public static final AntuIcon APP_PREFERENCES_KDE_CONNECT_48 = create("APP_PREFERENCES_KDE_CONNECT_48", "apps/48/preferences-kde-connect");
    public static final AntuIcon APP_PREFERENCES_MANAGEMENT_SERVICE_48 = create("APP_PREFERENCES_MANAGEMENT_SERVICE_48", "apps/48/preferences-management-service");
    public static final AntuIcon APP_PREFERENCES_OTHER_32 = create("APP_PREFERENCES_OTHER_32", "apps/32/preferences-other");
    public static final AntuIcon APP_PREFERENCES_OTHER_48 = create("APP_PREFERENCES_OTHER_48", "apps/48/preferences-other");
    public static final AntuIcon APP_PREFERENCES_PLUGIN_22 = create("APP_PREFERENCES_PLUGIN_22", "apps/22/preferences-plugin");
    public static final AntuIcon APP_PREFERENCES_PLUGIN_32 = create("APP_PREFERENCES_PLUGIN_32", "apps/32/preferences-plugin");
    public static final AntuIcon APP_PREFERENCES_PLUGIN_48 = create("APP_PREFERENCES_PLUGIN_48", "apps/48/preferences-plugin");
    public static final AntuIcon APP_PREFERENCES_PLUGIN_SCRIPT_48 = create("APP_PREFERENCES_PLUGIN_SCRIPT_48", "apps/48/preferences-plugin-script");
    public static final AntuIcon APP_PREFERENCES_REMOTE_DESKTOP_48 = create("APP_PREFERENCES_REMOTE_DESKTOP_48", "apps/48/preferences-remote-desktop");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_48 = create("APP_PREFERENCES_SYSTEM_48", "apps/48/preferences-system");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_BLUETOOTH_16 = create("APP_PREFERENCES_SYSTEM_BLUETOOTH_16", "apps/16/preferences-system-bluetooth");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_BLUETOOTH_32 = create("APP_PREFERENCES_SYSTEM_BLUETOOTH_32", "apps/32/preferences-system-bluetooth");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_BLUETOOTH_48 = create("APP_PREFERENCES_SYSTEM_BLUETOOTH_48", "apps/48/preferences-system-bluetooth");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_BLUETOOTH_INACTIVE_48 = create("APP_PREFERENCES_SYSTEM_BLUETOOTH_INACTIVE_48", "apps/48/preferences-system-bluetooth-inactive");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_BLUETOOTH_INACTIVE2_48 = create("APP_PREFERENCES_SYSTEM_BLUETOOTH_INACTIVE2_48", "apps/48/preferences-system-bluetooth-inactive2");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_FIREWALL_48 = create("APP_PREFERENCES_SYSTEM_FIREWALL_48", "apps/48/preferences-system-firewall");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_HOTCORNERS_48 = create("APP_PREFERENCES_SYSTEM_HOTCORNERS_48", "apps/48/preferences-system-hotcorners");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_LOGIN_32 = create("APP_PREFERENCES_SYSTEM_LOGIN_32", "apps/32/preferences-system-login");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_LOGIN_48 = create("APP_PREFERENCES_SYSTEM_LOGIN_48", "apps/48/preferences-system-login");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_NETWORK_32 = create("APP_PREFERENCES_SYSTEM_NETWORK_32", "apps/32/preferences-system-network");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_NETWORK_48 = create("APP_PREFERENCES_SYSTEM_NETWORK_48", "apps/48/preferences-system-network");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_NETWORK_SHARING_32 = create("APP_PREFERENCES_SYSTEM_NETWORK_SHARING_32", "apps/32/preferences-system-network-sharing");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_NETWORK_SHARING_48 = create("APP_PREFERENCES_SYSTEM_NETWORK_SHARING_48", "apps/48/preferences-system-network-sharing");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_PERFORMANCE_48 = create("APP_PREFERENCES_SYSTEM_PERFORMANCE_48", "apps/48/preferences-system-performance");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_POWER_48 = create("APP_PREFERENCES_SYSTEM_POWER_48", "apps/48/preferences-system-power");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_POWER_MANAGEMENT_32 = create("APP_PREFERENCES_SYSTEM_POWER_MANAGEMENT_32", "apps/32/preferences-system-power-management");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_POWER_MANAGEMENT_48 = create("APP_PREFERENCES_SYSTEM_POWER_MANAGEMENT_48", "apps/48/preferences-system-power-management");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_PRIVACY_32 = create("APP_PREFERENCES_SYSTEM_PRIVACY_32", "apps/32/preferences-system-privacy");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_PRIVACY_48 = create("APP_PREFERENCES_SYSTEM_PRIVACY_48", "apps/48/preferences-system-privacy");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_SEARCH_48 = create("APP_PREFERENCES_SYSTEM_SEARCH_48", "apps/48/preferences-system-search");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_SESSION_48 = create("APP_PREFERENCES_SYSTEM_SESSION_48", "apps/48/preferences-system-session");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_SESSION_SERVICES_32 = create("APP_PREFERENCES_SYSTEM_SESSION_SERVICES_32", "apps/32/preferences-system-session-services");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_SESSION_SERVICES_48 = create("APP_PREFERENCES_SYSTEM_SESSION_SERVICES_48", "apps/48/preferences-system-session-services");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_TIME_32 = create("APP_PREFERENCES_SYSTEM_TIME_32", "apps/32/preferences-system-time");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_TIME_48 = create("APP_PREFERENCES_SYSTEM_TIME_48", "apps/48/preferences-system-time");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_32 = create("APP_PREFERENCES_SYSTEM_WINDOWS_32", "apps/32/preferences-system-windows");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_48 = create("APP_PREFERENCES_SYSTEM_WINDOWS_48", "apps/48/preferences-system-windows");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_ACTIONS_32 = create("APP_PREFERENCES_SYSTEM_WINDOWS_ACTIONS_32", "apps/32/preferences-system-windows-actions");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_ACTIONS_48 = create("APP_PREFERENCES_SYSTEM_WINDOWS_ACTIONS_48", "apps/48/preferences-system-windows-actions");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_MOVE_32 = create("APP_PREFERENCES_SYSTEM_WINDOWS_MOVE_32", "apps/32/preferences-system-windows-move");
    public static final AntuIcon APP_PREFERENCES_SYSTEM_WINDOWS_MOVE_48 = create("APP_PREFERENCES_SYSTEM_WINDOWS_MOVE_48", "apps/48/preferences-system-windows-move");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_ADBLOCK_32 = create("APP_PREFERENCES_WEB_BROWSER_ADBLOCK_32", "apps/32/preferences-web-browser-adblock");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_ADBLOCK_48 = create("APP_PREFERENCES_WEB_BROWSER_ADBLOCK_48", "apps/48/preferences-web-browser-adblock");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_CACHE_32 = create("APP_PREFERENCES_WEB_BROWSER_CACHE_32", "apps/32/preferences-web-browser-cache");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_CACHE_48 = create("APP_PREFERENCES_WEB_BROWSER_CACHE_48", "apps/48/preferences-web-browser-cache");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_COOKIES_32 = create("APP_PREFERENCES_WEB_BROWSER_COOKIES_32", "apps/32/preferences-web-browser-cookies");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_COOKIES_48 = create("APP_PREFERENCES_WEB_BROWSER_COOKIES_48", "apps/48/preferences-web-browser-cookies");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_IDENTIFICATION_32 = create("APP_PREFERENCES_WEB_BROWSER_IDENTIFICATION_32", "apps/32/preferences-web-browser-identification");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_IDENTIFICATION_48 = create("APP_PREFERENCES_WEB_BROWSER_IDENTIFICATION_48", "apps/48/preferences-web-browser-identification");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_SHORTCUTS_32 = create("APP_PREFERENCES_WEB_BROWSER_SHORTCUTS_32", "apps/32/preferences-web-browser-shortcuts");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_SHORTCUTS_48 = create("APP_PREFERENCES_WEB_BROWSER_SHORTCUTS_48", "apps/48/preferences-web-browser-shortcuts");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_SSL_32 = create("APP_PREFERENCES_WEB_BROWSER_SSL_32", "apps/32/preferences-web-browser-ssl");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_SSL_48 = create("APP_PREFERENCES_WEB_BROWSER_SSL_48", "apps/48/preferences-web-browser-ssl");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_STYLESHEETS_32 = create("APP_PREFERENCES_WEB_BROWSER_STYLESHEETS_32", "apps/32/preferences-web-browser-stylesheets");
    public static final AntuIcon APP_PREFERENCES_WEB_BROWSER_STYLESHEETS_48 = create("APP_PREFERENCES_WEB_BROWSER_STYLESHEETS_48", "apps/48/preferences-web-browser-stylesheets");
    public static final AntuIcon APP_PRINTER_48 = create("APP_PRINTER_48", "apps/48/printer");
    public static final AntuIcon APP_PRJM16_TRANSPARENT_48 = create("APP_PRJM16_TRANSPARENT_48", "apps/48/prjm16-transparent");
    public static final AntuIcon APP_PSENSOR_48 = create("APP_PSENSOR_48", "apps/48/psensor");
    public static final AntuIcon APP_PTBATCHER_48 = create("APP_PTBATCHER_48", "apps/48/ptbatcher");
    public static final AntuIcon APP_PUTTY_48 = create("APP_PUTTY_48", "apps/48/putty");
    public static final AntuIcon APP_PWSAFE_48 = create("APP_PWSAFE_48", "apps/48/pwsafe");
    public static final AntuIcon APP_PYCAD_48 = create("APP_PYCAD_48", "apps/48/pycad");
    public static final AntuIcon APP_PYCHARM_48 = create("APP_PYCHARM_48", "apps/48/pycharm");
    public static final AntuIcon APP_PYRENAMER_48 = create("APP_PYRENAMER_48", "apps/48/pyrenamer");
    public static final AntuIcon APP_PYTHON_48 = create("APP_PYTHON_48", "apps/48/python");
    public static final AntuIcon APP_PYTHON_IDLE_48 = create("APP_PYTHON_IDLE_48", "apps/48/python-idle");
    public static final AntuIcon APP_PYTHON2_7_48 = create("APP_PYTHON2_7_48", "apps/48/python2.7");
    public static final AntuIcon APP_Q4WINE_48 = create("APP_Q4WINE_48", "apps/48/q4wine");
    public static final AntuIcon APP_QALCULATE_48 = create("APP_QALCULATE_48", "apps/48/qalculate");
    public static final AntuIcon APP_QASMIXER_48 = create("APP_QASMIXER_48", "apps/48/qasmixer");
    public static final AntuIcon APP_QBITTORRENT_48 = create("APP_QBITTORRENT_48", "apps/48/qbittorrent");
    public static final AntuIcon APP_QCOMICBOOK_48 = create("APP_QCOMICBOOK_48", "apps/48/qcomicbook");
    public static final AntuIcon APP_QDBUSVIEWER_48 = create("APP_QDBUSVIEWER_48", "apps/48/qdbusviewer");
    public static final AntuIcon APP_QDBUSVIEWER_QT5_48 = create("APP_QDBUSVIEWER_QT5_48", "apps/48/qdbusviewer-qt5");
    public static final AntuIcon APP_QDIRSTAT_48 = create("APP_QDIRSTAT_48", "apps/48/qdirstat");
    public static final AntuIcon APP_QELECTROTECH_48 = create("APP_QELECTROTECH_48", "apps/48/qelectrotech");
    public static final AntuIcon APP_QGIS_48 = create("APP_QGIS_48", "apps/48/qgis");
    public static final AntuIcon APP_QGIT_48 = create("APP_QGIT_48", "apps/48/qgit");
    public static final AntuIcon APP_QGO_48 = create("APP_QGO_48", "apps/48/qgo");
    public static final AntuIcon APP_QJACKCTL_48 = create("APP_QJACKCTL_48", "apps/48/qjackctl");
    public static final AntuIcon APP_QMKVEXTRACTOR_48 = create("APP_QMKVEXTRACTOR_48", "apps/48/qmkvextractor");
    public static final AntuIcon APP_QMLSCENE_48 = create("APP_QMLSCENE_48", "apps/48/qmlscene");
    public static final AntuIcon APP_QMMP_48 = create("APP_QMMP_48", "apps/48/qmmp");
    public static final AntuIcon APP_QNAPI_48_48 = create("APP_QNAPI_48_48", "apps/48/qnapi-48");
    public static final AntuIcon APP_QPHOTOREC_48 = create("APP_QPHOTOREC_48", "apps/48/qphotorec");
    public static final AntuIcon APP_QSYNTH_48 = create("APP_QSYNTH_48", "apps/48/qsynth");
    public static final AntuIcon APP_QT4LOGO_48 = create("APP_QT4LOGO_48", "apps/48/qt4logo");
    public static final AntuIcon APP_QTASSISTANT_48 = create("APP_QTASSISTANT_48", "apps/48/qtassistant");
    public static final AntuIcon APP_QTCREATOR_48 = create("APP_QTCREATOR_48", "apps/48/qtcreator");
    public static final AntuIcon APP_QTDBUSVIEWER_48 = create("APP_QTDBUSVIEWER_48", "apps/48/qtdbusviewer");
    public static final AntuIcon APP_QTDESIGNER_48 = create("APP_QTDESIGNER_48", "apps/48/qtdesigner");
    public static final AntuIcon APP_QTIPLOT_48 = create("APP_QTIPLOT_48", "apps/48/qtiplot");
    public static final AntuIcon APP_QTLINGUIST_48 = create("APP_QTLINGUIST_48", "apps/48/qtlinguist");
    public static final AntuIcon APP_QTLINGUISTIC_48 = create("APP_QTLINGUISTIC_48", "apps/48/qtlinguistic");
    public static final AntuIcon APP_QTLOGO_48 = create("APP_QTLOGO_48", "apps/48/qtlogo");
    public static final AntuIcon APP_QTOX_48 = create("APP_QTOX_48", "apps/48/qtox");
    public static final AntuIcon APP_QTRACTOR_48 = create("APP_QTRACTOR_48", "apps/48/qtractor");
    public static final AntuIcon APP_QUASSEL_48 = create("APP_QUASSEL_48", "apps/48/quassel");
    public static final AntuIcon APP_QUASSEL_L_48 = create("APP_QUASSEL_L_48", "apps/48/quassel_l");
    public static final AntuIcon APP_QUASSEL1_48 = create("APP_QUASSEL1_48", "apps/48/quassel1");
    public static final AntuIcon APP_QUCS_48 = create("APP_QUCS_48", "apps/48/qucs");
    public static final AntuIcon APP_QUICKUSBFORMATTER_48 = create("APP_QUICKUSBFORMATTER_48", "apps/48/quickusbformatter");
    public static final AntuIcon APP_QUICKVIEW_48 = create("APP_QUICKVIEW_48", "apps/48/quickview");
    public static final AntuIcon APP_QUIMUP_48 = create("APP_QUIMUP_48", "apps/48/quimup");
    public static final AntuIcon APP_QUITERSS_48 = create("APP_QUITERSS_48", "apps/48/quiterss");
    public static final AntuIcon APP_QUPZILLA_48 = create("APP_QUPZILLA_48", "apps/48/qupzilla");
    public static final AntuIcon APP_QUTIM_48 = create("APP_QUTIM_48", "apps/48/qutim");
    public static final AntuIcon APP_QV4L2_48 = create("APP_QV4L2_48", "apps/48/qv4l2");
    public static final AntuIcon APP_QWINFF_48 = create("APP_QWINFF_48", "apps/48/qwinff");
    public static final AntuIcon APP_RAWCONVERTER_48 = create("APP_RAWCONVERTER_48", "apps/48/rawconverter");
    public static final AntuIcon APP_RAWTHERAPEE_48 = create("APP_RAWTHERAPEE_48", "apps/48/rawtherapee");
    public static final AntuIcon APP_REALTIMESYNC_48 = create("APP_REALTIMESYNC_48", "apps/48/realtimesync");
    public static final AntuIcon APP_RECOLL_48 = create("APP_RECOLL_48", "apps/48/recoll");
    public static final AntuIcon APP_REDDIT_48 = create("APP_REDDIT_48", "apps/48/reddit");
    public static final AntuIcon APP_REDECLIPSE_48 = create("APP_REDECLIPSE_48", "apps/48/redeclipse");
    public static final AntuIcon APP_REDITR_48 = create("APP_REDITR_48", "apps/48/reditr");
    public static final AntuIcon APP_REDNOTEBOOK_48 = create("APP_REDNOTEBOOK_48", "apps/48/rednotebook");
    public static final AntuIcon APP_REDSHIFT_48 = create("APP_REDSHIFT_48", "apps/48/redshift");
    public static final AntuIcon APP_REKONQ_48 = create("APP_REKONQ_48", "apps/48/rekonq");
    public static final AntuIcon APP_REMMINA_48 = create("APP_REMMINA_48", "apps/48/remmina");
    public static final AntuIcon APP_REMOTE_DESKTOP_48 = create("APP_REMOTE_DESKTOP_48", "apps/48/remote-desktop");
    public static final AntuIcon APP_RENAME_48 = create("APP_RENAME_48", "apps/48/rename");
    public static final AntuIcon APP_RETROARCH_48 = create("APP_RETROARCH_48", "apps/48/retroarch");
    public static final AntuIcon APP_RETROSHARE_48 = create("APP_RETROSHARE_48", "apps/48/retroshare");
    public static final AntuIcon APP_ROBOMONGO_48 = create("APP_ROBOMONGO_48", "apps/48/robomongo");
    public static final AntuIcon APP_ROCCAT_48 = create("APP_ROCCAT_48", "apps/48/roccat");
    public static final AntuIcon APP_ROSAMP_48 = create("APP_ROSAMP_48", "apps/48/rosamp");
    public static final AntuIcon APP_RSTUDIO_48 = create("APP_RSTUDIO_48", "apps/48/rstudio");
    public static final AntuIcon APP_RUBYMINE_48 = create("APP_RUBYMINE_48", "apps/48/rubymine");
    public static final AntuIcon APP_RUNNING_48 = create("APP_RUNNING_48", "apps/48/running");
    public static final AntuIcon APP_SAMSUNG_DRIVER_48 = create("APP_SAMSUNG_DRIVER_48", "apps/48/samsung-driver");
    public static final AntuIcon APP_SANATA_48 = create("APP_SANATA_48", "apps/48/sanata-");
    public static final AntuIcon APP_SAYONARA_48 = create("APP_SAYONARA_48", "apps/48/sayonara");
    public static final AntuIcon APP_SCANNER_48 = create("APP_SCANNER_48", "apps/48/scanner");
    public static final AntuIcon APP_SCANTAILOR_48 = create("APP_SCANTAILOR_48", "apps/48/scantailor");
    public static final AntuIcon APP_SCILABBACKEND_48 = create("APP_SCILABBACKEND_48", "apps/48/scilabbackend");
    public static final AntuIcon APP_SCREEN_RECORDER_48 = create("APP_SCREEN_RECORDER_48", "apps/48/screen-recorder");
    public static final AntuIcon APP_SCREENRULER_48 = create("APP_SCREENRULER_48", "apps/48/screenruler");
    public static final AntuIcon APP_SCREENRULER_ICON_48 = create("APP_SCREENRULER_ICON_48", "apps/48/screenruler-icon");
    public static final AntuIcon APP_SCRIBUS_48 = create("APP_SCRIBUS_48", "apps/48/scribus");
    public static final AntuIcon APP_SCRIVENER_48 = create("APP_SCRIVENER_48", "apps/48/Scrivener");
    public static final AntuIcon APP_SCUOLABOOK_48 = create("APP_SCUOLABOOK_48", "apps/48/scuolabook");
    public static final AntuIcon APP_SEAFILE_48 = create("APP_SEAFILE_48", "apps/48/seafile");
    public static final AntuIcon APP_SEARCH_48 = create("APP_SEARCH_48", "apps/48/search");
    public static final AntuIcon APP_SECURITY_MEDIUM_48 = create("APP_SECURITY_MEDIUM_48", "apps/48/security-medium");
    public static final AntuIcon APP_SELENE_48 = create("APP_SELENE_48", "apps/48/selene");
    public static final AntuIcon APP_SELINUX_48 = create("APP_SELINUX_48", "apps/48/selinux");
    public static final AntuIcon APP_SEMN_48 = create("APP_SEMN_48", "apps/48/semn");
    public static final AntuIcon APP_SENCHA_ANIMATOR_48 = create("APP_SENCHA_ANIMATOR_48", "apps/48/sencha-animator");
    public static final AntuIcon APP_SESSION_PROPERTIES_48 = create("APP_SESSION_PROPERTIES_48", "apps/48/session-properties");
    public static final AntuIcon APP_SETROUBLESHOOT_48 = create("APP_SETROUBLESHOOT_48", "apps/48/setroubleshoot");
    public static final AntuIcon APP_SETROUBLESHOOT_ICON_48 = create("APP_SETROUBLESHOOT_ICON_48", "apps/48/setroubleshoot_icon");
    public static final AntuIcon APP_SHEETS_48 = create("APP_SHEETS_48", "apps/48/sheets");
    public static final AntuIcon APP_SHELL_48 = create("APP_SHELL_48", "apps/48/shell");
    public static final AntuIcon APP_SHOTWELL_48 = create("APP_SHOTWELL_48", "apps/48/shotwell");
    public static final AntuIcon APP_SHOWFOTO_48 = create("APP_SHOWFOTO_48", "apps/48/showfoto");
    public static final AntuIcon APP_SIGIL_48 = create("APP_SIGIL_48", "apps/48/sigil");
    public static final AntuIcon APP_SIMPLESCREENRECORDER_48 = create("APP_SIMPLESCREENRECORDER_48", "apps/48/simplescreenrecorder");
    public static final AntuIcon APP_SIMPLESCREENRECORDER_PAUSED_48 = create("APP_SIMPLESCREENRECORDER_PAUSED_48", "apps/48/simplescreenrecorder-paused");
    public static final AntuIcon APP_SIMPLESCREENRECORDER_RECORDING_48 = create("APP_SIMPLESCREENRECORDER_RECORDING_48", "apps/48/simplescreenrecorder-recording");
    public static final AntuIcon APP_SKANLITE_48 = create("APP_SKANLITE_48", "apps/48/skanlite");
    public static final AntuIcon APP_SKROOGE_48 = create("APP_SKROOGE_48", "apps/48/skrooge");
    public static final AntuIcon APP_SKROOGE_BLACK_48 = create("APP_SKROOGE_BLACK_48", "apps/48/skrooge-black");
    public static final AntuIcon APP_SKYDRIVE_48 = create("APP_SKYDRIVE_48", "apps/48/skydrive");
    public static final AntuIcon APP_SKYPE_48 = create("APP_SKYPE_48", "apps/48/skype");
    public static final AntuIcon APP_SKYPE_CALL_RECORDER_48 = create("APP_SKYPE_CALL_RECORDER_48", "apps/48/skype-call-recorder");
    public static final AntuIcon APP_SKYPE_PROTOCOL_48 = create("APP_SKYPE_PROTOCOL_48", "apps/48/skype_protocol");
    public static final AntuIcon APP_SMARTGIT_48 = create("APP_SMARTGIT_48", "apps/48/smartgit");
    public static final AntuIcon APP_SMARTGITHG_48 = create("APP_SMARTGITHG_48", "apps/48/smartgithg");
    public static final AntuIcon APP_SMARTSVN_48 = create("APP_SMARTSVN_48", "apps/48/smartsvn");
    public static final AntuIcon APP_SMC_48 = create("APP_SMC_48", "apps/48/smc");
    public static final AntuIcon APP_SMPLAYER_48 = create("APP_SMPLAYER_48", "apps/48/smplayer");
    public static final AntuIcon APP_SMS_PROTOCOL_48 = create("APP_SMS_PROTOCOL_48", "apps/48/sms_protocol");
    public static final AntuIcon APP_SMTUBE_48 = create("APP_SMTUBE_48", "apps/48/smtube");
    public static final AntuIcon APP_SNAPPY_48 = create("APP_SNAPPY_48", "apps/48/snappy");
    public static final AntuIcon APP_SNES9X_48 = create("APP_SNES9X_48", "apps/48/snes9x");
    public static final AntuIcon APP_SOFTWARE_PROPERTIES_48 = create("APP_SOFTWARE_PROPERTIES_48", "apps/48/software-properties");
    public static final AntuIcon APP_SOLAAR_48 = create("APP_SOLAAR_48", "apps/48/solaar");
    public static final AntuIcon APP_SONATA_48 = create("APP_SONATA_48", "apps/48/sonata");
    public static final AntuIcon APP_SOULSEEK_48 = create("APP_SOULSEEK_48", "apps/48/soulseek");
    public static final AntuIcon APP_SOUND_JUICER_48 = create("APP_SOUND_JUICER_48", "apps/48/sound-juicer");
    public static final AntuIcon APP_SOUNDCLOUD_48 = create("APP_SOUNDCLOUD_48", "apps/48/soundcloud");
    public static final AntuIcon APP_SOUNDCONVERTER_48 = create("APP_SOUNDCONVERTER_48", "apps/48/soundconverter");
    public static final AntuIcon APP_SOUNDKONVERTER_48 = create("APP_SOUNDKONVERTER_48", "apps/48/soundkonverter");
    public static final AntuIcon APP_SPACEFM_48 = create("APP_SPACEFM_48", "apps/48/spacefm");
    public static final AntuIcon APP_SPACEFM_FIND_48 = create("APP_SPACEFM_FIND_48", "apps/48/spacefm-find");
    public static final AntuIcon APP_SPARKLESHARE_48 = create("APP_SPARKLESHARE_48", "apps/48/sparkleshare");
    public static final AntuIcon APP_SPE_48 = create("APP_SPE_48", "apps/48/spe");
    public static final AntuIcon APP_SPECTACLE_48 = create("APP_SPECTACLE_48", "apps/48/spectacle");
    public static final AntuIcon APP_SPIDEROAK_48 = create("APP_SPIDEROAK_48", "apps/48/spideroak");
    public static final AntuIcon APP_SPOTIFY_48 = create("APP_SPOTIFY_48", "apps/48/spotify");
    public static final AntuIcon APP_SPOTIFY_CLIENT_48 = create("APP_SPOTIFY_CLIENT_48", "apps/48/spotify-client");
    public static final AntuIcon APP_SPRING_48 = create("APP_SPRING_48", "apps/48/Spring");
    public static final AntuIcon APP_SSH_ADD_48 = create("APP_SSH_ADD_48", "apps/48/ssh-add");
    public static final AntuIcon APP_STAGE_48 = create("APP_STAGE_48", "apps/48/stage");
    public static final AntuIcon APP_STARDICT_48 = create("APP_STARDICT_48", "apps/48/stardict");
    public static final AntuIcon APP_STEAM_48 = create("APP_STEAM_48", "apps/48/steam");
    public static final AntuIcon APP_STEAMSKINMANAGER_48 = create("APP_STEAMSKINMANAGER_48", "apps/48/steamskinmanager");
    public static final AntuIcon APP_STELLARIUM_48 = create("APP_STELLARIUM_48", "apps/48/stellarium");
    public static final AntuIcon APP_STEP_48 = create("APP_STEP_48", "apps/48/step");
    public static final AntuIcon APP_STOCK_KEYRING_48 = create("APP_STOCK_KEYRING_48", "apps/48/stock_keyring");
    public static final AntuIcon APP_STOCK_TASKS_48 = create("APP_STOCK_TASKS_48", "apps/48/stock_tasks");
    public static final AntuIcon APP_STORMCLOUD_48 = create("APP_STORMCLOUD_48", "apps/48/stormcloud");
    public static final AntuIcon APP_STRIGI_48 = create("APP_STRIGI_48", "apps/48/strigi");
    public static final AntuIcon APP_SUBDOWNLOADER_48 = create("APP_SUBDOWNLOADER_48", "apps/48/subdownloader");
    public static final AntuIcon APP_SUBLIME_48 = create("APP_SUBLIME_48", "apps/48/sublime");
    public static final AntuIcon APP_SUBLIME_TEXT_48 = create("APP_SUBLIME_TEXT_48", "apps/48/sublime-text");
    public static final AntuIcon APP_SUBTITLECOMPOSER_48 = create("APP_SUBTITLECOMPOSER_48", "apps/48/subtitlecomposer");
    public static final AntuIcon APP_SUBTITLEEDITOR_48 = create("APP_SUBTITLEEDITOR_48", "apps/48/subtitleeditor");
    public static final AntuIcon APP_SVN_48 = create("APP_SVN_48", "apps/48/svn");
    public static final AntuIcon APP_SWAMI_48 = create("APP_SWAMI_48", "apps/48/swami");
    public static final AntuIcon APP_SWEEPER_48 = create("APP_SWEEPER_48", "apps/48/sweeper");
    public static final AntuIcon APP_SYNAPTIC_48 = create("APP_SYNAPTIC_48", "apps/48/synaptic");
    public static final AntuIcon APP_SYNAPTICS_48 = create("APP_SYNAPTICS_48", "apps/48/synaptics");
    public static final AntuIcon APP_SYNAPTIKS_48 = create("APP_SYNAPTIKS_48", "apps/48/synaptiks");
    public static final AntuIcon APP_SYNFIG_ICON_48 = create("APP_SYNFIG_ICON_48", "apps/48/synfig_icon");
    public static final AntuIcon APP_SYNFIGSTUDIO_48 = create("APP_SYNFIGSTUDIO_48", "apps/48/synfigstudio");
    public static final AntuIcon APP_SYSPEEK_48 = create("APP_SYSPEEK_48", "apps/48/syspeek");
    public static final AntuIcon APP_SYSTEM_DIAGNOSIS_48 = create("APP_SYSTEM_DIAGNOSIS_48", "apps/48/system-diagnosis");
    public static final AntuIcon APP_SYSTEM_ERROR_48 = create("APP_SYSTEM_ERROR_48", "apps/48/system-error");
    public static final AntuIcon APP_SYSTEM_FILE_MANAGER_48 = create("APP_SYSTEM_FILE_MANAGER_48", "apps/48/system-file-manager");
    public static final AntuIcon APP_SYSTEM_HELP_16 = create("APP_SYSTEM_HELP_16", "apps/16/system-help");
    public static final AntuIcon APP_SYSTEM_HELP_22 = create("APP_SYSTEM_HELP_22", "apps/22/system-help");
    public static final AntuIcon APP_SYSTEM_HELP_48 = create("APP_SYSTEM_HELP_48", "apps/48/system-help");
    public static final AntuIcon APP_SYSTEM_LOCK_SCREEN_22 = create("APP_SYSTEM_LOCK_SCREEN_22", "apps/22/system-lock-screen");
    public static final AntuIcon APP_SYSTEM_LOCK_SCREEN_32 = create("APP_SYSTEM_LOCK_SCREEN_32", "apps/32/system-lock-screen");
    public static final AntuIcon APP_SYSTEM_LOCK_SCREEN_48 = create("APP_SYSTEM_LOCK_SCREEN_48", "apps/48/system-lock-screen");
    public static final AntuIcon APP_SYSTEM_LOG_OUT_22 = create("APP_SYSTEM_LOG_OUT_22", "apps/22/system-log-out");
    public static final AntuIcon APP_SYSTEM_LOG_OUT_32 = create("APP_SYSTEM_LOG_OUT_32", "apps/32/system-log-out");
    public static final AntuIcon APP_SYSTEM_LOG_OUT_48 = create("APP_SYSTEM_LOG_OUT_48", "apps/48/system-log-out");
    public static final AntuIcon APP_SYSTEM_REBOOT_22 = create("APP_SYSTEM_REBOOT_22", "apps/22/system-reboot");
    public static final AntuIcon APP_SYSTEM_REBOOT_32 = create("APP_SYSTEM_REBOOT_32", "apps/32/system-reboot");
    public static final AntuIcon APP_SYSTEM_REBOOT_48 = create("APP_SYSTEM_REBOOT_48", "apps/48/system-reboot");
    public static final AntuIcon APP_SYSTEM_RUN_32 = create("APP_SYSTEM_RUN_32", "apps/32/system-run");
    public static final AntuIcon APP_SYSTEM_RUN_48 = create("APP_SYSTEM_RUN_48", "apps/48/system-run");
    public static final AntuIcon APP_SYSTEM_SHUTDOWN_22 = create("APP_SYSTEM_SHUTDOWN_22", "apps/22/system-shutdown");
    public static final AntuIcon APP_SYSTEM_SHUTDOWN_32 = create("APP_SYSTEM_SHUTDOWN_32", "apps/32/system-shutdown");
    public static final AntuIcon APP_SYSTEM_SHUTDOWN_48 = create("APP_SYSTEM_SHUTDOWN_48", "apps/48/system-shutdown");
    public static final AntuIcon APP_SYSTEM_SOFTWARE_UPDATE_48 = create("APP_SYSTEM_SOFTWARE_UPDATE_48", "apps/48/system-software-update");
    public static final AntuIcon APP_SYSTEM_SUSPEND_22 = create("APP_SYSTEM_SUSPEND_22", "apps/22/system-suspend");
    public static final AntuIcon APP_SYSTEM_SUSPEND_32 = create("APP_SYSTEM_SUSPEND_32", "apps/32/system-suspend");
    public static final AntuIcon APP_SYSTEM_SUSPEND_48 = create("APP_SYSTEM_SUSPEND_48", "apps/48/system-suspend");
    public static final AntuIcon APP_SYSTEM_SUSPEND_HIBERNATE_22 = create("APP_SYSTEM_SUSPEND_HIBERNATE_22", "apps/22/system-suspend-hibernate");
    public static final AntuIcon APP_SYSTEM_SUSPEND_HIBERNATE_32 = create("APP_SYSTEM_SUSPEND_HIBERNATE_32", "apps/32/system-suspend-hibernate");
    public static final AntuIcon APP_SYSTEM_SUSPEND_HIBERNATE_48 = create("APP_SYSTEM_SUSPEND_HIBERNATE_48", "apps/48/system-suspend-hibernate");
    public static final AntuIcon APP_SYSTEM_SWITCH_USER_22 = create("APP_SYSTEM_SWITCH_USER_22", "apps/22/system-switch-user");
    public static final AntuIcon APP_SYSTEM_SWITCH_USER_32 = create("APP_SYSTEM_SWITCH_USER_32", "apps/32/system-switch-user");
    public static final AntuIcon APP_SYSTEM_SWITCH_USER_48 = create("APP_SYSTEM_SWITCH_USER_48", "apps/48/system-switch-user");
    public static final AntuIcon APP_SYSTEM_USERS_32 = create("APP_SYSTEM_USERS_32", "apps/32/system-users");
    public static final AntuIcon APP_SYSTEM_USERS_48 = create("APP_SYSTEM_USERS_48", "apps/48/system-users");
    public static final AntuIcon APP_SYSTEMSETTINGS_48 = create("APP_SYSTEMSETTINGS_48", "apps/48/systemsettings");
    public static final AntuIcon APP_T_LASKU_48 = create("APP_T_LASKU_48", "apps/48/t-lasku");
    public static final AntuIcon APP_TAGUA_48 = create("APP_TAGUA_48", "apps/48/tagua");
    public static final AntuIcon APP_TEAMVIEWER_48 = create("APP_TEAMVIEWER_48", "apps/48/teamviewer");
    public static final AntuIcon APP_TELEGRAM_48 = create("APP_TELEGRAM_48", "apps/48/telegram");
    public static final AntuIcon APP_TELEPATHY_KDE_48 = create("APP_TELEPATHY_KDE_48", "apps/48/telepathy-kde");
    public static final AntuIcon APP_TELIF_48 = create("APP_TELIF_48", "apps/48/telif");
    public static final AntuIcon APP_TERMINATOR_48 = create("APP_TERMINATOR_48", "apps/48/terminator");
    public static final AntuIcon APP_TEXMAKER_48 = create("APP_TEXMAKER_48", "apps/48/texmaker");
    public static final AntuIcon APP_TEXSTUDIO_48 = create("APP_TEXSTUDIO_48", "apps/48/texstudio");
    public static final AntuIcon APP_THUNDERBIRD_48 = create("APP_THUNDERBIRD_48", "apps/48/thunderbird");
    public static final AntuIcon APP_TIMESHIFT_48 = create("APP_TIMESHIFT_48", "apps/48/timeshift");
    public static final AntuIcon APP_TIMEVAULT_48 = create("APP_TIMEVAULT_48", "apps/48/timevault");
    public static final AntuIcon APP_TOMAHAWK_48 = create("APP_TOMAHAWK_48", "apps/48/tomahawk");
    public static final AntuIcon APP_TOMATE_48 = create("APP_TOMATE_48", "apps/48/tomate");
    public static final AntuIcon APP_TOR_BROWSER_48 = create("APP_TOR_BROWSER_48", "apps/48/tor-browser");
    public static final AntuIcon APP_TOTEM_48 = create("APP_TOTEM_48", "apps/48/totem");
    public static final AntuIcon APP_TRANSCODER_48 = create("APP_TRANSCODER_48", "apps/48/transcoder");
    public static final AntuIcon APP_TRANSMAGEDDON_48 = create("APP_TRANSMAGEDDON_48", "apps/48/transmageddon");
    public static final AntuIcon APP_TRANSMISSION_48 = create("APP_TRANSMISSION_48", "apps/48/transmission");
    public static final AntuIcon APP_TRANSMISSION_TRAY_48 = create("APP_TRANSMISSION_TRAY_48", "apps/48/transmission-tray");
    public static final AntuIcon APP_TRELLO_48 = create("APP_TRELLO_48", "apps/48/trello");
    public static final AntuIcon APP_TROJITA_48 = create("APP_TROJITA_48", "apps/48/trojita");
    public static final AntuIcon APP_TRUECRYPT_48 = create("APP_TRUECRYPT_48", "apps/48/truecrypt");
    public static final AntuIcon APP_TUMBLR_48 = create("APP_TUMBLR_48", "apps/48/tumblr");
    public static final AntuIcon APP_TUPI_48 = create("APP_TUPI_48", "apps/48/tupi");
    public static final AntuIcon APP_TURPIAL_48 = create("APP_TURPIAL_48", "apps/48/turpial");
    public static final AntuIcon APP_TUXGUITAR_48 = create("APP_TUXGUITAR_48", "apps/48/tuxguitar");
    public static final AntuIcon APP_TVMAXE_48 = create("APP_TVMAXE_48", "apps/48/tvmaxe");
    public static final AntuIcon APP_TWITTER_48 = create("APP_TWITTER_48", "apps/48/twitter");
    public static final AntuIcon APP_TYPECATCHER_48 = create("APP_TYPECATCHER_48", "apps/48/typecatcher");
    public static final AntuIcon APP_UBIQUITY_48 = create("APP_UBIQUITY_48", "apps/48/ubiquity");
    public static final AntuIcon APP_UNETBOOTIN_48 = create("APP_UNETBOOTIN_48", "apps/48/unetbootin");
    public static final AntuIcon APP_UNKNOWNAPP_48 = create("APP_UNKNOWNAPP_48", "apps/48/unknownapp");
    public static final AntuIcon APP_USB_CREATOR_KDE_48 = create("APP_USB_CREATOR_KDE_48", "apps/48/usb-creator-kde");
    public static final AntuIcon APP_UTILITIES_DESKTOP_EXTRA_48 = create("APP_UTILITIES_DESKTOP_EXTRA_48", "apps/48/utilities-desktop-extra");
    public static final AntuIcon APP_UTILITIES_ENERGY_MONITOR_48 = create("APP_UTILITIES_ENERGY_MONITOR_48", "apps/48/utilities-energy-monitor");
    public static final AntuIcon APP_UTILITIES_FILE_ARCHIVER_48 = create("APP_UTILITIES_FILE_ARCHIVER_48", "apps/48/utilities-file-archiver");
    public static final AntuIcon APP_UTILITIES_LOG_VIEWER_48 = create("APP_UTILITIES_LOG_VIEWER_48", "apps/48/utilities-log-viewer");
    public static final AntuIcon APP_UTILITIES_SYSTEM_MONITOR_48 = create("APP_UTILITIES_SYSTEM_MONITOR_48", "apps/48/utilities-system-monitor");
    public static final AntuIcon APP_UTILITIES_TERMINAL_48 = create("APP_UTILITIES_TERMINAL_48", "apps/48/utilities-terminal");
    public static final AntuIcon APP_UTILITIES_TERMINAL_SU_48 = create("APP_UTILITIES_TERMINAL_SU_48", "apps/48/utilities-terminal_su");
    public static final AntuIcon APP_VBAM_48 = create("APP_VBAM_48", "apps/48/vbam");
    public static final AntuIcon APP_VBOX_48 = create("APP_VBOX_48", "apps/48/vbox");
    public static final AntuIcon APP_VEROMIX_48 = create("APP_VEROMIX_48", "apps/48/Veromix");
    public static final AntuIcon APP_VIBER_48 = create("APP_VIBER_48", "apps/48/viber");
    public static final AntuIcon APP_VIDALIA_48 = create("APP_VIDALIA_48", "apps/48/vidalia");
    public static final AntuIcon APP_VIDEOPLAYER_48 = create("APP_VIDEOPLAYER_48", "apps/48/videoplayer");
    public static final AntuIcon APP_VIEW_MEDIA_PLAYLIST_32 = create("APP_VIEW_MEDIA_PLAYLIST_32", "apps/32/view-media-playlist");
    public static final AntuIcon APP_VIEW_MEDIA_PLAYLIST_48 = create("APP_VIEW_MEDIA_PLAYLIST_48", "apps/48/view-media-playlist");
    public static final AntuIcon APP_VIM_48 = create("APP_VIM_48", "apps/48/vim");
    public static final AntuIcon APP_VIM_QT_48 = create("APP_VIM_QT_48", "apps/48/vim-qt");
    public static final AntuIcon APP_VIMEO_48 = create("APP_VIMEO_48", "apps/48/vimeo");
    public static final AntuIcon APP_VIMEO_ALT_48 = create("APP_VIMEO_ALT_48", "apps/48/vimeo_alt");
    public static final AntuIcon APP_VINEYARD_48 = create("APP_VINEYARD_48", "apps/48/vineyard");
    public static final AntuIcon APP_VINEYARD_PREFERENCES_48 = create("APP_VINEYARD_PREFERENCES_48", "apps/48/vineyard-preferences");
    public static final AntuIcon APP_VIRTUALBOX_48 = create("APP_VIRTUALBOX_48", "apps/48/virtualbox");
    public static final AntuIcon APP_VISUALVM_48 = create("APP_VISUALVM_48", "apps/48/visualvm");
    public static final AntuIcon APP_VIVALDI_48 = create("APP_VIVALDI_48", "apps/48/vivaldi");
    public static final AntuIcon APP_VIVALDI_SNAPSHOT_48 = create("APP_VIVALDI_SNAPSHOT_48", "apps/48/vivaldi-snapshot");
    public static final AntuIcon APP_VLC_48 = create("APP_VLC_48", "apps/48/vlc");
    public static final AntuIcon APP_VLC_XMAS_48 = create("APP_VLC_XMAS_48", "apps/48/vlc-xmas");
    public static final AntuIcon APP_VMWARE_48 = create("APP_VMWARE_48", "apps/48/vmware");
    public static final AntuIcon APP_VMWARE_NETCFG_48 = create("APP_VMWARE_NETCFG_48", "apps/48/vmware-netcfg");
    public static final AntuIcon APP_VMWARE_PLAYER_48 = create("APP_VMWARE_PLAYER_48", "apps/48/vmware-player");
    public static final AntuIcon APP_VOCAL_48 = create("APP_VOCAL_48", "apps/48/vocal");
    public static final AntuIcon APP_VOKOSCREEN_48 = create("APP_VOKOSCREEN_48", "apps/48/vokoscreen");
    public static final AntuIcon APP_VSTUDIO_48 = create("APP_VSTUDIO_48", "apps/48/vstudio");
    public static final AntuIcon APP_WAITING_48 = create("APP_WAITING_48", "apps/48/waiting");
    public static final AntuIcon APP_WALLPAPER_48 = create("APP_WALLPAPER_48", "apps/48/wallpaper");
    public static final AntuIcon APP_WAVE_ACCOUNTING_48 = create("APP_WAVE_ACCOUNTING_48", "apps/48/wave-accounting");
    public static final AntuIcon APP_WAYLAND_48 = create("APP_WAYLAND_48", "apps/48/wayland");
    public static final AntuIcon APP_WEB_48 = create("APP_WEB_48", "apps/48/web");
    public static final AntuIcon APP_WEB_BROWSER_48 = create("APP_WEB_BROWSER_48", "apps/48/web-browser");
    public static final AntuIcon APP_WEB_GOOGLE_APPS_SCRIPT_48 = create("APP_WEB_GOOGLE_APPS_SCRIPT_48", "apps/48/web-google-apps-script");
    public static final AntuIcon APP_WEB_GOOGLE_DRAWINGS_48 = create("APP_WEB_GOOGLE_DRAWINGS_48", "apps/48/web-google-drawings");
    public static final AntuIcon APP_WEB_GOOGLE_FORMS_48 = create("APP_WEB_GOOGLE_FORMS_48", "apps/48/web-google-forms");
    public static final AntuIcon APP_WEB_GOOGLE_SLIDES_48 = create("APP_WEB_GOOGLE_SLIDES_48", "apps/48/web-google-slides");
    public static final AntuIcon APP_WEB_INBOX_48 = create("APP_WEB_INBOX_48", "apps/48/web-inbox");
    public static final AntuIcon APP_WEB_JOLICLOUD_48 = create("APP_WEB_JOLICLOUD_48", "apps/48/web-jolicloud");
    public static final AntuIcon APP_WEB_KINDLE_48 = create("APP_WEB_KINDLE_48", "apps/48/web-kindle");
    public static final AntuIcon APP_WEBBROWSER_APP_48 = create("APP_WEBBROWSER_APP_48", "apps/48/webbrowser-app");
    public static final AntuIcon APP_WEBKIT_48 = create("APP_WEBKIT_48", "apps/48/webkit");
    public static final AntuIcon APP_WESNOTH_1_10_EDITOR_ICON_48 = create("APP_WESNOTH_1_10_EDITOR_ICON_48", "apps/48/wesnoth-1.10_editor-icon");
    public static final AntuIcon APP_WESNOTH_1_10_ICON_48 = create("APP_WESNOTH_1_10_ICON_48", "apps/48/wesnoth-1.10-icon");
    public static final AntuIcon APP_WESNOTH_48 = create("APP_WESNOTH_48", "apps/48/wesnoth");
    public static final AntuIcon APP_WIKIMEDIA_COMMONS_48 = create("APP_WIKIMEDIA_COMMONS_48", "apps/48/wikimedia-commons");
    public static final AntuIcon APP_WIKIPEDIA_48 = create("APP_WIKIPEDIA_48", "apps/48/wikipedia");
    public static final AntuIcon APP_WINDOW_DUPLICATE_32 = create("APP_WINDOW_DUPLICATE_32", "apps/32/window-duplicate");
    public static final AntuIcon APP_WINDOW_DUPLICATE_48 = create("APP_WINDOW_DUPLICATE_48", "apps/48/window-duplicate");
    public static final AntuIcon APP_WINE_48 = create("APP_WINE_48", "apps/48/wine");
    public static final AntuIcon APP_WINE_HELP_48 = create("APP_WINE_HELP_48", "apps/48/wine-help");
    public static final AntuIcon APP_WINE_UNINSTALLER_48 = create("APP_WINE_UNINSTALLER_48", "apps/48/wine-uninstaller");
    public static final AntuIcon APP_WINE_WINECFG_48 = create("APP_WINE_WINECFG_48", "apps/48/wine-winecfg");
    public static final AntuIcon APP_WINECFG_48 = create("APP_WINECFG_48", "apps/48/winecfg");
    public static final AntuIcon APP_WINETRICKS_48 = create("APP_WINETRICKS_48", "apps/48/winetricks");
    public static final AntuIcon APP_WINFF_48 = create("APP_WINFF_48", "apps/48/winff");
    public static final AntuIcon APP_WINUSB_48 = create("APP_WINUSB_48", "apps/48/winusb");
    public static final AntuIcon APP_WINUSBGUI_ICON_48 = create("APP_WINUSBGUI_ICON_48", "apps/48/winusbgui-icon");
    public static final AntuIcon APP_WORDPRESS_48 = create("APP_WORDPRESS_48", "apps/48/wordpress");
    public static final AntuIcon APP_WORDPRESS_WORDPRESS_COM_48 = create("APP_WORDPRESS_WORDPRESS_COM_48", "apps/48/wordpress-wordpress.com");
    public static final AntuIcon APP_WORDS_48 = create("APP_WORDS_48", "apps/48/words");
    public static final AntuIcon APP_WORDVIEW_48 = create("APP_WORDVIEW_48", "apps/48/wordview");
    public static final AntuIcon APP_WPCOM_48 = create("APP_WPCOM_48", "apps/48/wpcom");
    public static final AntuIcon APP_WPS_OFFICE_ETMAIN_48 = create("APP_WPS_OFFICE_ETMAIN_48", "apps/48/wps-office-etmain");
    public static final AntuIcon APP_WPS_OFFICE_WPPMAIN_48 = create("APP_WPS_OFFICE_WPPMAIN_48", "apps/48/wps-office-wppmain");
    public static final AntuIcon APP_WPS_OFFICE_WPSMAIN_48 = create("APP_WPS_OFFICE_WPSMAIN_48", "apps/48/wps-office-wpsmain");
    public static final AntuIcon APP_X_PLANE_48 = create("APP_X_PLANE_48", "apps/48/x-plane");
    public static final AntuIcon APP_X_TERMINAL_EMULATOR_48 = create("APP_X_TERMINAL_EMULATOR_48", "apps/48/x-terminal-emulator");
    public static final AntuIcon APP_XCHAT_48 = create("APP_XCHAT_48", "apps/48/xchat");
    public static final AntuIcon APP_XDG_BROWSER_LAUNCHER_48 = create("APP_XDG_BROWSER_LAUNCHER_48", "apps/48/xdg-browser-launcher");
    public static final AntuIcon APP_XDIAGNOSE_48 = create("APP_XDIAGNOSE_48", "apps/48/xdiagnose");
    public static final AntuIcon APP_XINE_48 = create("APP_XINE_48", "apps/48/xine");
    public static final AntuIcon APP_XMIND_48 = create("APP_XMIND_48", "apps/48/xmind");
    public static final AntuIcon APP_XMRADIO_48 = create("APP_XMRADIO_48", "apps/48/xmradio");
    public static final AntuIcon APP_XNCONVERT_48 = create("APP_XNCONVERT_48", "apps/48/xnconvert");
    public static final AntuIcon APP_XNVIEW_48 = create("APP_XNVIEW_48", "apps/48/xnview");
    public static final AntuIcon APP_XONOTIC_48 = create("APP_XONOTIC_48", "apps/48/xonotic");
    public static final AntuIcon APP_XORG_48 = create("APP_XORG_48", "apps/48/xorg");
    public static final AntuIcon APP_XOURNAL_48 = create("APP_XOURNAL_48", "apps/48/xournal");
    public static final AntuIcon APP_XPLANE_48 = create("APP_XPLANE_48", "apps/48/xplane");
    public static final AntuIcon APP_XSANE_48 = create("APP_XSANE_48", "apps/48/xsane");
    public static final AntuIcon APP_XSENSORS_48 = create("APP_XSENSORS_48", "apps/48/xsensors");
    public static final AntuIcon APP_XTERM_48 = create("APP_XTERM_48", "apps/48/xterm");
    public static final AntuIcon APP_XTERM_COLOR_48 = create("APP_XTERM_COLOR_48", "apps/48/xterm-color");
    public static final AntuIcon APP_Y_PPA_MANAGER_48 = create("APP_Y_PPA_MANAGER_48", "apps/48/y-ppa-manager");
    public static final AntuIcon APP_YAD_48 = create("APP_YAD_48", "apps/48/yad");
    public static final AntuIcon APP_YAHOO_48 = create("APP_YAHOO_48", "apps/48/yahoo");
    public static final AntuIcon APP_YAHOO_PROTOCOL_48 = create("APP_YAHOO_PROTOCOL_48", "apps/48/yahoo_protocol");
    public static final AntuIcon APP_YAKUAKE_48 = create("APP_YAKUAKE_48", "apps/48/yakuake");
    public static final AntuIcon APP_YANDEX_BROWSER_48 = create("APP_YANDEX_BROWSER_48", "apps/48/yandex-browser");
    public static final AntuIcon APP_YANDEX_BROWSER_BETA_48 = create("APP_YANDEX_BROWSER_BETA_48", "apps/48/yandex-browser-beta");
    public static final AntuIcon APP_YAROCK_48 = create("APP_YAROCK_48", "apps/48/yarock");
    public static final AntuIcon APP_YAST_48 = create("APP_YAST_48", "apps/48/yast");
    public static final AntuIcon APP_YAST_ADDON_48 = create("APP_YAST_ADDON_48", "apps/48/yast-addon");
    public static final AntuIcon APP_YAST_BOOTLOADER_48 = create("APP_YAST_BOOTLOADER_48", "apps/48/yast-bootloader");
    public static final AntuIcon APP_YAST_CHECKMEDIA_48 = create("APP_YAST_CHECKMEDIA_48", "apps/48/yast-checkmedia");
    public static final AntuIcon APP_YAST_DISK_48 = create("APP_YAST_DISK_48", "apps/48/yast-disk");
    public static final AntuIcon APP_YAST_FIREWALL_48 = create("APP_YAST_FIREWALL_48", "apps/48/yast-firewall");
    public static final AntuIcon APP_YAST_HARDWARE_48 = create("APP_YAST_HARDWARE_48", "apps/48/yast-hardware");
    public static final AntuIcon APP_YAST_HIGH_AVAILABILITY_48 = create("APP_YAST_HIGH_AVAILABILITY_48", "apps/48/yast-high_availability");
    public static final AntuIcon APP_YAST_HOST_48 = create("APP_YAST_HOST_48", "apps/48/yast-host");
    public static final AntuIcon APP_YAST_HWINFO_48 = create("APP_YAST_HWINFO_48", "apps/48/yast-hwinfo");
    public static final AntuIcon APP_YAST_INETD_48 = create("APP_YAST_INETD_48", "apps/48/yast-inetd");
    public static final AntuIcon APP_YAST_ISCSI_CLIENT_48 = create("APP_YAST_ISCSI_CLIENT_48", "apps/48/yast-iscsi-client");
    public static final AntuIcon APP_YAST_KEYBOARD_48 = create("APP_YAST_KEYBOARD_48", "apps/48/yast-keyboard");
    public static final AntuIcon APP_YAST_LAN_48 = create("APP_YAST_LAN_48", "apps/48/yast-lan");
    public static final AntuIcon APP_YAST_LANGUAGE_48 = create("APP_YAST_LANGUAGE_48", "apps/48/yast-language");
    public static final AntuIcon APP_YAST_LIVE_INSTALL_48 = create("APP_YAST_LIVE_INSTALL_48", "apps/48/yast-live-install");
    public static final AntuIcon APP_YAST_MAIL_48 = create("APP_YAST_MAIL_48", "apps/48/yast-mail");
    public static final AntuIcon APP_YAST_MESSAGES_48 = create("APP_YAST_MESSAGES_48", "apps/48/yast-messages");
    public static final AntuIcon APP_YAST_MISC_48 = create("APP_YAST_MISC_48", "apps/48/yast-misc");
    public static final AntuIcon APP_YAST_NETWORK_SERVICES_48 = create("APP_YAST_NETWORK_SERVICES_48", "apps/48/yast-network-services");
    public static final AntuIcon APP_YAST_NFS_48 = create("APP_YAST_NFS_48", "apps/48/yast-nfs");
    public static final AntuIcon APP_YAST_NIS_48 = create("APP_YAST_NIS_48", "apps/48/yast-nis");
    public static final AntuIcon APP_YAST_NTP_CLIENT_48 = create("APP_YAST_NTP_CLIENT_48", "apps/48/yast-ntp-client");
    public static final AntuIcon APP_YAST_ONLINE_UPDATE_48 = create("APP_YAST_ONLINE_UPDATE_48", "apps/48/yast-online-update");
    public static final AntuIcon APP_YAST_POWERTWEAK_48 = create("APP_YAST_POWERTWEAK_48", "apps/48/yast-powertweak");
    public static final AntuIcon APP_YAST_PRINTER_48 = create("APP_YAST_PRINTER_48", "apps/48/yast-printer");
    public static final AntuIcon APP_YAST_PROXY_48 = create("APP_YAST_PROXY_48", "apps/48/yast-proxy");
    public static final AntuIcon APP_YAST_RELEASE_NOTES_48 = create("APP_YAST_RELEASE_NOTES_48", "apps/48/yast-release-notes");
    public static final AntuIcon APP_YAST_REMOTE_48 = create("APP_YAST_REMOTE_48", "apps/48/yast-remote");
    public static final AntuIcon APP_YAST_RUNLEVEL_48 = create("APP_YAST_RUNLEVEL_48", "apps/48/yast-runlevel");
    public static final AntuIcon APP_YAST_SAMBA_CLIENT_48 = create("APP_YAST_SAMBA_CLIENT_48", "apps/48/yast-samba-client");
    public static final AntuIcon APP_YAST_SAMBA_SERVER_48 = create("APP_YAST_SAMBA_SERVER_48", "apps/48/yast-samba-server");
    public static final AntuIcon APP_YAST_SCANNER_48 = create("APP_YAST_SCANNER_48", "apps/48/yast-scanner");
    public static final AntuIcon APP_YAST_SECURITY_48 = create("APP_YAST_SECURITY_48", "apps/48/yast-security");
    public static final AntuIcon APP_YAST_SNAPPER_48 = create("APP_YAST_SNAPPER_48", "apps/48/yast-snapper");
    public static final AntuIcon APP_YAST_SOFTWARE_48 = create("APP_YAST_SOFTWARE_48", "apps/48/yast-software");
    public static final AntuIcon APP_YAST_SOUND_48 = create("APP_YAST_SOUND_48", "apps/48/yast-sound");
    public static final AntuIcon APP_YAST_SUDO_48 = create("APP_YAST_SUDO_48", "apps/48/yast-sudo");
    public static final AntuIcon APP_YAST_SUPPORT_48 = create("APP_YAST_SUPPORT_48", "apps/48/yast-support");
    public static final AntuIcon APP_YAST_SW_SINGLE_48 = create("APP_YAST_SW_SINGLE_48", "apps/48/yast-sw_single");
    public static final AntuIcon APP_YAST_SW_SOURCE_48 = create("APP_YAST_SW_SOURCE_48", "apps/48/yast-sw_source");
    public static final AntuIcon APP_YAST_SYSCONFIG_48 = create("APP_YAST_SYSCONFIG_48", "apps/48/yast-sysconfig");
    public static final AntuIcon APP_YAST_SYSTEM_48 = create("APP_YAST_SYSTEM_48", "apps/48/yast-system");
    public static final AntuIcon APP_YAST_TIMEZONE_48 = create("APP_YAST_TIMEZONE_48", "apps/48/yast-timezone");
    public static final AntuIcon APP_YAST_USERS_48 = create("APP_YAST_USERS_48", "apps/48/yast-users");
    public static final AntuIcon APP_YAST_VENDOR_48 = create("APP_YAST_VENDOR_48", "apps/48/yast-vendor");
    public static final AntuIcon APP_YAST_VM_INSTALL_48 = create("APP_YAST_VM_INSTALL_48", "apps/48/yast-vm-install");
    public static final AntuIcon APP_YED_48 = create("APP_YED_48", "apps/48/yed");
    public static final AntuIcon APP_YOUTUBE_48 = create("APP_YOUTUBE_48", "apps/48/youtube");
    public static final AntuIcon APP_YOUTUBE_DL_48 = create("APP_YOUTUBE_DL_48", "apps/48/youtube-dl");
    public static final AntuIcon APP_YOUTUBE_DL_GUI_48 = create("APP_YOUTUBE_DL_GUI_48", "apps/48/youtube-dl-gui");
    public static final AntuIcon APP_YOUTUBE_DLG_48 = create("APP_YOUTUBE_DLG_48", "apps/48/youtube-dlg");
    public static final AntuIcon APP_YOUTUBE_TO_MP3_48 = create("APP_YOUTUBE_TO_MP3_48", "apps/48/youtube-to-mp3");
    public static final AntuIcon APP_YUMEX_48 = create("APP_YUMEX_48", "apps/48/yumex");
    public static final AntuIcon APP_ZANSHIN_48 = create("APP_ZANSHIN_48", "apps/48/zanshin");
    public static final AntuIcon APP_ZAZ_48 = create("APP_ZAZ_48", "apps/48/zaz");
    public static final AntuIcon APP_ZEKR_48 = create("APP_ZEKR_48", "apps/48/zekr");
    public static final AntuIcon APP_ZEROINSTALL_48 = create("APP_ZEROINSTALL_48", "apps/48/zeroinstall");
    public static final AntuIcon APP_ZSNES_48 = create("APP_ZSNES_48", "apps/48/zsnes");

    // CATEGORIES
    public static final AntuIcon CATEGORY_APPLICATIONS_DEVELOPMENT_22 = create("CATEGORY_APPLICATIONS_DEVELOPMENT_22", "categories/22/applications-development");
    public static final AntuIcon CATEGORY_APPLICATIONS_ACCESSORIES_32 = create("CATEGORY_APPLICATIONS_ACCESSORIES_32", "categories/32/applications-accessories");
    public static final AntuIcon CATEGORY_APPLICATIONS_DEVELOPMENT_TRANSLATION_32 = create("CATEGORY_APPLICATIONS_DEVELOPMENT_TRANSLATION_32", "categories/32/applications-development-translation");
    public static final AntuIcon CATEGORY_APPLICATIONS_DEVELOPMENT_WEB_32 = create("CATEGORY_APPLICATIONS_DEVELOPMENT_WEB_32", "categories/32/applications-development-web");
    public static final AntuIcon CATEGORY_APPLICATIONS_DEVELOPMENT_32 = create("CATEGORY_APPLICATIONS_DEVELOPMENT_32", "categories/32/applications-development");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_LANGUAGE_32 = create("CATEGORY_APPLICATIONS_EDUCATION_LANGUAGE_32", "categories/32/applications-education-language");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_MATHEMATICS_32 = create("CATEGORY_APPLICATIONS_EDUCATION_MATHEMATICS_32", "categories/32/applications-education-mathematics");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_MISCELLANEOUS_32 = create("CATEGORY_APPLICATIONS_EDUCATION_MISCELLANEOUS_32", "categories/32/applications-education-miscellaneous");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_PRESCHOOL_32 = create("CATEGORY_APPLICATIONS_EDUCATION_PRESCHOOL_32", "categories/32/applications-education-preschool");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_SCHOOL_32 = create("CATEGORY_APPLICATIONS_EDUCATION_SCHOOL_32", "categories/32/applications-education-school");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_SCIENCE_32 = create("CATEGORY_APPLICATIONS_EDUCATION_SCIENCE_32", "categories/32/applications-education-science");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_UNIVERSITY_32 = create("CATEGORY_APPLICATIONS_EDUCATION_UNIVERSITY_32", "categories/32/applications-education-university");
    public static final AntuIcon CATEGORY_APPLICATIONS_EDUCATION_32 = create("CATEGORY_APPLICATIONS_EDUCATION_32", "categories/32/applications-education");
    public static final AntuIcon CATEGORY_APPLICATIONS_ENGINEERING_32 = create("CATEGORY_APPLICATIONS_ENGINEERING_32", "categories/32/applications-engineering");
    public static final AntuIcon CATEGORY_APPLICATIONS_GAMES_32 = create("CATEGORY_APPLICATIONS_GAMES_32", "categories/32/applications-games");
    public static final AntuIcon CATEGORY_APPLICATIONS_GRAPHICS_32 = create("CATEGORY_APPLICATIONS_GRAPHICS_32", "categories/32/applications-graphics");
    public static final AntuIcon CATEGORY_APPLICATIONS_INTERNET_32 = create("CATEGORY_APPLICATIONS_INTERNET_32", "categories/32/applications-internet");
    public static final AntuIcon CATEGORY_APPLICATIONS_MULTIMEDIA_32 = create("CATEGORY_APPLICATIONS_MULTIMEDIA_32", "categories/32/applications-multimedia");
    public static final AntuIcon CATEGORY_APPLICATIONS_OFFICE_32 = create("CATEGORY_APPLICATIONS_OFFICE_32", "categories/32/applications-office");
    public static final AntuIcon CATEGORY_APPLICATIONS_OTHER_32 = create("CATEGORY_APPLICATIONS_OTHER_32", "categories/32/applications-other");
    public static final AntuIcon CATEGORY_APPLICATIONS_SCIENCE_32 = create("CATEGORY_APPLICATIONS_SCIENCE_32", "categories/32/applications-science");
    public static final AntuIcon CATEGORY_APPLICATIONS_SYSTEM_32 = create("CATEGORY_APPLICATIONS_SYSTEM_32", "categories/32/applications-system");
    public static final AntuIcon CATEGORY_APPLICATIONS_TOYS_32 = create("CATEGORY_APPLICATIONS_TOYS_32", "categories/32/applications-toys");
    public static final AntuIcon CATEGORY_APPLICATIONS_UTILITIES_32 = create("CATEGORY_APPLICATIONS_UTILITIES_32", "categories/32/applications-utilities");
    public static final AntuIcon CATEGORY_PREFERENCES_DESKTOP_PERIPHERALS_32 = create("CATEGORY_PREFERENCES_DESKTOP_PERIPHERALS_32", "categories/32/preferences-desktop-peripherals");
    public static final AntuIcon CATEGORY_PREFERENCES_DESKTOP_PERSONAL_32 = create("CATEGORY_PREFERENCES_DESKTOP_PERSONAL_32", "categories/32/preferences-desktop-personal");
    public static final AntuIcon CATEGORY_PREFERENCES_DESKTOP_32 = create("CATEGORY_PREFERENCES_DESKTOP_32", "categories/32/preferences-desktop");
    public static final AntuIcon CATEGORY_PREFERENCES_OTHER_32 = create("CATEGORY_PREFERENCES_OTHER_32", "categories/32/preferences-other");
    public static final AntuIcon CATEGORY_PREFERENCES_SYSTEM_NETWORK_32 = create("CATEGORY_PREFERENCES_SYSTEM_NETWORK_32", "categories/32/preferences-system-network");
    public static final AntuIcon CATEGORY_PREFERENCES_SYSTEM_32 = create("CATEGORY_PREFERENCES_SYSTEM_32", "categories/32/preferences-system");
    public static final AntuIcon CATEGORY_SYSTEM_HELP_32 = create("CATEGORY_SYSTEM_HELP_32", "categories/32/system-help");
    public static final AntuIcon CATEGORY_APPLICATIONS_OTHER_48 = create("CATEGORY_APPLICATIONS_OTHER_48", "categories/48/applications-other");


    // DEVICES
    public static final AntuIcon DEVICE_AUDIO_CARD_64 = create("DEVICE_AUDIO_CARD_64", "devices/64/audio-card");
    public static final AntuIcon DEVICE_AUDIO_HEADPHONES_64 = create("DEVICE_AUDIO_HEADPHONES_64", "devices/64/audio-headphones");
    public static final AntuIcon DEVICE_AUDIO_INPUT_MICROPHONE_64 = create("DEVICE_AUDIO_INPUT_MICROPHONE_64", "devices/64/audio-input-microphone");
    public static final AntuIcon DEVICE_BATTERY_64 = create("DEVICE_BATTERY_64", "devices/64/battery");
    public static final AntuIcon DEVICE_CAMERA_PHOTO_22 = create("DEVICE_CAMERA_PHOTO_22", "devices/22/camera-photo");
    public static final AntuIcon DEVICE_CAMERA_PHOTO_64 = create("DEVICE_CAMERA_PHOTO_64", "devices/64/camera-photo");
    public static final AntuIcon DEVICE_CAMERA_WEB_64 = create("DEVICE_CAMERA_WEB_64", "devices/64/camera-web");
    public static final AntuIcon DEVICE_COMPUTER_22 = create("DEVICE_COMPUTER_22", "devices/22/computer");
    public static final AntuIcon DEVICE_COMPUTER_48 = create("DEVICE_COMPUTER_48", "devices/48/computer");
    public static final AntuIcon DEVICE_COMPUTER_64 = create("DEVICE_COMPUTER_64", "devices/64/computer");
    public static final AntuIcon DEVICE_COMPUTER_LAPTOP_22 = create("DEVICE_COMPUTER_LAPTOP_22", "devices/22/computer-laptop");
    public static final AntuIcon DEVICE_COMPUTER_LAPTOP_64 = create("DEVICE_COMPUTER_LAPTOP_64", "devices/64/computer-laptop");
    public static final AntuIcon DEVICE_CPU_64 = create("DEVICE_CPU_64", "devices/64/cpu");
    public static final AntuIcon DEVICE_DRIVE_HARDDISK_22 = create("DEVICE_DRIVE_HARDDISK_22", "devices/22/drive-harddisk");
    public static final AntuIcon DEVICE_DRIVE_HARDDISK_64 = create("DEVICE_DRIVE_HARDDISK_64", "devices/64/drive-harddisk");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_22 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_22", "devices/22/drive-removable-media");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_64 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_64", "devices/64/drive-removable-media");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_PENDRIVE_64 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_PENDRIVE_64", "devices/64/drive-removable-media-pendrive");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_USB_22 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_USB_22", "devices/22/drive-removable-media-usb");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_USB_64 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_USB_64", "devices/64/drive-removable-media-usb");
    public static final AntuIcon DEVICE_DRIVE_REMOVABLE_MEDIA_USB_PENDRIVE_22 = create("DEVICE_DRIVE_REMOVABLE_MEDIA_USB_PENDRIVE_22", "devices/22/drive-removable-media-usb-pendrive");
    public static final AntuIcon DEVICE_INPUT_GAMING_64 = create("DEVICE_INPUT_GAMING_64", "devices/64/input-gaming");
    public static final AntuIcon DEVICE_INPUT_KEYBOARD_64 = create("DEVICE_INPUT_KEYBOARD_64", "devices/64/input-keyboard");
    public static final AntuIcon DEVICE_INPUT_MOUSE_64 = create("DEVICE_INPUT_MOUSE_64", "devices/64/input-mouse");
    public static final AntuIcon DEVICE_INPUT_TOUCHPAD_64 = create("DEVICE_INPUT_TOUCHPAD_64", "devices/64/input-touchpad");
    public static final AntuIcon DEVICE_KDECONNECT_22 = create("DEVICE_KDECONNECT_22", "devices/22/kdeconnect");
    public static final AntuIcon DEVICE_MEDIA_FLASH_64 = create("DEVICE_MEDIA_FLASH_64", "devices/64/media-flash");
    public static final AntuIcon DEVICE_MEDIA_FLASH_MEMORY_STICK_64 = create("DEVICE_MEDIA_FLASH_MEMORY_STICK_64", "devices/64/media-flash-memory-stick");
    public static final AntuIcon DEVICE_MEDIA_FLASH_SD_MMC_64 = create("DEVICE_MEDIA_FLASH_SD_MMC_64", "devices/64/media-flash-sd-mmc");
    public static final AntuIcon DEVICE_MEDIA_FLASH_SMART_MEDIA_64 = create("DEVICE_MEDIA_FLASH_SMART_MEDIA_64", "devices/64/media-flash-smart-media");
    public static final AntuIcon DEVICE_MEDIA_FLOPPY_22 = create("DEVICE_MEDIA_FLOPPY_22", "devices/22/media-floppy");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_22 = create("DEVICE_MEDIA_OPTICAL_22", "devices/22/media-optical");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_64 = create("DEVICE_MEDIA_OPTICAL_64", "devices/64/media-optical");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_AUDIO_22 = create("DEVICE_MEDIA_OPTICAL_AUDIO_22", "devices/22/media-optical-audio");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_AUDIO_64 = create("DEVICE_MEDIA_OPTICAL_AUDIO_64", "devices/64/media-optical-audio");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_BLU_RAY_64 = create("DEVICE_MEDIA_OPTICAL_BLU_RAY_64", "devices/64/media-optical-blu-ray");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_DATA_22 = create("DEVICE_MEDIA_OPTICAL_DATA_22", "devices/22/media-optical-data");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_DATA_64 = create("DEVICE_MEDIA_OPTICAL_DATA_64", "devices/64/media-optical-data");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_DVD_64 = create("DEVICE_MEDIA_OPTICAL_DVD_64", "devices/64/media-optical-dvd");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_MIXED_CD_22 = create("DEVICE_MEDIA_OPTICAL_MIXED_CD_22", "devices/22/media-optical-mixed-cd");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_RECORDABLE_22 = create("DEVICE_MEDIA_OPTICAL_RECORDABLE_22", "devices/22/media-optical-recordable");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_RECORDABLE_64 = create("DEVICE_MEDIA_OPTICAL_RECORDABLE_64", "devices/64/media-optical-recordable");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_VIDEO_22 = create("DEVICE_MEDIA_OPTICAL_VIDEO_22", "devices/22/media-optical-video");
    public static final AntuIcon DEVICE_MEDIA_OPTICAL_VIDEO_64 = create("DEVICE_MEDIA_OPTICAL_VIDEO_64", "devices/64/media-optical-video");
    public static final AntuIcon DEVICE_MULTIMEDIA_PLAYER_22 = create("DEVICE_MULTIMEDIA_PLAYER_22", "devices/22/multimedia-player");
    public static final AntuIcon DEVICE_MULTIMEDIA_PLAYER_64 = create("DEVICE_MULTIMEDIA_PLAYER_64", "devices/64/multimedia-player");
    public static final AntuIcon DEVICE_MULTIMEDIA_PLAYER_IPOD_MINI_BLUE_22 = create("DEVICE_MULTIMEDIA_PLAYER_IPOD_MINI_BLUE_22", "devices/22/multimedia-player-ipod-mini-blue");
    public static final AntuIcon DEVICE_NETWORK_BLUETOOTH_22 = create("DEVICE_NETWORK_BLUETOOTH_22", "devices/22/network-bluetooth");
    public static final AntuIcon DEVICE_NETWORK_CARD_64 = create("DEVICE_NETWORK_CARD_64", "devices/64/network-card");
    public static final AntuIcon DEVICE_NETWORK_MODEM_22 = create("DEVICE_NETWORK_MODEM_22", "devices/22/network-modem");
    public static final AntuIcon DEVICE_NETWORK_VPN_22 = create("DEVICE_NETWORK_VPN_22", "devices/22/network-vpn");
    public static final AntuIcon DEVICE_NETWORK_WIRED_22 = create("DEVICE_NETWORK_WIRED_22", "devices/22/network-wired");
    public static final AntuIcon DEVICE_NETWORK_WIRED_ACTIVATED_22 = create("DEVICE_NETWORK_WIRED_ACTIVATED_22", "devices/22/network-wired-activated");
    public static final AntuIcon DEVICE_NETWORK_WIRED_UNAVAILABLE_22 = create("DEVICE_NETWORK_WIRED_UNAVAILABLE_22", "devices/22/network-wired-unavailable");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_22 = create("DEVICE_NETWORK_WIRELESS_22", "devices/22/network-wireless");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_CONNECTED_25_22 = create("DEVICE_NETWORK_WIRELESS_CONNECTED_25_22", "devices/22/network-wireless-connected-25");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_CONNECTED_50_22 = create("DEVICE_NETWORK_WIRELESS_CONNECTED_50_22", "devices/22/network-wireless-connected-50");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_CONNECTED_75_22 = create("DEVICE_NETWORK_WIRELESS_CONNECTED_75_22", "devices/22/network-wireless-connected-75");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_CONNECTED_100_22 = create("DEVICE_NETWORK_WIRELESS_CONNECTED_100_22", "devices/22/network-wireless-connected-100");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_CONNECTED_00_22 = create("DEVICE_NETWORK_WIRELESS_CONNECTED_00_22", "devices/22/network-wireless-connected-00");
    public static final AntuIcon DEVICE_NETWORK_WIRELESS_DISCONNECTED_22 = create("DEVICE_NETWORK_WIRELESS_DISCONNECTED_22", "devices/22/network-wireless-disconnected");
    public static final AntuIcon DEVICE_PHONE_22 = create("DEVICE_PHONE_22", "devices/22/phone");
    public static final AntuIcon DEVICE_PHONE_64 = create("DEVICE_PHONE_64", "devices/64/phone");
    public static final AntuIcon DEVICE_PRINTER_64 = create("DEVICE_PRINTER_64", "devices/64/printer");
    public static final AntuIcon DEVICE_SCANNER_64 = create("DEVICE_SCANNER_64", "devices/64/scanner");
    public static final AntuIcon DEVICE_SMARTPHONE_22 = create("DEVICE_SMARTPHONE_22", "devices/22/smartphone");
    public static final AntuIcon DEVICE_SMARTPHONE_64 = create("DEVICE_SMARTPHONE_64", "devices/64/smartphone");
    public static final AntuIcon DEVICE_TABLET_22 = create("DEVICE_TABLET_22", "devices/22/tablet");
    public static final AntuIcon DEVICE_TABLET_64 = create("DEVICE_TABLET_64", "devices/64/tablet");
    public static final AntuIcon DEVICE_VIDEO_DISPLAY_64 = create("DEVICE_VIDEO_DISPLAY_64", "devices/64/video-display");
    public static final AntuIcon DEVICE_VIDEO_TELEVISION_64 = create("DEVICE_VIDEO_TELEVISION_64", "devices/64/video-television");

    // EMBLEMS
    public static final AntuIcon EMBLEM_CHECKED_8 = create("EMBLEM_CHECKED_8", "emblems/8/emblem-checked");
    public static final AntuIcon EMBLEM_CHECKMARK_8 = create("EMBLEM_CHECKMARK_8", "emblems/8/checkmark");
    public static final AntuIcon EMBLEM_FAVORITE_22 = create("EMBLEM_FAVORITE_22", "emblems/22/emblem-favorite");
    public static final AntuIcon EMBLEM_IMPORTANT_22 = create("EMBLEM_IMPORTANT_22", "emblems/22/emblem-important");
    public static final AntuIcon EMBLEM_LOCKED_8 = create("EMBLEM_LOCKED_8", "emblems/8/emblem-locked");
    public static final AntuIcon EMBLEM_LOCKED_22 = create("EMBLEM_LOCKED_22", "emblems/22/emblem-locked");
    public static final AntuIcon EMBLEM_MOUNTED_8 = create("EMBLEM_MOUNTED_8", "emblems/8/emblem-mounted");
    public static final AntuIcon EMBLEM_PAUSE_8 = create("EMBLEM_PAUSE_8", "emblems/8/emblem-pause");
    public static final AntuIcon EMBLEM_SYMBOLIC_LINK_16 = create("EMBLEM_SYMBOLIC_LINK_16", "emblems/16/emblem-symbolic-link");
    public static final AntuIcon EMBLEM_SYMBOLIC_LINK_22 = create("EMBLEM_SYMBOLIC_LINK_22", "emblems/22/emblem-symbolic-link");
    public static final AntuIcon EMBLEM_UNAVAILABLE_8 = create("EMBLEM_UNAVAILABLE_8", "emblems/8/emblem-unavailable");
    public static final AntuIcon EMBLEM_UNLOCKED_8 = create("EMBLEM_UNLOCKED_8", "emblems/8/emblem-unlocked");
    public static final AntuIcon EMBLEM_UNLOCKED_22 = create("EMBLEM_UNLOCKED_22", "emblems/22/emblem-unlocked");
    public static final AntuIcon EMBLEM_UNMOUNTED_8 = create("EMBLEM_UNMOUNTED_8", "emblems/8/emblem-unmounted");
    public static final AntuIcon EMBLEM_VCS_ADDED_16 = create("EMBLEM_VCS_ADDED_16", "emblems/16/vcs-added");
    public static final AntuIcon EMBLEM_VCS_CONFLICTING_16 = create("EMBLEM_VCS_CONFLICTING_16", "emblems/16/vcs-conflicting");
    public static final AntuIcon EMBLEM_VCS_LOCALLY_MODIFIED_16 = create("EMBLEM_VCS_LOCALLY_MODIFIED_16", "emblems/16/vcs-locally-modified");
    public static final AntuIcon EMBLEM_VCS_LOCALLY_MODIFIED_UNSTAGED_16 = create("EMBLEM_VCS_LOCALLY_MODIFIED_UNSTAGED_16", "emblems/16/vcs-locally-modified-unstaged");
    public static final AntuIcon EMBLEM_VCS_NORMAL_16 = create("EMBLEM_VCS_NORMAL_16", "emblems/16/vcs-normal");
    public static final AntuIcon EMBLEM_VCS_REMOVED_16 = create("EMBLEM_VCS_REMOVED_16", "emblems/16/vcs-removed");
    public static final AntuIcon EMBLEM_VCS_UPDATE_REQUIRED_16 = create("EMBLEM_VCS_UPDATE_REQUIRED_16", "emblems/16/vcs-update-required");

    // EMOTES
    public static final AntuIcon EMOTE_DRINK_BEER_22 = create("EMOTE_DRINK_BEER_22", "emotes/22/drink-beer");
    public static final AntuIcon EMOTE_DRINK_MARTINI_22 = create("EMOTE_DRINK_MARTINI_22", "emotes/22/drink-martini");
    public static final AntuIcon EMOTE_FACE_ANGEL_22 = create("EMOTE_FACE_ANGEL_22", "emotes/22/face-angel");
    public static final AntuIcon EMOTE_FACE_ANGRY_22 = create("EMOTE_FACE_ANGRY_22", "emotes/22/face-angry");
    public static final AntuIcon EMOTE_FACE_CLOWN_22 = create("EMOTE_FACE_CLOWN_22", "emotes/22/face-clown");
    public static final AntuIcon EMOTE_FACE_CONFUSED_22 = create("EMOTE_FACE_CONFUSED_22", "emotes/22/face-confused");
    public static final AntuIcon EMOTE_FACE_COOL_22 = create("EMOTE_FACE_COOL_22", "emotes/22/face-cool");
    public static final AntuIcon EMOTE_FACE_CRYING_22 = create("EMOTE_FACE_CRYING_22", "emotes/22/face-crying");
    public static final AntuIcon EMOTE_FACE_DEVILISH_22 = create("EMOTE_FACE_DEVILISH_22", "emotes/22/face-devilish");
    public static final AntuIcon EMOTE_FACE_EMBARRASSED_22 = create("EMOTE_FACE_EMBARRASSED_22", "emotes/22/face-embarrassed");
    public static final AntuIcon EMOTE_FACE_GLASSES_22 = create("EMOTE_FACE_GLASSES_22", "emotes/22/face-glasses");
    public static final AntuIcon EMOTE_FACE_IN_LOVE_22 = create("EMOTE_FACE_IN_LOVE_22", "emotes/22/face-in-love");
    public static final AntuIcon EMOTE_FACE_KISS_22 = create("EMOTE_FACE_KISS_22", "emotes/22/face-kiss");
    public static final AntuIcon EMOTE_FACE_LAUGHING_22 = create("EMOTE_FACE_LAUGHING_22", "emotes/22/face-laughing");
    public static final AntuIcon EMOTE_FACE_NINJA_22 = create("EMOTE_FACE_NINJA_22", "emotes/22/face-ninja");
    public static final AntuIcon EMOTE_FACE_PIRATE_22 = create("EMOTE_FACE_PIRATE_22", "emotes/22/face-pirate");
    public static final AntuIcon EMOTE_FACE_PLAIN_22 = create("EMOTE_FACE_PLAIN_22", "emotes/22/face-plain");
    public static final AntuIcon EMOTE_FACE_QUIET_22 = create("EMOTE_FACE_QUIET_22", "emotes/22/face-quiet");
    public static final AntuIcon EMOTE_FACE_RASPBERRY_22 = create("EMOTE_FACE_RASPBERRY_22", "emotes/22/face-raspberry");
    public static final AntuIcon EMOTE_FACE_SAD_22 = create("EMOTE_FACE_SAD_22", "emotes/22/face-sad");
    public static final AntuIcon EMOTE_FACE_SICK_22 = create("EMOTE_FACE_SICK_22", "emotes/22/face-sick");
    public static final AntuIcon EMOTE_FACE_SLEEPING_22 = create("EMOTE_FACE_SLEEPING_22", "emotes/22/face-sleeping");
    public static final AntuIcon EMOTE_FACE_SMILE_22 = create("EMOTE_FACE_SMILE_22", "emotes/22/face-smile");
    public static final AntuIcon EMOTE_FACE_SMILE_BIG_22 = create("EMOTE_FACE_SMILE_BIG_22", "emotes/22/face-smile-big");
    public static final AntuIcon EMOTE_FACE_SMILE_GRIN_22 = create("EMOTE_FACE_SMILE_GRIN_22", "emotes/22/face-smile-grin");
    public static final AntuIcon EMOTE_FACE_SMIRK_22 = create("EMOTE_FACE_SMIRK_22", "emotes/22/face-smirk");
    public static final AntuIcon EMOTE_FACE_SURPRISE_22 = create("EMOTE_FACE_SURPRISE_22", "emotes/22/face-surprise");
    public static final AntuIcon EMOTE_FACE_UNCERTAIN_22 = create("EMOTE_FACE_UNCERTAIN_22", "emotes/22/face-uncertain");
    public static final AntuIcon EMOTE_FACE_WINK_22 = create("EMOTE_FACE_WINK_22", "emotes/22/face-wink");
    public static final AntuIcon EMOTE_FACE_WORRIED_22 = create("EMOTE_FACE_WORRIED_22", "emotes/22/face-worried");
    public static final AntuIcon EMOTE_FACE_YAWN_22 = create("EMOTE_FACE_YAWN_22", "emotes/22/face-yawn");
    public static final AntuIcon EMOTE_FOOD_CAKE_22 = create("EMOTE_FOOD_CAKE_22", "emotes/22/food-cake");

    // PLACES
    public static final AntuIcon PLACE_BOOKMARKS_22 = create("PLACE_BOOKMARKS_22", "places/22/bookmarks");
    public static final AntuIcon PLACE_BOOKMARKS_32 = create("PLACE_BOOKMARKS_32", "places/32/bookmarks");
    public static final AntuIcon PLACE_BOOKMARKS_64 = create("PLACE_BOOKMARKS_64", "places/64/bookmarks");
    public static final AntuIcon PLACE_CERTIFICATE_SERVER_22 = create("PLACE_CERTIFICATE_SERVER_22", "places/22/certificate-server");
    public static final AntuIcon PLACE_CERTIFICATE_SERVER_32 = create("PLACE_CERTIFICATE_SERVER_32", "places/32/certificate-server");
    public static final AntuIcon PLACE_CERTIFICATE_SERVER_64 = create("PLACE_CERTIFICATE_SERVER_64", "places/64/certificate-server");
    public static final AntuIcon PLACE_DATABASE_CHAKRA_64 = create("PLACE_DATABASE_CHAKRA_64", "places/64/database-chakra");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_64 = create("PLACE_DISTRIBUTOR_LOGO_64", "places/64/distributor-logo");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_ARCHLINUX_64 = create("PLACE_DISTRIBUTOR_LOGO_ARCHLINUX_64", "places/64/distributor-logo-archlinux");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_BLAG_64 = create("PLACE_DISTRIBUTOR_LOGO_BLAG_64", "places/64/distributor-logo-blag");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_CHAKRA_64 = create("PLACE_DISTRIBUTOR_LOGO_CHAKRA_64", "places/64/distributor-logo-chakra");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_DEBIAN_64 = create("PLACE_DISTRIBUTOR_LOGO_DEBIAN_64", "places/64/distributor-logo-debian");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_DRAGORA_64 = create("PLACE_DISTRIBUTOR_LOGO_DRAGORA_64", "places/64/distributor-logo-dragora");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_FEDORA_64 = create("PLACE_DISTRIBUTOR_LOGO_FEDORA_64", "places/64/distributor-logo-fedora");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_FREEBSD_64 = create("PLACE_DISTRIBUTOR_LOGO_FREEBSD_64", "places/64/distributor-logo-freebsd");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_GENTOO_64 = create("PLACE_DISTRIBUTOR_LOGO_GENTOO_64", "places/64/distributor-logo-gentoo");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_GUIX_64 = create("PLACE_DISTRIBUTOR_LOGO_GUIX_64", "places/64/distributor-logo-guix");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_KAOS_64 = create("PLACE_DISTRIBUTOR_LOGO_KAOS_64", "places/64/distributor-logo-kaos");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_KUBUNTU_64 = create("PLACE_DISTRIBUTOR_LOGO_KUBUNTU_64", "places/64/distributor-logo-kubuntu");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_MAGEIA_64 = create("PLACE_DISTRIBUTOR_LOGO_MAGEIA_64", "places/64/distributor-logo-mageia");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_MANJARO_64 = create("PLACE_DISTRIBUTOR_LOGO_MANJARO_64", "places/64/distributor-logo-manjaro");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_OPENSUSE_64 = create("PLACE_DISTRIBUTOR_LOGO_OPENSUSE_64", "places/64/distributor-logo-opensuse");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_PARABOLA_64 = create("PLACE_DISTRIBUTOR_LOGO_PARABOLA_64", "places/64/distributor-logo-parabola");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_SLACKWARE_64 = create("PLACE_DISTRIBUTOR_LOGO_SLACKWARE_64", "places/64/distributor-logo-slackware");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_TRISQUEL_64 = create("PLACE_DISTRIBUTOR_LOGO_TRISQUEL_64", "places/64/distributor-logo-trisquel");
    public static final AntuIcon PLACE_DISTRIBUTOR_LOGO_UBUNTU_64 = create("PLACE_DISTRIBUTOR_LOGO_UBUNTU_64", "places/64/distributor-logo-ubuntu");
    public static final AntuIcon PLACE_DOCUMENT_MULTIPLE_22 = create("PLACE_DOCUMENT_MULTIPLE_22", "places/22/document-multiple");
    public static final AntuIcon PLACE_FAVORITES_22 = create("PLACE_FAVORITES_22", "places/22/favorites");
    public static final AntuIcon PLACE_FAVORITES_32 = create("PLACE_FAVORITES_32", "places/32/favorites");
    public static final AntuIcon PLACE_FAVORITES_64 = create("PLACE_FAVORITES_64", "places/64/favorites");
    public static final AntuIcon PLACE_FOLDER_22 = create("PLACE_FOLDER_22", "places/22/folder");
    public static final AntuIcon PLACE_FOLDER_32 = create("PLACE_FOLDER_32", "places/32/folder");
    public static final AntuIcon PLACE_FOLDER_64 = create("PLACE_FOLDER_64", "places/64/folder");
    public static final AntuIcon PLACE_FOLDER_ACTIVITIES_22 = create("PLACE_FOLDER_ACTIVITIES_22", "places/22/folder-activities");
    public static final AntuIcon PLACE_FOLDER_ACTIVITIES_32 = create("PLACE_FOLDER_ACTIVITIES_32", "places/32/folder-activities");
    public static final AntuIcon PLACE_FOLDER_ACTIVITIES_64 = create("PLACE_FOLDER_ACTIVITIES_64", "places/64/folder-activities");
    public static final AntuIcon PLACE_FOLDER_ADD_22 = create("PLACE_FOLDER_ADD_22", "places/22/folder-add");
    public static final AntuIcon PLACE_FOLDER_BLACK_22 = create("PLACE_FOLDER_BLACK_22", "places/22/folder-black");
    public static final AntuIcon PLACE_FOLDER_BLACK_32 = create("PLACE_FOLDER_BLACK_32", "places/32/folder-black");
    public static final AntuIcon PLACE_FOLDER_BLACK_64 = create("PLACE_FOLDER_BLACK_64", "places/64/folder-black");
    public static final AntuIcon PLACE_FOLDER_BLUE_22 = create("PLACE_FOLDER_BLUE_22", "places/22/folder-blue");
    public static final AntuIcon PLACE_FOLDER_BLUE_32 = create("PLACE_FOLDER_BLUE_32", "places/32/folder-blue");
    public static final AntuIcon PLACE_FOLDER_BLUE_64 = create("PLACE_FOLDER_BLUE_64", "places/64/folder-blue");
    public static final AntuIcon PLACE_FOLDER_BOOKMARK_22 = create("PLACE_FOLDER_BOOKMARK_22", "places/22/folder-bookmark");
    public static final AntuIcon PLACE_FOLDER_BOOKMARK_32 = create("PLACE_FOLDER_BOOKMARK_32", "places/32/folder-bookmark");
    public static final AntuIcon PLACE_FOLDER_BOOKMARK_64 = create("PLACE_FOLDER_BOOKMARK_64", "places/64/folder-bookmark");
    public static final AntuIcon PLACE_FOLDER_BROWN_22 = create("PLACE_FOLDER_BROWN_22", "places/22/folder-brown");
    public static final AntuIcon PLACE_FOLDER_BROWN_32 = create("PLACE_FOLDER_BROWN_32", "places/32/folder-brown");
    public static final AntuIcon PLACE_FOLDER_BROWN_64 = create("PLACE_FOLDER_BROWN_64", "places/64/folder-brown");
    public static final AntuIcon PLACE_FOLDER_CAMERA_22 = create("PLACE_FOLDER_CAMERA_22", "places/22/folder-camera");
    public static final AntuIcon PLACE_FOLDER_CAMERA_32 = create("PLACE_FOLDER_CAMERA_32", "places/32/folder-camera");
    public static final AntuIcon PLACE_FOLDER_CAMERA_64 = create("PLACE_FOLDER_CAMERA_64", "places/64/folder-camera");
    public static final AntuIcon PLACE_FOLDER_CHAKRA_22 = create("PLACE_FOLDER_CHAKRA_22", "places/22/folder-chakra");
    public static final AntuIcon PLACE_FOLDER_CHAKRA_32 = create("PLACE_FOLDER_CHAKRA_32", "places/32/folder-chakra");
    public static final AntuIcon PLACE_FOLDER_CHAKRA_64 = create("PLACE_FOLDER_CHAKRA_64", "places/64/folder-chakra");
    public static final AntuIcon PLACE_FOLDER_CLOUD_22 = create("PLACE_FOLDER_CLOUD_22", "places/22/folder-cloud");
    public static final AntuIcon PLACE_FOLDER_CLOUD_32 = create("PLACE_FOLDER_CLOUD_32", "places/32/folder-cloud");
    public static final AntuIcon PLACE_FOLDER_CLOUD_64 = create("PLACE_FOLDER_CLOUD_64", "places/64/folder-cloud");
    public static final AntuIcon PLACE_FOLDER_CLOUDS_32 = create("PLACE_FOLDER_CLOUDS_32", "places/32/folder-clouds");
    public static final AntuIcon PLACE_FOLDER_CYAN_22 = create("PLACE_FOLDER_CYAN_22", "places/22/folder-cyan");
    public static final AntuIcon PLACE_FOLDER_CYAN_32 = create("PLACE_FOLDER_CYAN_32", "places/32/folder-cyan");
    public static final AntuIcon PLACE_FOLDER_CYAN_64 = create("PLACE_FOLDER_CYAN_64", "places/64/folder-cyan");
    public static final AntuIcon PLACE_FOLDER_DEVELOPMENT_22 = create("PLACE_FOLDER_DEVELOPMENT_22", "places/22/folder-development");
    public static final AntuIcon PLACE_FOLDER_DEVELOPMENT_32 = create("PLACE_FOLDER_DEVELOPMENT_32", "places/32/folder-development");
    public static final AntuIcon PLACE_FOLDER_DEVELOPMENT_64 = create("PLACE_FOLDER_DEVELOPMENT_64", "places/64/folder-development");
    public static final AntuIcon PLACE_FOLDER_DOCUMENTS_22 = create("PLACE_FOLDER_DOCUMENTS_22", "places/22/folder-documents");
    public static final AntuIcon PLACE_FOLDER_DOCUMENTS_32 = create("PLACE_FOLDER_DOCUMENTS_32", "places/32/folder-documents");
    public static final AntuIcon PLACE_FOLDER_DOCUMENTS_64 = create("PLACE_FOLDER_DOCUMENTS_64", "places/64/folder-documents");
    public static final AntuIcon PLACE_FOLDER_DOWNLOAD_22 = create("PLACE_FOLDER_DOWNLOAD_22", "places/22/folder-download");
    public static final AntuIcon PLACE_FOLDER_DOWNLOAD_32 = create("PLACE_FOLDER_DOWNLOAD_32", "places/32/folder-download");
    public static final AntuIcon PLACE_FOLDER_DOWNLOAD_64 = create("PLACE_FOLDER_DOWNLOAD_64", "places/64/folder-download");
    public static final AntuIcon PLACE_FOLDER_DOWNLOADS_22 = create("PLACE_FOLDER_DOWNLOADS_22", "places/22/folder-downloads");
    public static final AntuIcon PLACE_FOLDER_DOWNLOADS_32 = create("PLACE_FOLDER_DOWNLOADS_32", "places/32/folder-downloads");
    public static final AntuIcon PLACE_FOLDER_DOWNLOADS_64 = create("PLACE_FOLDER_DOWNLOADS_64", "places/64/folder-downloads");
    public static final AntuIcon PLACE_FOLDER_DROPBOX_22 = create("PLACE_FOLDER_DROPBOX_22", "places/22/folder-dropbox");
    public static final AntuIcon PLACE_FOLDER_DROPBOX_32 = create("PLACE_FOLDER_DROPBOX_32", "places/32/folder-dropbox");
    public static final AntuIcon PLACE_FOLDER_DROPBOX_64 = create("PLACE_FOLDER_DROPBOX_64", "places/64/folder-dropbox");
    public static final AntuIcon PLACE_FOLDER_FAVORITES_22 = create("PLACE_FOLDER_FAVORITES_22", "places/22/folder-favorites");
    public static final AntuIcon PLACE_FOLDER_FAVORITES_32 = create("PLACE_FOLDER_FAVORITES_32", "places/32/folder-favorites");
    public static final AntuIcon PLACE_FOLDER_FAVORITES_64 = create("PLACE_FOLDER_FAVORITES_64", "places/64/folder-favorites");
    public static final AntuIcon PLACE_FOLDER_GAMES_22 = create("PLACE_FOLDER_GAMES_22", "places/22/folder-games");
    public static final AntuIcon PLACE_FOLDER_GAMES_32 = create("PLACE_FOLDER_GAMES_32", "places/32/folder-games");
    public static final AntuIcon PLACE_FOLDER_GAMES_64 = create("PLACE_FOLDER_GAMES_64", "places/64/folder-games");
    public static final AntuIcon PLACE_FOLDER_GDRIVE_22 = create("PLACE_FOLDER_GDRIVE_22", "places/22/folder-gdrive");
    public static final AntuIcon PLACE_FOLDER_GDRIVE_32 = create("PLACE_FOLDER_GDRIVE_32", "places/32/folder-gdrive");
    public static final AntuIcon PLACE_FOLDER_GDRIVE_64 = create("PLACE_FOLDER_GDRIVE_64", "places/64/folder-gdrive");
    public static final AntuIcon PLACE_FOLDER_GIT_22 = create("PLACE_FOLDER_GIT_22", "places/22/folder-git");
    public static final AntuIcon PLACE_FOLDER_GIT_32 = create("PLACE_FOLDER_GIT_32", "places/32/folder-git");
    public static final AntuIcon PLACE_FOLDER_GIT_64 = create("PLACE_FOLDER_GIT_64", "places/64/folder-git");
    public static final AntuIcon PLACE_FOLDER_GITHUB_22 = create("PLACE_FOLDER_GITHUB_22", "places/22/folder-github");
    public static final AntuIcon PLACE_FOLDER_GITHUB_32 = create("PLACE_FOLDER_GITHUB_32", "places/32/folder-github");
    public static final AntuIcon PLACE_FOLDER_GITHUB_64 = create("PLACE_FOLDER_GITHUB_64", "places/64/folder-github");
    public static final AntuIcon PLACE_FOLDER_GNOME_22 = create("PLACE_FOLDER_GNOME_22", "places/22/folder-gnome");
    public static final AntuIcon PLACE_FOLDER_GNOME_32 = create("PLACE_FOLDER_GNOME_32", "places/32/folder-gnome");
    public static final AntuIcon PLACE_FOLDER_GNOME_64 = create("PLACE_FOLDER_GNOME_64", "places/64/folder-gnome");
    public static final AntuIcon PLACE_FOLDER_GREEN_22 = create("PLACE_FOLDER_GREEN_22", "places/22/folder-green");
    public static final AntuIcon PLACE_FOLDER_GREEN_32 = create("PLACE_FOLDER_GREEN_32", "places/32/folder-green");
    public static final AntuIcon PLACE_FOLDER_GREEN_64 = create("PLACE_FOLDER_GREEN_64", "places/64/folder-green");
    public static final AntuIcon PLACE_FOLDER_GREY_22 = create("PLACE_FOLDER_GREY_22", "places/22/folder-grey");
    public static final AntuIcon PLACE_FOLDER_GREY_32 = create("PLACE_FOLDER_GREY_32", "places/32/folder-grey");
    public static final AntuIcon PLACE_FOLDER_GREY_64 = create("PLACE_FOLDER_GREY_64", "places/64/folder-grey");
    public static final AntuIcon PLACE_FOLDER_HTML_22 = create("PLACE_FOLDER_HTML_22", "places/22/folder-html");
    public static final AntuIcon PLACE_FOLDER_HTML_32 = create("PLACE_FOLDER_HTML_32", "places/32/folder-html");
    public static final AntuIcon PLACE_FOLDER_HTML_64 = create("PLACE_FOLDER_HTML_64", "places/64/folder-html");
    public static final AntuIcon PLACE_FOLDER_IMAGE_22 = create("PLACE_FOLDER_IMAGE_22", "places/22/folder-image");
    public static final AntuIcon PLACE_FOLDER_IMAGE_32 = create("PLACE_FOLDER_IMAGE_32", "places/32/folder-image");
    public static final AntuIcon PLACE_FOLDER_IMAGE_64 = create("PLACE_FOLDER_IMAGE_64", "places/64/folder-image");
    public static final AntuIcon PLACE_FOLDER_IMAGE_PEOPLE_22 = create("PLACE_FOLDER_IMAGE_PEOPLE_22", "places/22/folder-image-people");
    public static final AntuIcon PLACE_FOLDER_IMAGE_PEOPLE_32 = create("PLACE_FOLDER_IMAGE_PEOPLE_32", "places/32/folder-image-people");
    public static final AntuIcon PLACE_FOLDER_IMAGE_PEOPLE_64 = create("PLACE_FOLDER_IMAGE_PEOPLE_64", "places/64/folder-image-people");
    public static final AntuIcon PLACE_FOLDER_IMAGES_22 = create("PLACE_FOLDER_IMAGES_22", "places/22/folder-images");
    public static final AntuIcon PLACE_FOLDER_IMAGES_32 = create("PLACE_FOLDER_IMAGES_32", "places/32/folder-images");
    public static final AntuIcon PLACE_FOLDER_IMAGES_64 = create("PLACE_FOLDER_IMAGES_64", "places/64/folder-images");
    public static final AntuIcon PLACE_FOLDER_IMPORTANT_22 = create("PLACE_FOLDER_IMPORTANT_22", "places/22/folder-important");
    public static final AntuIcon PLACE_FOLDER_IMPORTANT_32 = create("PLACE_FOLDER_IMPORTANT_32", "places/32/folder-important");
    public static final AntuIcon PLACE_FOLDER_IMPORTANT_64 = create("PLACE_FOLDER_IMPORTANT_64", "places/64/folder-important");
    public static final AntuIcon PLACE_FOLDER_KAOS_22 = create("PLACE_FOLDER_KAOS_22", "places/22/folder-kaos");
    public static final AntuIcon PLACE_FOLDER_KAOS_32 = create("PLACE_FOLDER_KAOS_32", "places/32/folder-kaos");
    public static final AntuIcon PLACE_FOLDER_KAOS_64 = create("PLACE_FOLDER_KAOS_64", "places/64/folder-kaos");
    public static final AntuIcon PLACE_FOLDER_LOCKED_22 = create("PLACE_FOLDER_LOCKED_22", "places/22/folder-locked");
    public static final AntuIcon PLACE_FOLDER_LOCKED_32 = create("PLACE_FOLDER_LOCKED_32", "places/32/folder-locked");
    public static final AntuIcon PLACE_FOLDER_LOCKED_64 = create("PLACE_FOLDER_LOCKED_64", "places/64/folder-locked");
    public static final AntuIcon PLACE_FOLDER_MAGENTA_22 = create("PLACE_FOLDER_MAGENTA_22", "places/22/folder-magenta");
    public static final AntuIcon PLACE_FOLDER_MAGENTA_32 = create("PLACE_FOLDER_MAGENTA_32", "places/32/folder-magenta");
    public static final AntuIcon PLACE_FOLDER_MAGENTA_64 = create("PLACE_FOLDER_MAGENTA_64", "places/64/folder-magenta");
    public static final AntuIcon PLACE_FOLDER_MAIL_22 = create("PLACE_FOLDER_MAIL_22", "places/22/folder-mail");
    public static final AntuIcon PLACE_FOLDER_MAIL_32 = create("PLACE_FOLDER_MAIL_32", "places/32/folder-mail");
    public static final AntuIcon PLACE_FOLDER_MAIL_64 = create("PLACE_FOLDER_MAIL_64", "places/64/folder-mail");
    public static final AntuIcon PLACE_FOLDER_MUSIC_22 = create("PLACE_FOLDER_MUSIC_22", "places/22/folder-music");
    public static final AntuIcon PLACE_FOLDER_MUSIC_32 = create("PLACE_FOLDER_MUSIC_32", "places/32/folder-music");
    public static final AntuIcon PLACE_FOLDER_MUSIC_64 = create("PLACE_FOLDER_MUSIC_64", "places/64/folder-music");
    public static final AntuIcon PLACE_FOLDER_NETWORK_22 = create("PLACE_FOLDER_NETWORK_22", "places/22/folder-network");
    public static final AntuIcon PLACE_FOLDER_NETWORK_32 = create("PLACE_FOLDER_NETWORK_32", "places/32/folder-network");
    public static final AntuIcon PLACE_FOLDER_NETWORK_64 = create("PLACE_FOLDER_NETWORK_64", "places/64/folder-network");
    public static final AntuIcon PLACE_FOLDER_OPEN_22 = create("PLACE_FOLDER_OPEN_22", "places/22/folder-open");
    public static final AntuIcon PLACE_FOLDER_OPEN_32 = create("PLACE_FOLDER_OPEN_32", "places/32/folder-open");
    public static final AntuIcon PLACE_FOLDER_OPEN_64 = create("PLACE_FOLDER_OPEN_64", "places/64/folder-open");
    public static final AntuIcon PLACE_FOLDER_ORANGE_22 = create("PLACE_FOLDER_ORANGE_22", "places/22/folder-orange");
    public static final AntuIcon PLACE_FOLDER_ORANGE_32 = create("PLACE_FOLDER_ORANGE_32", "places/32/folder-orange");
    public static final AntuIcon PLACE_FOLDER_ORANGE_64 = create("PLACE_FOLDER_ORANGE_64", "places/64/folder-orange");
    public static final AntuIcon PLACE_FOLDER_OWNCLOUD_22 = create("PLACE_FOLDER_OWNCLOUD_22", "places/22/folder-owncloud");
    public static final AntuIcon PLACE_FOLDER_OWNCLOUD_32 = create("PLACE_FOLDER_OWNCLOUD_32", "places/32/folder-owncloud");
    public static final AntuIcon PLACE_FOLDER_OWNCLOUD_64 = create("PLACE_FOLDER_OWNCLOUD_64", "places/64/folder-owncloud");
    public static final AntuIcon PLACE_FOLDER_PICTURE_22 = create("PLACE_FOLDER_PICTURE_22", "places/22/folder-picture");
    public static final AntuIcon PLACE_FOLDER_PICTURE_32 = create("PLACE_FOLDER_PICTURE_32", "places/32/folder-picture");
    public static final AntuIcon PLACE_FOLDER_PICTURE_64 = create("PLACE_FOLDER_PICTURE_64", "places/64/folder-picture");
    public static final AntuIcon PLACE_FOLDER_PICTURES_22 = create("PLACE_FOLDER_PICTURES_22", "places/22/folder-pictures");
    public static final AntuIcon PLACE_FOLDER_PICTURES_32 = create("PLACE_FOLDER_PICTURES_32", "places/32/folder-pictures");
    public static final AntuIcon PLACE_FOLDER_PICTURES_64 = create("PLACE_FOLDER_PICTURES_64", "places/64/folder-pictures");
    public static final AntuIcon PLACE_FOLDER_PRINT_22 = create("PLACE_FOLDER_PRINT_22", "places/22/folder-print");
    public static final AntuIcon PLACE_FOLDER_PRINT_32 = create("PLACE_FOLDER_PRINT_32", "places/32/folder-print");
    public static final AntuIcon PLACE_FOLDER_PRINT_64 = create("PLACE_FOLDER_PRINT_64", "places/64/folder-print");
    public static final AntuIcon PLACE_FOLDER_PRIVATE_22 = create("PLACE_FOLDER_PRIVATE_22", "places/22/folder-private");
    public static final AntuIcon PLACE_FOLDER_PRIVATE_32 = create("PLACE_FOLDER_PRIVATE_32", "places/32/folder-private");
    public static final AntuIcon PLACE_FOLDER_PRIVATE_64 = create("PLACE_FOLDER_PRIVATE_64", "places/64/folder-private");
    public static final AntuIcon PLACE_FOLDER_PUBLIC_22 = create("PLACE_FOLDER_PUBLIC_22", "places/22/folder-public");
    public static final AntuIcon PLACE_FOLDER_PUBLIC_32 = create("PLACE_FOLDER_PUBLIC_32", "places/32/folder-public");
    public static final AntuIcon PLACE_FOLDER_PUBLIC_64 = create("PLACE_FOLDER_PUBLIC_64", "places/64/folder-public");
    public static final AntuIcon PLACE_FOLDER_PUBLICSHARE_22 = create("PLACE_FOLDER_PUBLICSHARE_22", "places/22/folder-publicshare");
    public static final AntuIcon PLACE_FOLDER_PUBLICSHARE_32 = create("PLACE_FOLDER_PUBLICSHARE_32", "places/32/folder-publicshare");
    public static final AntuIcon PLACE_FOLDER_PUBLICSHARE_64 = create("PLACE_FOLDER_PUBLICSHARE_64", "places/64/folder-publicshare");
    public static final AntuIcon PLACE_FOLDER_RED_22 = create("PLACE_FOLDER_RED_22", "places/22/folder-red");
    public static final AntuIcon PLACE_FOLDER_RED_32 = create("PLACE_FOLDER_RED_32", "places/32/folder-red");
    public static final AntuIcon PLACE_FOLDER_RED_64 = create("PLACE_FOLDER_RED_64", "places/64/folder-red");
    public static final AntuIcon PLACE_FOLDER_REMOTE_22 = create("PLACE_FOLDER_REMOTE_22", "places/22/folder-remote");
    public static final AntuIcon PLACE_FOLDER_REMOTE_32 = create("PLACE_FOLDER_REMOTE_32", "places/32/folder-remote");
    public static final AntuIcon PLACE_FOLDER_REMOTE_64 = create("PLACE_FOLDER_REMOTE_64", "places/64/folder-remote");
    public static final AntuIcon PLACE_FOLDER_SCRIPT_22 = create("PLACE_FOLDER_SCRIPT_22", "places/22/folder-script");
    public static final AntuIcon PLACE_FOLDER_SCRIPT_32 = create("PLACE_FOLDER_SCRIPT_32", "places/32/folder-script");
    public static final AntuIcon PLACE_FOLDER_SCRIPT_64 = create("PLACE_FOLDER_SCRIPT_64", "places/64/folder-script");
    public static final AntuIcon PLACE_FOLDER_SOUND_22 = create("PLACE_FOLDER_SOUND_22", "places/22/folder-sound");
    public static final AntuIcon PLACE_FOLDER_SOUND_32 = create("PLACE_FOLDER_SOUND_32", "places/32/folder-sound");
    public static final AntuIcon PLACE_FOLDER_SOUND_64 = create("PLACE_FOLDER_SOUND_64", "places/64/folder-sound");
    public static final AntuIcon PLACE_FOLDER_TAR_22 = create("PLACE_FOLDER_TAR_22", "places/22/folder-tar");
    public static final AntuIcon PLACE_FOLDER_TAR_32 = create("PLACE_FOLDER_TAR_32", "places/32/folder-tar");
    public static final AntuIcon PLACE_FOLDER_TAR_64 = create("PLACE_FOLDER_TAR_64", "places/64/folder-tar");
    public static final AntuIcon PLACE_FOLDER_TEMP_22 = create("PLACE_FOLDER_TEMP_22", "places/22/folder-temp");
    public static final AntuIcon PLACE_FOLDER_TEMP_32 = create("PLACE_FOLDER_TEMP_32", "places/32/folder-temp");
    public static final AntuIcon PLACE_FOLDER_TEMP_64 = create("PLACE_FOLDER_TEMP_64", "places/64/folder-temp");
    public static final AntuIcon PLACE_FOLDER_TEMPLATES_22 = create("PLACE_FOLDER_TEMPLATES_22", "places/22/folder-templates");
    public static final AntuIcon PLACE_FOLDER_TEMPLATES_32 = create("PLACE_FOLDER_TEMPLATES_32", "places/32/folder-templates");
    public static final AntuIcon PLACE_FOLDER_TEMPLATES_64 = create("PLACE_FOLDER_TEMPLATES_64", "places/64/folder-templates");
    public static final AntuIcon PLACE_FOLDER_TEXT_22 = create("PLACE_FOLDER_TEXT_22", "places/22/folder-text");
    public static final AntuIcon PLACE_FOLDER_TEXT_32 = create("PLACE_FOLDER_TEXT_32", "places/32/folder-text");
    public static final AntuIcon PLACE_FOLDER_TEXT_64 = create("PLACE_FOLDER_TEXT_64", "places/64/folder-text");
    public static final AntuIcon PLACE_FOLDER_TORRENTS_22 = create("PLACE_FOLDER_TORRENTS_22", "places/22/folder-torrents");
    public static final AntuIcon PLACE_FOLDER_TORRENTS_32 = create("PLACE_FOLDER_TORRENTS_32", "places/32/folder-torrents");
    public static final AntuIcon PLACE_FOLDER_TORRENTS_64 = create("PLACE_FOLDER_TORRENTS_64", "places/64/folder-torrents");
    public static final AntuIcon PLACE_FOLDER_TXT_22 = create("PLACE_FOLDER_TXT_22", "places/22/folder-txt");
    public static final AntuIcon PLACE_FOLDER_TXT_32 = create("PLACE_FOLDER_TXT_32", "places/32/folder-txt");
    public static final AntuIcon PLACE_FOLDER_TXT_64 = create("PLACE_FOLDER_TXT_64", "places/64/folder-txt");
    public static final AntuIcon PLACE_FOLDER_VBOX_22 = create("PLACE_FOLDER_VBOX_22", "places/22/folder-vbox");
    public static final AntuIcon PLACE_FOLDER_VBOX_32 = create("PLACE_FOLDER_VBOX_32", "places/32/folder-vbox");
    public static final AntuIcon PLACE_FOLDER_VBOX_64 = create("PLACE_FOLDER_VBOX_64", "places/64/folder-vbox");
    public static final AntuIcon PLACE_FOLDER_VIDEO_22 = create("PLACE_FOLDER_VIDEO_22", "places/22/folder-video");
    public static final AntuIcon PLACE_FOLDER_VIDEO_32 = create("PLACE_FOLDER_VIDEO_32", "places/32/folder-video");
    public static final AntuIcon PLACE_FOLDER_VIDEO_64 = create("PLACE_FOLDER_VIDEO_64", "places/64/folder-video");
    public static final AntuIcon PLACE_FOLDER_VIDEOCAMERA_22 = create("PLACE_FOLDER_VIDEOCAMERA_22", "places/22/folder-videocamera");
    public static final AntuIcon PLACE_FOLDER_VIDEOCAMERA_32 = create("PLACE_FOLDER_VIDEOCAMERA_32", "places/32/folder-videocamera");
    public static final AntuIcon PLACE_FOLDER_VIDEOCAMERA_64 = create("PLACE_FOLDER_VIDEOCAMERA_64", "places/64/folder-videocamera");
    public static final AntuIcon PLACE_FOLDER_VIDEOS_22 = create("PLACE_FOLDER_VIDEOS_22", "places/22/folder-videos");
    public static final AntuIcon PLACE_FOLDER_VIDEOS_32 = create("PLACE_FOLDER_VIDEOS_32", "places/32/folder-videos");
    public static final AntuIcon PLACE_FOLDER_VIDEOS_64 = create("PLACE_FOLDER_VIDEOS_64", "places/64/folder-videos");
    public static final AntuIcon PLACE_FOLDER_VIOLET_22 = create("PLACE_FOLDER_VIOLET_22", "places/22/folder-violet");
    public static final AntuIcon PLACE_FOLDER_VIOLET_32 = create("PLACE_FOLDER_VIOLET_32", "places/32/folder-violet");
    public static final AntuIcon PLACE_FOLDER_VIOLET_64 = create("PLACE_FOLDER_VIOLET_64", "places/64/folder-violet");
    public static final AntuIcon PLACE_FOLDER_WIFI_22 = create("PLACE_FOLDER_WIFI_22", "places/22/folder-wifi");
    public static final AntuIcon PLACE_FOLDER_WIFI_32 = create("PLACE_FOLDER_WIFI_32", "places/32/folder-wifi");
    public static final AntuIcon PLACE_FOLDER_WIFI_64 = create("PLACE_FOLDER_WIFI_64", "places/64/folder-wifi");
    public static final AntuIcon PLACE_FOLDER_YELLOW_22 = create("PLACE_FOLDER_YELLOW_22", "places/22/folder-yellow");
    public static final AntuIcon PLACE_FOLDER_YELLOW_32 = create("PLACE_FOLDER_YELLOW_32", "places/32/folder-yellow");
    public static final AntuIcon PLACE_FOLDER_YELLOW_64 = create("PLACE_FOLDER_YELLOW_64", "places/64/folder-yellow");
    public static final AntuIcon PLACE_MAIL_FOLDER_INBOX_22 = create("PLACE_MAIL_FOLDER_INBOX_22", "places/22/mail-folder-inbox");
    public static final AntuIcon PLACE_MAIL_FOLDER_OUTBOX_22 = create("PLACE_MAIL_FOLDER_OUTBOX_22", "places/22/mail-folder-outbox");
    public static final AntuIcon PLACE_MAIL_FOLDER_SENT_22 = create("PLACE_MAIL_FOLDER_SENT_22", "places/22/mail-folder-sent");
    public static final AntuIcon PLACE_MAIL_MESSAGE_22 = create("PLACE_MAIL_MESSAGE_22", "places/22/mail-message");
    public static final AntuIcon PLACE_NETWORK_SERVER_22 = create("PLACE_NETWORK_SERVER_22", "places/22/network-server");
    public static final AntuIcon PLACE_NETWORK_SERVER_32 = create("PLACE_NETWORK_SERVER_32", "places/32/network-server");
    public static final AntuIcon PLACE_NETWORK_SERVER_64 = create("PLACE_NETWORK_SERVER_64", "places/64/network-server");
    public static final AntuIcon PLACE_NETWORK_SERVER_DATABASE_22 = create("PLACE_NETWORK_SERVER_DATABASE_22", "places/22/network-server-database");
    public static final AntuIcon PLACE_NETWORK_SERVER_DATABASE_32 = create("PLACE_NETWORK_SERVER_DATABASE_32", "places/32/network-server-database");
    public static final AntuIcon PLACE_NETWORK_SERVER_DATABASE_64 = create("PLACE_NETWORK_SERVER_DATABASE_64", "places/64/network-server-database");
    public static final AntuIcon PLACE_NETWORK_WORKGROUP_22 = create("PLACE_NETWORK_WORKGROUP_22", "places/22/network-workgroup");
    public static final AntuIcon PLACE_NETWORK_WORKGROUP_32 = create("PLACE_NETWORK_WORKGROUP_32", "places/32/network-workgroup");
    public static final AntuIcon PLACE_NETWORK_WORKGROUP_64 = create("PLACE_NETWORK_WORKGROUP_64", "places/64/network-workgroup");
    public static final AntuIcon PLACE_REPOSITORY_22 = create("PLACE_REPOSITORY_22", "places/22/repository");
    public static final AntuIcon PLACE_REPOSITORY_32 = create("PLACE_REPOSITORY_32", "places/32/repository");
    public static final AntuIcon PLACE_REPOSITORY_64 = create("PLACE_REPOSITORY_64", "places/64/repository");
    public static final AntuIcon PLACE_SERVER_DATABASE_22 = create("PLACE_SERVER_DATABASE_22", "places/22/server-database");
    public static final AntuIcon PLACE_SERVER_DATABASE_32 = create("PLACE_SERVER_DATABASE_32", "places/32/server-database");
    public static final AntuIcon PLACE_SERVER_DATABASE_64 = create("PLACE_SERVER_DATABASE_64", "places/64/server-database");
    public static final AntuIcon PLACE_START_HERE_64 = create("PLACE_START_HERE_64", "places/64/start-here");
    public static final AntuIcon PLACE_START_HERE_ARCHLINUX_64 = create("PLACE_START_HERE_ARCHLINUX_64", "places/64/start-here-archlinux");
    public static final AntuIcon PLACE_START_HERE_BLAG_64 = create("PLACE_START_HERE_BLAG_64", "places/64/start-here-blag");
    public static final AntuIcon PLACE_START_HERE_CHAKRA_64 = create("PLACE_START_HERE_CHAKRA_64", "places/64/start-here-chakra");
    public static final AntuIcon PLACE_START_HERE_DEBIAN_64 = create("PLACE_START_HERE_DEBIAN_64", "places/64/start-here-debian");
    public static final AntuIcon PLACE_START_HERE_DRAGORA_64 = create("PLACE_START_HERE_DRAGORA_64", "places/64/start-here-dragora");
    public static final AntuIcon PLACE_START_HERE_FEDORA_64 = create("PLACE_START_HERE_FEDORA_64", "places/64/start-here-fedora");
    public static final AntuIcon PLACE_START_HERE_FREEBSD_64 = create("PLACE_START_HERE_FREEBSD_64", "places/64/start-here-freebsd");
    public static final AntuIcon PLACE_START_HERE_GENTOO_64 = create("PLACE_START_HERE_GENTOO_64", "places/64/start-here-gentoo");
    public static final AntuIcon PLACE_START_HERE_GUIX_64 = create("PLACE_START_HERE_GUIX_64", "places/64/start-here-guix");
    public static final AntuIcon PLACE_START_HERE_KAOS_64 = create("PLACE_START_HERE_KAOS_64", "places/64/start-here-kaos");
    public static final AntuIcon PLACE_START_HERE_KDE_22 = create("PLACE_START_HERE_KDE_22", "places/22/start-here-kde");
    public static final AntuIcon PLACE_START_HERE_KDE_64 = create("PLACE_START_HERE_KDE_64", "places/64/start-here-kde");
    public static final AntuIcon PLACE_START_HERE_KUBUNTU_64 = create("PLACE_START_HERE_KUBUNTU_64", "places/64/start-here-kubuntu");
    public static final AntuIcon PLACE_START_HERE_MAGEIA_64 = create("PLACE_START_HERE_MAGEIA_64", "places/64/start-here-mageia");
    public static final AntuIcon PLACE_START_HERE_MANJARO_64 = create("PLACE_START_HERE_MANJARO_64", "places/64/start-here-manjaro");
    public static final AntuIcon PLACE_START_HERE_OPENSUSE_64 = create("PLACE_START_HERE_OPENSUSE_64", "places/64/start-here-opensuse");
    public static final AntuIcon PLACE_START_HERE_PARABOLA_64 = create("PLACE_START_HERE_PARABOLA_64", "places/64/start-here-parabola");
    public static final AntuIcon PLACE_START_HERE_SLACKWARE_64 = create("PLACE_START_HERE_SLACKWARE_64", "places/64/start-here-slackware");
    public static final AntuIcon PLACE_START_HERE_TRISQUEL_64 = create("PLACE_START_HERE_TRISQUEL_64", "places/64/start-here-trisquel");
    public static final AntuIcon PLACE_START_HERE_UBUNTU_64 = create("PLACE_START_HERE_UBUNTU_64", "places/64/start-here-ubuntu");
    public static final AntuIcon PLACE_STOCK_FOLDER_22 = create("PLACE_STOCK_FOLDER_22", "places/22/stock_folder");
    public static final AntuIcon PLACE_STOCK_FOLDER_32 = create("PLACE_STOCK_FOLDER_32", "places/32/stock_folder");
    public static final AntuIcon PLACE_STOCK_FOLDER_64 = create("PLACE_STOCK_FOLDER_64", "places/64/stock_folder");
    public static final AntuIcon PLACE_USER_DESKTOP_22 = create("PLACE_USER_DESKTOP_22", "places/22/user-desktop");
    public static final AntuIcon PLACE_USER_DESKTOP_32 = create("PLACE_USER_DESKTOP_32", "places/32/user-desktop");
    public static final AntuIcon PLACE_USER_DESKTOP_64 = create("PLACE_USER_DESKTOP_64", "places/64/user-desktop");
    public static final AntuIcon PLACE_USER_HOME_22 = create("PLACE_USER_HOME_22", "places/22/user-home");
    public static final AntuIcon PLACE_USER_HOME_32 = create("PLACE_USER_HOME_32", "places/32/user-home");
    public static final AntuIcon PLACE_USER_HOME_64 = create("PLACE_USER_HOME_64", "places/64/user-home");
    public static final AntuIcon PLACE_USER_TRASH_22 = create("PLACE_USER_TRASH_22", "places/22/user-trash");
    public static final AntuIcon PLACE_USER_TRASH_32 = create("PLACE_USER_TRASH_32", "places/32/user-trash");
    public static final AntuIcon PLACE_USER_TRASH_64 = create("PLACE_USER_TRASH_64", "places/64/user-trash");
    public static final AntuIcon PLACE_USER_TRASH_FULL_22 = create("PLACE_USER_TRASH_FULL_22", "places/22/user-trash-full");
    public static final AntuIcon PLACE_USER_TRASH_FULL_32 = create("PLACE_USER_TRASH_FULL_32", "places/32/user-trash-full");
    public static final AntuIcon PLACE_USER_TRASH_FULL_64 = create("PLACE_USER_TRASH_FULL_64", "places/64/user-trash-full");

    // STATUS
    public static final AntuIcon STATUS_APPOINTMENT_RECURRING_64 = create("STATUS_APPOINTMENT_RECURRING_64", "status/64/appointment-recurring");
    public static final AntuIcon STATUS_APPOINTMENT_REMINDER_64 = create("STATUS_APPOINTMENT_REMINDER_64", "status/64/appointment-reminder");
    public static final AntuIcon STATUS_AUDIO_READY_22 = create("STATUS_AUDIO_READY_22", "status/22/audio-ready");
    public static final AntuIcon STATUS_AUDIO_VOLUME_HIGH_22 = create("STATUS_AUDIO_VOLUME_HIGH_22", "status/22/audio-volume-high");
    public static final AntuIcon STATUS_AUDIO_VOLUME_LOW_22 = create("STATUS_AUDIO_VOLUME_LOW_22", "status/22/audio-volume-low");
    public static final AntuIcon STATUS_AUDIO_VOLUME_MEDIUM_22 = create("STATUS_AUDIO_VOLUME_MEDIUM_22", "status/22/audio-volume-medium");
    public static final AntuIcon STATUS_AUDIO_VOLUME_MUTED_22 = create("STATUS_AUDIO_VOLUME_MUTED_22", "status/22/audio-volume-muted");
    public static final AntuIcon STATUS_BATTERY_040_64 = create("STATUS_BATTERY_040_64", "status/64/battery-040");
    public static final AntuIcon STATUS_BATTERY_060_64 = create("STATUS_BATTERY_060_64", "status/64/battery-060");
    public static final AntuIcon STATUS_BATTERY_080_64 = create("STATUS_BATTERY_080_64", "status/64/battery-080");
    public static final AntuIcon STATUS_BATTERY_100_64 = create("STATUS_BATTERY_100_64", "status/64/battery-100");
    public static final AntuIcon STATUS_BATTERY_CAUTION_64 = create("STATUS_BATTERY_CAUTION_64", "status/64/battery-caution");
    public static final AntuIcon STATUS_BATTERY_CHARGING_040_64 = create("STATUS_BATTERY_CHARGING_040_64", "status/64/battery-charging-040");
    public static final AntuIcon STATUS_BATTERY_CHARGING_060_64 = create("STATUS_BATTERY_CHARGING_060_64", "status/64/battery-charging-060");
    public static final AntuIcon STATUS_BATTERY_CHARGING_64 = create("STATUS_BATTERY_CHARGING_64", "status/64/battery-charging");
    public static final AntuIcon STATUS_BATTERY_CHARGING_080_64 = create("STATUS_BATTERY_CHARGING_080_64", "status/64/battery-charging-080");
    public static final AntuIcon STATUS_BATTERY_CHARGING_CAUTION_64 = create("STATUS_BATTERY_CHARGING_CAUTION_64", "status/64/battery-charging-caution");
    public static final AntuIcon STATUS_BATTERY_CHARGING_LOW_64 = create("STATUS_BATTERY_CHARGING_LOW_64", "status/64/battery-charging-low");
    public static final AntuIcon STATUS_BATTERY_LOW_64 = create("STATUS_BATTERY_LOW_64", "status/64/battery-low");
    public static final AntuIcon STATUS_BATTERY_MISSING_64 = create("STATUS_BATTERY_MISSING_64", "status/64/battery-missing");
    public static final AntuIcon STATUS_CAMERA_OFF_22 = create("STATUS_CAMERA_OFF_22", "status/22/camera-off");
    public static final AntuIcon STATUS_CAMERA_ON_22 = create("STATUS_CAMERA_ON_22", "status/22/camera-on");
    public static final AntuIcon STATUS_CAMERA_READY_22 = create("STATUS_CAMERA_READY_22", "status/22/camera-ready");
    public static final AntuIcon STATUS_DIALOG_ERROR_64 = create("STATUS_DIALOG_ERROR_64", "status/64/dialog-error");
    public static final AntuIcon STATUS_DIALOG_INFORMATION_22 = create("STATUS_DIALOG_INFORMATION_22", "status/22/dialog-information");
    public static final AntuIcon STATUS_DIALOG_INFORMATION_64 = create("STATUS_DIALOG_INFORMATION_64", "status/64/dialog-information");
    public static final AntuIcon STATUS_DIALOG_PASSWORD_64 = create("STATUS_DIALOG_PASSWORD_64", "status/64/dialog-password");
    public static final AntuIcon STATUS_DIALOG_QUESTION_64 = create("STATUS_DIALOG_QUESTION_64", "status/64/dialog-question");
    public static final AntuIcon STATUS_DIALOG_WARNING_64 = create("STATUS_DIALOG_WARNING_64", "status/64/dialog-warning");
    public static final AntuIcon STATUS_FOLDER_CLOUD_22 = create("STATUS_FOLDER_CLOUD_22", "status/22/folder-cloud");
    public static final AntuIcon STATUS_FOLDER_OPEN_64 = create("STATUS_FOLDER_OPEN_64", "status/64/folder-open");
    public static final AntuIcon STATUS_IMAGE_LOADING_64 = create("STATUS_IMAGE_LOADING_64", "status/64/image-loading");
    public static final AntuIcon STATUS_IMAGE_MISSING_22 = create("STATUS_IMAGE_MISSING_22", "status/22/image-missing");
    public static final AntuIcon STATUS_IRC_CHANNEL_JOINED_22 = create("STATUS_IRC_CHANNEL_JOINED_22", "status/22/irc-channel-joined");
    public static final AntuIcon STATUS_IRC_CHANNEL_PARTED_22 = create("STATUS_IRC_CHANNEL_PARTED_22", "status/22/irc-channel-parted");
    public static final AntuIcon STATUS_KDECONNECT_TRAY_22 = create("STATUS_KDECONNECT_TRAY_22", "status/22/kdeconnect-tray");
    public static final AntuIcon STATUS_LAPTOP_CONNECTED_22 = create("STATUS_LAPTOP_CONNECTED_22", "status/22/laptop-connected");
    public static final AntuIcon STATUS_LAPTOP_DISCONNECTED_22 = create("STATUS_LAPTOP_DISCONNECTED_22", "status/22/laptop-disconnected");
    public static final AntuIcon STATUS_LAPTOP_TRUSTED_22 = create("STATUS_LAPTOP_TRUSTED_22", "status/22/laptop-trusted");
    public static final AntuIcon STATUS_MAIL_ATTACHMENT_22 = create("STATUS_MAIL_ATTACHMENT_22", "status/22/mail-attachment");
    public static final AntuIcon STATUS_MAIL_QUEUED_64 = create("STATUS_MAIL_QUEUED_64", "status/64/mail-queued");
    public static final AntuIcon STATUS_MAIL_READ_64 = create("STATUS_MAIL_READ_64", "status/64/mail-read");
    public static final AntuIcon STATUS_MAIL_REPLIED_64 = create("STATUS_MAIL_REPLIED_64", "status/64/mail-replied");
    public static final AntuIcon STATUS_MAIL_SENT_64 = create("STATUS_MAIL_SENT_64", "status/64/mail-sent");
    public static final AntuIcon STATUS_MAIL_TAGGED_64 = create("STATUS_MAIL_TAGGED_64", "status/64/mail-tagged");
    public static final AntuIcon STATUS_MAIL_TASK_64 = create("STATUS_MAIL_TASK_64", "status/64/mail-task");
    public static final AntuIcon STATUS_MAIL_UNREAD_64 = create("STATUS_MAIL_UNREAD_64", "status/64/mail-unread");
    public static final AntuIcon STATUS_MAIL_UNREAD_NEW_64 = create("STATUS_MAIL_UNREAD_NEW_64", "status/64/mail-unread-new");
    public static final AntuIcon STATUS_MEDIA_PLAYLIST_REPEAT_64 = create("STATUS_MEDIA_PLAYLIST_REPEAT_64", "status/64/media-playlist-repeat");
    public static final AntuIcon STATUS_MEDIA_PLAYLIST_SHUFFLE_64 = create("STATUS_MEDIA_PLAYLIST_SHUFFLE_64", "status/64/media-playlist-shuffle");
    public static final AntuIcon STATUS_MEETING_CHAIR_64 = create("STATUS_MEETING_CHAIR_64", "status/64/meeting-chair");
    public static final AntuIcon STATUS_MEETING_OBSERVER_64 = create("STATUS_MEETING_OBSERVER_64", "status/64/meeting-observer");
    public static final AntuIcon STATUS_MEETING_ORGANIZER_64 = create("STATUS_MEETING_ORGANIZER_64", "status/64/meeting-organizer");
    public static final AntuIcon STATUS_MEETING_PARTICIPANT_64 = create("STATUS_MEETING_PARTICIPANT_64", "status/64/meeting-participant");
    public static final AntuIcon STATUS_MEETING_PARTICIPANT_OPTIONAL_64 = create("STATUS_MEETING_PARTICIPANT_OPTIONAL_64", "status/64/meeting-participant-optional");
    public static final AntuIcon STATUS_MEETING_PARTICIPANT_REPLY_64 = create("STATUS_MEETING_PARTICIPANT_REPLY_64", "status/64/meeting-participant-reply");
    public static final AntuIcon STATUS_MIC_OFF_22 = create("STATUS_MIC_OFF_22", "status/22/mic-off");
    public static final AntuIcon STATUS_MIC_ON_22 = create("STATUS_MIC_ON_22", "status/22/mic-on");
    public static final AntuIcon STATUS_MIC_READY_22 = create("STATUS_MIC_READY_22", "status/22/mic-ready");
    public static final AntuIcon STATUS_OBJECT_LOCKED_64 = create("STATUS_OBJECT_LOCKED_64", "status/64/object-locked");
    public static final AntuIcon STATUS_OBJECT_UNLOCKED_64 = create("STATUS_OBJECT_UNLOCKED_64", "status/64/object-unlocked");
    public static final AntuIcon STATUS_PRINTER_ERROR_64 = create("STATUS_PRINTER_ERROR_64", "status/64/printer-error");
    public static final AntuIcon STATUS_PRINTER_PRINTING_64 = create("STATUS_PRINTER_PRINTING_64", "status/64/printer-printing");
    public static final AntuIcon STATUS_SCRIPT_ERROR_22 = create("STATUS_SCRIPT_ERROR_22", "status/22/script-error");
    public static final AntuIcon STATUS_SECURITY_HIGH_22 = create("STATUS_SECURITY_HIGH_22", "status/22/security-high");
    public static final AntuIcon STATUS_SECURITY_HIGH_64 = create("STATUS_SECURITY_HIGH_64", "status/64/security-high");
    public static final AntuIcon STATUS_SECURITY_LOW_22  = create("STATUS_SECURITY_LOW_22", "status/22/security-low");
    public static final AntuIcon STATUS_SECURITY_LOW_64 = create("STATUS_SECURITY_LOW_64", "status/64/security-low");
    public static final AntuIcon STATUS_SECURITY_MEDIUM_22 = create("STATUS_SECURITY_MEDIUM_22", "status/22/security-medium");
    public static final AntuIcon STATUS_SECURITY_MEDIUM_64 = create("STATUS_SECURITY_MEDIUM_64", "status/64/security-medium");
    public static final AntuIcon STATUS_SMARTPHONE_CONNECTED_22 = create("STATUS_SMARTPHONE_CONNECTED_22", "status/22/smartphone-connected");
    public static final AntuIcon STATUS_SMARTPHONE_DISCONNECTED_22 = create("STATUS_SMARTPHONE_DISCONNECTED_22", "status/22/smartphone-disconnected");
    public static final AntuIcon STATUS_SMARTPHONE_TRUSTED_22 = create("STATUS_SMARTPHONE_TRUSTED_22", "status/22/smartphone-trusted");
    public static final AntuIcon STATUS_STATE_BUSY_22 = create("STATUS_STATE_BUSY_22", "status/22/dropboxstatus-busy");
    public static final AntuIcon STATUS_STATE_DOWNLOAD_22 = create("STATUS_STATE_DOWNLOAD_22", "status/22/state-download");
    public static final AntuIcon STATUS_STATE_ERROR_22 = create("STATUS_STATE_ERROR_22", "status/22/dropboxstatus-x");
    public static final AntuIcon STATUS_STATE_IDLE_22 = create("STATUS_STATE_IDLE_22", "status/22/dropboxstatus-idle");
    public static final AntuIcon STATUS_STATE_INFORMATION_22 = create("STATUS_STATE_INFORMATION_22", "status/22/state-information");
    public static final AntuIcon STATUS_STATE_OFFLINE_22 = create("STATUS_STATE_OFFLINE_22", "status/22/state-offline");
    public static final AntuIcon STATUS_STATE_PAUSE_22 = create("STATUS_STATE_PAUSE_22", "status/22/state-pause");
    public static final AntuIcon STATUS_STATE_WARNING_22 = create("STATUS_STATE_WARNING_22", "status/22/state-warning");
    public static final AntuIcon STATUS_TABLET_CONNECTED_22 = create("STATUS_TABLET_CONNECTED_22", "status/22/tablet-connected");
    public static final AntuIcon STATUS_TABLET_DISCONNECTED_22 = create("STATUS_TABLET_DISCONNECTED_22", "status/22/tablet-disconnected");
    public static final AntuIcon STATUS_TABLET_TRUSTED_22 = create("STATUS_TABLET_TRUSTED_22", "status/22/tablet-trusted");
    public static final AntuIcon STATUS_TASK_ACCEPTED_64 = create("STATUS_TASK_ACCEPTED_64", "status/64/task-accepted");
    public static final AntuIcon STATUS_TASK_ATTEMPT_64 = create("STATUS_TASK_ATTEMPT_64", "status/64/task-attempt");
    public static final AntuIcon STATUS_TASK_ATTENTION_64 = create("STATUS_TASK_ATTENTION_64", "status/64/task-attention");
    public static final AntuIcon STATUS_TASK_COMPLETE_64 = create("STATUS_TASK_COMPLETE_64", "status/64/task-complete");
    public static final AntuIcon STATUS_TASK_DELEGATE_64 = create("STATUS_TASK_DELEGATE_64", "status/64/task-delegate");
    public static final AntuIcon STATUS_TASK_ONGOING_64 = create("STATUS_TASK_ONGOING_64", "status/64/task-ongoing");
    public static final AntuIcon STATUS_TASK_RECURRING_64 = create("STATUS_TASK_RECURRING_64", "status/64/task-recurring");
    public static final AntuIcon STATUS_TASK_REJECT_64 = create("STATUS_TASK_REJECT_64", "status/64/task-reject");
    public static final AntuIcon STATUS_TASK_REMINDER_64 = create("STATUS_TASK_REMINDER_64", "status/64/task-reminder");
    public static final AntuIcon STATUS_USER_AVAILABLE_22 = create("STATUS_USER_AVAILABLE_22", "status/22/user-available");
    public static final AntuIcon STATUS_USER_AWAY_22 = create("STATUS_USER_AWAY_22", "status/22/user-away");
    public static final AntuIcon STATUS_USER_AWAY_EXTENDED_22 = create("STATUS_USER_AWAY_EXTENDED_22", "status/22/user-away-extended");
    public static final AntuIcon STATUS_USER_BUSY_22 = create("STATUS_USER_BUSY_22", "status/22/user-busy");
    public static final AntuIcon STATUS_USER_IDLE_22 = create("STATUS_USER_IDLE_22", "status/22/user-idle");
    public static final AntuIcon STATUS_USER_INVISIBLE_22 = create("STATUS_USER_INVISIBLE_22", "status/22/user-invisible");
    public static final AntuIcon STATUS_USER_OFFLINE_22 = create("STATUS_USER_OFFLINE_22", "status/22/user-offline");
    public static final AntuIcon STATUS_USER_ONLINE_22 = create("STATUS_USER_ONLINE_22", "status/22/user-online");
    public static final AntuIcon STATUS_USER_TRASH_FULL_64 = create("STATUS_USER_TRASH_FULL_64", "status/64/user-trash-full");
    public static final AntuIcon STATUS_WALLET_CLOSED_64 = create("STATUS_WALLET_CLOSED_64", "status/64/wallet-closed");
    public static final AntuIcon STATUS_WALLET_OPEN_64 = create("STATUS_WALLET_OPEN_64", "status/64/wallet-open");
    public static final AntuIcon STATUS_WEATHER_CLEAR_64 = create("STATUS_WEATHER_CLEAR_64", "status/64/weather-clear");
    public static final AntuIcon STATUS_WEATHER_CLEAR_NIGHT_64 = create("STATUS_WEATHER_CLEAR_NIGHT_64", "status/64/weather-clear-night");
    public static final AntuIcon STATUS_WEATHER_CLOUDS_64 = create("STATUS_WEATHER_CLOUDS_64", "status/64/weather-clouds");
    public static final AntuIcon STATUS_WEATHER_CLOUDS_NIGHT_64 = create("STATUS_WEATHER_CLOUDS_NIGHT_64", "status/64/weather-clouds-night");
    public static final AntuIcon STATUS_WEATHER_FEW_CLOUDS_64 = create("STATUS_WEATHER_FEW_CLOUDS_64", "status/64/weather-few-clouds");
    public static final AntuIcon STATUS_WEATHER_FEW_CLOUDS_NIGHT_64 = create("STATUS_WEATHER_FEW_CLOUDS_NIGHT_64", "status/64/weather-few-clouds-night");
    public static final AntuIcon STATUS_WEATHER_FREEZING_RAIN_64 = create("STATUS_WEATHER_FREEZING_RAIN_64", "status/64/weather-freezing-rain");
    public static final AntuIcon STATUS_WEATHER_HAIL_64 = create("STATUS_WEATHER_HAIL_64", "status/64/weather-hail");
    public static final AntuIcon STATUS_WEATHER_MANY_CLOUDS_64 = create("STATUS_WEATHER_MANY_CLOUDS_64", "status/64/weather-many-clouds");
    public static final AntuIcon STATUS_WEATHER_MIST_64 = create("STATUS_WEATHER_MIST_64", "status/64/weather-mist");
    public static final AntuIcon STATUS_WEATHER_NONE_AVAILABLE_64 = create("STATUS_WEATHER_NONE_AVAILABLE_64", "status/64/weather-none-available");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_64 = create("STATUS_WEATHER_SHOWERS_64", "status/64/weather-showers");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_DAY_64 = create("STATUS_WEATHER_SHOWERS_DAY_64", "status/64/weather-showers-day");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_NIGHT_64 = create("STATUS_WEATHER_SHOWERS_NIGHT_64", "status/64/weather-showers-night");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_SCATTERED_64 = create("STATUS_WEATHER_SHOWERS_SCATTERED_64", "status/64/weather-showers-scattered");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_SCATTERED_DAY_64 = create("STATUS_WEATHER_SHOWERS_SCATTERED_DAY_64", "status/64/weather-showers-scattered-day");
    public static final AntuIcon STATUS_WEATHER_SHOWERS_SCATTERED_NIGHT_64 = create("STATUS_WEATHER_SHOWERS_SCATTERED_NIGHT_64", "status/64/weather-showers-scattered-night");
    public static final AntuIcon STATUS_WEATHER_SNOW_64 = create("STATUS_WEATHER_SNOW_64", "status/64/weather-snow");
    public static final AntuIcon STATUS_WEATHER_SNOW_RAIN_64 = create("STATUS_WEATHER_SNOW_RAIN_64", "status/64/weather-snow-rain");
    public static final AntuIcon STATUS_WEATHER_SNOW_SCATTERED_64 = create("STATUS_WEATHER_SNOW_SCATTERED_64", "status/64/weather-snow-scattered");
    public static final AntuIcon STATUS_WEATHER_SNOW_SCATTERED_DAY_64 = create("STATUS_WEATHER_SNOW_SCATTERED_DAY_64", "status/64/weather-snow-scattered-day");
    public static final AntuIcon STATUS_WEATHER_SNOW_SCATTERED_NIGHT_64 = create("STATUS_WEATHER_SNOW_SCATTERED_NIGHT_64", "status/64/weather-snow-scattered-night");
    public static final AntuIcon STATUS_WEATHER_STORM_64 = create("STATUS_WEATHER_STORM_64", "status/64/weather-storm");
    public static final AntuIcon STATUS_WEATHER_STORM_DAY_64 = create("STATUS_WEATHER_STORM_DAY_64", "status/64/weather-storm-day");
    public static final AntuIcon STATUS_WEATHER_STORM_NIGHT_64 = create("STATUS_WEATHER_STORM_NIGHT_64", "status/64/weather-storm-night");
    public static final AntuIcon STATUS_WINDOW_SUPPRESSED_64 = create("STATUS_WINDOW_SUPPRESSED_64", "status/64/window-suppressed");

    // ACTIONS
    public static final AntuIcon ACTION_ACCEPT_SIGNAL_24 = create("ACTION_ACCEPT_SIGNAL_24", "actions/24/accept_signal");
    public static final AntuIcon ACTION_ACROBAT_24 = create("ACTION_ACROBAT_24", "actions/24/acrobat");
    public static final AntuIcon ACTION_ACROBAT_32 = create("ACTION_ACROBAT_32", "actions/32/acrobat");
    public static final AntuIcon ACTION_ACTION_ALBUMFOLDER_IMPORTDIR2_24 = create("ACTION_ACTION_ALBUMFOLDER_IMPORTDIR2_24", "actions/24/action-albumfolder-importdir2");
    public static final AntuIcon ACTION_ACTION_RSS_TAG_24 = create("ACTION_ACTION_RSS_TAG_24", "actions/24/action-rss_tag");
    public static final AntuIcon ACTION_ADDRESS_BOOK_NEW_24 = create("ACTION_ADDRESS_BOOK_NEW_24", "actions/24/address-book-new");
    public static final AntuIcon ACTION_ADDRESS_BOOK_NEW_32 = create("ACTION_ADDRESS_BOOK_NEW_32", "actions/32/address-book-new");
    public static final AntuIcon ACTION_ADJUSTCURVES_24 = create("ACTION_ADJUSTCURVES_24", "actions/24/adjustcurves");
    public static final AntuIcon ACTION_ADJUSTHSL_24 = create("ACTION_ADJUSTHSL_24", "actions/24/adjusthsl");
    public static final AntuIcon ACTION_ADJUSTLEVELS_24 = create("ACTION_ADJUSTLEVELS_24", "actions/24/adjustlevels");
    public static final AntuIcon ACTION_ADJUSTRGB_24 = create("ACTION_ADJUSTRGB_24", "actions/24/adjustrgb");
    public static final AntuIcon ACTION_ADRESS_BOOK_NEW_24 = create("ACTION_ADRESS_BOOK_NEW_24", "actions/24/adress-book-new");
    public static final AntuIcon ACTION_AGGREGATION_24 = create("ACTION_AGGREGATION_24", "actions/24/aggregation");
    public static final AntuIcon ACTION_AKONADI_PHONE_HOME_24 = create("ACTION_AKONADI_PHONE_HOME_24", "actions/24/akonadi-phone-home");
    public static final AntuIcon ACTION_AKONADICONSOLE_24 = create("ACTION_AKONADICONSOLE_24", "actions/24/akonadiconsole");
    public static final AntuIcon ACTION_ALBUMFOLDER_IMPORTDIR_24 = create("ACTION_ALBUMFOLDER_IMPORTDIR_24", "actions/24/albumfolder-importdir");
    public static final AntuIcon ACTION_ALBUMFOLDER_IMPORTIMAGES_24 = create("ACTION_ALBUMFOLDER_IMPORTIMAGES_24", "actions/24/albumfolder-importimages");
    public static final AntuIcon ACTION_ALBUMFOLDER_NEW_24 = create("ACTION_ALBUMFOLDER_NEW_24", "actions/24/albumfolder-new");
    public static final AntuIcon ACTION_ALBUMFOLDER_PROPERTIES_24 = create("ACTION_ALBUMFOLDER_PROPERTIES_24", "actions/24/albumfolder-properties");
    public static final AntuIcon ACTION_ALBUMFOLDER_USER_TRASH_24 = create("ACTION_ALBUMFOLDER_USER_TRASH_24", "actions/24/albumfolder-user-trash");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_BASELINE_24 = create("ACTION_ALIGN_HORIZONTAL_BASELINE_24", "actions/24/align-horizontal-baseline");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_BOTTOM_OUT_24 = create("ACTION_ALIGN_HORIZONTAL_BOTTOM_OUT_24", "actions/24/align-horizontal-bottom-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_BOTTOM_OUT_32 = create("ACTION_ALIGN_HORIZONTAL_BOTTOM_OUT_32", "actions/32/align-horizontal-bottom-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_CENTER_24 = create("ACTION_ALIGN_HORIZONTAL_CENTER_24", "actions/24/align-horizontal-center");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_CENTER_32 = create("ACTION_ALIGN_HORIZONTAL_CENTER_32", "actions/32/align-horizontal-center");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_24 = create("ACTION_ALIGN_HORIZONTAL_LEFT_24", "actions/24/align-horizontal-left");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_32 = create("ACTION_ALIGN_HORIZONTAL_LEFT_32", "actions/32/align-horizontal-left");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_OUT_24 = create("ACTION_ALIGN_HORIZONTAL_LEFT_OUT_24", "actions/24/align-horizontal-left-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_OUT_32 = create("ACTION_ALIGN_HORIZONTAL_LEFT_OUT_32", "actions/32/align-horizontal-left-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_TO_ANCHOR_24 = create("ACTION_ALIGN_HORIZONTAL_LEFT_TO_ANCHOR_24", "actions/24/align-horizontal-left-to-anchor");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_LEFT_TO_ANCHOR_32 = create("ACTION_ALIGN_HORIZONTAL_LEFT_TO_ANCHOR_32", "actions/32/align-horizontal-left-to-anchor");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_24 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_24", "actions/24/align-horizontal-right");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_32 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_32", "actions/32/align-horizontal-right");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_OUT_24 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_OUT_24", "actions/24/align-horizontal-right-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_OUT_32 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_OUT_32", "actions/32/align-horizontal-right-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_TO_ANCHOR_24 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_TO_ANCHOR_24", "actions/24/align-horizontal-right-to-anchor");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_RIGHT_TO_ANCHOR_32 = create("ACTION_ALIGN_HORIZONTAL_RIGHT_TO_ANCHOR_32", "actions/32/align-horizontal-right-to-anchor");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_TOP_OUT_24 = create("ACTION_ALIGN_HORIZONTAL_TOP_OUT_24", "actions/24/align-horizontal-top-out");
    public static final AntuIcon ACTION_ALIGN_HORIZONTAL_TOP_OUT_32 = create("ACTION_ALIGN_HORIZONTAL_TOP_OUT_32", "actions/32/align-horizontal-top-out");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BASELINE_24 = create("ACTION_ALIGN_VERTICAL_BASELINE_24", "actions/24/align-vertical-baseline");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_24 = create("ACTION_ALIGN_VERTICAL_BOTTOM_24", "actions/24/align-vertical-bottom");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_32 = create("ACTION_ALIGN_VERTICAL_BOTTOM_32", "actions/32/align-vertical-bottom");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_OUT_24 = create("ACTION_ALIGN_VERTICAL_BOTTOM_OUT_24", "actions/24/align-vertical-bottom-out");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_OUT_32 = create("ACTION_ALIGN_VERTICAL_BOTTOM_OUT_32", "actions/32/align-vertical-bottom-out");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_TO_ANCHOR_24 = create("ACTION_ALIGN_VERTICAL_BOTTOM_TO_ANCHOR_24", "actions/24/align-vertical-bottom-to-anchor");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_BOTTOM_TO_ANCHOR_32 = create("ACTION_ALIGN_VERTICAL_BOTTOM_TO_ANCHOR_32", "actions/32/align-vertical-bottom-to-anchor");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_CENTER_24 = create("ACTION_ALIGN_VERTICAL_CENTER_24", "actions/24/align-vertical-center");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_CENTER_32 = create("ACTION_ALIGN_VERTICAL_CENTER_32", "actions/32/align-vertical-center");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_24 = create("ACTION_ALIGN_VERTICAL_TOP_24", "actions/24/align-vertical-top");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_32 = create("ACTION_ALIGN_VERTICAL_TOP_32", "actions/32/align-vertical-top");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_OUT_24 = create("ACTION_ALIGN_VERTICAL_TOP_OUT_24", "actions/24/align-vertical-top-out");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_OUT_32 = create("ACTION_ALIGN_VERTICAL_TOP_OUT_32", "actions/32/align-vertical-top-out");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_TO_ANCHOR_24 = create("ACTION_ALIGN_VERTICAL_TOP_TO_ANCHOR_24", "actions/24/align-vertical-top-to-anchor");
    public static final AntuIcon ACTION_ALIGN_VERTICAL_TOP_TO_ANCHOR_32 = create("ACTION_ALIGN_VERTICAL_TOP_TO_ANCHOR_32", "actions/32/align-vertical-top-to-anchor");
    public static final AntuIcon ACTION_AMAROK_ARTIST_24 = create("ACTION_AMAROK_ARTIST_24", "actions/24/amarok_artist");
    public static final AntuIcon ACTION_AMAROK_CART_ADD_24 = create("ACTION_AMAROK_CART_ADD_24", "actions/24/amarok_cart_add");
    public static final AntuIcon ACTION_AMAROK_CART_REMOVE_24 = create("ACTION_AMAROK_CART_REMOVE_24", "actions/24/amarok_cart_remove");
    public static final AntuIcon ACTION_AMAROK_CART_VIEW_24 = create("ACTION_AMAROK_CART_VIEW_24", "actions/24/amarok_cart_view");
    public static final AntuIcon ACTION_AMAROK_CHANGE_LANGUAGE_24 = create("ACTION_AMAROK_CHANGE_LANGUAGE_24", "actions/24/amarok_change_language");
    public static final AntuIcon ACTION_AMAROK_CLOCK_24 = create("ACTION_AMAROK_CLOCK_24", "actions/24/amarok_clock");
    public static final AntuIcon ACTION_AMAROK_LYRICS_24 = create("ACTION_AMAROK_LYRICS_24", "actions/24/amarok_lyrics");
    public static final AntuIcon ACTION_AMAROK_PLAYCOUNT_24 = create("ACTION_AMAROK_PLAYCOUNT_24", "actions/24/amarok_playcount");
    public static final AntuIcon ACTION_AMAROK_PLAYLIST_24 = create("ACTION_AMAROK_PLAYLIST_24", "actions/24/amarok_playlist");
    public static final AntuIcon ACTION_AMAROK_PLAYLIST_REFRESH_24 = create("ACTION_AMAROK_PLAYLIST_REFRESH_24", "actions/24/amarok_playlist_refresh");
    public static final AntuIcon ACTION_AMAROK_SCRIPTS_24 = create("ACTION_AMAROK_SCRIPTS_24", "actions/24/amarok_scripts");
    public static final AntuIcon ACTION_ANCHOR_24 = create("ACTION_ANCHOR_24", "actions/24/anchor");
    public static final AntuIcon ACTION_ANTIVIGNETTING_24 = create("ACTION_ANTIVIGNETTING_24", "actions/24/antivignetting");
    public static final AntuIcon ACTION_APPLICATION_EXIT_24 = create("ACTION_APPLICATION_EXIT_24", "actions/24/application-exit");
    public static final AntuIcon ACTION_APPLICATION_MENU_24 = create("ACTION_APPLICATION_MENU_24", "actions/24/application-menu");
    public static final AntuIcon ACTION_APPLICATIONS_SYSTEM_24 = create("ACTION_APPLICATIONS_SYSTEM_24", "actions/24/applications-system");
    public static final AntuIcon ACTION_APPOINTMENT_NEW_24 = create("ACTION_APPOINTMENT_NEW_24", "actions/24/appointment-new");
    public static final AntuIcon ACTION_APPOINTMENT_NEW_32 = create("ACTION_APPOINTMENT_NEW_32", "actions/32/appointment-new");
    public static final AntuIcon ACTION_ARCHIVE_EXTRACT_24 = create("ACTION_ARCHIVE_EXTRACT_24", "actions/24/archive-extract");
    public static final AntuIcon ACTION_ARCHIVE_EXTRACT_32 = create("ACTION_ARCHIVE_EXTRACT_32", "actions/32/archive-extract");
    public static final AntuIcon ACTION_ARCHIVE_INSERT_24 = create("ACTION_ARCHIVE_INSERT_24", "actions/24/archive-insert");
    public static final AntuIcon ACTION_ARCHIVE_INSERT_32 = create("ACTION_ARCHIVE_INSERT_32", "actions/32/archive-insert");
    public static final AntuIcon ACTION_ARCHIVE_INSERT_DIRECTORY_24 = create("ACTION_ARCHIVE_INSERT_DIRECTORY_24", "actions/24/archive-insert-directory");
    public static final AntuIcon ACTION_ARCHIVE_INSERT_DIRECTORY_32 = create("ACTION_ARCHIVE_INSERT_DIRECTORY_32", "actions/32/archive-insert-directory");
    public static final AntuIcon ACTION_ARCHIVE_REMOVE_24 = create("ACTION_ARCHIVE_REMOVE_24", "actions/24/archive-remove");
    public static final AntuIcon ACTION_ARCHIVE_REMOVE_32 = create("ACTION_ARCHIVE_REMOVE_32", "actions/32/archive-remove");
    public static final AntuIcon ACTION_ARROW_DOWN_24 = create("ACTION_ARROW_DOWN_24", "actions/24/arrow-down");
    public static final AntuIcon ACTION_ARROW_DOWN_DOUBLE_24 = create("ACTION_ARROW_DOWN_DOUBLE_24", "actions/24/arrow-down-double");
    public static final AntuIcon ACTION_ARROW_LEFT_24 = create("ACTION_ARROW_LEFT_24", "actions/24/arrow-left");
    public static final AntuIcon ACTION_ARROW_LEFT_DOUBLE_24 = create("ACTION_ARROW_LEFT_DOUBLE_24", "actions/24/arrow-left-double");
    public static final AntuIcon ACTION_ARROW_LEFT_DOUBLE_32 = create("ACTION_ARROW_LEFT_DOUBLE_32", "actions/32/arrow-left-double");
    public static final AntuIcon ACTION_ARROW_RIGHT_24 = create("ACTION_ARROW_RIGHT_24", "actions/24/arrow-right");
    public static final AntuIcon ACTION_ARROW_RIGHT_DOUBLE_24 = create("ACTION_ARROW_RIGHT_DOUBLE_24", "actions/24/arrow-right-double");
    public static final AntuIcon ACTION_ARROW_RIGHT_DOUBLE_32 = create("ACTION_ARROW_RIGHT_DOUBLE_32", "actions/32/arrow-right-double");
    public static final AntuIcon ACTION_ARROW_UP_24 = create("ACTION_ARROW_UP_24", "actions/24/arrow-up");
    public static final AntuIcon ACTION_ARROW_UP_DOUBLE_24 = create("ACTION_ARROW_UP_DOUBLE_24", "actions/24/arrow-up-double");
    public static final AntuIcon ACTION_ARTIFACT_24 = create("ACTION_ARTIFACT_24", "actions/24/artifact");
    public static final AntuIcon ACTION_ASSOCIATION_24 = create("ACTION_ASSOCIATION_24", "actions/24/association");
    public static final AntuIcon ACTION_ATMOSPHERE_24 = create("ACTION_ATMOSPHERE_24", "actions/24/atmosphere");
    public static final AntuIcon ACTION_AUDIO_OFF_24 = create("ACTION_AUDIO_OFF_24", "actions/24/audio-off");
    public static final AntuIcon ACTION_AUDIO_ON_24 = create("ACTION_AUDIO_ON_24", "actions/24/audio-on");
    public static final AntuIcon ACTION_AUDIO_READY_24 = create("ACTION_AUDIO_READY_24", "actions/24/audio-ready");
    public static final AntuIcon ACTION_AUTO_SCALE_ALL_24 = create("ACTION_AUTO_SCALE_ALL_24", "actions/24/auto-scale-all");
    public static final AntuIcon ACTION_AUTO_SCALE_X_24 = create("ACTION_AUTO_SCALE_X_24", "actions/24/auto-scale-x");
    public static final AntuIcon ACTION_AUTO_SCALE_Y_24 = create("ACTION_AUTO_SCALE_Y_24", "actions/24/auto-scale-y");
    public static final AntuIcon ACTION_AUTO_TYPE_24 = create("ACTION_AUTO_TYPE_24", "actions/24/auto-type");
    public static final AntuIcon ACTION_AUTOCORRECTION_24 = create("ACTION_AUTOCORRECTION_24", "actions/24/autocorrection");
    public static final AntuIcon ACTION_AUTOCORRECTION_32 = create("ACTION_AUTOCORRECTION_32", "actions/32/autocorrection");
    public static final AntuIcon ACTION_AUTOMATED_TASKS_24 = create("ACTION_AUTOMATED_TASKS_24", "actions/24/automated-tasks");
    public static final AntuIcon ACTION_AXIS_HORIZONTAL_24 = create("ACTION_AXIS_HORIZONTAL_24", "actions/24/axis-horizontal");
    public static final AntuIcon ACTION_AXIS_VERTICAL_24 = create("ACTION_AXIS_VERTICAL_24", "actions/24/axis-vertical");
    public static final AntuIcon ACTION_BITMAP_TRACE_24 = create("ACTION_BITMAP_TRACE_24", "actions/24/bitmap-trace");
    public static final AntuIcon ACTION_BLURFX_24 = create("ACTION_BLURFX_24", "actions/24/blurfx");
    public static final AntuIcon ACTION_BLURIMAGE_24 = create("ACTION_BLURIMAGE_24", "actions/24/blurimage");
    public static final AntuIcon ACTION_BLURIMAGE_32 = create("ACTION_BLURIMAGE_32", "actions/32/blurimage");
    public static final AntuIcon ACTION_BOOKMARK_NEW_24 = create("ACTION_BOOKMARK_NEW_24", "actions/24/bookmark-new");
    public static final AntuIcon ACTION_BOOKMARK_NEW_32 = create("ACTION_BOOKMARK_NEW_32", "actions/32/bookmark-new");
    public static final AntuIcon ACTION_BOOKMARK_NEW_LIST_24 = create("ACTION_BOOKMARK_NEW_LIST_24", "actions/24/bookmark-new-list");
    public static final AntuIcon ACTION_BOOKMARK_REMOVE_24 = create("ACTION_BOOKMARK_REMOVE_24", "actions/24/bookmark-remove");
    public static final AntuIcon ACTION_BOOKMARK_REMOVE_32 = create("ACTION_BOOKMARK_REMOVE_32", "actions/32/bookmark-remove");
    public static final AntuIcon ACTION_BOOKMARK_TOOLBAR_24 = create("ACTION_BOOKMARK_TOOLBAR_24", "actions/24/bookmark-toolbar");
    public static final AntuIcon ACTION_BOOKMARKS_24 = create("ACTION_BOOKMARKS_24", "actions/24/bookmarks");
    public static final AntuIcon ACTION_BOOKMARKS_48 = create("ACTION_BOOKMARKS_48", "actions/48/bookmarks");
    public static final AntuIcon ACTION_BOOKMARKS_ORGANIZE_24 = create("ACTION_BOOKMARKS_ORGANIZE_24", "actions/24/bookmarks-organize");
    public static final AntuIcon ACTION_BORDERTOOL_24 = create("ACTION_BORDERTOOL_24", "actions/24/bordertool");
    public static final AntuIcon ACTION_BORDERTOOL_32 = create("ACTION_BORDERTOOL_32", "actions/32/bordertool");
    public static final AntuIcon ACTION_BQM_ADD_24 = create("ACTION_BQM_ADD_24", "actions/24/bqm-add");
    public static final AntuIcon ACTION_BQM_ADDQUEUE_24 = create("ACTION_BQM_ADDQUEUE_24", "actions/24/bqm-addqueue");
    public static final AntuIcon ACTION_BQM_COMMIT_24 = create("ACTION_BQM_COMMIT_24", "actions/24/bqm-commit");
    public static final AntuIcon ACTION_BQM_DIFF_24 = create("ACTION_BQM_DIFF_24", "actions/24/bqm-diff");
    public static final AntuIcon ACTION_BQM_REMOVE_24 = create("ACTION_BQM_REMOVE_24", "actions/24/bqm-remove");
    public static final AntuIcon ACTION_BQM_RMQUEUE_24 = create("ACTION_BQM_RMQUEUE_24", "actions/24/bqm-rmqueue");
    public static final AntuIcon ACTION_BQM_UPDATE_24 = create("ACTION_BQM_UPDATE_24", "actions/24/bqm-update");
    public static final AntuIcon ACTION_BWTONAL_24 = create("ACTION_BWTONAL_24", "actions/24/bwtonal");
    public static final AntuIcon ACTION_CALL_START_24 = create("ACTION_CALL_START_24", "actions/24/call-start");
    public static final AntuIcon ACTION_CALL_START_32 = create("ACTION_CALL_START_32", "actions/32/call-start");
    public static final AntuIcon ACTION_CALL_STOP_24 = create("ACTION_CALL_STOP_24", "actions/24/call-stop");
    public static final AntuIcon ACTION_CALL_STOP_32 = create("ACTION_CALL_STOP_32", "actions/32/call-stop");
    public static final AntuIcon ACTION_CAMERA_OFF_24 = create("ACTION_CAMERA_OFF_24", "actions/24/camera-off");
    public static final AntuIcon ACTION_CAMERA_ON_24 = create("ACTION_CAMERA_ON_24", "actions/24/camera-on");
    public static final AntuIcon ACTION_CAMERA_READY_24 = create("ACTION_CAMERA_READY_24", "actions/24/camera-ready");
    public static final AntuIcon ACTION_CARTESIAN_PLOT_FOUR_AXES_24 = create("ACTION_CARTESIAN_PLOT_FOUR_AXES_24", "actions/24/cartesian-plot-four-axes");
    public static final AntuIcon ACTION_CARTESIAN_PLOT_TWO_AXES_24 = create("ACTION_CARTESIAN_PLOT_TWO_AXES_24", "actions/24/cartesian-plot-two-axes");
    public static final AntuIcon ACTION_CARTESIAN_PLOT_TWO_AXES_CENTERED_24 = create("ACTION_CARTESIAN_PLOT_TWO_AXES_CENTERED_24", "actions/24/cartesian-plot-two-axes-centered");
    public static final AntuIcon ACTION_CARTESIAN_PLOT_TWO_AXES_CENTERED_ORIGIN_24 = create("ACTION_CARTESIAN_PLOT_TWO_AXES_CENTERED_ORIGIN_24", "actions/24/cartesian-plot-two-axes-centered-origin");
    public static final AntuIcon ACTION_CATEGORY2PARENT_24 = create("ACTION_CATEGORY2PARENT_24", "actions/24/category2parent");
    public static final AntuIcon ACTION_CATEGORY_24 = create("ACTION_CATEGORY_24", "actions/24/category");
    public static final AntuIcon ACTION_CHANNELMIXER_24 = create("ACTION_CHANNELMIXER_24", "actions/24/channelmixer");
    public static final AntuIcon ACTION_CHARACTER_SET_24 = create("ACTION_CHARACTER_SET_24", "actions/24/character-set");
    public static final AntuIcon ACTION_CHARCOALTOOL_24 = create("ACTION_CHARCOALTOOL_24", "actions/24/charcoaltool");
    public static final AntuIcon ACTION_CHECK_CONSTRAINT_24 = create("ACTION_CHECK_CONSTRAINT_24", "actions/24/check_constraint");
    public static final AntuIcon ACTION_CHECKBOX_24 = create("ACTION_CHECKBOX_24", "actions/24/checkbox");
    public static final AntuIcon ACTION_CHECKMARK_24 = create("ACTION_CHECKMARK_24", "actions/24/checkmark");
    public static final AntuIcon ACTION_CHILD2CATEGORY_24 = create("ACTION_CHILD2CATEGORY_24", "actions/24/child2category");
    public static final AntuIcon ACTION_CHOICE_RHOMB_24 = create("ACTION_CHOICE_RHOMB_24", "actions/24/choice-rhomb");
    public static final AntuIcon ACTION_CHOICE_ROUND_24 = create("ACTION_CHOICE_ROUND_24", "actions/24/choice-round");
    public static final AntuIcon ACTION_CHRONOMETER_24 = create("ACTION_CHRONOMETER_24", "actions/24/chronometer");
    public static final AntuIcon ACTION_CHRONOMETER_48 = create("ACTION_CHRONOMETER_48", "actions/48/chronometer");
    public static final AntuIcon ACTION_CHRONOMETER_LAP_24 = create("ACTION_CHRONOMETER_LAP_24", "actions/24/chronometer-lap");
    public static final AntuIcon ACTION_CHRONOMETER_PAUSE_24 = create("ACTION_CHRONOMETER_PAUSE_24", "actions/24/chronometer-pause");
    public static final AntuIcon ACTION_CHRONOMETER_RESET_24 = create("ACTION_CHRONOMETER_RESET_24", "actions/24/chronometer-reset");
    public static final AntuIcon ACTION_CHRONOMETER_START_24 = create("ACTION_CHRONOMETER_START_24", "actions/24/chronometer-start");
    public static final AntuIcon ACTION_CODE_BLOCK_24 = create("ACTION_CODE_BLOCK_24", "actions/24/code-block");
    public static final AntuIcon ACTION_CODE_CLASS_24 = create("ACTION_CODE_CLASS_24", "actions/24/code-class");
    public static final AntuIcon ACTION_CODE_CONTEXT_24 = create("ACTION_CODE_CONTEXT_24", "actions/24/code-context");
    public static final AntuIcon ACTION_CODE_FUNCTION_24 = create("ACTION_CODE_FUNCTION_24", "actions/24/code-function");
    public static final AntuIcon ACTION_CODE_TYPEDEF_24 = create("ACTION_CODE_TYPEDEF_24", "actions/24/code-typedef");
    public static final AntuIcon ACTION_CODE_VARIABLE_24 = create("ACTION_CODE_VARIABLE_24", "actions/24/code-variable");
    public static final AntuIcon ACTION_COLLECTION_RESCAN_AMAROK_24 = create("ACTION_COLLECTION_RESCAN_AMAROK_24", "actions/24/collection-rescan-amarok");
    public static final AntuIcon ACTION_COLOR_FILL_24 = create("ACTION_COLOR_FILL_24", "actions/24/color-fill");
    public static final AntuIcon ACTION_COLOR_GRADIENT_24 = create("ACTION_COLOR_GRADIENT_24", "actions/24/color-gradient");
    public static final AntuIcon ACTION_COLOR_MANAGEMENT_24 = create("ACTION_COLOR_MANAGEMENT_24", "actions/24/color-management");
    public static final AntuIcon ACTION_COLOR_MANAGEMENT_32 = create("ACTION_COLOR_MANAGEMENT_32", "actions/32/color-management");
    public static final AntuIcon ACTION_COLOR_PICKER_24 = create("ACTION_COLOR_PICKER_24", "actions/24/color-picker");
    public static final AntuIcon ACTION_COLOR_PICKER_32 = create("ACTION_COLOR_PICKER_32", "actions/32/color-picker");
    public static final AntuIcon ACTION_COLOR_PICKER_BLACK_24 = create("ACTION_COLOR_PICKER_BLACK_24", "actions/24/color-picker-black");
    public static final AntuIcon ACTION_COLOR_PICKER_BLACK_32 = create("ACTION_COLOR_PICKER_BLACK_32", "actions/32/color-picker-black");
    public static final AntuIcon ACTION_COLOR_PICKER_GREY_24 = create("ACTION_COLOR_PICKER_GREY_24", "actions/24/color-picker-grey");
    public static final AntuIcon ACTION_COLOR_PICKER_GREY_32 = create("ACTION_COLOR_PICKER_GREY_32", "actions/32/color-picker-grey");
    public static final AntuIcon ACTION_COLOR_PICKER_WHITE_24 = create("ACTION_COLOR_PICKER_WHITE_24", "actions/24/color-picker-white");
    public static final AntuIcon ACTION_COLOR_PICKER_WHITE_32 = create("ACTION_COLOR_PICKER_WHITE_32", "actions/32/color-picker-white");
    public static final AntuIcon ACTION_COLORFX_24 = create("ACTION_COLORFX_24", "actions/24/colorfx");
    public static final AntuIcon ACTION_COLORMANAGEMENT_24 = create("ACTION_COLORMANAGEMENT_24", "actions/24/colormanagement");
    public static final AntuIcon ACTION_COLORNEG_24 = create("ACTION_COLORNEG_24", "actions/24/colorneg");
    public static final AntuIcon ACTION_COLORS_CHROMABLUE_24 = create("ACTION_COLORS_CHROMABLUE_24", "actions/24/colors-chromablue");
    public static final AntuIcon ACTION_COLORS_CHROMABLUE_32 = create("ACTION_COLORS_CHROMABLUE_32", "actions/32/colors-chromablue");
    public static final AntuIcon ACTION_COLORS_CHROMAGREEN_24 = create("ACTION_COLORS_CHROMAGREEN_24", "actions/24/colors-chromagreen");
    public static final AntuIcon ACTION_COLORS_CHROMAGREEN_32 = create("ACTION_COLORS_CHROMAGREEN_32", "actions/32/colors-chromagreen");
    public static final AntuIcon ACTION_COLORS_CHROMARED_24 = create("ACTION_COLORS_CHROMARED_24", "actions/24/colors-chromared");
    public static final AntuIcon ACTION_COLORS_CHROMARED_32 = create("ACTION_COLORS_CHROMARED_32", "actions/32/colors-chromared");
    public static final AntuIcon ACTION_COLORS_LUMA_24 = create("ACTION_COLORS_LUMA_24", "actions/24/colors-luma");
    public static final AntuIcon ACTION_COLORS_LUMA_32 = create("ACTION_COLORS_LUMA_32", "actions/32/colors-luma");
    public static final AntuIcon ACTION_COMBINED_FRAGMENT_24 = create("ACTION_COMBINED_FRAGMENT_24", "actions/24/combined_fragment");
    public static final AntuIcon ACTION_COMPASS_24 = create("ACTION_COMPASS_24", "actions/24/compass");
    public static final AntuIcon ACTION_COMPASS_32 = create("ACTION_COMPASS_32", "actions/32/compass");
    public static final AntuIcon ACTION_COMPOSITION_24 = create("ACTION_COMPOSITION_24", "actions/24/composition");
    public static final AntuIcon ACTION_CONFIGURE_24 = create("ACTION_CONFIGURE_24", "actions/24/configure");
    public static final AntuIcon ACTION_CONFIGURE_32 = create("ACTION_CONFIGURE_32", "actions/32/configure");
    public static final AntuIcon ACTION_CONFIGURE_SHORTCUTS_24 = create("ACTION_CONFIGURE_SHORTCUTS_24", "actions/24/configure-shortcuts");
    public static final AntuIcon ACTION_CONFIGURE_TOOLBARS_24 = create("ACTION_CONFIGURE_TOOLBARS_24", "actions/24/configure-toolbars");
    public static final AntuIcon ACTION_CONTACT_NEW_24 = create("ACTION_CONTACT_NEW_24", "actions/24/contact-new");
    public static final AntuIcon ACTION_CONTAINMENT_24 = create("ACTION_CONTAINMENT_24", "actions/24/containment");
    public static final AntuIcon ACTION_CONTRAST_24 = create("ACTION_CONTRAST_24", "actions/24/contrast");
    public static final AntuIcon ACTION_COORDINATE_24 = create("ACTION_COORDINATE_24", "actions/24/coordinate");
    public static final AntuIcon ACTION_COPY_COORDINATES_24 = create("ACTION_COPY_COORDINATES_24", "actions/24/copy-coordinates");
    public static final AntuIcon ACTION_CROSSHAIRS_24 = create("ACTION_CROSSHAIRS_24", "actions/24/crosshairs");
    public static final AntuIcon ACTION_CURSOR_ARROW_24 = create("ACTION_CURSOR_ARROW_24", "actions/24/cursor-arrow");
    public static final AntuIcon ACTION_CURVE_CONNECTOR_24 = create("ACTION_CURVE_CONNECTOR_24", "actions/24/curve-connector");
    public static final AntuIcon ACTION_DASHBOARD_SHOW_24 = create("ACTION_DASHBOARD_SHOW_24", "actions/24/dashboard-show");
    public static final AntuIcon ACTION_DATABASE_CHANGE_KEY_24 = create("ACTION_DATABASE_CHANGE_KEY_24", "actions/24/database-change-key");
    public static final AntuIcon ACTION_DATABASE_INDEX_24 = create("ACTION_DATABASE_INDEX_24", "actions/24/database-index");
    public static final AntuIcon ACTION_DEBUG_EXECUTE_FROM_CURSOR_24 = create("ACTION_DEBUG_EXECUTE_FROM_CURSOR_24", "actions/24/debug-execute-from-cursor");
    public static final AntuIcon ACTION_DEBUG_EXECUTE_TO_CURSOR_24 = create("ACTION_DEBUG_EXECUTE_TO_CURSOR_24", "actions/24/debug-execute-to-cursor");
    public static final AntuIcon ACTION_DEBUG_RUN_24 = create("ACTION_DEBUG_RUN_24", "actions/24/debug-run");
    public static final AntuIcon ACTION_DEBUG_RUN_CURSOR_24 = create("ACTION_DEBUG_RUN_CURSOR_24", "actions/24/debug-run-cursor");
    public static final AntuIcon ACTION_DEBUG_STEP_INSTRUCTION_24 = create("ACTION_DEBUG_STEP_INSTRUCTION_24", "actions/24/debug-step-instruction");
    public static final AntuIcon ACTION_DEBUG_STEP_INTO_24 = create("ACTION_DEBUG_STEP_INTO_24", "actions/24/debug-step-into");
    public static final AntuIcon ACTION_DEBUG_STEP_INTO_INSTRUCTION_24 = create("ACTION_DEBUG_STEP_INTO_INSTRUCTION_24", "actions/24/debug-step-into-instruction");
    public static final AntuIcon ACTION_DEBUG_STEP_OUT_24 = create("ACTION_DEBUG_STEP_OUT_24", "actions/24/debug-step-out");
    public static final AntuIcon ACTION_DEBUG_STEP_OVER_24 = create("ACTION_DEBUG_STEP_OVER_24", "actions/24/debug-step-over");
    public static final AntuIcon ACTION_DEEP_HISTORY_24 = create("ACTION_DEEP_HISTORY_24", "actions/24/deep-history");
    public static final AntuIcon ACTION_DELETE_COMMENT_24 = create("ACTION_DELETE_COMMENT_24", "actions/24/delete-comment");
    public static final AntuIcon ACTION_DEPENDENCY_24 = create("ACTION_DEPENDENCY_24", "actions/24/dependency");
    public static final AntuIcon ACTION_DEPTH8TO16_24 = create("ACTION_DEPTH8TO16_24", "actions/24/depth8to16");
    public static final AntuIcon ACTION_DEPTH16TO8_24 = create("ACTION_DEPTH16TO8_24", "actions/24/depth16to8");
    public static final AntuIcon ACTION_DIALOG_ALIGN_AND_DISTRIBUTE_24 = create("ACTION_DIALOG_ALIGN_AND_DISTRIBUTE_24", "actions/24/dialog-align-and-distribute");
    public static final AntuIcon ACTION_DIALOG_CANCEL_24 = create("ACTION_DIALOG_CANCEL_24", "actions/24/dialog-cancel");
    public static final AntuIcon ACTION_DIALOG_CANCEL_32 = create("ACTION_DIALOG_CANCEL_32", "actions/32/dialog-cancel");
    public static final AntuIcon ACTION_DIALOG_CLOSE_24 = create("ACTION_DIALOG_CLOSE_24", "actions/24/dialog-close");
    public static final AntuIcon ACTION_DIALOG_FILL_AND_STROKE_24 = create("ACTION_DIALOG_FILL_AND_STROKE_24", "actions/24/dialog-fill-and-stroke");
    public static final AntuIcon ACTION_DIALOG_FILTERS_24 = create("ACTION_DIALOG_FILTERS_24", "actions/24/dialog-filters");
    public static final AntuIcon ACTION_DIALOG_ICON_PREVIEW_24 = create("ACTION_DIALOG_ICON_PREVIEW_24", "actions/24/dialog-icon-preview");
    public static final AntuIcon ACTION_DIALOG_INPUT_DEVICES_24 = create("ACTION_DIALOG_INPUT_DEVICES_24", "actions/24/dialog-input-devices");
    public static final AntuIcon ACTION_DIALOG_LAYERS_24 = create("ACTION_DIALOG_LAYERS_24", "actions/24/dialog-layers");
    public static final AntuIcon ACTION_DIALOG_MEMORY_24 = create("ACTION_DIALOG_MEMORY_24", "actions/24/dialog-memory");
    public static final AntuIcon ACTION_DIALOG_MESSAGES_24 = create("ACTION_DIALOG_MESSAGES_24", "actions/24/dialog-messages");
    public static final AntuIcon ACTION_DIALOG_MESSAGES_32 = create("ACTION_DIALOG_MESSAGES_32", "actions/32/dialog-messages");
    public static final AntuIcon ACTION_DIALOG_OBJECT_PROPERTIES_24 = create("ACTION_DIALOG_OBJECT_PROPERTIES_24", "actions/24/dialog-object-properties");
    public static final AntuIcon ACTION_DIALOG_OK_24 = create("ACTION_DIALOG_OK_24", "actions/24/dialog-ok");
    public static final AntuIcon ACTION_DIALOG_OK_APPLY_24 = create("ACTION_DIALOG_OK_APPLY_24", "actions/24/dialog-ok-apply");
    public static final AntuIcon ACTION_DIALOG_OK_APPLY_32 = create("ACTION_DIALOG_OK_APPLY_32", "actions/32/dialog-ok-apply");
    public static final AntuIcon ACTION_DIALOG_ROWS_AND_COLUMNS_24 = create("ACTION_DIALOG_ROWS_AND_COLUMNS_24", "actions/24/dialog-rows-and-columns");
    public static final AntuIcon ACTION_DIALOG_SCRIPTS_24 = create("ACTION_DIALOG_SCRIPTS_24", "actions/24/dialog-scripts");
    public static final AntuIcon ACTION_DIALOG_TEXT_AND_FONT_24 = create("ACTION_DIALOG_TEXT_AND_FONT_24", "actions/24/dialog-text-and-font");
    public static final AntuIcon ACTION_DIALOG_TILE_CLONES_24 = create("ACTION_DIALOG_TILE_CLONES_24", "actions/24/dialog-tile-clones");
    public static final AntuIcon ACTION_DIALOG_TRANSFORM_24 = create("ACTION_DIALOG_TRANSFORM_24", "actions/24/dialog-transform");
    public static final AntuIcon ACTION_DIALOG_XML_EDITOR_24 = create("ACTION_DIALOG_XML_EDITOR_24", "actions/24/dialog-xml-editor");
    public static final AntuIcon ACTION_DIRSYNC_24 = create("ACTION_DIRSYNC_24", "actions/24/dirsync");
    public static final AntuIcon ACTION_DISTORTIONFX_24 = create("ACTION_DISTORTIONFX_24", "actions/24/distortionfx");
    public static final AntuIcon ACTION_DISTRIBUTE_GRAPH_24 = create("ACTION_DISTRIBUTE_GRAPH_24", "actions/24/distribute-graph");
    public static final AntuIcon ACTION_DISTRIBUTE_GRAPH_DIRECTED_24 = create("ACTION_DISTRIBUTE_GRAPH_DIRECTED_24", "actions/24/distribute-graph-directed");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_24", "actions/24/distribute-horizontal");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_BASELINE_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_BASELINE_24", "actions/24/distribute-horizontal-baseline");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_CENTER_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_CENTER_24", "actions/24/distribute-horizontal-center");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_EQUAL_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_EQUAL_24", "actions/24/distribute-horizontal-equal");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_GAPS_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_GAPS_24", "actions/24/distribute-horizontal-gaps");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_LEFT_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_LEFT_24", "actions/24/distribute-horizontal-left");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_MARGIN_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_MARGIN_24", "actions/24/distribute-horizontal-margin");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_PAGE_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_PAGE_24", "actions/24/distribute-horizontal-page");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_RIGHT_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_RIGHT_24", "actions/24/distribute-horizontal-right");
    public static final AntuIcon ACTION_DISTRIBUTE_HORIZONTAL_X_24 = create("ACTION_DISTRIBUTE_HORIZONTAL_X_24", "actions/24/distribute-horizontal-x");
    public static final AntuIcon ACTION_DISTRIBUTE_RANDOMIZE_24 = create("ACTION_DISTRIBUTE_RANDOMIZE_24", "actions/24/distribute-randomize");
    public static final AntuIcon ACTION_DISTRIBUTE_REMOVE_OVERLAPS_24 = create("ACTION_DISTRIBUTE_REMOVE_OVERLAPS_24", "actions/24/distribute-remove-overlaps");
    public static final AntuIcon ACTION_DISTRIBUTE_UNCLUMP_24 = create("ACTION_DISTRIBUTE_UNCLUMP_24", "actions/24/distribute-unclump");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_24 = create("ACTION_DISTRIBUTE_VERTICAL_24", "actions/24/distribute-vertical");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_BASELINE_24 = create("ACTION_DISTRIBUTE_VERTICAL_BASELINE_24", "actions/24/distribute-vertical-baseline");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_BOTTOM_24 = create("ACTION_DISTRIBUTE_VERTICAL_BOTTOM_24", "actions/24/distribute-vertical-bottom");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_CENTER_24 = create("ACTION_DISTRIBUTE_VERTICAL_CENTER_24", "actions/24/distribute-vertical-center");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_EQUAL_24 = create("ACTION_DISTRIBUTE_VERTICAL_EQUAL_24", "actions/24/distribute-vertical-equal");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_GAPS_24 = create("ACTION_DISTRIBUTE_VERTICAL_GAPS_24", "actions/24/distribute-vertical-gaps");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_MARGIN_24 = create("ACTION_DISTRIBUTE_VERTICAL_MARGIN_24", "actions/24/distribute-vertical-margin");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_PAGE_24 = create("ACTION_DISTRIBUTE_VERTICAL_PAGE_24", "actions/24/distribute-vertical-page");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_TOP_24 = create("ACTION_DISTRIBUTE_VERTICAL_TOP_24", "actions/24/distribute-vertical-top");
    public static final AntuIcon ACTION_DISTRIBUTE_VERTICAL_Y_24 = create("ACTION_DISTRIBUTE_VERTICAL_Y_24", "actions/24/distribute-vertical-y");
    public static final AntuIcon ACTION_DOCUMENT_CLEANUP_24 = create("ACTION_DOCUMENT_CLEANUP_24", "actions/24/document-cleanup");
    public static final AntuIcon ACTION_DOCUMENT_CLOSE_24 = create("ACTION_DOCUMENT_CLOSE_24", "actions/24/document-close");
    public static final AntuIcon ACTION_DOCUMENT_CLOSE_32 = create("ACTION_DOCUMENT_CLOSE_32", "actions/32/document-close");
    public static final AntuIcon ACTION_DOCUMENT_DECRYPT_24 = create("ACTION_DOCUMENT_DECRYPT_24", "actions/24/document-decrypt");
    public static final AntuIcon ACTION_DOCUMENT_DECRYPT_32 = create("ACTION_DOCUMENT_DECRYPT_32", "actions/32/document-decrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_24 = create("ACTION_DOCUMENT_EDIT_24", "actions/24/document-edit");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_32 = create("ACTION_DOCUMENT_EDIT_32", "actions/32/document-edit");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_DECRYPT_24 = create("ACTION_DOCUMENT_EDIT_DECRYPT_24", "actions/24/document-edit-decrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_DECRYPT_32 = create("ACTION_DOCUMENT_EDIT_DECRYPT_32", "actions/32/document-edit-decrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_DECRYPT_VERIFY_24 = create("ACTION_DOCUMENT_EDIT_DECRYPT_VERIFY_24", "actions/24/document-edit-decrypt-verify");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_DECRYPT_VERIFY_32 = create("ACTION_DOCUMENT_EDIT_DECRYPT_VERIFY_32", "actions/32/document-edit-decrypt-verify");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_ENCRYPT_24 = create("ACTION_DOCUMENT_EDIT_ENCRYPT_24", "actions/24/document-edit-encrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_ENCRYPT_32 = create("ACTION_DOCUMENT_EDIT_ENCRYPT_32", "actions/32/document-edit-encrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_SIGN_24 = create("ACTION_DOCUMENT_EDIT_SIGN_24", "actions/24/document-edit-sign");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_SIGN_32 = create("ACTION_DOCUMENT_EDIT_SIGN_32", "actions/32/document-edit-sign");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_SIGN_ENCRYPT_24 = create("ACTION_DOCUMENT_EDIT_SIGN_ENCRYPT_24", "actions/24/document-edit-sign-encrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_SIGN_ENCRYPT_32 = create("ACTION_DOCUMENT_EDIT_SIGN_ENCRYPT_32", "actions/32/document-edit-sign-encrypt");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_VERIFY_24 = create("ACTION_DOCUMENT_EDIT_VERIFY_24", "actions/24/document-edit-verify");
    public static final AntuIcon ACTION_DOCUMENT_EDIT_VERIFY_32 = create("ACTION_DOCUMENT_EDIT_VERIFY_32", "actions/32/document-edit-verify");
    public static final AntuIcon ACTION_DOCUMENT_ENCRYPT_24 = create("ACTION_DOCUMENT_ENCRYPT_24", "actions/24/document-encrypt");
    public static final AntuIcon ACTION_DOCUMENT_ENCRYPTED_24 = create("ACTION_DOCUMENT_ENCRYPTED_24", "actions/24/document-encrypted");
    public static final AntuIcon ACTION_DOCUMENT_ENCRYPTED_32 = create("ACTION_DOCUMENT_ENCRYPTED_32", "actions/32/document-encrypted");
    public static final AntuIcon ACTION_DOCUMENT_EXPORT_24 = create("ACTION_DOCUMENT_EXPORT_24", "actions/24/document-export");
    public static final AntuIcon ACTION_DOCUMENT_EXPORT_32 = create("ACTION_DOCUMENT_EXPORT_32", "actions/32/document-export");
    public static final AntuIcon ACTION_DOCUMENT_EXPORT_OCAL_24 = create("ACTION_DOCUMENT_EXPORT_OCAL_24", "actions/24/document-export-ocal");
    public static final AntuIcon ACTION_DOCUMENT_EXPORT_TABLE_24 = create("ACTION_DOCUMENT_EXPORT_TABLE_24", "actions/24/document-export-table");
    public static final AntuIcon ACTION_DOCUMENT_IMPORT_24 = create("ACTION_DOCUMENT_IMPORT_24", "actions/24/document-import");
    public static final AntuIcon ACTION_DOCUMENT_IMPORT_32 = create("ACTION_DOCUMENT_IMPORT_32", "actions/32/document-import");
    public static final AntuIcon ACTION_DOCUMENT_IMPORT_OCAL_24 = create("ACTION_DOCUMENT_IMPORT_OCAL_24", "actions/24/document-import-ocal");
    public static final AntuIcon ACTION_DOCUMENT_MULTIPLE_24 = create("ACTION_DOCUMENT_MULTIPLE_24", "actions/24/document-multiple");
    public static final AntuIcon ACTION_DOCUMENT_NEW_24 = create("ACTION_DOCUMENT_NEW_24", "actions/24/document-new");
    public static final AntuIcon ACTION_DOCUMENT_NEW_32 = create("ACTION_DOCUMENT_NEW_32", "actions/32/document-new");
    public static final AntuIcon ACTION_DOCUMENT_NEW_FROM_TEMPLATE_24 = create("ACTION_DOCUMENT_NEW_FROM_TEMPLATE_24", "actions/24/document-new-from-template");
    public static final AntuIcon ACTION_DOCUMENT_NEW_FROM_TEMPLATE_32 = create("ACTION_DOCUMENT_NEW_FROM_TEMPLATE_32", "actions/32/document-new-from-template");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_24 = create("ACTION_DOCUMENT_OPEN_24", "actions/24/document-open");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_32 = create("ACTION_DOCUMENT_OPEN_32", "actions/32/document-open");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_DATA_24 = create("ACTION_DOCUMENT_OPEN_DATA_24", "actions/24/document-open-data");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_FOLDER_24 = create("ACTION_DOCUMENT_OPEN_FOLDER_24", "actions/24/document-open-folder");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_RECENT_24 = create("ACTION_DOCUMENT_OPEN_RECENT_24", "actions/24/document-open-recent");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_RECENT_32 = create("ACTION_DOCUMENT_OPEN_RECENT_32", "actions/32/document-open-recent");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_RECENT_48 = create("ACTION_DOCUMENT_OPEN_RECENT_48", "actions/48/document-open-recent");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_REMOTE_24 = create("ACTION_DOCUMENT_OPEN_REMOTE_24", "actions/24/document-open-remote");
    public static final AntuIcon ACTION_DOCUMENT_OPEN_REMOTE_32 = create("ACTION_DOCUMENT_OPEN_REMOTE_32", "actions/32/document-open-remote");
    public static final AntuIcon ACTION_DOCUMENT_PREVIEW_24 = create("ACTION_DOCUMENT_PREVIEW_24", "actions/24/document-preview");
    public static final AntuIcon ACTION_DOCUMENT_PREVIEW_32 = create("ACTION_DOCUMENT_PREVIEW_32", "actions/32/document-preview");
    public static final AntuIcon ACTION_DOCUMENT_PREVIEW_ARCHIVE_24 = create("ACTION_DOCUMENT_PREVIEW_ARCHIVE_24", "actions/24/document-preview-archive");
    public static final AntuIcon ACTION_DOCUMENT_PREVIEW_ARCHIVE_32 = create("ACTION_DOCUMENT_PREVIEW_ARCHIVE_32", "actions/32/document-preview-archive");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_24 = create("ACTION_DOCUMENT_PRINT_24", "actions/24/document-print");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_32 = create("ACTION_DOCUMENT_PRINT_32", "actions/32/document-print");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_DIRECT_24 = create("ACTION_DOCUMENT_PRINT_DIRECT_24", "actions/24/document-print-direct");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_DIRECT_32 = create("ACTION_DOCUMENT_PRINT_DIRECT_32", "actions/32/document-print-direct");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_FRAME_24 = create("ACTION_DOCUMENT_PRINT_FRAME_24", "actions/24/document-print-frame");
    public static final AntuIcon ACTION_DOCUMENT_PRINT_PREVIEW_24 = create("ACTION_DOCUMENT_PRINT_PREVIEW_24", "actions/24/document-print-preview");
    public static final AntuIcon ACTION_DOCUMENT_PROPERTIES_24 = create("ACTION_DOCUMENT_PROPERTIES_24", "actions/24/document-properties");
    public static final AntuIcon ACTION_DOCUMENT_REVERT_24 = create("ACTION_DOCUMENT_REVERT_24", "actions/24/document-revert");
    public static final AntuIcon ACTION_DOCUMENT_REVERT_32 = create("ACTION_DOCUMENT_REVERT_32", "actions/32/document-revert");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_24 = create("ACTION_DOCUMENT_SAVE_24", "actions/24/document-save");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_32 = create("ACTION_DOCUMENT_SAVE_32", "actions/32/document-save");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_ALL_24 = create("ACTION_DOCUMENT_SAVE_ALL_24", "actions/24/document-save-all");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_ALL_32 = create("ACTION_DOCUMENT_SAVE_ALL_32", "actions/32/document-save-all");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_AS_24 = create("ACTION_DOCUMENT_SAVE_AS_24", "actions/24/document-save-as");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_AS_32 = create("ACTION_DOCUMENT_SAVE_AS_32", "actions/32/document-save-as");
    public static final AntuIcon ACTION_DOCUMENT_SAVE_AS_TEMPLATE_24 = create("ACTION_DOCUMENT_SAVE_AS_TEMPLATE_24", "actions/24/document-save-as-template");
    public static final AntuIcon ACTION_DOCUMENT_SHARE_24 = create("ACTION_DOCUMENT_SHARE_24", "actions/24/document-share");
    public static final AntuIcon ACTION_DOCUMENT_SHARE_32 = create("ACTION_DOCUMENT_SHARE_32", "actions/32/document-share");
    public static final AntuIcon ACTION_DOCUMENT_SIGN_24 = create("ACTION_DOCUMENT_SIGN_24", "actions/24/document-sign");
    public static final AntuIcon ACTION_DOCUMENT_SWAP_24 = create("ACTION_DOCUMENT_SWAP_24", "actions/24/document-swap");
    public static final AntuIcon ACTION_DOCUMENTATION_24 = create("ACTION_DOCUMENTATION_24", "actions/24/documentation");
    public static final AntuIcon ACTION_DOCUMENTINFO_24 = create("ACTION_DOCUMENTINFO_24", "actions/24/documentinfo");
    public static final AntuIcon ACTION_DOWNLOAD_24 = create("ACTION_DOWNLOAD_24", "actions/24/download");
    public static final AntuIcon ACTION_DOWNLOAD_AMAROK_24 = create("ACTION_DOWNLOAD_AMAROK_24", "actions/24/download-amarok");
    public static final AntuIcon ACTION_DOWNLOAD_LATER_24 = create("ACTION_DOWNLOAD_LATER_24", "actions/24/download-later");
    public static final AntuIcon ACTION_DRAW_ARROW_BACK_24 = create("ACTION_DRAW_ARROW_BACK_24", "actions/24/draw-arrow-back");
    public static final AntuIcon ACTION_DRAW_ARROW_DOWN_24 = create("ACTION_DRAW_ARROW_DOWN_24", "actions/24/draw-arrow-down");
    public static final AntuIcon ACTION_DRAW_ARROW_FORWARD_24 = create("ACTION_DRAW_ARROW_FORWARD_24", "actions/24/draw-arrow-forward");
    public static final AntuIcon ACTION_DRAW_ARROW_UP_24 = create("ACTION_DRAW_ARROW_UP_24", "actions/24/draw-arrow-up");
    public static final AntuIcon ACTION_DRAW_BEZIER_CURVES_24 = create("ACTION_DRAW_BEZIER_CURVES_24", "actions/24/draw-bezier-curves");
    public static final AntuIcon ACTION_DRAW_BRUSH_24 = create("ACTION_DRAW_BRUSH_24", "actions/24/draw-brush");
    public static final AntuIcon ACTION_DRAW_CALLIGRAPHIC_24 = create("ACTION_DRAW_CALLIGRAPHIC_24", "actions/24/draw-calligraphic");
    public static final AntuIcon ACTION_DRAW_CIRCLE_24 = create("ACTION_DRAW_CIRCLE_24", "actions/24/draw-circle");
    public static final AntuIcon ACTION_DRAW_CONNECTOR_24 = create("ACTION_DRAW_CONNECTOR_24", "actions/24/draw-connector");
    public static final AntuIcon ACTION_DRAW_CROSS_24 = create("ACTION_DRAW_CROSS_24", "actions/24/draw-cross");
    public static final AntuIcon ACTION_DRAW_CUBOID_24 = create("ACTION_DRAW_CUBOID_24", "actions/24/draw-cuboid");
    public static final AntuIcon ACTION_DRAW_DONUT_24 = create("ACTION_DRAW_DONUT_24", "actions/24/draw-donut");
    public static final AntuIcon ACTION_DRAW_ELLIPSE_24 = create("ACTION_DRAW_ELLIPSE_24", "actions/24/draw-ellipse");
    public static final AntuIcon ACTION_DRAW_ELLIPSE_ARC_24 = create("ACTION_DRAW_ELLIPSE_ARC_24", "actions/24/draw-ellipse-arc");
    public static final AntuIcon ACTION_DRAW_ELLIPSE_SEGMENT_24 = create("ACTION_DRAW_ELLIPSE_SEGMENT_24", "actions/24/draw-ellipse-segment");
    public static final AntuIcon ACTION_DRAW_ELLIPSE_WHOLE_24 = create("ACTION_DRAW_ELLIPSE_WHOLE_24", "actions/24/draw-ellipse-whole");
    public static final AntuIcon ACTION_DRAW_ERASER_24 = create("ACTION_DRAW_ERASER_24", "actions/24/draw-eraser");
    public static final AntuIcon ACTION_DRAW_FREEHAND_24 = create("ACTION_DRAW_FREEHAND_24", "actions/24/draw-freehand");
    public static final AntuIcon ACTION_DRAW_FREEHAND_48 = create("ACTION_DRAW_FREEHAND_48", "actions/48/draw-freehand");
    public static final AntuIcon ACTION_DRAW_HALFCIRCLE1_24 = create("ACTION_DRAW_HALFCIRCLE1_24", "actions/24/draw-halfcircle1");
    public static final AntuIcon ACTION_DRAW_HALFCIRCLE2_24 = create("ACTION_DRAW_HALFCIRCLE2_24", "actions/24/draw-halfcircle2");
    public static final AntuIcon ACTION_DRAW_HALFCIRCLE3_24 = create("ACTION_DRAW_HALFCIRCLE3_24", "actions/24/draw-halfcircle3");
    public static final AntuIcon ACTION_DRAW_HALFCIRCLE4_24 = create("ACTION_DRAW_HALFCIRCLE4_24", "actions/24/draw-halfcircle4");
    public static final AntuIcon ACTION_DRAW_LINE_24 = create("ACTION_DRAW_LINE_24", "actions/24/draw-line");
    public static final AntuIcon ACTION_DRAW_PATH_24 = create("ACTION_DRAW_PATH_24", "actions/24/draw-path");
    public static final AntuIcon ACTION_DRAW_POLYGON_24 = create("ACTION_DRAW_POLYGON_24", "actions/24/draw-polygon");
    public static final AntuIcon ACTION_DRAW_POLYGON_STAR_24 = create("ACTION_DRAW_POLYGON_STAR_24", "actions/24/draw-polygon-star");
    public static final AntuIcon ACTION_DRAW_POLYLINE_24 = create("ACTION_DRAW_POLYLINE_24", "actions/24/draw-polyline");
    public static final AntuIcon ACTION_DRAW_RECTANGLE_24 = create("ACTION_DRAW_RECTANGLE_24", "actions/24/draw-rectangle");
    public static final AntuIcon ACTION_DRAW_SPIRAL_24 = create("ACTION_DRAW_SPIRAL_24", "actions/24/draw-spiral");
    public static final AntuIcon ACTION_DRAW_SQUARE_INVERTED_CORNERS_24 = create("ACTION_DRAW_SQUARE_INVERTED_CORNERS_24", "actions/24/draw-square-inverted-corners");
    public static final AntuIcon ACTION_DRAW_STAR_24 = create("ACTION_DRAW_STAR_24", "actions/24/draw-star");
    public static final AntuIcon ACTION_DRAW_TEXT_24 = create("ACTION_DRAW_TEXT_24", "actions/24/draw-text");
    public static final AntuIcon ACTION_DRAW_TRIANGLE1_24 = create("ACTION_DRAW_TRIANGLE1_24", "actions/24/draw-triangle1");
    public static final AntuIcon ACTION_DRAW_TRIANGLE2_24 = create("ACTION_DRAW_TRIANGLE2_24", "actions/24/draw-triangle2");
    public static final AntuIcon ACTION_DRAW_TRIANGLE3_24 = create("ACTION_DRAW_TRIANGLE3_24", "actions/24/draw-triangle3");
    public static final AntuIcon ACTION_DRAW_TRIANGLE4_24 = create("ACTION_DRAW_TRIANGLE4_24", "actions/24/draw-triangle4");
    public static final AntuIcon ACTION_DRAW_TRIANGLE_24 = create("ACTION_DRAW_TRIANGLE_24", "actions/24/draw-triangle");
    public static final AntuIcon ACTION_DRAW_WATERCOLOR_24 = create("ACTION_DRAW_WATERCOLOR_24", "actions/24/draw-watercolor");
    public static final AntuIcon ACTION_DYNAMIC_AMAROK_24 = create("ACTION_DYNAMIC_AMAROK_24", "actions/24/dynamic-amarok");
    public static final AntuIcon ACTION_EARTHQUAKE_24 = create("ACTION_EARTHQUAKE_24", "actions/24/earthquake");
    public static final AntuIcon ACTION_EDIT_BOMB_24 = create("ACTION_EDIT_BOMB_24", "actions/24/edit-bomb");
    public static final AntuIcon ACTION_EDIT_CLEAR_24 = create("ACTION_EDIT_CLEAR_24", "actions/24/edit-clear");
    public static final AntuIcon ACTION_EDIT_CLEAR_HISTORY_24 = create("ACTION_EDIT_CLEAR_HISTORY_24", "actions/24/edit-clear-history");
    public static final AntuIcon ACTION_EDIT_CLEAR_LIST_24 = create("ACTION_EDIT_CLEAR_LIST_24", "actions/24/edit-clear-list");
    public static final AntuIcon ACTION_EDIT_CLEAR_LOCATIONBAR_LTR_24 = create("ACTION_EDIT_CLEAR_LOCATIONBAR_LTR_24", "actions/24/edit-clear-locationbar-ltr");
    public static final AntuIcon ACTION_EDIT_CLEAR_LOCATIONBAR_RTL_24 = create("ACTION_EDIT_CLEAR_LOCATIONBAR_RTL_24", "actions/24/edit-clear-locationbar-rtl");
    public static final AntuIcon ACTION_EDIT_CLONE_24 = create("ACTION_EDIT_CLONE_24", "actions/24/edit-clone");
    public static final AntuIcon ACTION_EDIT_CLONE_UNLINK_24 = create("ACTION_EDIT_CLONE_UNLINK_24", "actions/24/edit-clone-unlink");
    public static final AntuIcon ACTION_EDIT_COPY_24 = create("ACTION_EDIT_COPY_24", "actions/24/edit-copy");
    public static final AntuIcon ACTION_EDIT_CUT_24 = create("ACTION_EDIT_CUT_24", "actions/24/edit-cut");
    public static final AntuIcon ACTION_EDIT_DELETE_24 = create("ACTION_EDIT_DELETE_24", "actions/24/edit-delete");
    public static final AntuIcon ACTION_EDIT_DELETE_32 = create("ACTION_EDIT_DELETE_32", "actions/32/edit-delete");
    public static final AntuIcon ACTION_EDIT_DELETE_SHRED_24 = create("ACTION_EDIT_DELETE_SHRED_24", "actions/24/edit-delete-shred");
    public static final AntuIcon ACTION_EDIT_DOWNLOAD_24 = create("ACTION_EDIT_DOWNLOAD_24", "actions/24/edit-download");
    public static final AntuIcon ACTION_EDIT_DUPLICATE_24 = create("ACTION_EDIT_DUPLICATE_24", "actions/24/edit-duplicate");
    public static final AntuIcon ACTION_EDIT_FIND_24 = create("ACTION_EDIT_FIND_24", "actions/24/edit-find");
    public static final AntuIcon ACTION_EDIT_FIND_MAIL_24 = create("ACTION_EDIT_FIND_MAIL_24", "actions/24/edit-find-mail");
    public static final AntuIcon ACTION_EDIT_FIND_PROJECT_24 = create("ACTION_EDIT_FIND_PROJECT_24", "actions/24/edit-find-project");
    public static final AntuIcon ACTION_EDIT_FIND_REPLACE_24 = create("ACTION_EDIT_FIND_REPLACE_24", "actions/24/edit-find-replace");
    public static final AntuIcon ACTION_EDIT_FIND_USER_24 = create("ACTION_EDIT_FIND_USER_24", "actions/24/edit-find-user");
    public static final AntuIcon ACTION_EDIT_GUIDES_24 = create("ACTION_EDIT_GUIDES_24", "actions/24/edit-guides");
    public static final AntuIcon ACTION_EDIT_IMAGE_FACE_ADD_24 = create("ACTION_EDIT_IMAGE_FACE_ADD_24", "actions/24/edit-image-face-add");
    public static final AntuIcon ACTION_EDIT_IMAGE_FACE_DETECT_24 = create("ACTION_EDIT_IMAGE_FACE_DETECT_24", "actions/24/edit-image-face-detect");
    public static final AntuIcon ACTION_EDIT_IMAGE_FACE_RECOGNIZE_24 = create("ACTION_EDIT_IMAGE_FACE_RECOGNIZE_24", "actions/24/edit-image-face-recognize");
    public static final AntuIcon ACTION_EDIT_IMAGE_FACE_SHOW_24 = create("ACTION_EDIT_IMAGE_FACE_SHOW_24", "actions/24/edit-image-face-show");
    public static final AntuIcon ACTION_EDIT_LINK_24 = create("ACTION_EDIT_LINK_24", "actions/24/edit-link");
    public static final AntuIcon ACTION_EDIT_NODE_24 = create("ACTION_EDIT_NODE_24", "actions/24/edit-node");
    public static final AntuIcon ACTION_EDIT_PASTE_24 = create("ACTION_EDIT_PASTE_24", "actions/24/edit-paste");
    public static final AntuIcon ACTION_EDIT_PASTE_IN_PLACE_24 = create("ACTION_EDIT_PASTE_IN_PLACE_24", "actions/24/edit-paste-in-place");
    public static final AntuIcon ACTION_EDIT_PASTE_STYLE_24 = create("ACTION_EDIT_PASTE_STYLE_24", "actions/24/edit-paste-style");
    public static final AntuIcon ACTION_EDIT_REDO_24 = create("ACTION_EDIT_REDO_24", "actions/24/edit-redo");
    public static final AntuIcon ACTION_EDIT_REDO_32 = create("ACTION_EDIT_REDO_32", "actions/32/edit-redo");
    public static final AntuIcon ACTION_EDIT_RENAME_24 = create("ACTION_EDIT_RENAME_24", "actions/24/edit-rename");
    public static final AntuIcon ACTION_EDIT_SELECT_24 = create("ACTION_EDIT_SELECT_24", "actions/24/edit-select");
    public static final AntuIcon ACTION_EDIT_SELECT_ALL_24 = create("ACTION_EDIT_SELECT_ALL_24", "actions/24/edit-select-all");
    public static final AntuIcon ACTION_EDIT_SELECT_ALL_LAYERS_24 = create("ACTION_EDIT_SELECT_ALL_LAYERS_24", "actions/24/edit-select-all-layers");
    public static final AntuIcon ACTION_EDIT_SELECT_INVERT_24 = create("ACTION_EDIT_SELECT_INVERT_24", "actions/24/edit-select-invert");
    public static final AntuIcon ACTION_EDIT_SELECT_LASSO_24 = create("ACTION_EDIT_SELECT_LASSO_24", "actions/24/edit-select-lasso");
    public static final AntuIcon ACTION_EDIT_SELECT_NONE_24 = create("ACTION_EDIT_SELECT_NONE_24", "actions/24/edit-select-none");
    public static final AntuIcon ACTION_EDIT_SELECT_ORIGINAL_24 = create("ACTION_EDIT_SELECT_ORIGINAL_24", "actions/24/edit-select-original");
    public static final AntuIcon ACTION_EDIT_TABLE_CELL_MERGE_24 = create("ACTION_EDIT_TABLE_CELL_MERGE_24", "actions/24/edit-table-cell-merge");
    public static final AntuIcon ACTION_EDIT_TABLE_CELL_SPLIT_24 = create("ACTION_EDIT_TABLE_CELL_SPLIT_24", "actions/24/edit-table-cell-split");
    public static final AntuIcon ACTION_EDIT_TABLE_DELETE_COLUMN_24 = create("ACTION_EDIT_TABLE_DELETE_COLUMN_24", "actions/24/edit-table-delete-column");
    public static final AntuIcon ACTION_EDIT_TABLE_DELETE_ROW_24 = create("ACTION_EDIT_TABLE_DELETE_ROW_24", "actions/24/edit-table-delete-row");
    public static final AntuIcon ACTION_EDIT_TABLE_INSERT_COLUMN_LEFT_24 = create("ACTION_EDIT_TABLE_INSERT_COLUMN_LEFT_24", "actions/24/edit-table-insert-column-left");
    public static final AntuIcon ACTION_EDIT_TABLE_INSERT_COLUMN_RIGHT_24 = create("ACTION_EDIT_TABLE_INSERT_COLUMN_RIGHT_24", "actions/24/edit-table-insert-column-right");
    public static final AntuIcon ACTION_EDIT_TABLE_INSERT_ROW_ABOVE_24 = create("ACTION_EDIT_TABLE_INSERT_ROW_ABOVE_24", "actions/24/edit-table-insert-row-above");
    public static final AntuIcon ACTION_EDIT_TABLE_INSERT_ROW_BELOW_24 = create("ACTION_EDIT_TABLE_INSERT_ROW_BELOW_24", "actions/24/edit-table-insert-row-below");
    public static final AntuIcon ACTION_EDIT_TABLE_INSERT_ROW_UNDER_24 = create("ACTION_EDIT_TABLE_INSERT_ROW_UNDER_24", "actions/24/edit-table-insert-row-under");
    public static final AntuIcon ACTION_EDIT_TEXT_FRAME_UPDATE_24 = create("ACTION_EDIT_TEXT_FRAME_UPDATE_24", "actions/24/edit-text-frame-update");
    public static final AntuIcon ACTION_EDIT_UNDO_24 = create("ACTION_EDIT_UNDO_24", "actions/24/edit-undo");
    public static final AntuIcon ACTION_EDIT_UNDO_32 = create("ACTION_EDIT_UNDO_32", "actions/32/edit-undo");
    public static final AntuIcon ACTION_EDIT_UNDO_HISTORY_24 = create("ACTION_EDIT_UNDO_HISTORY_24", "actions/24/edit-undo-history");
    public static final AntuIcon ACTION_EDITIMAGE_24 = create("ACTION_EDITIMAGE_24", "actions/24/editimage");
    public static final AntuIcon ACTION_EMBOSSTOOL_24 = create("ACTION_EMBOSSTOOL_24", "actions/24/embosstool");
    public static final AntuIcon ACTION_END_STATE_24 = create("ACTION_END_STATE_24", "actions/24/end_state");
    public static final AntuIcon ACTION_ENTITY_24 = create("ACTION_ENTITY_24", "actions/24/entity");
    public static final AntuIcon ACTION_ENTRY_NEW_24 = create("ACTION_ENTRY_NEW_24", "actions/24/entry-new");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_ALL_24 = create("ACTION_ESCAPE_DIRECTION_ALL_24", "actions/24/escape-direction-all");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_DOWN_24 = create("ACTION_ESCAPE_DIRECTION_DOWN_24", "actions/24/escape-direction-down");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_HORIZONTAL_24 = create("ACTION_ESCAPE_DIRECTION_HORIZONTAL_24", "actions/24/escape-direction-horizontal");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_LEFT_24 = create("ACTION_ESCAPE_DIRECTION_LEFT_24", "actions/24/escape-direction-left");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_RIGHT_24 = create("ACTION_ESCAPE_DIRECTION_RIGHT_24", "actions/24/escape-direction-right");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_UP_24 = create("ACTION_ESCAPE_DIRECTION_UP_24", "actions/24/escape-direction-up");
    public static final AntuIcon ACTION_ESCAPE_DIRECTION_VERTICAL_24 = create("ACTION_ESCAPE_DIRECTION_VERTICAL_24", "actions/24/escape-direction-vertical");
    public static final AntuIcon ACTION_EXCEPTION_24 = create("ACTION_EXCEPTION_24", "actions/24/exception");
    public static final AntuIcon ACTION_EXCHANGE_POSITIONS_24 = create("ACTION_EXCHANGE_POSITIONS_24", "actions/24/exchange-positions");
    public static final AntuIcon ACTION_EXCHANGE_POSITIONS_CLOCKWISE_24 = create("ACTION_EXCHANGE_POSITIONS_CLOCKWISE_24", "actions/24/exchange-positions-clockwise");
    public static final AntuIcon ACTION_EXCHANGE_POSITIONS_ZORDER_24 = create("ACTION_EXCHANGE_POSITIONS_ZORDER_24", "actions/24/exchange-positions-zorder");
    public static final AntuIcon ACTION_EXIFINFO_24 = create("ACTION_EXIFINFO_24", "actions/24/exifinfo");
    public static final AntuIcon ACTION_FAVORITE_GENRES_AMAROK_24 = create("ACTION_FAVORITE_GENRES_AMAROK_24", "actions/24/favorite-genres-amarok");
    public static final AntuIcon ACTION_FEED_SUBSCRIBE_24 = create("ACTION_FEED_SUBSCRIBE_24", "actions/24/feed-subscribe");
    public static final AntuIcon ACTION_FILE_ZOOM_IN_24 = create("ACTION_FILE_ZOOM_IN_24", "actions/24/file-zoom-in");
    public static final AntuIcon ACTION_FILE_ZOOM_OUT_24 = create("ACTION_FILE_ZOOM_OUT_24", "actions/24/file-zoom-out");
    public static final AntuIcon ACTION_FILENAME_ALBUM_AMAROK_24 = create("ACTION_FILENAME_ALBUM_AMAROK_24", "actions/24/filename-album-amarok");
    public static final AntuIcon ACTION_FILENAME_AND_AMAROK_24 = create("ACTION_FILENAME_AND_AMAROK_24", "actions/24/filename-and-amarok");
    public static final AntuIcon ACTION_FILENAME_ARTIST_AMAROK_24 = create("ACTION_FILENAME_ARTIST_AMAROK_24", "actions/24/filename-artist-amarok");
    public static final AntuIcon ACTION_FILENAME_BPM_AMAROK_24 = create("ACTION_FILENAME_BPM_AMAROK_24", "actions/24/filename-bpm-amarok");
    public static final AntuIcon ACTION_FILENAME_COMMENT_AMAROK_24 = create("ACTION_FILENAME_COMMENT_AMAROK_24", "actions/24/filename-comment-amarok");
    public static final AntuIcon ACTION_FILENAME_COMPOSER_AMAROK_24 = create("ACTION_FILENAME_COMPOSER_AMAROK_24", "actions/24/filename-composer-amarok");
    public static final AntuIcon ACTION_FILENAME_DASH_AMAROK_24 = create("ACTION_FILENAME_DASH_AMAROK_24", "actions/24/filename-dash-amarok");
    public static final AntuIcon ACTION_FILENAME_DISCNUMBER_AMAROK_24 = create("ACTION_FILENAME_DISCNUMBER_AMAROK_24", "actions/24/filename-discnumber-amarok");
    public static final AntuIcon ACTION_FILENAME_DIVIDER_24 = create("ACTION_FILENAME_DIVIDER_24", "actions/24/filename-divider");
    public static final AntuIcon ACTION_FILENAME_DOT_AMAROK_24 = create("ACTION_FILENAME_DOT_AMAROK_24", "actions/24/filename-dot-amarok");
    public static final AntuIcon ACTION_FILENAME_FILETYPE_AMAROK_24 = create("ACTION_FILENAME_FILETYPE_AMAROK_24", "actions/24/filename-filetype-amarok");
    public static final AntuIcon ACTION_FILENAME_GENRE_AMAROK_24 = create("ACTION_FILENAME_GENRE_AMAROK_24", "actions/24/filename-genre-amarok");
    public static final AntuIcon ACTION_FILENAME_GROUP_LENGTH_24 = create("ACTION_FILENAME_GROUP_LENGTH_24", "actions/24/filename-group-length");
    public static final AntuIcon ACTION_FILENAME_GROUP_TRACKS_24 = create("ACTION_FILENAME_GROUP_TRACKS_24", "actions/24/filename-group-tracks");
    public static final AntuIcon ACTION_FILENAME_IGNORE_AMAROK_24 = create("ACTION_FILENAME_IGNORE_AMAROK_24", "actions/24/filename-ignore-amarok");
    public static final AntuIcon ACTION_FILENAME_INITIAL_AMAROK_24 = create("ACTION_FILENAME_INITIAL_AMAROK_24", "actions/24/filename-initial-amarok");
    public static final AntuIcon ACTION_FILENAME_LAST_PLAYED_24 = create("ACTION_FILENAME_LAST_PLAYED_24", "actions/24/filename-last-played");
    public static final AntuIcon ACTION_FILENAME_MOODBAR_24 = create("ACTION_FILENAME_MOODBAR_24", "actions/24/filename-moodbar");
    public static final AntuIcon ACTION_FILENAME_SAMPLE_RATE_24 = create("ACTION_FILENAME_SAMPLE_RATE_24", "actions/24/filename-sample-rate");
    public static final AntuIcon ACTION_FILENAME_SLASH_AMAROK_24 = create("ACTION_FILENAME_SLASH_AMAROK_24", "actions/24/filename-slash-amarok");
    public static final AntuIcon ACTION_FILENAME_SPACE_AMAROK_24 = create("ACTION_FILENAME_SPACE_AMAROK_24", "actions/24/filename-space-amarok");
    public static final AntuIcon ACTION_FILENAME_TITLE_AMAROK_24 = create("ACTION_FILENAME_TITLE_AMAROK_24", "actions/24/filename-title-amarok");
    public static final AntuIcon ACTION_FILENAME_TRACK_AMAROK_24 = create("ACTION_FILENAME_TRACK_AMAROK_24", "actions/24/filename-track-amarok");
    public static final AntuIcon ACTION_FILENAME_UNDERSCORE_AMAROK_24 = create("ACTION_FILENAME_UNDERSCORE_AMAROK_24", "actions/24/filename-underscore-amarok");
    public static final AntuIcon ACTION_FILENAME_YEAR_AMAROK_24 = create("ACTION_FILENAME_YEAR_AMAROK_24", "actions/24/filename-year-amarok");
    public static final AntuIcon ACTION_FILEVIEW_PREVIEW_24 = create("ACTION_FILEVIEW_PREVIEW_24", "actions/24/fileview-preview");
    public static final AntuIcon ACTION_FILL_COLOR_24 = create("ACTION_FILL_COLOR_24", "actions/24/fill-color");
    public static final AntuIcon ACTION_FILL_RULE_EVEN_ODD_24 = create("ACTION_FILL_RULE_EVEN_ODD_24", "actions/24/fill-rule-even-odd");
    public static final AntuIcon ACTION_FILL_RULE_NONZERO_24 = create("ACTION_FILL_RULE_NONZERO_24", "actions/24/fill-rule-nonzero");
    public static final AntuIcon ACTION_FILMGRAIN_24 = create("ACTION_FILMGRAIN_24", "actions/24/filmgrain");
    public static final AntuIcon ACTION_FINAL_ACTIVITY_24 = create("ACTION_FINAL_ACTIVITY_24", "actions/24/final_activity");
    public static final AntuIcon ACTION_FLAG_24 = create("ACTION_FLAG_24", "actions/24/flag");
    public static final AntuIcon ACTION_FLAG_BLACK_24 = create("ACTION_FLAG_BLACK_24", "actions/24/flag-black");
    public static final AntuIcon ACTION_FLAG_BLUE_24 = create("ACTION_FLAG_BLUE_24", "actions/24/flag-blue");
    public static final AntuIcon ACTION_FLAG_GREEN_24 = create("ACTION_FLAG_GREEN_24", "actions/24/flag-green");
    public static final AntuIcon ACTION_FLAG_RED_24 = create("ACTION_FLAG_RED_24", "actions/24/flag-red");
    public static final AntuIcon ACTION_FLAG_YELLOW_24 = create("ACTION_FLAG_YELLOW_24", "actions/24/flag-yellow");
    public static final AntuIcon ACTION_FOLDER_NEW_24 = create("ACTION_FOLDER_NEW_24", "actions/24/folder-new");
    public static final AntuIcon ACTION_FOLDER_NEW_32 = create("ACTION_FOLDER_NEW_32", "actions/32/folder-new");
    public static final AntuIcon ACTION_FOLDER_SYNC_24 = create("ACTION_FOLDER_SYNC_24", "actions/24/folder-sync");
    public static final AntuIcon ACTION_FOLDER_SYNC_32 = create("ACTION_FOLDER_SYNC_32", "actions/32/folder-sync");
    public static final AntuIcon ACTION_FOLLOWMOUSE_24 = create("ACTION_FOLLOWMOUSE_24", "actions/24/followmouse");
    public static final AntuIcon ACTION_FOREIGNKEY_CONSTRAINT_24 = create("ACTION_FOREIGNKEY_CONSTRAINT_24", "actions/24/foreignkey_constraint");
    public static final AntuIcon ACTION_FORK_24 = create("ACTION_FORK_24", "actions/24/fork");
    public static final AntuIcon ACTION_FORMAT_ADD_NODE_24 = create("ACTION_FORMAT_ADD_NODE_24", "actions/24/format-add-node");
    public static final AntuIcon ACTION_FORMAT_ALIGN_VERTICAL_BOTTOM_24 = create("ACTION_FORMAT_ALIGN_VERTICAL_BOTTOM_24", "actions/24/format-align-vertical-bottom");
    public static final AntuIcon ACTION_FORMAT_ALIGN_VERTICAL_CENTER_24 = create("ACTION_FORMAT_ALIGN_VERTICAL_CENTER_24", "actions/24/format-align-vertical-center");
    public static final AntuIcon ACTION_FORMAT_ALIGN_VERTICAL_TOP_24 = create("ACTION_FORMAT_ALIGN_VERTICAL_TOP_24", "actions/24/format-align-vertical-top");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_ALL_24 = create("ACTION_FORMAT_BORDER_SET_ALL_24", "actions/24/format-border-set-all");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_BOTTOM_24 = create("ACTION_FORMAT_BORDER_SET_BOTTOM_24", "actions/24/format-border-set-bottom");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_DIAGONAL_BL_TR_24 = create("ACTION_FORMAT_BORDER_SET_DIAGONAL_BL_TR_24", "actions/24/format-border-set-diagonal-bl-tr");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_DIAGONAL_TL_BR_24 = create("ACTION_FORMAT_BORDER_SET_DIAGONAL_TL_BR_24", "actions/24/format-border-set-diagonal-tl-br");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_EXTERNAL_24 = create("ACTION_FORMAT_BORDER_SET_EXTERNAL_24", "actions/24/format-border-set-external");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_INTERNAL_24 = create("ACTION_FORMAT_BORDER_SET_INTERNAL_24", "actions/24/format-border-set-internal");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_INTERNAL_HORIZONTAL_24 = create("ACTION_FORMAT_BORDER_SET_INTERNAL_HORIZONTAL_24", "actions/24/format-border-set-internal-horizontal");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_INTERNAL_VERTICAL_24 = create("ACTION_FORMAT_BORDER_SET_INTERNAL_VERTICAL_24", "actions/24/format-border-set-internal-vertical");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_LEFT_24 = create("ACTION_FORMAT_BORDER_SET_LEFT_24", "actions/24/format-border-set-left");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_NONE_24 = create("ACTION_FORMAT_BORDER_SET_NONE_24", "actions/24/format-border-set-none");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_RIGHT_24 = create("ACTION_FORMAT_BORDER_SET_RIGHT_24", "actions/24/format-border-set-right");
    public static final AntuIcon ACTION_FORMAT_BORDER_SET_TOP_24 = create("ACTION_FORMAT_BORDER_SET_TOP_24", "actions/24/format-border-set-top");
    public static final AntuIcon ACTION_FORMAT_BORDER_STYLE_24 = create("ACTION_FORMAT_BORDER_STYLE_24", "actions/24/format-border-style");
    public static final AntuIcon ACTION_FORMAT_BREAK_NODE_24 = create("ACTION_FORMAT_BREAK_NODE_24", "actions/24/format-break-node");
    public static final AntuIcon ACTION_FORMAT_CONNECT_NODE_24 = create("ACTION_FORMAT_CONNECT_NODE_24", "actions/24/format-connect-node");
    public static final AntuIcon ACTION_FORMAT_CONVERT_TO_PATH_24 = create("ACTION_FORMAT_CONVERT_TO_PATH_24", "actions/24/format-convert-to-path");
    public static final AntuIcon ACTION_FORMAT_CURRENCY_24 = create("ACTION_FORMAT_CURRENCY_24", "actions/24/format-currency");
    public static final AntuIcon ACTION_FORMAT_DISCONNECT_NODE_24 = create("ACTION_FORMAT_DISCONNECT_NODE_24", "actions/24/format-disconnect-node");
    public static final AntuIcon ACTION_FORMAT_FILL_COLOR_24 = create("ACTION_FORMAT_FILL_COLOR_24", "actions/24/format-fill-color");
    public static final AntuIcon ACTION_FORMAT_FONT_SIZE_LESS_24 = create("ACTION_FORMAT_FONT_SIZE_LESS_24", "actions/24/format-font-size-less");
    public static final AntuIcon ACTION_FORMAT_FONT_SIZE_MORE_24 = create("ACTION_FORMAT_FONT_SIZE_MORE_24", "actions/24/format-font-size-more");
    public static final AntuIcon ACTION_FORMAT_INDENT_LESS_24 = create("ACTION_FORMAT_INDENT_LESS_24", "actions/24/format-indent-less");
    public static final AntuIcon ACTION_FORMAT_INDENT_MORE_24 = create("ACTION_FORMAT_INDENT_MORE_24", "actions/24/format-indent-more");
    public static final AntuIcon ACTION_FORMAT_INSERT_NODE_24 = create("ACTION_FORMAT_INSERT_NODE_24", "actions/24/format-insert-node");
    public static final AntuIcon ACTION_FORMAT_JOIN_NODE_24 = create("ACTION_FORMAT_JOIN_NODE_24", "actions/24/format-join-node");
    public static final AntuIcon ACTION_FORMAT_JUSTIFY_CENTER_24 = create("ACTION_FORMAT_JUSTIFY_CENTER_24", "actions/24/format-justify-center");
    public static final AntuIcon ACTION_FORMAT_JUSTIFY_FILL_24 = create("ACTION_FORMAT_JUSTIFY_FILL_24", "actions/24/format-justify-fill");
    public static final AntuIcon ACTION_FORMAT_JUSTIFY_LEFT_24 = create("ACTION_FORMAT_JUSTIFY_LEFT_24", "actions/24/format-justify-left");
    public static final AntuIcon ACTION_FORMAT_JUSTIFY_RIGHT_24 = create("ACTION_FORMAT_JUSTIFY_RIGHT_24", "actions/24/format-justify-right");
    public static final AntuIcon ACTION_FORMAT_LINE_SPACING_DOUBLE_24 = create("ACTION_FORMAT_LINE_SPACING_DOUBLE_24", "actions/24/format-line-spacing-double");
    public static final AntuIcon ACTION_FORMAT_LINE_SPACING_NORMAL_24 = create("ACTION_FORMAT_LINE_SPACING_NORMAL_24", "actions/24/format-line-spacing-normal");
    public static final AntuIcon ACTION_FORMAT_LINE_SPACING_TRIPLE_24 = create("ACTION_FORMAT_LINE_SPACING_TRIPLE_24", "actions/24/format-line-spacing-triple");
    public static final AntuIcon ACTION_FORMAT_LIST_ORDERED_24 = create("ACTION_FORMAT_LIST_ORDERED_24", "actions/24/format-list-ordered");
    public static final AntuIcon ACTION_FORMAT_LIST_UNORDERED_24 = create("ACTION_FORMAT_LIST_UNORDERED_24", "actions/24/format-list-unordered");
    public static final AntuIcon ACTION_FORMAT_NODE_CORNER_24 = create("ACTION_FORMAT_NODE_CORNER_24", "actions/24/format-node-corner");
    public static final AntuIcon ACTION_FORMAT_NODE_CURVE_24 = create("ACTION_FORMAT_NODE_CURVE_24", "actions/24/format-node-curve");
    public static final AntuIcon ACTION_FORMAT_NODE_LINE_24 = create("ACTION_FORMAT_NODE_LINE_24", "actions/24/format-node-line");
    public static final AntuIcon ACTION_FORMAT_NODE_SMOOTH_24 = create("ACTION_FORMAT_NODE_SMOOTH_24", "actions/24/format-node-smooth");
    public static final AntuIcon ACTION_FORMAT_NODE_SYMMETRIC_24 = create("ACTION_FORMAT_NODE_SYMMETRIC_24", "actions/24/format-node-symmetric");
    public static final AntuIcon ACTION_FORMAT_NUMBER_PERCENT_24 = create("ACTION_FORMAT_NUMBER_PERCENT_24", "actions/24/format-number-percent");
    public static final AntuIcon ACTION_FORMAT_PRECISION_LESS_24 = create("ACTION_FORMAT_PRECISION_LESS_24", "actions/24/format-precision-less");
    public static final AntuIcon ACTION_FORMAT_PRECISION_MORE_24 = create("ACTION_FORMAT_PRECISION_MORE_24", "actions/24/format-precision-more");
    public static final AntuIcon ACTION_FORMAT_REMOVE_NODE_24 = create("ACTION_FORMAT_REMOVE_NODE_24", "actions/24/format-remove-node");
    public static final AntuIcon ACTION_FORMAT_SEGMENT_CURVE_24 = create("ACTION_FORMAT_SEGMENT_CURVE_24", "actions/24/format-segment-curve");
    public static final AntuIcon ACTION_FORMAT_SEGMENT_LINE_24 = create("ACTION_FORMAT_SEGMENT_LINE_24", "actions/24/format-segment-line");
    public static final AntuIcon ACTION_FORMAT_STROKE_COLOR_24 = create("ACTION_FORMAT_STROKE_COLOR_24", "actions/24/format-stroke-color");
    public static final AntuIcon ACTION_FORMAT_STROKE_COLOR_32 = create("ACTION_FORMAT_STROKE_COLOR_32", "actions/32/format-stroke-color");
    public static final AntuIcon ACTION_FORMAT_TEXT_BLOCKQUOTE_24 = create("ACTION_FORMAT_TEXT_BLOCKQUOTE_24", "actions/24/format-text-blockquote");
    public static final AntuIcon ACTION_FORMAT_TEXT_BOLD_24 = create("ACTION_FORMAT_TEXT_BOLD_24", "actions/24/format-text-bold");
    public static final AntuIcon ACTION_FORMAT_TEXT_CAPITALIZE_24 = create("ACTION_FORMAT_TEXT_CAPITALIZE_24", "actions/24/format-text-capitalize");
    public static final AntuIcon ACTION_FORMAT_TEXT_CODE_24 = create("ACTION_FORMAT_TEXT_CODE_24", "actions/24/format-text-code");
    public static final AntuIcon ACTION_FORMAT_TEXT_COLOR_24 = create("ACTION_FORMAT_TEXT_COLOR_24", "actions/24/format-text-color");
    public static final AntuIcon ACTION_FORMAT_TEXT_DIRECTION_HORIZONTAL_24 = create("ACTION_FORMAT_TEXT_DIRECTION_HORIZONTAL_24", "actions/24/format-text-direction-horizontal");
    public static final AntuIcon ACTION_FORMAT_TEXT_DIRECTION_LTR_24 = create("ACTION_FORMAT_TEXT_DIRECTION_LTR_24", "actions/24/format-text-direction-ltr");
    public static final AntuIcon ACTION_FORMAT_TEXT_DIRECTION_RTL_24 = create("ACTION_FORMAT_TEXT_DIRECTION_RTL_24", "actions/24/format-text-direction-rtl");
    public static final AntuIcon ACTION_FORMAT_TEXT_DIRECTION_VERTICAL_24 = create("ACTION_FORMAT_TEXT_DIRECTION_VERTICAL_24", "actions/24/format-text-direction-vertical");
    public static final AntuIcon ACTION_FORMAT_TEXT_ITALIC_24 = create("ACTION_FORMAT_TEXT_ITALIC_24", "actions/24/format-text-italic");
    public static final AntuIcon ACTION_FORMAT_TEXT_LOWERCASE_24 = create("ACTION_FORMAT_TEXT_LOWERCASE_24", "actions/24/format-text-lowercase");
    public static final AntuIcon ACTION_FORMAT_TEXT_STRIKETHROUGH_24 = create("ACTION_FORMAT_TEXT_STRIKETHROUGH_24", "actions/24/format-text-strikethrough");
    public static final AntuIcon ACTION_FORMAT_TEXT_SUBSCRIPT_24 = create("ACTION_FORMAT_TEXT_SUBSCRIPT_24", "actions/24/format-text-subscript");
    public static final AntuIcon ACTION_FORMAT_TEXT_SUPERSCRIPT_24 = create("ACTION_FORMAT_TEXT_SUPERSCRIPT_24", "actions/24/format-text-superscript");
    public static final AntuIcon ACTION_FORMAT_TEXT_SYMBOL_24 = create("ACTION_FORMAT_TEXT_SYMBOL_24", "actions/24/format-text-symbol");
    public static final AntuIcon ACTION_FORMAT_TEXT_UNDERLINE_24 = create("ACTION_FORMAT_TEXT_UNDERLINE_24", "actions/24/format-text-underline");
    public static final AntuIcon ACTION_FORMAT_TEXT_UPPERCASE_24 = create("ACTION_FORMAT_TEXT_UPPERCASE_24", "actions/24/format-text-uppercase");
    public static final AntuIcon ACTION_FORMULA_24 = create("ACTION_FORMULA_24", "actions/24/formula");
    public static final AntuIcon ACTION_FREEROTATION_24 = create("ACTION_FREEROTATION_24", "actions/24/freerotation");
    public static final AntuIcon ACTION_GAMES_ACHIEVEMENTS_24 = create("ACTION_GAMES_ACHIEVEMENTS_24", "actions/24/games-achievements");
    public static final AntuIcon ACTION_GAMES_CONFIG_BACKGROUND_24 = create("ACTION_GAMES_CONFIG_BACKGROUND_24", "actions/24/games-config-background");
    public static final AntuIcon ACTION_GAMES_CONFIG_BOARD_24 = create("ACTION_GAMES_CONFIG_BOARD_24", "actions/24/games-config-board");
    public static final AntuIcon ACTION_GAMES_CONFIG_CUSTOM_24 = create("ACTION_GAMES_CONFIG_CUSTOM_24", "actions/24/games-config-custom");
    public static final AntuIcon ACTION_GAMES_CONFIG_OPTIONS_24 = create("ACTION_GAMES_CONFIG_OPTIONS_24", "actions/24/games-config-options");
    public static final AntuIcon ACTION_GAMES_CONFIG_THEME_24 = create("ACTION_GAMES_CONFIG_THEME_24", "actions/24/games-config-theme");
    public static final AntuIcon ACTION_GAMES_CONFIG_TILES_24 = create("ACTION_GAMES_CONFIG_TILES_24", "actions/24/games-config-tiles");
    public static final AntuIcon ACTION_GAMES_DIFFICULT_24 = create("ACTION_GAMES_DIFFICULT_24", "actions/24/games-difficult");
    public static final AntuIcon ACTION_GAMES_ENDTURN_24 = create("ACTION_GAMES_ENDTURN_24", "actions/24/games-endturn");
    public static final AntuIcon ACTION_GAMES_HIGHSCORES_24 = create("ACTION_GAMES_HIGHSCORES_24", "actions/24/games-highscores");
    public static final AntuIcon ACTION_GAMES_HINT_24 = create("ACTION_GAMES_HINT_24", "actions/24/games-hint");
    public static final AntuIcon ACTION_GAMES_SOLVE_24 = create("ACTION_GAMES_SOLVE_24", "actions/24/games-solve");
    public static final AntuIcon ACTION_GEANY_BUILD_24 = create("ACTION_GEANY_BUILD_24", "actions/24/geany-build");
    public static final AntuIcon ACTION_GEANY_CLOSE_ALL_24 = create("ACTION_GEANY_CLOSE_ALL_24", "actions/24/geany-close-all");
    public static final AntuIcon ACTION_GEANY_SAVE_ALL_24 = create("ACTION_GEANY_SAVE_ALL_24", "actions/24/geany-save-all");
    public static final AntuIcon ACTION_GENERALISE_24 = create("ACTION_GENERALISE_24", "actions/24/generalise");
    public static final AntuIcon ACTION_GET_HOT_NEW_STUFF_24 = create("ACTION_GET_HOT_NEW_STUFF_24", "actions/24/get-hot-new-stuff");
    public static final AntuIcon ACTION_GO_BOTTOM_24 = create("ACTION_GO_BOTTOM_24", "actions/24/go-bottom");
    public static final AntuIcon ACTION_GO_DOWN_24 = create("ACTION_GO_DOWN_24", "actions/24/go-down");
    public static final AntuIcon ACTION_GO_DOWN_SEARCH_24 = create("ACTION_GO_DOWN_SEARCH_24", "actions/24/go-down-search");
    public static final AntuIcon ACTION_GO_FIRST_24 = create("ACTION_GO_FIRST_24", "actions/24/go-first");
    public static final AntuIcon ACTION_GO_FIRST_VIEW_24 = create("ACTION_GO_FIRST_VIEW_24", "actions/24/go-first-view");
    public static final AntuIcon ACTION_GO_FIRST_VIEW_PAGE_24 = create("ACTION_GO_FIRST_VIEW_PAGE_24", "actions/24/go-first-view-page");
    public static final AntuIcon ACTION_GO_HOME_24 = create("ACTION_GO_HOME_24", "actions/24/go-home");
    public static final AntuIcon ACTION_GO_JUMP_24 = create("ACTION_GO_JUMP_24", "actions/24/go-jump");
    public static final AntuIcon ACTION_GO_JUMP_DECLARATION_24 = create("ACTION_GO_JUMP_DECLARATION_24", "actions/24/go-jump-declaration");
    public static final AntuIcon ACTION_GO_JUMP_DEFINITION_24 = create("ACTION_GO_JUMP_DEFINITION_24", "actions/24/go-jump-definition");
    public static final AntuIcon ACTION_GO_JUMP_LOCATIONBAR_24 = create("ACTION_GO_JUMP_LOCATIONBAR_24", "actions/24/go-jump-locationbar");
    public static final AntuIcon ACTION_GO_JUMP_TODAY_24 = create("ACTION_GO_JUMP_TODAY_24", "actions/24/go-jump-today");
    public static final AntuIcon ACTION_GO_LAST_24 = create("ACTION_GO_LAST_24", "actions/24/go-last");
    public static final AntuIcon ACTION_GO_LAST_VIEW_24 = create("ACTION_GO_LAST_VIEW_24", "actions/24/go-last-view");
    public static final AntuIcon ACTION_GO_LAST_VIEW_PAGE_24 = create("ACTION_GO_LAST_VIEW_PAGE_24", "actions/24/go-last-view-page");
    public static final AntuIcon ACTION_GO_NEXT_24 = create("ACTION_GO_NEXT_24", "actions/24/go-next");
    public static final AntuIcon ACTION_GO_NEXT_CONTEXT_24 = create("ACTION_GO_NEXT_CONTEXT_24", "actions/24/go-next-context");
    public static final AntuIcon ACTION_GO_NEXT_USE_24 = create("ACTION_GO_NEXT_USE_24", "actions/24/go-next-use");
    public static final AntuIcon ACTION_GO_NEXT_VIEW_24 = create("ACTION_GO_NEXT_VIEW_24", "actions/24/go-next-view");
    public static final AntuIcon ACTION_GO_NEXT_VIEW_PAGE_24 = create("ACTION_GO_NEXT_VIEW_PAGE_24", "actions/24/go-next-view-page");
    public static final AntuIcon ACTION_GO_PARENT_FOLDER_24 = create("ACTION_GO_PARENT_FOLDER_24", "actions/24/go-parent-folder");
    public static final AntuIcon ACTION_GO_PARENT_FOLDER_32 = create("ACTION_GO_PARENT_FOLDER_32", "actions/32/go-parent-folder");
    public static final AntuIcon ACTION_GO_PREVIOUS_24 = create("ACTION_GO_PREVIOUS_24", "actions/24/go-previous");
    public static final AntuIcon ACTION_GO_PREVIOUS_CONTEXT_24 = create("ACTION_GO_PREVIOUS_CONTEXT_24", "actions/24/go-previous-context");
    public static final AntuIcon ACTION_GO_PREVIOUS_USE_24 = create("ACTION_GO_PREVIOUS_USE_24", "actions/24/go-previous-use");
    public static final AntuIcon ACTION_GO_PREVIOUS_VIEW_24 = create("ACTION_GO_PREVIOUS_VIEW_24", "actions/24/go-previous-view");
    public static final AntuIcon ACTION_GO_PREVIOUS_VIEW_PAGE_24 = create("ACTION_GO_PREVIOUS_VIEW_PAGE_24", "actions/24/go-previous-view-page");
    public static final AntuIcon ACTION_GO_TOP_24 = create("ACTION_GO_TOP_24", "actions/24/go-top");
    public static final AntuIcon ACTION_GO_UP_24 = create("ACTION_GO_UP_24", "actions/24/go-up");
    public static final AntuIcon ACTION_GO_UP_SEARCH_24 = create("ACTION_GO_UP_SEARCH_24", "actions/24/go-up-search");
    public static final AntuIcon ACTION_GPG_24 = create("ACTION_GPG_24", "actions/24/gpg");
    public static final AntuIcon ACTION_GPGSM_24 = create("ACTION_GPGSM_24", "actions/24/gpgsm");
    public static final AntuIcon ACTION_GRID_AXONOMETRIC_24 = create("ACTION_GRID_AXONOMETRIC_24", "actions/24/grid-axonometric");
    public static final AntuIcon ACTION_GRID_RECTANGULAR_24 = create("ACTION_GRID_RECTANGULAR_24", "actions/24/grid-rectangular");
    public static final AntuIcon ACTION_GTK_ADD_24 = create("ACTION_GTK_ADD_24", "actions/24/gtk-add");
    public static final AntuIcon ACTION_GTK_APPLY_24 = create("ACTION_GTK_APPLY_24", "actions/24/gtk-apply");
    public static final AntuIcon ACTION_GTK_AUTHENTICATION_24 = create("ACTION_GTK_AUTHENTICATION_24", "actions/24/gtk-authentication");
    public static final AntuIcon ACTION_GTK_BOLD_24 = create("ACTION_GTK_BOLD_24", "actions/24/gtk-bold");
    public static final AntuIcon ACTION_GTK_CANCEL_24 = create("ACTION_GTK_CANCEL_24", "actions/24/gtk-cancel");
    public static final AntuIcon ACTION_GTK_CDROM_24 = create("ACTION_GTK_CDROM_24", "actions/24/gtk-cdrom");
    public static final AntuIcon ACTION_GTK_CLEAR_24 = create("ACTION_GTK_CLEAR_24", "actions/24/gtk-clear");
    public static final AntuIcon ACTION_GTK_CLOSE_24 = create("ACTION_GTK_CLOSE_24", "actions/24/gtk-close");
    public static final AntuIcon ACTION_GTK_COLOR_PICKER_24 = create("ACTION_GTK_COLOR_PICKER_24", "actions/24/gtk-color-picker");
    public static final AntuIcon ACTION_GTK_CONNECT_24 = create("ACTION_GTK_CONNECT_24", "actions/24/gtk-connect");
    public static final AntuIcon ACTION_GTK_CONVERT_24 = create("ACTION_GTK_CONVERT_24", "actions/24/gtk-convert");
    public static final AntuIcon ACTION_GTK_DISCONNECT_24 = create("ACTION_GTK_DISCONNECT_24", "actions/24/gtk-disconnect");
    public static final AntuIcon ACTION_GTK_EDIT_24 = create("ACTION_GTK_EDIT_24", "actions/24/gtk-edit");
    public static final AntuIcon ACTION_GTK_EXECUTE_24 = create("ACTION_GTK_EXECUTE_24", "actions/24/gtk-execute");
    public static final AntuIcon ACTION_GTK_FLOPPY_24 = create("ACTION_GTK_FLOPPY_24", "actions/24/gtk-floppy");
    public static final AntuIcon ACTION_GTK_INDEX_24 = create("ACTION_GTK_INDEX_24", "actions/24/gtk-index");
    public static final AntuIcon ACTION_GTK_NO_24 = create("ACTION_GTK_NO_24", "actions/24/gtk-no");
    public static final AntuIcon ACTION_GTK_OK_24 = create("ACTION_GTK_OK_24", "actions/24/gtk-ok");
    public static final AntuIcon ACTION_GTK_PREFERENCES_24 = create("ACTION_GTK_PREFERENCES_24", "actions/24/gtk-preferences");
    public static final AntuIcon ACTION_GTK_PROPERTIES_24 = create("ACTION_GTK_PROPERTIES_24", "actions/24/gtk-properties");
    public static final AntuIcon ACTION_GTK_QUIT_24 = create("ACTION_GTK_QUIT_24", "actions/24/gtk-quit");
    public static final AntuIcon ACTION_GTK_SELECT_COLOR_24 = create("ACTION_GTK_SELECT_COLOR_24", "actions/24/gtk-select-color");
    public static final AntuIcon ACTION_GTK_SELECT_FONT_24 = create("ACTION_GTK_SELECT_FONT_24", "actions/24/gtk-select-font");
    public static final AntuIcon ACTION_GTK_STOP_24 = create("ACTION_GTK_STOP_24", "actions/24/gtk-stop");
    public static final AntuIcon ACTION_GTK_TAB_DUPLICATE_24 = create("ACTION_GTK_TAB_DUPLICATE_24", "actions/24/gtk-tab-duplicate");
    public static final AntuIcon ACTION_GTK_TAB_NEW_24 = create("ACTION_GTK_TAB_NEW_24", "actions/24/gtk-tab-new");
    public static final AntuIcon ACTION_GTK_UNDELETE_LTR_24 = create("ACTION_GTK_UNDELETE_LTR_24", "actions/24/gtk-undelete-ltr");
    public static final AntuIcon ACTION_GTK_UNDELETE_RTL_24 = create("ACTION_GTK_UNDELETE_RTL_24", "actions/24/gtk-undelete-rtl");
    public static final AntuIcon ACTION_GTK_YES_24 = create("ACTION_GTK_YES_24", "actions/24/gtk-yes");
    public static final AntuIcon ACTION_HELP_ABOUT_24 = create("ACTION_HELP_ABOUT_24", "actions/24/help-about");
    public static final AntuIcon ACTION_HELP_ABOUT_32 = create("ACTION_HELP_ABOUT_32", "actions/32/help-about");
    public static final AntuIcon ACTION_HELP_CONTENTS_24 = create("ACTION_HELP_CONTENTS_24", "actions/24/help-contents");
    public static final AntuIcon ACTION_HELP_CONTEXTUAL_24 = create("ACTION_HELP_CONTEXTUAL_24", "actions/24/help-contextual");
    public static final AntuIcon ACTION_HELP_DONATE_24 = create("ACTION_HELP_DONATE_24", "actions/24/help-donate");
    public static final AntuIcon ACTION_HELP_FEEDBACK_24 = create("ACTION_HELP_FEEDBACK_24", "actions/24/help-feedback");
    public static final AntuIcon ACTION_HELP_HINT_24 = create("ACTION_HELP_HINT_24", "actions/24/help-hint");
    public static final AntuIcon ACTION_HELP_KEYBORD_SHORTCUTS_24 = create("ACTION_HELP_KEYBORD_SHORTCUTS_24", "actions/24/help-keybord-shortcuts");
    public static final AntuIcon ACTION_HELP_WHATSTHIS_24 = create("ACTION_HELP_WHATSTHIS_24", "actions/24/help-whatsthis");
    public static final AntuIcon ACTION_HELP_WHATSTHIS_32 = create("ACTION_HELP_WHATSTHIS_32", "actions/32/help-whatsthis");
    public static final AntuIcon ACTION_HIDEMOUSE_24 = create("ACTION_HIDEMOUSE_24", "actions/24/hidemouse");
    public static final AntuIcon ACTION_HINT_24 = create("ACTION_HINT_24", "actions/24/hint");
    public static final AntuIcon ACTION_HOTPIXELS_24 = create("ACTION_HOTPIXELS_24", "actions/24/hotpixels");
    public static final AntuIcon ACTION_IDENTITY_24 = create("ACTION_IDENTITY_24", "actions/24/identity");
    public static final AntuIcon ACTION_IM_AIM_24 = create("ACTION_IM_AIM_24", "actions/24/im-aim");
    public static final AntuIcon ACTION_IM_BAN_KICK_USER_24 = create("ACTION_IM_BAN_KICK_USER_24", "actions/24/im-ban-kick-user");
    public static final AntuIcon ACTION_IM_BAN_USER_24 = create("ACTION_IM_BAN_USER_24", "actions/24/im-ban-user");
    public static final AntuIcon ACTION_IM_FACEBOOK_24 = create("ACTION_IM_FACEBOOK_24", "actions/24/im-facebook");
    public static final AntuIcon ACTION_IM_GADUGADU_24 = create("ACTION_IM_GADUGADU_24", "actions/24/im-gadugadu");
    public static final AntuIcon ACTION_IM_GOOGLE_24 = create("ACTION_IM_GOOGLE_24", "actions/24/im-google");
    public static final AntuIcon ACTION_IM_GOOGLE_TALK_24 = create("ACTION_IM_GOOGLE_TALK_24", "actions/24/im-google-talk");
    public static final AntuIcon ACTION_IM_ICQ_24 = create("ACTION_IM_ICQ_24", "actions/24/im-icq");
    public static final AntuIcon ACTION_IM_IDENTICA_24 = create("ACTION_IM_IDENTICA_24", "actions/24/im-identi.ca");
    public static final AntuIcon ACTION_IM_INVISIBLE_USER_24 = create("ACTION_IM_INVISIBLE_USER_24", "actions/24/im-invisible-user");
    public static final AntuIcon ACTION_IM_IRC_24 = create("ACTION_IM_IRC_24", "actions/24/im-irc");
    public static final AntuIcon ACTION_IM_JABBER_24 = create("ACTION_IM_JABBER_24", "actions/24/im-jabber");
    public static final AntuIcon ACTION_IM_KICK_USER_24 = create("ACTION_IM_KICK_USER_24", "actions/24/im-kick-user");
    public static final AntuIcon ACTION_IM_MSN_24 = create("ACTION_IM_MSN_24", "actions/24/im-msn");
    public static final AntuIcon ACTION_IM_QQ_24 = create("ACTION_IM_QQ_24", "actions/24/im-qq");
    public static final AntuIcon ACTION_IM_SKYPE_24 = create("ACTION_IM_SKYPE_24", "actions/24/im-skype");
    public static final AntuIcon ACTION_IM_TWITTER_24 = create("ACTION_IM_TWITTER_24", "actions/24/im-twitter");
    public static final AntuIcon ACTION_IM_USER_24 = create("ACTION_IM_USER_24", "actions/24/im-user");
    public static final AntuIcon ACTION_IM_USER_AWAY_24 = create("ACTION_IM_USER_AWAY_24", "actions/24/im-user-away");
    public static final AntuIcon ACTION_IM_USER_BUSY_24 = create("ACTION_IM_USER_BUSY_24", "actions/24/im-user-busy");
    public static final AntuIcon ACTION_IM_USER_OFFLINE_24 = create("ACTION_IM_USER_OFFLINE_24", "actions/24/im-user-offline");
    public static final AntuIcon ACTION_IM_USER_ONLINE_24 = create("ACTION_IM_USER_ONLINE_24", "actions/24/im-user-online");
    public static final AntuIcon ACTION_IM_YAHOO_24 = create("ACTION_IM_YAHOO_24", "actions/24/im-yahoo");
    public static final AntuIcon ACTION_IMAGECOMMENT_24 = create("ACTION_IMAGECOMMENT_24", "actions/24/imagecomment");
    public static final AntuIcon ACTION_INFO_AMAROK_24 = create("ACTION_INFO_AMAROK_24", "actions/24/Info-amarok");
    public static final AntuIcon ACTION_INITIAL_STATE_24 = create("ACTION_INITIAL_STATE_24", "actions/24/initial_state");
    public static final AntuIcon ACTION_INPAINTING_24 = create("ACTION_INPAINTING_24", "actions/24/inpainting");
    public static final AntuIcon ACTION_INSERT_BUTTON_24 = create("ACTION_INSERT_BUTTON_24", "actions/24/insert-button");
    public static final AntuIcon ACTION_INSERT_ENDNOTE_24 = create("ACTION_INSERT_ENDNOTE_24", "actions/24/insert-endnote");
    public static final AntuIcon ACTION_INSERT_FOOTNOTE_24 = create("ACTION_INSERT_FOOTNOTE_24", "actions/24/insert-footnote");
    public static final AntuIcon ACTION_INSERT_HORIZONTAL_RULE_24 = create("ACTION_INSERT_HORIZONTAL_RULE_24", "actions/24/insert-horizontal-rule");
    public static final AntuIcon ACTION_INSERT_IMAGE_24 = create("ACTION_INSERT_IMAGE_24", "actions/24/insert-image");
    public static final AntuIcon ACTION_INSERT_LINK_24 = create("ACTION_INSERT_LINK_24", "actions/24/insert-link");
    public static final AntuIcon ACTION_INSERT_MATH_EXPRESSION_24 = create("ACTION_INSERT_MATH_EXPRESSION_24", "actions/24/insert-math-expression");
    public static final AntuIcon ACTION_INSERT_MORE_MARK_24 = create("ACTION_INSERT_MORE_MARK_24", "actions/24/insert-more-mark");
    public static final AntuIcon ACTION_INSERT_PAGE_BREAK_24 = create("ACTION_INSERT_PAGE_BREAK_24", "actions/24/insert-page-break");
    public static final AntuIcon ACTION_INSERT_TABLE_24 = create("ACTION_INSERT_TABLE_24", "actions/24/insert-table");
    public static final AntuIcon ACTION_INSERT_TABLE_OF_CONTENTS_24 = create("ACTION_INSERT_TABLE_OF_CONTENTS_24", "actions/24/insert-table-of-contents");
    public static final AntuIcon ACTION_INSERT_TABLEOFCONTENTS_24 = create("ACTION_INSERT_TABLEOFCONTENTS_24", "actions/24/insert-tableofcontents");
    public static final AntuIcon ACTION_INSERT_TEXT_24 = create("ACTION_INSERT_TEXT_24", "actions/24/insert-text");
    public static final AntuIcon ACTION_INSERT_TEXT_FRAME_24 = create("ACTION_INSERT_TEXT_FRAME_24", "actions/24/insert-text-frame");
    public static final AntuIcon ACTION_INTERNET_AMAROK_24 = create("ACTION_INTERNET_AMAROK_24", "actions/24/internet-amarok");
    public static final AntuIcon ACTION_INVERTIMAGE_24 = create("ACTION_INVERTIMAGE_24", "actions/24/invertimage");
    public static final AntuIcon ACTION_IRC_CHANNEL_ACTIVE_24 = create("ACTION_IRC_CHANNEL_ACTIVE_24", "actions/24/irc-channel-active");
    public static final AntuIcon ACTION_IRC_CHANNEL_INACTIVE_24 = create("ACTION_IRC_CHANNEL_INACTIVE_24", "actions/24/irc-channel-inactive");
    public static final AntuIcon ACTION_IRC_CLOSE_CHANNEL_24 = create("ACTION_IRC_CLOSE_CHANNEL_24", "actions/24/irc-close-channel");
    public static final AntuIcon ACTION_IRC_JOIN_CHANNEL_24 = create("ACTION_IRC_JOIN_CHANNEL_24", "actions/24/irc-join-channel");
    public static final AntuIcon ACTION_IRC_OPERATOR_24 = create("ACTION_IRC_OPERATOR_24", "actions/24/irc-operator");
    public static final AntuIcon ACTION_IRC_REMOVE_OPERATOR_24 = create("ACTION_IRC_REMOVE_OPERATOR_24", "actions/24/irc-remove-operator");
    public static final AntuIcon ACTION_IRC_UNVOICE_24 = create("ACTION_IRC_UNVOICE_24", "actions/24/irc-unvoice");
    public static final AntuIcon ACTION_IRC_VOICE_24 = create("ACTION_IRC_VOICE_24", "actions/24/irc-voice");
    public static final AntuIcon ACTION_JOIN_24 = create("ACTION_JOIN_24", "actions/24/join");
    public static final AntuIcon ACTION_JOURNAL_NEW_24 = create("ACTION_JOURNAL_NEW_24", "actions/24/journal-new");
    public static final AntuIcon ACTION_JUMP_TODAY_24 = create("ACTION_JUMP_TODAY_24", "actions/24/jump-today");
    public static final AntuIcon ACTION_JUNCTION_24 = create("ACTION_JUNCTION_24", "actions/24/junction");
    public static final AntuIcon ACTION_KDENLIVE_ADD_CLIP_24 = create("ACTION_KDENLIVE_ADD_CLIP_24", "actions/24/kdenlive-add-clip");
    public static final AntuIcon ACTION_KDENLIVE_ADD_COLOR_CLIP_24 = create("ACTION_KDENLIVE_ADD_COLOR_CLIP_24", "actions/24/kdenlive-add-color-clip");
    public static final AntuIcon ACTION_KDENLIVE_ADD_SLIDE_CLIP_24 = create("ACTION_KDENLIVE_ADD_SLIDE_CLIP_24", "actions/24/kdenlive-add-slide-clip");
    public static final AntuIcon ACTION_KDENLIVE_ADD_TEXT_CLIP_24 = create("ACTION_KDENLIVE_ADD_TEXT_CLIP_24", "actions/24/kdenlive-add-text-clip");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_BOTTOM_24 = create("ACTION_KDENLIVE_ALIGN_BOTTOM_24", "actions/24/kdenlive-align-bottom");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_HOR_24 = create("ACTION_KDENLIVE_ALIGN_HOR_24", "actions/24/kdenlive-align-hor");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_LEFT_24 = create("ACTION_KDENLIVE_ALIGN_LEFT_24", "actions/24/kdenlive-align-left");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_NONE_24 = create("ACTION_KDENLIVE_ALIGN_NONE_24", "actions/24/kdenlive-align-none");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_RIGHT_24 = create("ACTION_KDENLIVE_ALIGN_RIGHT_24", "actions/24/kdenlive-align-right");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_TOP_24 = create("ACTION_KDENLIVE_ALIGN_TOP_24", "actions/24/kdenlive-align-top");
    public static final AntuIcon ACTION_KDENLIVE_ALIGN_VERT_24 = create("ACTION_KDENLIVE_ALIGN_VERT_24", "actions/24/kdenlive-align-vert");
    public static final AntuIcon ACTION_KDENLIVE_CUSTOM_EFFECT_24 = create("ACTION_KDENLIVE_CUSTOM_EFFECT_24", "actions/24/kdenlive-custom-effect");
    public static final AntuIcon ACTION_KDENLIVE_DELEFFECT_24 = create("ACTION_KDENLIVE_DELEFFECT_24", "actions/24/kdenlive-deleffect");
    public static final AntuIcon ACTION_KDENLIVE_DOWN_24 = create("ACTION_KDENLIVE_DOWN_24", "actions/24/kdenlive-down");
    public static final AntuIcon ACTION_KDENLIVE_HIDE_AUDIO_24 = create("ACTION_KDENLIVE_HIDE_AUDIO_24", "actions/24/kdenlive-hide-audio");
    public static final AntuIcon ACTION_KDENLIVE_HIDE_VIDEO_24 = create("ACTION_KDENLIVE_HIDE_VIDEO_24", "actions/24/kdenlive-hide-video");
    public static final AntuIcon ACTION_KDENLIVE_INSERT_EDIT_24 = create("ACTION_KDENLIVE_INSERT_EDIT_24", "actions/24/kdenlive-insert-edit");
    public static final AntuIcon ACTION_KDENLIVE_INSERT_RECT_24 = create("ACTION_KDENLIVE_INSERT_RECT_24", "actions/24/kdenlive-insert-rect");
    public static final AntuIcon ACTION_KDENLIVE_INSERT_UNICODE_24 = create("ACTION_KDENLIVE_INSERT_UNICODE_24", "actions/24/kdenlive-insert-unicode");
    public static final AntuIcon ACTION_KDENLIVE_LOCK_24 = create("ACTION_KDENLIVE_LOCK_24", "actions/24/kdenlive-lock");
    public static final AntuIcon ACTION_KDENLIVE_MENU_24 = create("ACTION_KDENLIVE_MENU_24", "actions/24/kdenlive-menu");
    public static final AntuIcon ACTION_KDENLIVE_NORMAL_EDIT_24 = create("ACTION_KDENLIVE_NORMAL_EDIT_24", "actions/24/kdenlive-normal-edit");
    public static final AntuIcon ACTION_KDENLIVE_OBJECT_HEIGHT_24 = create("ACTION_KDENLIVE_OBJECT_HEIGHT_24", "actions/24/kdenlive-object-height");
    public static final AntuIcon ACTION_KDENLIVE_OBJECT_WIDTH_24 = create("ACTION_KDENLIVE_OBJECT_WIDTH_24", "actions/24/kdenlive-object-width");
    public static final AntuIcon ACTION_KDENLIVE_OVERWRITE_EDIT_24 = create("ACTION_KDENLIVE_OVERWRITE_EDIT_24", "actions/24/kdenlive-overwrite-edit");
    public static final AntuIcon ACTION_KDENLIVE_SELECT_ALL_24 = create("ACTION_KDENLIVE_SELECT_ALL_24", "actions/24/kdenlive-select-all");
    public static final AntuIcon ACTION_KDENLIVE_SELECT_IMAGES_24 = create("ACTION_KDENLIVE_SELECT_IMAGES_24", "actions/24/kdenlive-select-images");
    public static final AntuIcon ACTION_KDENLIVE_SELECT_RECTS_24 = create("ACTION_KDENLIVE_SELECT_RECTS_24", "actions/24/kdenlive-select-rects");
    public static final AntuIcon ACTION_KDENLIVE_SELECT_TEXTS_24 = create("ACTION_KDENLIVE_SELECT_TEXTS_24", "actions/24/kdenlive-select-texts");
    public static final AntuIcon ACTION_KDENLIVE_SELECT_TOOL_24 = create("ACTION_KDENLIVE_SELECT_TOOL_24", "actions/24/kdenlive-select-tool");
    public static final AntuIcon ACTION_KDENLIVE_SHOW_AUDIO_24 = create("ACTION_KDENLIVE_SHOW_AUDIO_24", "actions/24/kdenlive-show-audio");
    public static final AntuIcon ACTION_KDENLIVE_SHOW_AUDIOTHUMB_24 = create("ACTION_KDENLIVE_SHOW_AUDIOTHUMB_24", "actions/24/kdenlive-show-audiothumb");
    public static final AntuIcon ACTION_KDENLIVE_SHOW_MARKERS_24 = create("ACTION_KDENLIVE_SHOW_MARKERS_24", "actions/24/kdenlive-show-markers");
    public static final AntuIcon ACTION_KDENLIVE_SHOW_VIDEO_24 = create("ACTION_KDENLIVE_SHOW_VIDEO_24", "actions/24/kdenlive-show-video");
    public static final AntuIcon ACTION_KDENLIVE_SHOW_VIDEOTHUMB_24 = create("ACTION_KDENLIVE_SHOW_VIDEOTHUMB_24", "actions/24/kdenlive-show-videothumb");
    public static final AntuIcon ACTION_KDENLIVE_SNAP_24 = create("ACTION_KDENLIVE_SNAP_24", "actions/24/kdenlive-snap");
    public static final AntuIcon ACTION_KDENLIVE_SPACER_TOOL_24 = create("ACTION_KDENLIVE_SPACER_TOOL_24", "actions/24/kdenlive-spacer-tool");
    public static final AntuIcon ACTION_KDENLIVE_SPLIT_AUDIO_24 = create("ACTION_KDENLIVE_SPLIT_AUDIO_24", "actions/24/kdenlive-split-audio");
    public static final AntuIcon ACTION_KDENLIVE_TRACK_HAS_EFFECT_24 = create("ACTION_KDENLIVE_TRACK_HAS_EFFECT_24", "actions/24/kdenlive-track_has_effect");
    public static final AntuIcon ACTION_KDENLIVE_UNLOCK_24 = create("ACTION_KDENLIVE_UNLOCK_24", "actions/24/kdenlive-unlock");
    public static final AntuIcon ACTION_KDENLIVE_UNSELECT_ALL_24 = create("ACTION_KDENLIVE_UNSELECT_ALL_24", "actions/24/kdenlive-unselect-all");
    public static final AntuIcon ACTION_KDENLIVE_UP_24 = create("ACTION_KDENLIVE_UP_24", "actions/24/kdenlive-up");
    public static final AntuIcon ACTION_KDENLIVE_ZINDEX_BOTTOM_24 = create("ACTION_KDENLIVE_ZINDEX_BOTTOM_24", "actions/24/kdenlive-zindex-bottom");
    public static final AntuIcon ACTION_KDENLIVE_ZINDEX_DOWN_24 = create("ACTION_KDENLIVE_ZINDEX_DOWN_24", "actions/24/kdenlive-zindex-down");
    public static final AntuIcon ACTION_KDENLIVE_ZINDEX_TOP_24 = create("ACTION_KDENLIVE_ZINDEX_TOP_24", "actions/24/kdenlive-zindex-top");
    public static final AntuIcon ACTION_KDENLIVE_ZINDEX_UP_24 = create("ACTION_KDENLIVE_ZINDEX_UP_24", "actions/24/kdenlive-zindex-up");
    public static final AntuIcon ACTION_KDENLIVE_ZONE_END_24 = create("ACTION_KDENLIVE_ZONE_END_24", "actions/24/kdenlive-zone-end");
    public static final AntuIcon ACTION_KDENLIVE_ZONE_START_24 = create("ACTION_KDENLIVE_ZONE_START_24", "actions/24/kdenlive-zone-start");
    public static final AntuIcon ACTION_KDENLIVE_ZOOM_LARGE_24 = create("ACTION_KDENLIVE_ZOOM_LARGE_24", "actions/24/kdenlive-zoom-large");
    public static final AntuIcon ACTION_KDENLIVE_ZOOM_SMALL_24 = create("ACTION_KDENLIVE_ZOOM_SMALL_24", "actions/24/kdenlive-zoom-small");
    public static final AntuIcon ACTION_KEY_ENTER_24 = create("ACTION_KEY_ENTER_24", "actions/24/key-enter");
    public static final AntuIcon ACTION_KMOUTH_PHRASE_24 = create("ACTION_KMOUTH_PHRASE_24", "actions/24/kmouth-phrase");
    public static final AntuIcon ACTION_KMOUTH_PHRASE_NEW_24 = create("ACTION_KMOUTH_PHRASE_NEW_24", "actions/24/kmouth-phrase-new");
    public static final AntuIcon ACTION_KMOUTH_PHRASEBOOK_24 = create("ACTION_KMOUTH_PHRASEBOOK_24", "actions/24/kmouth-phrasebook");
    public static final AntuIcon ACTION_KMOUTH_PHRESEBOOK_NEW_24 = create("ACTION_KMOUTH_PHRESEBOOK_NEW_24", "actions/24/kmouth-phresebook-new");
    public static final AntuIcon ACTION_KNOTES_ALARM_24 = create("ACTION_KNOTES_ALARM_24", "actions/24/knotes_alarm");
    public static final AntuIcon ACTION_KNOTES_DATE_24 = create("ACTION_KNOTES_DATE_24", "actions/24/knotes_date");
    public static final AntuIcon ACTION_KNOTES_DELETE_24 = create("ACTION_KNOTES_DELETE_24", "actions/24/knotes_delete");
    public static final AntuIcon ACTION_KONTES_CLOSE_24 = create("ACTION_KONTES_CLOSE_24", "actions/24/kontes_close");
    public static final AntuIcon ACTION_KR_COMBINE_24 = create("ACTION_KR_COMBINE_24", "actions/24/kr_combine");
    public static final AntuIcon ACTION_KR_COMPAREDIRS_24 = create("ACTION_KR_COMPAREDIRS_24", "actions/24/kr_comparedirs");
    public static final AntuIcon ACTION_KR_DISKUSAGE_24 = create("ACTION_KR_DISKUSAGE_24", "actions/24/kr_diskusage");
    public static final AntuIcon ACTION_KR_JUMPBACK_24 = create("ACTION_KR_JUMPBACK_24", "actions/24/kr_jumpback");
    public static final AntuIcon ACTION_KR_MOUNTMAN_24 = create("ACTION_KR_MOUNTMAN_24", "actions/24/kr_mountman");
    public static final AntuIcon ACTION_KR_SETJUMPBACK_24 = create("ACTION_KR_SETJUMPBACK_24", "actions/24/kr_setjumpback");
    public static final AntuIcon ACTION_KR_SYNCBROWSE_OFF_24 = create("ACTION_KR_SYNCBROWSE_OFF_24", "actions/24/kr_syncbrowse_off");
    public static final AntuIcon ACTION_KR_SYNCBROWSE_ON_24 = create("ACTION_KR_SYNCBROWSE_ON_24", "actions/24/kr_syncbrowse_on");
    public static final AntuIcon ACTION_KR_UNSELECT_24 = create("ACTION_KR_UNSELECT_24", "actions/24/kr_unselect");
    public static final AntuIcon ACTION_KSTARS_CBOUND_24 = create("ACTION_KSTARS_CBOUND_24", "actions/24/kstars_cbound");
    public static final AntuIcon ACTION_KSTARS_CLINES_24 = create("ACTION_KSTARS_CLINES_24", "actions/24/kstars_clines");
    public static final AntuIcon ACTION_KSTARS_CNAMES_24 = create("ACTION_KSTARS_CNAMES_24", "actions/24/kstars_cnames");
    public static final AntuIcon ACTION_KSTARS_DEEPSKY_24 = create("ACTION_KSTARS_DEEPSKY_24", "actions/24/kstars_deepsky");
    public static final AntuIcon ACTION_KSTARS_EKOS_24 = create("ACTION_KSTARS_EKOS_24", "actions/24/kstars_ekos");
    public static final AntuIcon ACTION_KSTARS_FITSVIEWER_24 = create("ACTION_KSTARS_FITSVIEWER_24", "actions/24/kstars_fitsviewer");
    public static final AntuIcon ACTION_KSTARS_GRID_24 = create("ACTION_KSTARS_GRID_24", "actions/24/kstars_grid");
    public static final AntuIcon ACTION_KSTARS_HGRID_24 = create("ACTION_KSTARS_HGRID_24", "actions/24/kstars_hgrid");
    public static final AntuIcon ACTION_KSTARS_HORIZON_24 = create("ACTION_KSTARS_HORIZON_24", "actions/24/kstars_horizon");
    public static final AntuIcon ACTION_KSTARS_INDI_24 = create("ACTION_KSTARS_INDI_24", "actions/24/kstars_indi");
    public static final AntuIcon ACTION_KSTARS_MW_24 = create("ACTION_KSTARS_MW_24", "actions/24/kstars_mw");
    public static final AntuIcon ACTION_KSTARS_SATELLITES_24 = create("ACTION_KSTARS_SATELLITES_24", "actions/24/kstars_satellites");
    public static final AntuIcon ACTION_KSTARS_SOLARSYSTEM_24 = create("ACTION_KSTARS_SOLARSYSTEM_24", "actions/24/kstars_solarsystem");
    public static final AntuIcon ACTION_KSTARS_SUPERNOVAE_24 = create("ACTION_KSTARS_SUPERNOVAE_24", "actions/24/kstars_supernovae");
    public static final AntuIcon ACTION_KT_ADD_FEEDS_24 = create("ACTION_KT_ADD_FEEDS_24", "actions/24/kt-add-feeds");
    public static final AntuIcon ACTION_KT_ADD_FILTERS_24 = create("ACTION_KT_ADD_FILTERS_24", "actions/24/kt-add-filters");
    public static final AntuIcon ACTION_KT_BANDWIDTH_SCHEDULER_24 = create("ACTION_KT_BANDWIDTH_SCHEDULER_24", "actions/24/kt-bandwidth-scheduler");
    public static final AntuIcon ACTION_KT_CHANGE_TRACKER_24 = create("ACTION_KT_CHANGE_TRACKER_24", "actions/24/kt-change-tracker");
    public static final AntuIcon ACTION_KT_CHECK_DATA_24 = create("ACTION_KT_CHECK_DATA_24", "actions/24/kt-check-data");
    public static final AntuIcon ACTION_KT_CHUNKS_24 = create("ACTION_KT_CHUNKS_24", "actions/24/kt-chunks");
    public static final AntuIcon ACTION_KT_ENCRYPTED_24 = create("ACTION_KT_ENCRYPTED_24", "actions/24/kt-encrypted");
    public static final AntuIcon ACTION_KT_INFO_WIDGET_24 = create("ACTION_KT_INFO_WIDGET_24", "actions/24/kt-info-widget");
    public static final AntuIcon ACTION_KT_MAGNET_24 = create("ACTION_KT_MAGNET_24", "actions/24/kt-magnet");
    public static final AntuIcon ACTION_KT_PAUSE_24 = create("ACTION_KT_PAUSE_24", "actions/24/kt-pause");
    public static final AntuIcon ACTION_KT_PLUGINS_24 = create("ACTION_KT_PLUGINS_24", "actions/24/kt-plugins");
    public static final AntuIcon ACTION_KT_QUEUE_MANAGER_24 = create("ACTION_KT_QUEUE_MANAGER_24", "actions/24/kt-queue-manager");
    public static final AntuIcon ACTION_KT_REMOVE_24 = create("ACTION_KT_REMOVE_24", "actions/24/kt-remove");
    public static final AntuIcon ACTION_KT_REMOVE_FEEDS_24 = create("ACTION_KT_REMOVE_FEEDS_24", "actions/24/kt-remove-feeds");
    public static final AntuIcon ACTION_KT_REMOVE_FILTERS_24 = create("ACTION_KT_REMOVE_FILTERS_24", "actions/24/kt-remove-filters");
    public static final AntuIcon ACTION_KT_RESTORE_DEFAULTS_24 = create("ACTION_KT_RESTORE_DEFAULTS_24", "actions/24/kt-restore-defaults");
    public static final AntuIcon ACTION_KT_SET_MAX_DOWNLOAD_SPEED_24 = create("ACTION_KT_SET_MAX_DOWNLOAD_SPEED_24", "actions/24/kt-set-max-download-speed");
    public static final AntuIcon ACTION_KT_SET_MAX_UPLOAD_SPEED_24 = create("ACTION_KT_SET_MAX_UPLOAD_SPEED_24", "actions/24/kt-set-max-upload-speed");
    public static final AntuIcon ACTION_KT_SHOW_STATUSBAR_24 = create("ACTION_KT_SHOW_STATUSBAR_24", "actions/24/kt-show-statusbar");
    public static final AntuIcon ACTION_KT_SPEED_LIMITS_24 = create("ACTION_KT_SPEED_LIMITS_24", "actions/24/kt-speed-limits");
    public static final AntuIcon ACTION_KT_START_24 = create("ACTION_KT_START_24", "actions/24/kt-start");
    public static final AntuIcon ACTION_KT_START_ALL_24 = create("ACTION_KT_START_ALL_24", "actions/24/kt-start-all");
    public static final AntuIcon ACTION_KT_STOP_24 = create("ACTION_KT_STOP_24", "actions/24/kt-stop");
    public static final AntuIcon ACTION_KT_STOP_ALL_24 = create("ACTION_KT_STOP_ALL_24", "actions/24/kt-stop-all");
    public static final AntuIcon ACTION_KTNEF_EXTRACT_TO_24 = create("ACTION_KTNEF_EXTRACT_TO_24", "actions/24/ktnef_extract_to");
    public static final AntuIcon ACTION_LABEL_AMAROK_24 = create("ACTION_LABEL_AMAROK_24", "actions/24/label-amarok");
    public static final AntuIcon ACTION_LABPLOT_AXIS_HORIZONTAL_24 = create("ACTION_LABPLOT_AXIS_HORIZONTAL_24", "actions/24/labplot-axis-horizontal");
    public static final AntuIcon ACTION_LABPLOT_AXIS_VERTICAL_24 = create("ACTION_LABPLOT_AXIS_VERTICAL_24", "actions/24/labplot-axis-vertical");
    public static final AntuIcon ACTION_LABPLOT_SHIFT_DOWN_Y_24 = create("ACTION_LABPLOT_SHIFT_DOWN_Y_24", "actions/24/labplot-shift-down-y");
    public static final AntuIcon ACTION_LABPLOT_SHIFT_LEFT_X_24 = create("ACTION_LABPLOT_SHIFT_LEFT_X_24", "actions/24/labplot-shift-left-x");
    public static final AntuIcon ACTION_LABPLOT_SHIFT_RIGHT_X_24 = create("ACTION_LABPLOT_SHIFT_RIGHT_X_24", "actions/24/labplot-shift-right-x");
    public static final AntuIcon ACTION_LABPLOT_SHIFT_UP_Y_24 = create("ACTION_LABPLOT_SHIFT_UP_Y_24", "actions/24/labplot-shift-up-y");
    public static final AntuIcon ACTION_LABPLOT_TEX_LOGO_24 = create("ACTION_LABPLOT_TEX_LOGO_24", "actions/24/labplot-TeX-logo");
    public static final AntuIcon ACTION_LABPLOT_XY_CURVE_24 = create("ACTION_LABPLOT_XY_CURVE_24", "actions/24/labplot-xy-curve");
    public static final AntuIcon ACTION_LABPLOT_XY_EQUATION_CURVE_24 = create("ACTION_LABPLOT_XY_EQUATION_CURVE_24", "actions/24/labplot-xy-equation-curve");
    public static final AntuIcon ACTION_LABPLOT_XY_FIT_CURVE_24 = create("ACTION_LABPLOT_XY_FIT_CURVE_24", "actions/24/labplot-xy-fit-curve");
    public static final AntuIcon ACTION_LABPLOT_XY_PLOT_FOUR_AXES_24 = create("ACTION_LABPLOT_XY_PLOT_FOUR_AXES_24", "actions/24/labplot-xy-plot-four-axes");
    public static final AntuIcon ACTION_LABPLOT_XY_PLOT_TWO_AXES_24 = create("ACTION_LABPLOT_XY_PLOT_TWO_AXES_24", "actions/24/labplot-xy-plot-two-axes");
    public static final AntuIcon ACTION_LABPLOT_XY_PLOT_TWO_AXES_CENTERED_24 = create("ACTION_LABPLOT_XY_PLOT_TWO_AXES_CENTERED_24", "actions/24/labplot-xy-plot-two-axes-centered");
    public static final AntuIcon ACTION_LABPLOT_XY_PLOT_TWO_AXES_CENTERED_ORIGIN_24 = create("ACTION_LABPLOT_XY_PLOT_TWO_AXES_CENTERED_ORIGIN_24", "actions/24/labplot-xy-plot-two-axes-centered-origin");
    public static final AntuIcon ACTION_LABPLOT_ZOOM_IN_X_24 = create("ACTION_LABPLOT_ZOOM_IN_X_24", "actions/24/labplot-zoom-in-x");
    public static final AntuIcon ACTION_LABPLOT_ZOOM_IN_Y_24 = create("ACTION_LABPLOT_ZOOM_IN_Y_24", "actions/24/labplot-zoom-in-y");
    public static final AntuIcon ACTION_LABPLOT_ZOOM_OUT_X_24 = create("ACTION_LABPLOT_ZOOM_OUT_X_24", "actions/24/labplot-zoom-out-x");
    public static final AntuIcon ACTION_LABPLOT_ZOOM_OUT_Y_24 = create("ACTION_LABPLOT_ZOOM_OUT_Y_24", "actions/24/labplot-zoom-out-y");
    public static final AntuIcon ACTION_LAYER_BOTTOM_24 = create("ACTION_LAYER_BOTTOM_24", "actions/24/layer-bottom");
    public static final AntuIcon ACTION_LAYER_DELETE_24 = create("ACTION_LAYER_DELETE_24", "actions/24/layer-delete");
    public static final AntuIcon ACTION_LAYER_DUPLICATE_24 = create("ACTION_LAYER_DUPLICATE_24", "actions/24/layer-duplicate");
    public static final AntuIcon ACTION_LAYER_LOWER_24 = create("ACTION_LAYER_LOWER_24", "actions/24/layer-lower");
    public static final AntuIcon ACTION_LAYER_NEW_24 = create("ACTION_LAYER_NEW_24", "actions/24/layer-new");
    public static final AntuIcon ACTION_LAYER_NEXT_24 = create("ACTION_LAYER_NEXT_24", "actions/24/layer-next");
    public static final AntuIcon ACTION_LAYER_PREVIOUS_24 = create("ACTION_LAYER_PREVIOUS_24", "actions/24/layer-previous");
    public static final AntuIcon ACTION_LAYER_RAISE_24 = create("ACTION_LAYER_RAISE_24", "actions/24/layer-raise");
    public static final AntuIcon ACTION_LAYER_RENAME_24 = create("ACTION_LAYER_RENAME_24", "actions/24/layer-rename");
    public static final AntuIcon ACTION_LAYER_TOP_24 = create("ACTION_LAYER_TOP_24", "actions/24/layer-top");
    public static final AntuIcon ACTION_LAYER_VISIBLE_OFF_24 = create("ACTION_LAYER_VISIBLE_OFF_24", "actions/24/layer-visible-off");
    public static final AntuIcon ACTION_LAYER_VISIBLE_ON_24 = create("ACTION_LAYER_VISIBLE_ON_24", "actions/24/layer-visible-on");
    public static final AntuIcon ACTION_LENSAUTOFIX_24 = create("ACTION_LENSAUTOFIX_24", "actions/24/lensautofix");
    public static final AntuIcon ACTION_LENSDISTORTION_24 = create("ACTION_LENSDISTORTION_24", "actions/24/lensdistortion");
    public static final AntuIcon ACTION_LICENSE_24 = create("ACTION_LICENSE_24", "actions/24/license");
    public static final AntuIcon ACTION_LIGHTTABLE_24 = create("ACTION_LIGHTTABLE_24", "actions/24/lighttable");
    public static final AntuIcon ACTION_LIGHTTABLEADD_24 = create("ACTION_LIGHTTABLEADD_24", "actions/24/lighttableadd");
    public static final AntuIcon ACTION_LINES_CONNECTOR_24 = create("ACTION_LINES_CONNECTOR_24", "actions/24/lines-connector");
    public static final AntuIcon ACTION_LIST_ADD_24 = create("ACTION_LIST_ADD_24", "actions/24/list-add");
    public static final AntuIcon ACTION_LIST_ADD_FONT_24 = create("ACTION_LIST_ADD_FONT_24", "actions/24/list-add-font");
    public static final AntuIcon ACTION_LIST_ADD_USER_24 = create("ACTION_LIST_ADD_USER_24", "actions/24/list-add-user");
    public static final AntuIcon ACTION_LIST_REMOVE_24 = create("ACTION_LIST_REMOVE_24", "actions/24/list-remove");
    public static final AntuIcon ACTION_LIST_REMOVE_USER_24 = create("ACTION_LIST_REMOVE_USER_24", "actions/24/list-remove-user");
    public static final AntuIcon ACTION_LIST_RESOURCE_ADD_24 = create("ACTION_LIST_RESOURCE_ADD_24", "actions/24/list-resource-add");
    public static final AntuIcon ACTION_LOVE_AMAROK_24 = create("ACTION_LOVE_AMAROK_24", "actions/24/love-amarok");
    public static final AntuIcon ACTION_MAIL_ATTACHMENT_24 = create("ACTION_MAIL_ATTACHMENT_24", "actions/24/mail-attachment");
    public static final AntuIcon ACTION_MAIL_ENCRYPTED_24 = create("ACTION_MAIL_ENCRYPTED_24", "actions/24/mail-encrypted");
    public static final AntuIcon ACTION_MAIL_ENCRYPTED_FULL_24 = create("ACTION_MAIL_ENCRYPTED_FULL_24", "actions/24/mail-encrypted-full");
    public static final AntuIcon ACTION_MAIL_ENCRYPTED_PART_24 = create("ACTION_MAIL_ENCRYPTED_PART_24", "actions/24/mail-encrypted-part");
    public static final AntuIcon ACTION_MAIL_FLAG_24 = create("ACTION_MAIL_FLAG_24", "actions/24/mail-flag");
    public static final AntuIcon ACTION_MAIL_FORWARD_24 = create("ACTION_MAIL_FORWARD_24", "actions/24/mail-forward");
    public static final AntuIcon ACTION_MAIL_FORWARDED_24 = create("ACTION_MAIL_FORWARDED_24", "actions/24/mail-forwarded");
    public static final AntuIcon ACTION_MAIL_FORWARDED_REPLIED_24 = create("ACTION_MAIL_FORWARDED_REPLIED_24", "actions/24/mail-forwarded-replied");
    public static final AntuIcon ACTION_MAIL_INVITATION_24 = create("ACTION_MAIL_INVITATION_24", "actions/24/mail-invitation");
    public static final AntuIcon ACTION_MAIL_MARK_IMPORTANT_24 = create("ACTION_MAIL_MARK_IMPORTANT_24", "actions/24/mail-mark-important");
    public static final AntuIcon ACTION_MAIL_MARK_JUNK_24 = create("ACTION_MAIL_MARK_JUNK_24", "actions/24/mail-mark-junk");
    public static final AntuIcon ACTION_MAIL_MARK_NOTJUNK_24 = create("ACTION_MAIL_MARK_NOTJUNK_24", "actions/24/mail-mark-notjunk");
    public static final AntuIcon ACTION_MAIL_MARK_READ_24 = create("ACTION_MAIL_MARK_READ_24", "actions/24/mail-mark-read");
    public static final AntuIcon ACTION_MAIL_MARK_TASK_24 = create("ACTION_MAIL_MARK_TASK_24", "actions/24/mail-mark-task");
    public static final AntuIcon ACTION_MAIL_MARK_UNREAD_24 = create("ACTION_MAIL_MARK_UNREAD_24", "actions/24/mail-mark-unread");
    public static final AntuIcon ACTION_MAIL_MARK_UNREAD_NEW_24 = create("ACTION_MAIL_MARK_UNREAD_NEW_24", "actions/24/mail-mark-unread-new");
    public static final AntuIcon ACTION_MAIL_MEETING_REQUEST_REPLY_24 = create("ACTION_MAIL_MEETING_REQUEST_REPLY_24", "actions/24/mail-meeting-request-reply");
    public static final AntuIcon ACTION_MAIL_MESSAGE_NEW_24 = create("ACTION_MAIL_MESSAGE_NEW_24", "actions/24/mail-message-new");
    public static final AntuIcon ACTION_MAIL_MESSAGE_NEW_LIST_24 = create("ACTION_MAIL_MESSAGE_NEW_LIST_24", "actions/24/mail-message-new-list");
    public static final AntuIcon ACTION_MAIL_QUEUE_24 = create("ACTION_MAIL_QUEUE_24", "actions/24/mail-queue");
    public static final AntuIcon ACTION_MAIL_RECEIVE_24 = create("ACTION_MAIL_RECEIVE_24", "actions/24/mail-receive");
    public static final AntuIcon ACTION_MAIL_REPLIED_24 = create("ACTION_MAIL_REPLIED_24", "actions/24/mail-replied");
    public static final AntuIcon ACTION_MAIL_REPLY_ALL_24 = create("ACTION_MAIL_REPLY_ALL_24", "actions/24/mail-reply-all");
    public static final AntuIcon ACTION_MAIL_REPLY_CUSTOM_24 = create("ACTION_MAIL_REPLY_CUSTOM_24", "actions/24/mail-reply-custom");
    public static final AntuIcon ACTION_MAIL_REPLY_CUSTOM_ALL_24 = create("ACTION_MAIL_REPLY_CUSTOM_ALL_24", "actions/24/mail-reply-custom-all");
    public static final AntuIcon ACTION_MAIL_REPLY_LIST_24 = create("ACTION_MAIL_REPLY_LIST_24", "actions/24/mail-reply-list");
    public static final AntuIcon ACTION_MAIL_REPLY_SENDER_24 = create("ACTION_MAIL_REPLY_SENDER_24", "actions/24/mail-reply-sender");
    public static final AntuIcon ACTION_MAIL_SEND_24 = create("ACTION_MAIL_SEND_24", "actions/24/mail-send");
    public static final AntuIcon ACTION_MAIL_SIGNATURE_UNKNOWN_24 = create("ACTION_MAIL_SIGNATURE_UNKNOWN_24", "actions/24/mail-signature-unknown");
    public static final AntuIcon ACTION_MAIL_SIGNED_24 = create("ACTION_MAIL_SIGNED_24", "actions/24/mail-signed");
    public static final AntuIcon ACTION_MAIL_SIGNED_FULL_24 = create("ACTION_MAIL_SIGNED_FULL_24", "actions/24/mail-signed-full");
    public static final AntuIcon ACTION_MAIL_SIGNED_FULLY_24 = create("ACTION_MAIL_SIGNED_FULLY_24", "actions/24/mail-signed-fully");
    public static final AntuIcon ACTION_MAIL_SIGNED_PART_24 = create("ACTION_MAIL_SIGNED_PART_24", "actions/24/mail-signed-part");
    public static final AntuIcon ACTION_MAIL_SIGNED_VERIFIED_24 = create("ACTION_MAIL_SIGNED_VERIFIED_24", "actions/24/mail-signed-verified");
    public static final AntuIcon ACTION_MAIL_TAGGED_24 = create("ACTION_MAIL_TAGGED_24", "actions/24/mail-tagged");
    public static final AntuIcon ACTION_MAP_FLAT_24 = create("ACTION_MAP_FLAT_24", "actions/24/map-flat");
    public static final AntuIcon ACTION_MAP_GLOBE_24 = create("ACTION_MAP_GLOBE_24", "actions/24/map-globe");
    public static final AntuIcon ACTION_MAP_GNOMONIC_24 = create("ACTION_MAP_GNOMONIC_24", "actions/24/map-gnomonic");
    public static final AntuIcon ACTION_MAP_MERCATOR_24 = create("ACTION_MAP_MERCATOR_24", "actions/24/map-mercator");
    public static final AntuIcon ACTION_MEASURE_24 = create("ACTION_MEASURE_24", "actions/24/measure");
    public static final AntuIcon ACTION_MEDIA_ALBUM_COVER_24 = create("ACTION_MEDIA_ALBUM_COVER_24", "actions/24/media-album-cover");
    public static final AntuIcon ACTION_MEDIA_ALBUM_COVER_MANAGER_AMAROK_24 = create("ACTION_MEDIA_ALBUM_COVER_MANAGER_AMAROK_24", "actions/24/media-album-cover-manager-amarok");
    public static final AntuIcon ACTION_MEDIA_ALBUM_REPEAT_AMAROK_24 = create("ACTION_MEDIA_ALBUM_REPEAT_AMAROK_24", "actions/24/media-album-repeat-amarok");
    public static final AntuIcon ACTION_MEDIA_ALBUM_TRACK_24 = create("ACTION_MEDIA_ALBUM_TRACK_24", "actions/24/media-album-track");
    public static final AntuIcon ACTION_MEDIA_EJECT_24 = create("ACTION_MEDIA_EJECT_24", "actions/24/media-eject");
    public static final AntuIcon ACTION_MEDIA_EJECT_32 = create("ACTION_MEDIA_EJECT_32", "actions/32/media-eject");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_PAUSE_24 = create("ACTION_MEDIA_PLAYBACK_PAUSE_24", "actions/24/media-playback-pause");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_PAUSE_32 = create("ACTION_MEDIA_PLAYBACK_PAUSE_32", "actions/32/media-playback-pause");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_START_24 = create("ACTION_MEDIA_PLAYBACK_START_24", "actions/24/media-playback-start");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_START_32 = create("ACTION_MEDIA_PLAYBACK_START_32", "actions/32/media-playback-start");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_STOP_24 = create("ACTION_MEDIA_PLAYBACK_STOP_24", "actions/24/media-playback-stop");
    public static final AntuIcon ACTION_MEDIA_PLAYBACK_STOP_32 = create("ACTION_MEDIA_PLAYBACK_STOP_32", "actions/32/media-playback-stop");
    public static final AntuIcon ACTION_MEDIA_PLAYLIST_REPEAT_24 = create("ACTION_MEDIA_PLAYLIST_REPEAT_24", "actions/24/media-playlist-repeat");
    public static final AntuIcon ACTION_MEDIA_PLAYLIST_REPEAT_32 = create("ACTION_MEDIA_PLAYLIST_REPEAT_32", "actions/32/media-playlist-repeat");
    public static final AntuIcon ACTION_MEDIA_PLAYLIST_REPEAT_AMAROK_24 = create("ACTION_MEDIA_PLAYLIST_REPEAT_AMAROK_24", "actions/24/media-playlist-repeat-amarok");
    public static final AntuIcon ACTION_MEDIA_PLAYLIST_SHUFFLE_24 = create("ACTION_MEDIA_PLAYLIST_SHUFFLE_24", "actions/24/media-playlist-shuffle");
    public static final AntuIcon ACTION_MEDIA_PLAYLIST_SHUFFLE_32 = create("ACTION_MEDIA_PLAYLIST_SHUFFLE_32", "actions/32/media-playlist-shuffle");
    public static final AntuIcon ACTION_MEDIA_RANDOM_ALBUMS_AMAROK_24 = create("ACTION_MEDIA_RANDOM_ALBUMS_AMAROK_24", "actions/24/media-random-albums-amarok");
    public static final AntuIcon ACTION_MEDIA_RANDOM_TRACKS_AMAROK_24 = create("ACTION_MEDIA_RANDOM_TRACKS_AMAROK_24", "actions/24/media-random-tracks-amarok");
    public static final AntuIcon ACTION_MEDIA_RECORD_24 = create("ACTION_MEDIA_RECORD_24", "actions/24/media-record");
    public static final AntuIcon ACTION_MEDIA_RECORD_32 = create("ACTION_MEDIA_RECORD_32", "actions/32/media-record");
    public static final AntuIcon ACTION_MEDIA_REPEAT_ALBUM_AMAROK_24 = create("ACTION_MEDIA_REPEAT_ALBUM_AMAROK_24", "actions/24/media-repeat-album-amarok");
    public static final AntuIcon ACTION_MEDIA_REPEAT_PLAYLIST_AMAROK_24 = create("ACTION_MEDIA_REPEAT_PLAYLIST_AMAROK_24", "actions/24/media-repeat-playlist-amarok");
    public static final AntuIcon ACTION_MEDIA_REPEAT_TRACK_AMAROK_24 = create("ACTION_MEDIA_REPEAT_TRACK_AMAROK_24", "actions/24/media-repeat-track-amarok");
    public static final AntuIcon ACTION_MEDIA_SEEK_BACKWARD_24 = create("ACTION_MEDIA_SEEK_BACKWARD_24", "actions/24/media-seek-backward");
    public static final AntuIcon ACTION_MEDIA_SEEK_BACKWARD_32 = create("ACTION_MEDIA_SEEK_BACKWARD_32", "actions/32/media-seek-backward");
    public static final AntuIcon ACTION_MEDIA_SEEK_FORWARD_24 = create("ACTION_MEDIA_SEEK_FORWARD_24", "actions/24/media-seek-forward");
    public static final AntuIcon ACTION_MEDIA_SEEK_FORWARD_32 = create("ACTION_MEDIA_SEEK_FORWARD_32", "actions/32/media-seek-forward");
    public static final AntuIcon ACTION_MEDIA_SHOW_ACTIVE_TRACK_AMAROK_24 = create("ACTION_MEDIA_SHOW_ACTIVE_TRACK_AMAROK_24", "actions/24/media-show-active-track-amarok");
    public static final AntuIcon ACTION_MEDIA_SKIP_BACKWARD_24 = create("ACTION_MEDIA_SKIP_BACKWARD_24", "actions/24/media-skip-backward");
    public static final AntuIcon ACTION_MEDIA_SKIP_BACKWARD_32 = create("ACTION_MEDIA_SKIP_BACKWARD_32", "actions/32/media-skip-backward");
    public static final AntuIcon ACTION_MEDIA_SKIP_FORWARD_24 = create("ACTION_MEDIA_SKIP_FORWARD_24", "actions/24/media-skip-forward");
    public static final AntuIcon ACTION_MEDIA_SKIP_FORWARD_32 = create("ACTION_MEDIA_SKIP_FORWARD_32", "actions/32/media-skip-forward");
    public static final AntuIcon ACTION_MEDIA_STANDARD_TRACK_PROGRESSION_AMAROK_24 = create("ACTION_MEDIA_STANDARD_TRACK_PROGRESSION_AMAROK_24", "actions/24/media-standard-track-progression-amarok");
    public static final AntuIcon ACTION_MEDIA_TRACK_ADD_AMAROK_24 = create("ACTION_MEDIA_TRACK_ADD_AMAROK_24", "actions/24/media-track-add-amarok");
    public static final AntuIcon ACTION_MEDIA_TRACK_EDIT_AMAROK_24 = create("ACTION_MEDIA_TRACK_EDIT_AMAROK_24", "actions/24/media-track-edit-amarok");
    public static final AntuIcon ACTION_MEDIA_TRACK_QUEUE_AMAROK_24 = create("ACTION_MEDIA_TRACK_QUEUE_AMAROK_24", "actions/24/media-track-queue-amarok");
    public static final AntuIcon ACTION_MEDIA_TRACK_REMOVE_AMAROK_24 = create("ACTION_MEDIA_TRACK_REMOVE_AMAROK_24", "actions/24/media-track-remove-amarok");
    public static final AntuIcon ACTION_MEETING_ATTENDING_24 = create("ACTION_MEETING_ATTENDING_24", "actions/24/meeting-attending");
    public static final AntuIcon ACTION_MEETING_ATTENDING_48 = create("ACTION_MEETING_ATTENDING_48", "actions/48/meeting-attending");
    public static final AntuIcon ACTION_MEETING_ATTENDING_TENTATIVE_24 = create("ACTION_MEETING_ATTENDING_TENTATIVE_24", "actions/24/meeting-attending-tentative");
    public static final AntuIcon ACTION_MEETING_PARTICIPANT_NO_RESPONSE_24 = create("ACTION_MEETING_PARTICIPANT_NO_RESPONSE_24", "actions/24/meeting-participant-no-response");
    public static final AntuIcon ACTION_MEETING_PARTICIPANT_REQUEST_RESPONSE_24 = create("ACTION_MEETING_PARTICIPANT_REQUEST_RESPONSE_24", "actions/24/meeting-participant-request-response");
    public static final AntuIcon ACTION_MENU_NEW_24 = create("ACTION_MENU_NEW_24", "actions/24/menu_new");
    public static final AntuIcon ACTION_MENU_NEW_SEP_24 = create("ACTION_MENU_NEW_SEP_24", "actions/24/menu_new_sep");
    public static final AntuIcon ACTION_MERGE_24 = create("ACTION_MERGE_24", "actions/24/merge");
    public static final AntuIcon ACTION_MESSAGEBOX_WARNING_24 = create("ACTION_MESSAGEBOX_WARNING_24", "actions/24/messagebox_warning");
    public static final AntuIcon ACTION_MIC_OFF_24 = create("ACTION_MIC_OFF_24", "actions/24/mic-off");
    public static final AntuIcon ACTION_MIC_ON_24 = create("ACTION_MIC_ON_24", "actions/24/mic-on");
    public static final AntuIcon ACTION_MIC_READY_24 = create("ACTION_MIC_READY_24", "actions/24/mic-ready");
    public static final AntuIcon ACTION_MILESTONE_24 = create("ACTION_MILESTONE_24", "actions/24/milestone");
    public static final AntuIcon ACTION_MUSIC_AMAROK_24 = create("ACTION_MUSIC_AMAROK_24", "actions/24/music-amarok");
    public static final AntuIcon ACTION_NETWORK_CONNECT_24 = create("ACTION_NETWORK_CONNECT_24", "actions/24/network-connect");
    public static final AntuIcon ACTION_NETWORK_DISCONNECT_24 = create("ACTION_NETWORK_DISCONNECT_24", "actions/24/network-disconnect");
    public static final AntuIcon ACTION_NEW_AUDIO_ALARM_24 = create("ACTION_NEW_AUDIO_ALARM_24", "actions/24/new-audio-alarm");
    public static final AntuIcon ACTION_NEWS_SUBSCRIBE_24 = create("ACTION_NEWS_SUBSCRIBE_24", "actions/24/news-subscribe");
    public static final AntuIcon ACTION_NEWS_UNSUBSCRIBE_24 = create("ACTION_NEWS_UNSUBSCRIBE_24", "actions/24/news-unsubscribe");
    public static final AntuIcon ACTION_NOCOVER_24 = create("ACTION_NOCOVER_24", "actions/24/nocover");
    public static final AntuIcon ACTION_NODE_24 = create("ACTION_NODE_24", "actions/24/node");
    public static final AntuIcon ACTION_NODE_SEGMENT_CURVE_24 = create("ACTION_NODE_SEGMENT_CURVE_24", "actions/24/node-segment-curve");
    public static final AntuIcon ACTION_NODE_SEGMENT_LINE_24 = create("ACTION_NODE_SEGMENT_LINE_24", "actions/24/node-segment-line");
    public static final AntuIcon ACTION_NODE_TRANSFORM_24 = create("ACTION_NODE_TRANSFORM_24", "actions/24/node-transform");
    public static final AntuIcon ACTION_NODE_TYPE_AUTO_SMOOTH_24 = create("ACTION_NODE_TYPE_AUTO_SMOOTH_24", "actions/24/node-type-auto-smooth");
    public static final AntuIcon ACTION_NODE_TYPE_CUSP_24 = create("ACTION_NODE_TYPE_CUSP_24", "actions/24/node-type-cusp");
    public static final AntuIcon ACTION_NODE_TYPE_SMOOTH_24 = create("ACTION_NODE_TYPE_SMOOTH_24", "actions/24/node-type-smooth");
    public static final AntuIcon ACTION_NODE_TYPE_SYMMETRIC_24 = create("ACTION_NODE_TYPE_SYMMETRIC_24", "actions/24/node-type-symmetric");
    public static final AntuIcon ACTION_NOISEREDUCTION_24 = create("ACTION_NOISEREDUCTION_24", "actions/24/noisereduction");
    public static final AntuIcon ACTION_OBJECT_ALIGN_HORIZONTAL_CENTER_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_HORIZONTAL_CENTER_CALLIGRA_24", "actions/24/object-align-horizontal-center-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_HORIZONTAL_LEFT_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_HORIZONTAL_LEFT_CALLIGRA_24", "actions/24/object-align-horizontal-left-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_HORIZONTAL_RIGHT_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_HORIZONTAL_RIGHT_CALLIGRA_24", "actions/24/object-align-horizontal-right-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_VERTICAL_BOTTOM_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_VERTICAL_BOTTOM_CALLIGRA_24", "actions/24/object-align-vertical-bottom-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_VERTICAL_BOTTOM_TOP_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_VERTICAL_BOTTOM_TOP_CALLIGRA_24", "actions/24/object-align-vertical-bottom-top-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_VERTICAL_CENTER_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_VERTICAL_CENTER_CALLIGRA_24", "actions/24/object-align-vertical-center-calligra");
    public static final AntuIcon ACTION_OBJECT_ALIGN_VERTICAL_TOP_CALLIGRA_24 = create("ACTION_OBJECT_ALIGN_VERTICAL_TOP_CALLIGRA_24", "actions/24/object-align-vertical-top-calligra");
    public static final AntuIcon ACTION_OBJECT_COLUMNS_24 = create("ACTION_OBJECT_COLUMNS_24", "actions/24/object-columns");
    public static final AntuIcon ACTION_OBJECT_FILL_24 = create("ACTION_OBJECT_FILL_24", "actions/24/object-fill");
    public static final AntuIcon ACTION_OBJECT_FLIP_HORIZONTAL_24 = create("ACTION_OBJECT_FLIP_HORIZONTAL_24", "actions/24/object-flip-horizontal");
    public static final AntuIcon ACTION_OBJECT_FLIP_VERTICAL_24 = create("ACTION_OBJECT_FLIP_VERTICAL_24", "actions/24/object-flip-vertical");
    public static final AntuIcon ACTION_OBJECT_GROUP_24 = create("ACTION_OBJECT_GROUP_24", "actions/24/object-group");
    public static final AntuIcon ACTION_OBJECT_GROUP_CALLIGRA_24 = create("ACTION_OBJECT_GROUP_CALLIGRA_24", "actions/24/object-group-calligra");
    public static final AntuIcon ACTION_OBJECT_HIDDEN_24 = create("ACTION_OBJECT_HIDDEN_24", "actions/24/object-hidden");
    public static final AntuIcon ACTION_OBJECT_LOCKED_24 = create("ACTION_OBJECT_LOCKED_24", "actions/24/object-locked");
    public static final AntuIcon ACTION_OBJECT_ORDER_BACK_24 = create("ACTION_OBJECT_ORDER_BACK_24", "actions/24/object-order-back");
    public static final AntuIcon ACTION_OBJECT_ORDER_BACK_32 = create("ACTION_OBJECT_ORDER_BACK_32", "actions/32/object-order-back");
    public static final AntuIcon ACTION_OBJECT_ORDER_BACK_CALLIGRA_24 = create("ACTION_OBJECT_ORDER_BACK_CALLIGRA_24", "actions/24/object-order-back-calligra");
    public static final AntuIcon ACTION_OBJECT_ORDER_FRONT_24 = create("ACTION_OBJECT_ORDER_FRONT_24", "actions/24/object-order-front");
    public static final AntuIcon ACTION_OBJECT_ORDER_FRONT_32 = create("ACTION_OBJECT_ORDER_FRONT_32", "actions/32/object-order-front");
    public static final AntuIcon ACTION_OBJECT_ORDER_FRONT_CALLIGRA_24 = create("ACTION_OBJECT_ORDER_FRONT_CALLIGRA_24", "actions/24/object-order-front-calligra");
    public static final AntuIcon ACTION_OBJECT_ORDER_LOWER_24 = create("ACTION_OBJECT_ORDER_LOWER_24", "actions/24/object-order-lower");
    public static final AntuIcon ACTION_OBJECT_ORDER_LOWER_32 = create("ACTION_OBJECT_ORDER_LOWER_32", "actions/32/object-order-lower");
    public static final AntuIcon ACTION_OBJECT_ORDER_LOWER_CALLIGRA_24 = create("ACTION_OBJECT_ORDER_LOWER_CALLIGRA_24", "actions/24/object-order-lower-calligra");
    public static final AntuIcon ACTION_OBJECT_ORDER_RAISE_24 = create("ACTION_OBJECT_ORDER_RAISE_24", "actions/24/object-order-raise");
    public static final AntuIcon ACTION_OBJECT_ORDER_RAISE_32 = create("ACTION_OBJECT_ORDER_RAISE_32", "actions/32/object-order-raise");
    public static final AntuIcon ACTION_OBJECT_ORDER_RAISE_CALLIGRA_24 = create("ACTION_OBJECT_ORDER_RAISE_CALLIGRA_24", "actions/24/object-order-raise-calligra");
    public static final AntuIcon ACTION_OBJECT_ROTATE_LEFT_24 = create("ACTION_OBJECT_ROTATE_LEFT_24", "actions/24/object-rotate-left");
    public static final AntuIcon ACTION_OBJECT_ROTATE_RIGHT_24 = create("ACTION_OBJECT_ROTATE_RIGHT_24", "actions/24/object-rotate-right");
    public static final AntuIcon ACTION_OBJECT_ROWS_24 = create("ACTION_OBJECT_ROWS_24", "actions/24/object-rows");
    public static final AntuIcon ACTION_OBJECT_STROKE_24 = create("ACTION_OBJECT_STROKE_24", "actions/24/object-stroke");
    public static final AntuIcon ACTION_OBJECT_STROKE_STYLE_24 = create("ACTION_OBJECT_STROKE_STYLE_24", "actions/24/object-stroke-style");
    public static final AntuIcon ACTION_OBJECT_TO_PATH_24 = create("ACTION_OBJECT_TO_PATH_24", "actions/24/object-to-path");
    public static final AntuIcon ACTION_OBJECT_UNGROUP_24 = create("ACTION_OBJECT_UNGROUP_24", "actions/24/object-ungroup");
    public static final AntuIcon ACTION_OBJECT_UNGROUP_CALLIGRA_24 = create("ACTION_OBJECT_UNGROUP_CALLIGRA_24", "actions/24/object-ungroup-calligra");
    public static final AntuIcon ACTION_OBJECT_UNLOCKED_24 = create("ACTION_OBJECT_UNLOCKED_24", "actions/24/object-unlocked");
    public static final AntuIcon ACTION_OBJECT_VISIBLE_24 = create("ACTION_OBJECT_VISIBLE_24", "actions/24/object-visible");
    public static final AntuIcon ACTION_OFFICE_CHART_AREA_24 = create("ACTION_OFFICE_CHART_AREA_24", "actions/24/office-chart-area");
    public static final AntuIcon ACTION_OFFICE_CHART_AREA_FOCUS_PEAK_NODE_24 = create("ACTION_OFFICE_CHART_AREA_FOCUS_PEAK_NODE_24", "actions/24/office-chart-area-focus-peak-node");
    public static final AntuIcon ACTION_OFFICE_CHART_AREA_PERCENTAGE_24 = create("ACTION_OFFICE_CHART_AREA_PERCENTAGE_24", "actions/24/office-chart-area-percentage");
    public static final AntuIcon ACTION_OFFICE_CHART_AREA_STACKED_24 = create("ACTION_OFFICE_CHART_AREA_STACKED_24", "actions/24/office-chart-area-stacked");
    public static final AntuIcon ACTION_OFFICE_CHART_BAR_24 = create("ACTION_OFFICE_CHART_BAR_24", "actions/24/office-chart-bar");
    public static final AntuIcon ACTION_OFFICE_CHART_BAR_PERCENTAGE_24 = create("ACTION_OFFICE_CHART_BAR_PERCENTAGE_24", "actions/24/office-chart-bar-percentage");
    public static final AntuIcon ACTION_OFFICE_CHART_BAR_STACKED_24 = create("ACTION_OFFICE_CHART_BAR_STACKED_24", "actions/24/office-chart-bar-stacked");
    public static final AntuIcon ACTION_OFFICE_CHART_LINE_24 = create("ACTION_OFFICE_CHART_LINE_24", "actions/24/office-chart-line");
    public static final AntuIcon ACTION_OFFICE_CHART_LINE_PERCENTAGE_24 = create("ACTION_OFFICE_CHART_LINE_PERCENTAGE_24", "actions/24/office-chart-line-percentage");
    public static final AntuIcon ACTION_OFFICE_CHART_LINE_STACKED_24 = create("ACTION_OFFICE_CHART_LINE_STACKED_24", "actions/24/office-chart-line-stacked");
    public static final AntuIcon ACTION_OFFICE_CHART_PIE_24 = create("ACTION_OFFICE_CHART_PIE_24", "actions/24/office-chart-pie");
    public static final AntuIcon ACTION_OFFICE_CHART_POLAR_24 = create("ACTION_OFFICE_CHART_POLAR_24", "actions/24/office-chart-polar");
    public static final AntuIcon ACTION_OFFICE_CHART_POLAR_STACKED_24 = create("ACTION_OFFICE_CHART_POLAR_STACKED_24", "actions/24/office-chart-polar-stacked");
    public static final AntuIcon ACTION_OFFICE_CHART_RING_24 = create("ACTION_OFFICE_CHART_RING_24", "actions/24/office-chart-ring");
    public static final AntuIcon ACTION_OFFICE_CHART_SCATTER_24 = create("ACTION_OFFICE_CHART_SCATTER_24", "actions/24/office-chart-scatter");
    public static final AntuIcon ACTION_OFFLINE_SETTINGS_24 = create("ACTION_OFFLINE_SETTINGS_24", "actions/24/offline-settings");
    public static final AntuIcon ACTION_OILPAINT_24 = create("ACTION_OILPAINT_24", "actions/24/oilpaint");
    public static final AntuIcon ACTION_OVEREXPOSURE_24 = create("ACTION_OVEREXPOSURE_24", "actions/24/overexposure");
    public static final AntuIcon ACTION_PAGE_2SIDES_24 = create("ACTION_PAGE_2SIDES_24", "actions/24/page-2sides");
    public static final AntuIcon ACTION_PAGE_3SIDES_24 = create("ACTION_PAGE_3SIDES_24", "actions/24/page-3sides");
    public static final AntuIcon ACTION_PAGE_4SIDES_24 = create("ACTION_PAGE_4SIDES_24", "actions/24/page-4sides");
    public static final AntuIcon ACTION_PAGE_SIMPLE_24 = create("ACTION_PAGE_SIMPLE_24", "actions/24/page-simple");
    public static final AntuIcon ACTION_PAGE_ZOOM_24 = create("ACTION_PAGE_ZOOM_24", "actions/24/page-zoom");
    public static final AntuIcon ACTION_PAINT_GRADIENT_LINEAR_24 = create("ACTION_PAINT_GRADIENT_LINEAR_24", "actions/24/paint-gradient-linear");
    public static final AntuIcon ACTION_PAINT_GRADIENT_RADIAL_24 = create("ACTION_PAINT_GRADIENT_RADIAL_24", "actions/24/paint-gradient-radial");
    public static final AntuIcon ACTION_PAINT_NONE_24 = create("ACTION_PAINT_NONE_24", "actions/24/paint-none");
    public static final AntuIcon ACTION_PAINT_PATTERN_24 = create("ACTION_PAINT_PATTERN_24", "actions/24/paint-pattern");
    public static final AntuIcon ACTION_PAINT_SOLID_24 = create("ACTION_PAINT_SOLID_24", "actions/24/paint-solid");
    public static final AntuIcon ACTION_PAINT_SWATCH_24 = create("ACTION_PAINT_SWATCH_24", "actions/24/paint-swatch");
    public static final AntuIcon ACTION_PAINT_UNKNOWN_24 = create("ACTION_PAINT_UNKNOWN_24", "actions/24/paint-unknown");
    public static final AntuIcon ACTION_PASSWORD_COPY_24 = create("ACTION_PASSWORD_COPY_24", "actions/24/password-copy");
    public static final AntuIcon ACTION_PATH_BREAK_APART_24 = create("ACTION_PATH_BREAK_APART_24", "actions/24/path-break-apart");
    public static final AntuIcon ACTION_PATH_CLIP_EDIT_24 = create("ACTION_PATH_CLIP_EDIT_24", "actions/24/path-clip-edit");
    public static final AntuIcon ACTION_PATH_COMBINE_24 = create("ACTION_PATH_COMBINE_24", "actions/24/path-combine");
    public static final AntuIcon ACTION_PATH_CUT_24 = create("ACTION_PATH_CUT_24", "actions/24/path-cut");
    public static final AntuIcon ACTION_PATH_DIFFERENCE_24 = create("ACTION_PATH_DIFFERENCE_24", "actions/24/path-difference");
    public static final AntuIcon ACTION_PATH_DIVISION_24 = create("ACTION_PATH_DIVISION_24", "actions/24/path-division");
    public static final AntuIcon ACTION_PATH_EFFECT_PARAMETER_NEXT_24 = create("ACTION_PATH_EFFECT_PARAMETER_NEXT_24", "actions/24/path-effect-parameter-next");
    public static final AntuIcon ACTION_PATH_EXCLUSION_24 = create("ACTION_PATH_EXCLUSION_24", "actions/24/path-exclusion");
    public static final AntuIcon ACTION_PATH_INSET_24 = create("ACTION_PATH_INSET_24", "actions/24/path-inset");
    public static final AntuIcon ACTION_PATH_INTERSECTION_24 = create("ACTION_PATH_INTERSECTION_24", "actions/24/path-intersection");
    public static final AntuIcon ACTION_PATH_MASK_EDIT_24 = create("ACTION_PATH_MASK_EDIT_24", "actions/24/path-mask-edit");
    public static final AntuIcon ACTION_PATH_MODE_POLYLINE_24 = create("ACTION_PATH_MODE_POLYLINE_24", "actions/24/path-mode-polyline");
    public static final AntuIcon ACTION_PATH_MODE_POLYLINE_PARAXIAL_24 = create("ACTION_PATH_MODE_POLYLINE_PARAXIAL_24", "actions/24/path-mode-polyline-paraxial");
    public static final AntuIcon ACTION_PATH_OFFSET_DYNAMIC_24 = create("ACTION_PATH_OFFSET_DYNAMIC_24", "actions/24/path-offset-dynamic");
    public static final AntuIcon ACTION_PATH_OFFSET_LINKED_24 = create("ACTION_PATH_OFFSET_LINKED_24", "actions/24/path-offset-linked");
    public static final AntuIcon ACTION_PATH_OUTSET_24 = create("ACTION_PATH_OUTSET_24", "actions/24/path-outset");
    public static final AntuIcon ACTION_PATH_REVERSE_24 = create("ACTION_PATH_REVERSE_24", "actions/24/path-reverse");
    public static final AntuIcon ACTION_PATH_SIMPLIFY_24 = create("ACTION_PATH_SIMPLIFY_24", "actions/24/path-simplify");
    public static final AntuIcon ACTION_PATH_UNION_24 = create("ACTION_PATH_UNION_24", "actions/24/path-union");
    public static final AntuIcon ACTION_PDF_ANNOTATIONS_24 = create("ACTION_PDF_ANNOTATIONS_24", "actions/24/pdf-annotations");
    public static final AntuIcon ACTION_PERCENT_24 = create("ACTION_PERCENT_24", "actions/24/percent");
    public static final AntuIcon ACTION_PERCENT_32 = create("ACTION_PERCENT_32", "actions/32/percent");
    public static final AntuIcon ACTION_PERSPECTIVE_24 = create("ACTION_PERSPECTIVE_24", "actions/24/perspective");
    public static final AntuIcon ACTION_PHOTOS_AMAROK_24 = create("ACTION_PHOTOS_AMAROK_24", "actions/24/photos-amarok");
    public static final AntuIcon ACTION_PIXELART_TRACE_24 = create("ACTION_PIXELART_TRACE_24", "actions/24/pixelart-trace");
    public static final AntuIcon ACTION_PLAYER_TIME_24 = create("ACTION_PLAYER_TIME_24", "actions/24/player-time");
    public static final AntuIcon ACTION_PLAYER_VOLUME_24 = create("ACTION_PLAYER_VOLUME_24", "actions/24/player-volume");
    public static final AntuIcon ACTION_PLAYER_VOLUME_MUTED_24 = create("ACTION_PLAYER_VOLUME_MUTED_24", "actions/24/player-volume-muted");
    public static final AntuIcon ACTION_PLAYLIST_GENERATOR_24 = create("ACTION_PLAYLIST_GENERATOR_24", "actions/24/playlist-generator");
    public static final AntuIcon ACTION_PLAYLIST_SORT_24 = create("ACTION_PLAYLIST_SORT_24", "actions/24/playlist-sort");
    public static final AntuIcon ACTION_PLUGINS_24 = create("ACTION_PLUGINS_24", "actions/24/plugins");
    public static final AntuIcon ACTION_PODCAST_AMAROK_24 = create("ACTION_PODCAST_AMAROK_24", "actions/24/podcast-amarok");
    public static final AntuIcon ACTION_PORT_24 = create("ACTION_PORT_24", "actions/24/port");
    public static final AntuIcon ACTION_PRECONDITION_24 = create("ACTION_PRECONDITION_24", "actions/24/precondition");
    public static final AntuIcon ACTION_PREFERENCES_ACTIVITIES_24 = create("ACTION_PREFERENCES_ACTIVITIES_24", "actions/24/preferences-activities");
    public static final AntuIcon ACTION_PREFERENCES_ACTIVITIES_48 = create("ACTION_PREFERENCES_ACTIVITIES_48", "actions/48/preferences-activities");
    public static final AntuIcon ACTION_PREFERENCES_INDICATOR_AMAROK_24 = create("ACTION_PREFERENCES_INDICATOR_AMAROK_24", "actions/24/preferences-indicator-amarok");
    public static final AntuIcon ACTION_PREFERENCES_MEDIA_PLAYBACK_AMAROK_24 = create("ACTION_PREFERENCES_MEDIA_PLAYBACK_AMAROK_24", "actions/24/preferences-media-playback-amarok");
    public static final AntuIcon ACTION_PREFERENCES_OTHER_24 = create("ACTION_PREFERENCES_OTHER_24", "actions/24/preferences-other");
    public static final AntuIcon ACTION_PREFLIGHT_VERIFIER_24 = create("ACTION_PREFLIGHT_VERIFIER_24", "actions/24/preflight-verifier");
    public static final AntuIcon ACTION_PREPOSTCONDITION_24 = create("ACTION_PREPOSTCONDITION_24", "actions/24/PrePostCondition");
    public static final AntuIcon ACTION_PRIMARYKEY_CONSTRAINT_24 = create("ACTION_PRIMARYKEY_CONSTRAINT_24", "actions/24/primarykey_constraint");
    public static final AntuIcon ACTION_PROCESS_STOP_24 = create("ACTION_PROCESS_STOP_24", "actions/24/process-stop");
    public static final AntuIcon ACTION_PROJECT_DEVELOPMENT_24 = create("ACTION_PROJECT_DEVELOPMENT_24", "actions/24/project-development");
    public static final AntuIcon ACTION_PROJECT_DEVELOPMENT_CLOSE_24 = create("ACTION_PROJECT_DEVELOPMENT_CLOSE_24", "actions/24/project-development-close");
    public static final AntuIcon ACTION_PROJECT_DEVELOPMENT_CLOSE_ALL_24 = create("ACTION_PROJECT_DEVELOPMENT_CLOSE_ALL_24", "actions/24/project-development-close-all");
    public static final AntuIcon ACTION_PROJECT_DEVELOPMENT_NEW_TEMPLATE_24 = create("ACTION_PROJECT_DEVELOPMENT_NEW_TEMPLATE_24", "actions/24/project-development-new-template");
    public static final AntuIcon ACTION_PROJECT_OPEN_24 = create("ACTION_PROJECT_OPEN_24", "actions/24/project-open");
    public static final AntuIcon ACTION_QUICKOPEN_24 = create("ACTION_QUICKOPEN_24", "actions/24/quickopen");
    public static final AntuIcon ACTION_QUICKOPEN_CLASS_24 = create("ACTION_QUICKOPEN_CLASS_24", "actions/24/quickopen-class");
    public static final AntuIcon ACTION_QUICKOPEN_FILE_24 = create("ACTION_QUICKOPEN_FILE_24", "actions/24/quickopen-file");
    public static final AntuIcon ACTION_QUICKOPEN_FUNCTION_24 = create("ACTION_QUICKOPEN_FUNCTION_24", "actions/24/quickopen-function");
    public static final AntuIcon ACTION_QUICKVIEW_24 = create("ACTION_QUICKVIEW_24", "actions/24/quickview");
    public static final AntuIcon ACTION_RAINDROP_24 = create("ACTION_RAINDROP_24", "actions/24/raindrop");
    public static final AntuIcon ACTION_RATING_24 = create("ACTION_RATING_24", "actions/24/rating");
    public static final AntuIcon ACTION_RATING_UNRATED_24 = create("ACTION_RATING_UNRATED_24", "actions/24/rating-unrated");
    public static final AntuIcon ACTION_RATIOCROP_24 = create("ACTION_RATIOCROP_24", "actions/24/ratiocrop");
    public static final AntuIcon ACTION_REALIZATION_24 = create("ACTION_REALIZATION_24", "actions/24/realization");
    public static final AntuIcon ACTION_RECTANGLE_MAKE_CORNERS_SHARP_24 = create("ACTION_RECTANGLE_MAKE_CORNERS_SHARP_24", "actions/24/rectangle-make-corners-sharp");
    public static final AntuIcon ACTION_REDEYES_24 = create("ACTION_REDEYES_24", "actions/24/redeyes");
    public static final AntuIcon ACTION_REMOVE_AMAROK_24 = create("ACTION_REMOVE_AMAROK_24", "actions/24/remove-amarok");
    public static final AntuIcon ACTION_REMOVE_LINK_24 = create("ACTION_REMOVE_LINK_24", "actions/24/remove-link");
    public static final AntuIcon ACTION_RESOURCE_CALENDAR_CHILD_24 = create("ACTION_RESOURCE_CALENDAR_CHILD_24", "actions/24/resource-calendar-child");
    public static final AntuIcon ACTION_RESOURCE_CALENDAR_CHILD_INSERT_24 = create("ACTION_RESOURCE_CALENDAR_CHILD_INSERT_24", "actions/24/resource-calendar-child-insert");
    public static final AntuIcon ACTION_RESOURCE_CALENDAR_INSERT_24 = create("ACTION_RESOURCE_CALENDAR_INSERT_24", "actions/24/resource-calendar-insert");
    public static final AntuIcon ACTION_RESOURCE_GROUP_24 = create("ACTION_RESOURCE_GROUP_24", "actions/24/resource-group");
    public static final AntuIcon ACTION_RESOURCE_GROUP_NEW_24 = create("ACTION_RESOURCE_GROUP_NEW_24", "actions/24/resource-group-new");
    public static final AntuIcon ACTION_RESTORATION_24 = create("ACTION_RESTORATION_24", "actions/24/restoration");
    public static final AntuIcon ACTION_RETWEET_24 = create("ACTION_RETWEET_24", "actions/24/retweet");
    public static final AntuIcon ACTION_REVERSE_24 = create("ACTION_REVERSE_24", "actions/24/reverse");
    public static final AntuIcon ACTION_ROLL_24 = create("ACTION_ROLL_24", "actions/24/roll");
    public static final AntuIcon ACTION_RUN_BUILD_24 = create("ACTION_RUN_BUILD_24", "actions/24/run-build");
    public static final AntuIcon ACTION_RUN_BUILD_CLEAN_24 = create("ACTION_RUN_BUILD_CLEAN_24", "actions/24/run-build-clean");
    public static final AntuIcon ACTION_RUN_BUILD_CONFIGURE_24 = create("ACTION_RUN_BUILD_CONFIGURE_24", "actions/24/run-build-configure");
    public static final AntuIcon ACTION_RUN_BUILD_FILE_24 = create("ACTION_RUN_BUILD_FILE_24", "actions/24/run-build-file");
    public static final AntuIcon ACTION_RUN_BUILD_INSTALL_24 = create("ACTION_RUN_BUILD_INSTALL_24", "actions/24/run-build-install");
    public static final AntuIcon ACTION_RUN_BUILD_INSTALL_ROOT_24 = create("ACTION_RUN_BUILD_INSTALL_ROOT_24", "actions/24/run-build-install-root");
    public static final AntuIcon ACTION_RUN_BUILD_PRUNE_24 = create("ACTION_RUN_BUILD_PRUNE_24", "actions/24/run-build-prune");
    public static final AntuIcon ACTION_RUN_CLEAN_24 = create("ACTION_RUN_CLEAN_24", "actions/24/run-clean");
    public static final AntuIcon ACTION_RUN_INSTALL_24 = create("ACTION_RUN_INSTALL_24", "actions/24/run-install");
    public static final AntuIcon ACTION_SEARCH_24 = create("ACTION_SEARCH_24", "actions/24/search");
    public static final AntuIcon ACTION_SELECT_RECTANGULAR_24 = create("ACTION_SELECT_RECTANGULAR_24", "actions/24/select-rectangular");
    public static final AntuIcon ACTION_SELECTION_BOTTOM_24 = create("ACTION_SELECTION_BOTTOM_24", "actions/24/selection-bottom");
    public static final AntuIcon ACTION_SELECTION_LOWER_24 = create("ACTION_SELECTION_LOWER_24", "actions/24/selection-lower");
    public static final AntuIcon ACTION_SELECTION_MAKE_BITMAP_COPY_24 = create("ACTION_SELECTION_MAKE_BITMAP_COPY_24", "actions/24/selection-make-bitmap-copy");
    public static final AntuIcon ACTION_SELECTION_MOVE_TO_LAYER_ABOVE_24 = create("ACTION_SELECTION_MOVE_TO_LAYER_ABOVE_24", "actions/24/selection-move-to-layer-above");
    public static final AntuIcon ACTION_SELECTION_MOVE_TO_LAYER_BELOW_24 = create("ACTION_SELECTION_MOVE_TO_LAYER_BELOW_24", "actions/24/selection-move-to-layer-below");
    public static final AntuIcon ACTION_SELECTION_RAISE_24 = create("ACTION_SELECTION_RAISE_24", "actions/24/selection-raise");
    public static final AntuIcon ACTION_SELECTION_TOP_24 = create("ACTION_SELECTION_TOP_24", "actions/24/selection-top");
    public static final AntuIcon ACTION_SEND_SIGNAL_24 = create("ACTION_SEND_SIGNAL_24", "actions/24/send_signal");
    public static final AntuIcon ACTION_SERVICES_24 = create("ACTION_SERVICES_24", "actions/24/services");
    public static final AntuIcon ACTION_SHAPE_CHOOSE_24 = create("ACTION_SHAPE_CHOOSE_24", "actions/24/shape-choose");
    public static final AntuIcon ACTION_SHAPES_24 = create("ACTION_SHAPES_24", "actions/24/shapes");
    public static final AntuIcon ACTION_SHARPENIMAGE_24 = create("ACTION_SHARPENIMAGE_24", "actions/24/sharpenimage");
    public static final AntuIcon ACTION_SHEAR_24 = create("ACTION_SHEAR_24", "actions/24/shear");
    public static final AntuIcon ACTION_SHIFT_DOWN_Y_24 = create("ACTION_SHIFT_DOWN_Y_24", "actions/24/shift-down-y");
    public static final AntuIcon ACTION_SHIFT_LEFT_X_24 = create("ACTION_SHIFT_LEFT_X_24", "actions/24/shift-left-x");
    public static final AntuIcon ACTION_SHIFT_RIGHT_X_24 = create("ACTION_SHIFT_RIGHT_X_24", "actions/24/shift-right-x");
    public static final AntuIcon ACTION_SHIFT_UP_Y_24 = create("ACTION_SHIFT_UP_Y_24", "actions/24/shift-up-y");
    public static final AntuIcon ACTION_SHOW_DIALOGS_24 = create("ACTION_SHOW_DIALOGS_24", "actions/24/show-dialogs");
    public static final AntuIcon ACTION_SHOW_GRID_24 = create("ACTION_SHOW_GRID_24", "actions/24/show-grid");
    public static final AntuIcon ACTION_SHOW_GUIDES_24 = create("ACTION_SHOW_GUIDES_24", "actions/24/show-guides");
    public static final AntuIcon ACTION_SHOW_MENU_24 = create("ACTION_SHOW_MENU_24", "actions/24/show-menu");
    public static final AntuIcon ACTION_SHOW_NODE_HANDLES_24 = create("ACTION_SHOW_NODE_HANDLES_24", "actions/24/show-node-handles");
    public static final AntuIcon ACTION_SHOW_OFFLINE_24 = create("ACTION_SHOW_OFFLINE_24", "actions/24/show-offline");
    public static final AntuIcon ACTION_SHOW_PATH_OUTLINE_24 = create("ACTION_SHOW_PATH_OUTLINE_24", "actions/24/show-path-outline");
    public static final AntuIcon ACTION_SIMILARARTISTS_AMAROK_24 = create("ACTION_SIMILARARTISTS_AMAROK_24", "actions/24/similarartists-amarok");
    public static final AntuIcon ACTION_SKG_CHART_BUBBLE_24 = create("ACTION_SKG_CHART_BUBBLE_24", "actions/24/skg-chart-bubble");
    public static final AntuIcon ACTION_SKROOGE_CREDIT_CARD_24 = create("ACTION_SKROOGE_CREDIT_CARD_24", "actions/24/skrooge_credit_card");
    public static final AntuIcon ACTION_SKROOGE_LESS_24 = create("ACTION_SKROOGE_LESS_24", "actions/24/skrooge_less");
    public static final AntuIcon ACTION_SKROOGE_MORE_24 = create("ACTION_SKROOGE_MORE_24", "actions/24/skrooge_more");
    public static final AntuIcon ACTION_SKROOGE_MUCH_LESS_24 = create("ACTION_SKROOGE_MUCH_LESS_24", "actions/24/skrooge_much_less");
    public static final AntuIcon ACTION_SKROOGE_MUCH_MORE_24 = create("ACTION_SKROOGE_MUCH_MORE_24", "actions/24/skrooge_much_more");
    public static final AntuIcon ACTION_SKROOGE_TYPE_24 = create("ACTION_SKROOGE_TYPE_24", "actions/24/skrooge_type");
    public static final AntuIcon ACTION_SMALLCLOCK_24 = create("ACTION_SMALLCLOCK_24", "actions/24/smallclock");
    public static final AntuIcon ACTION_SNAP_24 = create("ACTION_SNAP_24", "actions/24/snap");
    public static final AntuIcon ACTION_SNAP_BOUNDING_BOX_CENTER_24 = create("ACTION_SNAP_BOUNDING_BOX_CENTER_24", "actions/24/snap-bounding-box-center");
    public static final AntuIcon ACTION_SNAP_BOUNDING_BOX_CORNERS_24 = create("ACTION_SNAP_BOUNDING_BOX_CORNERS_24", "actions/24/snap-bounding-box-corners");
    public static final AntuIcon ACTION_SNAP_BOUNDING_BOX_EDGES_24 = create("ACTION_SNAP_BOUNDING_BOX_EDGES_24", "actions/24/snap-bounding-box-edges");
    public static final AntuIcon ACTION_SNAP_BOUNDING_BOX_MIDPOINTS_24 = create("ACTION_SNAP_BOUNDING_BOX_MIDPOINTS_24", "actions/24/snap-bounding-box-midpoints");
    public static final AntuIcon ACTION_SNAP_EXTENSION_24 = create("ACTION_SNAP_EXTENSION_24", "actions/24/snap-extension");
    public static final AntuIcon ACTION_SNAP_GRID_GUIDE_INTERSECTIONS_24 = create("ACTION_SNAP_GRID_GUIDE_INTERSECTIONS_24", "actions/24/snap-grid-guide-intersections");
    public static final AntuIcon ACTION_SNAP_GUIDELINE_24 = create("ACTION_SNAP_GUIDELINE_24", "actions/24/snap-guideline");
    public static final AntuIcon ACTION_SNAP_INTERSECTION_24 = create("ACTION_SNAP_INTERSECTION_24", "actions/24/snap-intersection");
    public static final AntuIcon ACTION_SNAP_NODE_24 = create("ACTION_SNAP_NODE_24", "actions/24/snap-node");
    public static final AntuIcon ACTION_SNAP_NODES_CENTER_24 = create("ACTION_SNAP_NODES_CENTER_24", "actions/24/snap-nodes-center");
    public static final AntuIcon ACTION_SNAP_NODES_CUSP_24 = create("ACTION_SNAP_NODES_CUSP_24", "actions/24/snap-nodes-cusp");
    public static final AntuIcon ACTION_SNAP_NODES_INTERSECTION_24 = create("ACTION_SNAP_NODES_INTERSECTION_24", "actions/24/snap-nodes-intersection");
    public static final AntuIcon ACTION_SNAP_NODES_MIDPOINT_24 = create("ACTION_SNAP_NODES_MIDPOINT_24", "actions/24/snap-nodes-midpoint");
    public static final AntuIcon ACTION_SNAP_NODES_PATH_24 = create("ACTION_SNAP_NODES_PATH_24", "actions/24/snap-nodes-path");
    public static final AntuIcon ACTION_SNAP_NODES_ROTATION_CENTER_24 = create("ACTION_SNAP_NODES_ROTATION_CENTER_24", "actions/24/snap-nodes-rotation-center");
    public static final AntuIcon ACTION_SNAP_NODES_SMOOTH_24 = create("ACTION_SNAP_NODES_SMOOTH_24", "actions/24/snap-nodes-smooth");
    public static final AntuIcon ACTION_SNAP_ORTHOGONAL_24 = create("ACTION_SNAP_ORTHOGONAL_24", "actions/24/snap-orthogonal");
    public static final AntuIcon ACTION_SNAP_PAGE_24 = create("ACTION_SNAP_PAGE_24", "actions/24/snap-page");
    public static final AntuIcon ACTION_SNAP_TEXT_BASELINE_24 = create("ACTION_SNAP_TEXT_BASELINE_24", "actions/24/snap-text-baseline");
    public static final AntuIcon ACTION_SORT_NAME_24 = create("ACTION_SORT_NAME_24", "actions/24/sort-name");
    public static final AntuIcon ACTION_SORT_PRESENCE_24 = create("ACTION_SORT_PRESENCE_24", "actions/24/sort-presence");
    public static final AntuIcon ACTION_SPEAKER_24 = create("ACTION_SPEAKER_24", "actions/24/speaker");
    public static final AntuIcon ACTION_SPLIT_24 = create("ACTION_SPLIT_24", "actions/24/split");
    public static final AntuIcon ACTION_STANDARD_CONNECTOR_24 = create("ACTION_STANDARD_CONNECTOR_24", "actions/24/standard-connector");
    public static final AntuIcon ACTION_STATE_FORK_24 = create("ACTION_STATE_FORK_24", "actions/24/state-fork");
    public static final AntuIcon ACTION_STICKY_ZOOM_24 = create("ACTION_STICKY_ZOOM_24", "actions/24/sticky_zoom");
    public static final AntuIcon ACTION_STORY_EDITOR_24 = create("ACTION_STORY_EDITOR_24", "actions/24/story-editor");
    public static final AntuIcon ACTION_STRAIGHT_CONNECTOR_24 = create("ACTION_STRAIGHT_CONNECTOR_24", "actions/24/straight-connector");
    public static final AntuIcon ACTION_STROKE_CAP_BEVEL_24 = create("ACTION_STROKE_CAP_BEVEL_24", "actions/24/stroke-cap-bevel");
    public static final AntuIcon ACTION_STROKE_CAP_BUTT_24 = create("ACTION_STROKE_CAP_BUTT_24", "actions/24/stroke-cap-butt");
    public static final AntuIcon ACTION_STROKE_CAP_MITER_24 = create("ACTION_STROKE_CAP_MITER_24", "actions/24/stroke-cap-miter");
    public static final AntuIcon ACTION_STROKE_CAP_ROUND_24 = create("ACTION_STROKE_CAP_ROUND_24", "actions/24/stroke-cap-round");
    public static final AntuIcon ACTION_STROKE_CAP_SQUARE_24 = create("ACTION_STROKE_CAP_SQUARE_24", "actions/24/stroke-cap-square");
    public static final AntuIcon ACTION_STROKE_JOIN_BEVEL_24 = create("ACTION_STROKE_JOIN_BEVEL_24", "actions/24/stroke-join-bevel");
    public static final AntuIcon ACTION_STROKE_JOIN_MITER_24 = create("ACTION_STROKE_JOIN_MITER_24", "actions/24/stroke-join-miter");
    public static final AntuIcon ACTION_STROKE_JOIN_ROUND_24 = create("ACTION_STROKE_JOIN_ROUND_24", "actions/24/stroke-join-round");
    public static final AntuIcon ACTION_STROKE_TO_PATH_24 = create("ACTION_STROKE_TO_PATH_24", "actions/24/stroke-to-path");
    public static final AntuIcon ACTION_SVN_COMMIT_24 = create("ACTION_SVN_COMMIT_24", "actions/24/svn-commit");
    public static final AntuIcon ACTION_SVN_UPDATE_24 = create("ACTION_SVN_UPDATE_24", "actions/24/svn-update");
    public static final AntuIcon ACTION_SYMBOLS_24 = create("ACTION_SYMBOLS_24", "actions/24/symbols");
    public static final AntuIcon ACTION_SYSTEM_RUN_24 = create("ACTION_SYSTEM_RUN_24", "actions/24/system-run");
    public static final AntuIcon ACTION_SYSTEM_SEARCH_24 = create("ACTION_SYSTEM_SEARCH_24", "actions/24/system-search");
    public static final AntuIcon ACTION_TAB_CLOSE_24 = create("ACTION_TAB_CLOSE_24", "actions/24/tab-close");
    public static final AntuIcon ACTION_TAB_CLOSE_OTHER_24 = create("ACTION_TAB_CLOSE_OTHER_24", "actions/24/tab-close-other");
    public static final AntuIcon ACTION_TAB_DETACH_24 = create("ACTION_TAB_DETACH_24", "actions/24/tab-detach");
    public static final AntuIcon ACTION_TAB_DUPLICATE_24 = create("ACTION_TAB_DUPLICATE_24", "actions/24/tab-duplicate");
    public static final AntuIcon ACTION_TAB_NEW_24 = create("ACTION_TAB_NEW_24", "actions/24/tab-new");
    public static final AntuIcon ACTION_TAB_NEW_48 = create("ACTION_TAB_NEW_48", "actions/48/tab-new");
    public static final AntuIcon ACTION_TAB_NEW_BACKGROUND_24 = create("ACTION_TAB_NEW_BACKGROUND_24", "actions/24/tab-new-background");
    public static final AntuIcon ACTION_TABLE_24 = create("ACTION_TABLE_24", "actions/24/table");
    public static final AntuIcon ACTION_TAG_24 = create("ACTION_TAG_24", "actions/24/tag");
    public static final AntuIcon ACTION_TAG_ADDRESSBOOK_24 = create("ACTION_TAG_ADDRESSBOOK_24", "actions/24/tag-addressbook");
    public static final AntuIcon ACTION_TAG_ASSIGNED_24 = create("ACTION_TAG_ASSIGNED_24", "actions/24/tag-assigned");
    public static final AntuIcon ACTION_TAG_DELETE_24 = create("ACTION_TAG_DELETE_24", "actions/24/tag-delete");
    public static final AntuIcon ACTION_TAG_EVENTS_24 = create("ACTION_TAG_EVENTS_24", "actions/24/tag-events");
    public static final AntuIcon ACTION_TAG_FOLDER_24 = create("ACTION_TAG_FOLDER_24", "actions/24/tag-folder");
    public static final AntuIcon ACTION_TAG_NEW_24 = create("ACTION_TAG_NEW_24", "actions/24/tag-new");
    public static final AntuIcon ACTION_TAG_PEOPLE_24 = create("ACTION_TAG_PEOPLE_24", "actions/24/tag-people");
    public static final AntuIcon ACTION_TAG_PLACES_24 = create("ACTION_TAG_PLACES_24", "actions/24/tag-places");
    public static final AntuIcon ACTION_TAG_PROPERTIES_24 = create("ACTION_TAG_PROPERTIES_24", "actions/24/tag-properties");
    public static final AntuIcon ACTION_TAG_RECENTS_24 = create("ACTION_TAG_RECENTS_24", "actions/24/tag-recents");
    public static final AntuIcon ACTION_TAG_RESET_24 = create("ACTION_TAG_RESET_24", "actions/24/tag-reset");
    public static final AntuIcon ACTION_TASK_NEW_24 = create("ACTION_TASK_NEW_24", "actions/24/task-new");
    public static final AntuIcon ACTION_TAXES_FINANCES_24 = create("ACTION_TAXES_FINANCES_24", "actions/24/taxes-finances");
    public static final AntuIcon ACTION_TEX_LOGO_24 = create("ACTION_TEX_LOGO_24", "actions/24/TeX_logo");
    public static final AntuIcon ACTION_TEXT_FIELD_24 = create("ACTION_TEXT_FIELD_24", "actions/24/text-field");
    public static final AntuIcon ACTION_TEXT_FLOW_INTO_FRAME_24 = create("ACTION_TEXT_FLOW_INTO_FRAME_24", "actions/24/text-flow-into-frame");
    public static final AntuIcon ACTION_TEXT_FRAME_LINK_24 = create("ACTION_TEXT_FRAME_LINK_24", "actions/24/text-frame-link");
    public static final AntuIcon ACTION_TEXT_FRAME_UNLINK_24 = create("ACTION_TEXT_FRAME_UNLINK_24", "actions/24/text-frame-unlink");
    public static final AntuIcon ACTION_TEXT_HORZ_KERN_24 = create("ACTION_TEXT_HORZ_KERN_24", "actions/24/text_horz_kern");
    public static final AntuIcon ACTION_TEXT_LETTER_SPACING_24 = create("ACTION_TEXT_LETTER_SPACING_24", "actions/24/text_letter_spacing");
    public static final AntuIcon ACTION_TEXT_LINE_SPACING_24 = create("ACTION_TEXT_LINE_SPACING_24", "actions/24/text_line_spacing");
    public static final AntuIcon ACTION_TEXT_PUT_ON_PATH_24 = create("ACTION_TEXT_PUT_ON_PATH_24", "actions/24/text-put-on-path");
    public static final AntuIcon ACTION_TEXT_REMOVE_FROM_PATH_24 = create("ACTION_TEXT_REMOVE_FROM_PATH_24", "actions/24/text-remove-from-path");
    public static final AntuIcon ACTION_TEXT_REMOVE_KERNS_24 = create("ACTION_TEXT_REMOVE_KERNS_24", "actions/24/text_remove_kerns");
    public static final AntuIcon ACTION_TEXT_ROTATION_24 = create("ACTION_TEXT_ROTATION_24", "actions/24/text_rotation");
    public static final AntuIcon ACTION_TEXT_SPEAK_24 = create("ACTION_TEXT_SPEAK_24", "actions/24/text-speak");
    public static final AntuIcon ACTION_TEXT_SUBSCRIPT_24 = create("ACTION_TEXT_SUBSCRIPT_24", "actions/24/text_subscript");
    public static final AntuIcon ACTION_TEXT_SUPERSCRIPT_24 = create("ACTION_TEXT_SUPERSCRIPT_24", "actions/24/text_superscript");
    public static final AntuIcon ACTION_TEXT_UNFLOW_24 = create("ACTION_TEXT_UNFLOW_24", "actions/24/text-unflow");
    public static final AntuIcon ACTION_TEXT_UNKERN_24 = create("ACTION_TEXT_UNKERN_24", "actions/24/text-unkern");
    public static final AntuIcon ACTION_TEXT_VERT_KERN_24 = create("ACTION_TEXT_VERT_KERN_24", "actions/24/text_vert_kern");
    public static final AntuIcon ACTION_TEXT_WORD_SPACING_24 = create("ACTION_TEXT_WORD_SPACING_24", "actions/24/text_word_spacing");
    public static final AntuIcon ACTION_TEXT_WRAP_24 = create("ACTION_TEXT_WRAP_24", "actions/24/text-wrap");
    public static final AntuIcon ACTION_TEXTURE_24 = create("ACTION_TEXTURE_24", "actions/24/texture");
    public static final AntuIcon ACTION_TOOL_ANIMATOR_24 = create("ACTION_TOOL_ANIMATOR_24", "actions/24/tool-animator");
    public static final AntuIcon ACTION_TOOL_COLOR_ERASER_24 = create("ACTION_TOOL_COLOR_ERASER_24", "actions/24/tool_color_eraser");
    public static final AntuIcon ACTION_TOOL_CURVE_24 = create("ACTION_TOOL_CURVE_24", "actions/24/tool_curve");
    public static final AntuIcon ACTION_TOOL_ELLIPTICAL_SELECTION_24 = create("ACTION_TOOL_ELLIPTICAL_SELECTION_24", "actions/24/tool_elliptical_selection");
    public static final AntuIcon ACTION_TOOL_MEASURE_24 = create("ACTION_TOOL_MEASURE_24", "actions/24/tool-measure");
    public static final AntuIcon ACTION_TOOL_NODE_EDITOR_24 = create("ACTION_TOOL_NODE_EDITOR_24", "actions/24/tool-node-editor");
    public static final AntuIcon ACTION_TOOL_POINTER_24 = create("ACTION_TOOL_POINTER_24", "actions/24/tool-pointer");
    public static final AntuIcon ACTION_TOOL_POLYGON_24 = create("ACTION_TOOL_POLYGON_24", "actions/24/tool_polygon");
    public static final AntuIcon ACTION_TOOL_ROUNDED_RECTANGLE_24 = create("ACTION_TOOL_ROUNDED_RECTANGLE_24", "actions/24/tool_rounded_rectangle");
    public static final AntuIcon ACTION_TOOL_SPRAY_24 = create("ACTION_TOOL_SPRAY_24", "actions/24/tool-spray");
    public static final AntuIcon ACTION_TOOL_TWEAK_24 = create("ACTION_TOOL_TWEAK_24", "actions/24/tool-tweak");
    public static final AntuIcon ACTION_TOOLS_CHECK_SPELLING_24 = create("ACTION_TOOLS_CHECK_SPELLING_24", "actions/24/tools-check-spelling");
    public static final AntuIcon ACTION_TOOLS_MEDIA_OPTICAL_BURN_24 = create("ACTION_TOOLS_MEDIA_OPTICAL_BURN_24", "actions/24/tools-media-optical-burn");
    public static final AntuIcon ACTION_TOOLS_MEDIA_OPTICAL_BURN_IMAGE_24 = create("ACTION_TOOLS_MEDIA_OPTICAL_BURN_IMAGE_24", "actions/24/tools-media-optical-burn-image");
    public static final AntuIcon ACTION_TOOLS_MEDIA_OPTICAL_COPY_24 = create("ACTION_TOOLS_MEDIA_OPTICAL_COPY_24", "actions/24/tools-media-optical-copy");
    public static final AntuIcon ACTION_TOOLS_MEDIA_OPTICAL_ERASE_24 = create("ACTION_TOOLS_MEDIA_OPTICAL_ERASE_24", "actions/24/tools-media-optical-erase");
    public static final AntuIcon ACTION_TOOLS_MEDIA_OPTICAL_FORMAT_24 = create("ACTION_TOOLS_MEDIA_OPTICAL_FORMAT_24", "actions/24/tools-media-optical-format");
    public static final AntuIcon ACTION_TOOLS_REPORT_BUG_24 = create("ACTION_TOOLS_REPORT_BUG_24", "actions/24/tools-report-bug");
    public static final AntuIcon ACTION_TOOLS_RIP_AUDIO_CD_24 = create("ACTION_TOOLS_RIP_AUDIO_CD_24", "actions/24/tools-rip-audio-cd");
    public static final AntuIcon ACTION_TOOLS_RIP_VIDEO_CD_24 = create("ACTION_TOOLS_RIP_VIDEO_CD_24", "actions/24/tools-rip-video-cd");
    public static final AntuIcon ACTION_TOOLS_RIP_VIDEO_DVD_24 = create("ACTION_TOOLS_RIP_VIDEO_DVD_24", "actions/24/tools-rip-video-dvd");
    public static final AntuIcon ACTION_TOOLS_WIZARD_24 = create("ACTION_TOOLS_WIZARD_24", "actions/24/tools-wizard");
    public static final AntuIcon ACTION_TRANSFORM_AFFECT_GRADIENT_24 = create("ACTION_TRANSFORM_AFFECT_GRADIENT_24", "actions/24/transform-affect-gradient");
    public static final AntuIcon ACTION_TRANSFORM_AFFECT_PATTERN_24 = create("ACTION_TRANSFORM_AFFECT_PATTERN_24", "actions/24/transform-affect-pattern");
    public static final AntuIcon ACTION_TRANSFORM_AFFECT_ROUNDED_CORNERS_24 = create("ACTION_TRANSFORM_AFFECT_ROUNDED_CORNERS_24", "actions/24/transform-affect-rounded-corners");
    public static final AntuIcon ACTION_TRANSFORM_AFFECT_STROKE_24 = create("ACTION_TRANSFORM_AFFECT_STROKE_24", "actions/24/transform-affect-stroke");
    public static final AntuIcon ACTION_TRANSFORM_CROP_24 = create("ACTION_TRANSFORM_CROP_24", "actions/24/transform-crop");
    public static final AntuIcon ACTION_TRANSFORM_CROP_AND_RESIZE_24 = create("ACTION_TRANSFORM_CROP_AND_RESIZE_24", "actions/24/transform-crop-and-resize");
    public static final AntuIcon ACTION_TRANSFORM_MIRROR_HORIZONTAL_24 = create("ACTION_TRANSFORM_MIRROR_HORIZONTAL_24", "actions/24/transform-mirror-horizontal");
    public static final AntuIcon ACTION_TRANSFORM_MIRROR_VERTICAL_24 = create("ACTION_TRANSFORM_MIRROR_VERTICAL_24", "actions/24/transform-mirror-vertical");
    public static final AntuIcon ACTION_TRANSFORM_MOVE_24 = create("ACTION_TRANSFORM_MOVE_24", "actions/24/transform-move");
    public static final AntuIcon ACTION_TRANSFORM_MOVE_HORIZONTAL_24 = create("ACTION_TRANSFORM_MOVE_HORIZONTAL_24", "actions/24/transform-move-horizontal");
    public static final AntuIcon ACTION_TRANSFORM_MOVE_VERTICAL_24 = create("ACTION_TRANSFORM_MOVE_VERTICAL_24", "actions/24/transform-move-vertical");
    public static final AntuIcon ACTION_TRANSFORM_ROTATE_24 = create("ACTION_TRANSFORM_ROTATE_24", "actions/24/transform-rotate");
    public static final AntuIcon ACTION_TRANSFORM_SCALE_24 = create("ACTION_TRANSFORM_SCALE_24", "actions/24/transform-scale");
    public static final AntuIcon ACTION_TRANSFORM_SCALE_TEXTBOX_POINTS_24 = create("ACTION_TRANSFORM_SCALE_TEXTBOX_POINTS_24", "actions/24/transform-scale-textbox-points");
    public static final AntuIcon ACTION_TRANSFORM_SHEAR_DOWN_24 = create("ACTION_TRANSFORM_SHEAR_DOWN_24", "actions/24/transform-shear-down");
    public static final AntuIcon ACTION_TRANSFORM_SHEAR_LEFT_24 = create("ACTION_TRANSFORM_SHEAR_LEFT_24", "actions/24/transform-shear-left");
    public static final AntuIcon ACTION_TRANSFORM_SHEAR_RIGHT_24 = create("ACTION_TRANSFORM_SHEAR_RIGHT_24", "actions/24/transform-shear-right");
    public static final AntuIcon ACTION_TRANSFORM_SHEAR_UP_24 = create("ACTION_TRANSFORM_SHEAR_UP_24", "actions/24/transform-shear-up");
    public static final AntuIcon ACTION_TRASH_EMPTY_24 = create("ACTION_TRASH_EMPTY_24", "actions/24/trash-empty");
    public static final AntuIcon ACTION_UMBRELLO_DIAGRAM_DEPLOYMENT_24 = create("ACTION_UMBRELLO_DIAGRAM_DEPLOYMENT_24", "actions/24/umbrello_diagram_deployment");
    public static final AntuIcon ACTION_UNDEREXPOSURE_24 = create("ACTION_UNDEREXPOSURE_24", "actions/24/underexposure");
    public static final AntuIcon ACTION_UNIASSOCIATION_24 = create("ACTION_UNIASSOCIATION_24", "actions/24/uniassociation");
    public static final AntuIcon ACTION_UNIQUE_CONSTRAINT_24 = create("ACTION_UNIQUE_CONSTRAINT_24", "actions/24/unique_constraint");
    public static final AntuIcon ACTION_UPCOMINGEVENTS_AMAROK_24 = create("ACTION_UPCOMINGEVENTS_AMAROK_24", "actions/24/upcomingevents-amarok");
    public static final AntuIcon ACTION_UPINDICATOR_24 = create("ACTION_UPINDICATOR_24", "actions/24/upindicator");
    public static final AntuIcon ACTION_UPLOAD_MEDIA_24 = create("ACTION_UPLOAD_MEDIA_24", "actions/24/upload-media");
    public static final AntuIcon ACTION_USER_GROUP_DELETE_24 = create("ACTION_USER_GROUP_DELETE_24", "actions/24/user-group-delete");
    public static final AntuIcon ACTION_USER_GROUP_NEW_24 = create("ACTION_USER_GROUP_NEW_24", "actions/24/user-group-new");
    public static final AntuIcon ACTION_USER_GROUP_PROPERTIES_24 = create("ACTION_USER_GROUP_PROPERTIES_24", "actions/24/user-group-properties");
    public static final AntuIcon ACTION_USER_IDENTITY_24 = create("ACTION_USER_IDENTITY_24", "actions/24/user-identity");
    public static final AntuIcon ACTION_USER_PROPERTIES_24 = create("ACTION_USER_PROPERTIES_24", "actions/24/user-properties");
    public static final AntuIcon ACTION_USERNAME_COPY_24 = create("ACTION_USERNAME_COPY_24", "actions/24/username-copy");
    public static final AntuIcon ACTION_VIDEOCLIP_AMAROK_24 = create("ACTION_VIDEOCLIP_AMAROK_24", "actions/24/videoclip-amarok");
    public static final AntuIcon ACTION_VIEW_BANK_24 = create("ACTION_VIEW_BANK_24", "actions/24/view-bank");
    public static final AntuIcon ACTION_VIEW_BANK_ACCOUNT_24 = create("ACTION_VIEW_BANK_ACCOUNT_24", "actions/24/view-bank-account");
    public static final AntuIcon ACTION_VIEW_BANK_ACCOUNT_CHECKING_24 = create("ACTION_VIEW_BANK_ACCOUNT_CHECKING_24", "actions/24/view-bank-account-checking");
    public static final AntuIcon ACTION_VIEW_BANK_ACCOUNT_SAVINGS_24 = create("ACTION_VIEW_BANK_ACCOUNT_SAVINGS_24", "actions/24/view-bank-account-savings");
    public static final AntuIcon ACTION_VIEW_BARCODE_24 = create("ACTION_VIEW_BARCODE_24", "actions/24/view-barcode");
    public static final AntuIcon ACTION_VIEW_BARCODE_ADD_24 = create("ACTION_VIEW_BARCODE_ADD_24", "actions/24/view-barcode-add");
    public static final AntuIcon ACTION_VIEW_CALENDAR_24 = create("ACTION_VIEW_CALENDAR_24", "actions/24/view-calendar");
    public static final AntuIcon ACTION_VIEW_CALENDAR_AGENDA_24 = create("ACTION_VIEW_CALENDAR_AGENDA_24", "actions/24/view-calendar-agenda");
    public static final AntuIcon ACTION_VIEW_CALENDAR_BIRTHDAY_24 = create("ACTION_VIEW_CALENDAR_BIRTHDAY_24", "actions/24/view-calendar-birthday");
    public static final AntuIcon ACTION_VIEW_CALENDAR_DAY_24 = create("ACTION_VIEW_CALENDAR_DAY_24", "actions/24/view-calendar-day");
    public static final AntuIcon ACTION_VIEW_CALENDAR_HOLIDAY_24 = create("ACTION_VIEW_CALENDAR_HOLIDAY_24", "actions/24/view-calendar-holiday");
    public static final AntuIcon ACTION_VIEW_CALENDAR_JOURNAL_24 = create("ACTION_VIEW_CALENDAR_JOURNAL_24", "actions/24/view-calendar-journal");
    public static final AntuIcon ACTION_VIEW_CALENDAR_LIST_24 = create("ACTION_VIEW_CALENDAR_LIST_24", "actions/24/view-calendar-list");
    public static final AntuIcon ACTION_VIEW_CALENDAR_MONTH_24 = create("ACTION_VIEW_CALENDAR_MONTH_24", "actions/24/view-calendar-month");
    public static final AntuIcon ACTION_VIEW_CALENDAR_SPECIAL_OCCASION_24 = create("ACTION_VIEW_CALENDAR_SPECIAL_OCCASION_24", "actions/24/view-calendar-special-occasion");
    public static final AntuIcon ACTION_VIEW_CALENDAR_TASKS_24 = create("ACTION_VIEW_CALENDAR_TASKS_24", "actions/24/view-calendar-tasks");
    public static final AntuIcon ACTION_VIEW_CALENDAR_TIME_SPENT_24 = create("ACTION_VIEW_CALENDAR_TIME_SPENT_24", "actions/24/view-calendar-time-spent");
    public static final AntuIcon ACTION_VIEW_CALENDAR_TIMELINE_24 = create("ACTION_VIEW_CALENDAR_TIMELINE_24", "actions/24/view-calendar-timeline");
    public static final AntuIcon ACTION_VIEW_CALENDAR_UPCOMING_DAYS_24 = create("ACTION_VIEW_CALENDAR_UPCOMING_DAYS_24", "actions/24/view-calendar-upcoming-days");
    public static final AntuIcon ACTION_VIEW_CALENDAR_UPCOMING_EVENTS_24 = create("ACTION_VIEW_CALENDAR_UPCOMING_EVENTS_24", "actions/24/view-calendar-upcoming-events");
    public static final AntuIcon ACTION_VIEW_CALENDAR_WEDDING_ANNIVERSARY_24 = create("ACTION_VIEW_CALENDAR_WEDDING_ANNIVERSARY_24", "actions/24/view-calendar-wedding-anniversary");
    public static final AntuIcon ACTION_VIEW_CALENDAR_WEEK_24 = create("ACTION_VIEW_CALENDAR_WEEK_24", "actions/24/view-calendar-week");
    public static final AntuIcon ACTION_VIEW_CALENDAR_WHATSNEXT_24 = create("ACTION_VIEW_CALENDAR_WHATSNEXT_24", "actions/24/view-calendar-whatsnext");
    public static final AntuIcon ACTION_VIEW_CALENDAR_WORKWEEK_24 = create("ACTION_VIEW_CALENDAR_WORKWEEK_24", "actions/24/view-calendar-workweek");
    public static final AntuIcon ACTION_VIEW_CATALOG_24 = create("ACTION_VIEW_CATALOG_24", "actions/24/view-catalog");
    public static final AntuIcon ACTION_VIEW_CATEGORIES_24 = create("ACTION_VIEW_CATEGORIES_24", "actions/24/view-categories");
    public static final AntuIcon ACTION_VIEW_CATEGORIES_EXPENDITURES_24 = create("ACTION_VIEW_CATEGORIES_EXPENDITURES_24", "actions/24/view-categories-expenditures");
    public static final AntuIcon ACTION_VIEW_CATEGORIES_INCOMES_24 = create("ACTION_VIEW_CATEGORIES_INCOMES_24", "actions/24/view-categories-incomes");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_24 = create("ACTION_VIEW_CERTIFICATE_24", "actions/24/view-certificate");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_ADD_24 = create("ACTION_VIEW_CERTIFICATE_ADD_24", "actions/24/view-certificate-add");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_EXPORT_24 = create("ACTION_VIEW_CERTIFICATE_EXPORT_24", "actions/24/view-certificate-export");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_EXPORT_SECRET_24 = create("ACTION_VIEW_CERTIFICATE_EXPORT_SECRET_24", "actions/24/view-certificate-export-secret");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_EXPORT_SERVER_24 = create("ACTION_VIEW_CERTIFICATE_EXPORT_SERVER_24", "actions/24/view-certificate-export-server");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_IMPORT_24 = create("ACTION_VIEW_CERTIFICATE_IMPORT_24", "actions/24/view-certificate-import");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_SERVER_CONFIGURE_24 = create("ACTION_VIEW_CERTIFICATE_SERVER_CONFIGURE_24", "actions/24/view-certificate-server-configure");
    public static final AntuIcon ACTION_VIEW_CERTIFICATE_SIGN_24 = create("ACTION_VIEW_CERTIFICATE_SIGN_24", "actions/24/view-certificate-sign");
    public static final AntuIcon ACTION_VIEW_CHOOSE_24 = create("ACTION_VIEW_CHOOSE_24", "actions/24/view-choose");
    public static final AntuIcon ACTION_VIEW_CLOSE_24 = create("ACTION_VIEW_CLOSE_24", "actions/24/view-close");
    public static final AntuIcon ACTION_VIEW_CONVERSATION_BALLOON_24 = create("ACTION_VIEW_CONVERSATION_BALLOON_24", "actions/24/view-conversation-balloon");
    public static final AntuIcon ACTION_VIEW_FILE_COLUMNS_24 = create("ACTION_VIEW_FILE_COLUMNS_24", "actions/24/view-file-columns");
    public static final AntuIcon ACTION_VIEW_FILTER_24 = create("ACTION_VIEW_FILTER_24", "actions/24/view-filter");
    public static final AntuIcon ACTION_VIEW_FILTER_32 = create("ACTION_VIEW_FILTER_32", "actions/32/view-filter");
    public static final AntuIcon ACTION_VIEW_FORM_24 = create("ACTION_VIEW_FORM_24", "actions/24/view-form");
    public static final AntuIcon ACTION_VIEW_FORM_ACTION_24 = create("ACTION_VIEW_FORM_ACTION_24", "actions/24/view-form-action");
    public static final AntuIcon ACTION_VIEW_FULLSCREEN_24 = create("ACTION_VIEW_FULLSCREEN_24", "actions/24/view-fullscreen");
    public static final AntuIcon ACTION_VIEW_GRID_24 = create("ACTION_VIEW_GRID_24", "actions/24/view-grid");
    public static final AntuIcon ACTION_VIEW_GROUP_24 = create("ACTION_VIEW_GROUP_24", "actions/24/view-group");
    public static final AntuIcon ACTION_VIEW_GROUP_32 = create("ACTION_VIEW_GROUP_32", "actions/32/view-group");
    public static final AntuIcon ACTION_VIEW_HISTORY_24 = create("ACTION_VIEW_HISTORY_24", "actions/24/view-history");
    public static final AntuIcon ACTION_VIEW_LEFT_CLOSE_24 = create("ACTION_VIEW_LEFT_CLOSE_24", "actions/24/view-left-close");
    public static final AntuIcon ACTION_VIEW_LEFT_CLOSE_32 = create("ACTION_VIEW_LEFT_CLOSE_32", "actions/32/view-left-close");
    public static final AntuIcon ACTION_VIEW_LIST_DETAILS_24 = create("ACTION_VIEW_LIST_DETAILS_24", "actions/24/view-list-details");
    public static final AntuIcon ACTION_VIEW_LIST_DETAILS_32 = create("ACTION_VIEW_LIST_DETAILS_32", "actions/32/view-list-details");
    public static final AntuIcon ACTION_VIEW_LIST_ICONS_24 = create("ACTION_VIEW_LIST_ICONS_24", "actions/24/view-list-icons");
    public static final AntuIcon ACTION_VIEW_LIST_ICONS_32 = create("ACTION_VIEW_LIST_ICONS_32", "actions/32/view-list-icons");
    public static final AntuIcon ACTION_VIEW_LIST_TEXT_24 = create("ACTION_VIEW_LIST_TEXT_24", "actions/24/view-list-text");
    public static final AntuIcon ACTION_VIEW_LIST_TEXT_32 = create("ACTION_VIEW_LIST_TEXT_32", "actions/32/view-list-text");
    public static final AntuIcon ACTION_VIEW_LIST_TREE_24 = create("ACTION_VIEW_LIST_TREE_24", "actions/24/view-list-tree");
    public static final AntuIcon ACTION_VIEW_LIST_TREE_32 = create("ACTION_VIEW_LIST_TREE_32", "actions/32/view-list-tree");
    public static final AntuIcon ACTION_VIEW_MEDIA_ARTIST_24 = create("ACTION_VIEW_MEDIA_ARTIST_24", "actions/24/view-media-artist");
    public static final AntuIcon ACTION_VIEW_MEDIA_EQUALIZER_24 = create("ACTION_VIEW_MEDIA_EQUALIZER_24", "actions/24/view-media-equalizer");
    public static final AntuIcon ACTION_VIEW_MEDIA_LYRICS_24 = create("ACTION_VIEW_MEDIA_LYRICS_24", "actions/24/view-media-lyrics");
    public static final AntuIcon ACTION_VIEW_MEDIA_PLAYLIST_24 = create("ACTION_VIEW_MEDIA_PLAYLIST_24", "actions/24/view-media-playlist");
    public static final AntuIcon ACTION_VIEW_MEDIA_VISUALIZATION_24 = create("ACTION_VIEW_MEDIA_VISUALIZATION_24", "actions/24/view-media-visualization");
    public static final AntuIcon ACTION_VIEW_MULTIPLE_OBJECTS_24 = create("ACTION_VIEW_MULTIPLE_OBJECTS_24", "actions/24/view-multiple-objects");
    public static final AntuIcon ACTION_VIEW_OBJECT_HISTOGRAM_LINEAR_24 = create("ACTION_VIEW_OBJECT_HISTOGRAM_LINEAR_24", "actions/24/view-object-histogram-linear");
    public static final AntuIcon ACTION_VIEW_OBJECT_HISTOGRAM_LOGARITHMIC_24 = create("ACTION_VIEW_OBJECT_HISTOGRAM_LOGARITHMIC_24", "actions/24/view-object-histogram-logarithmic");
    public static final AntuIcon ACTION_VIEW_PIM_CALENDAR_24 = create("ACTION_VIEW_PIM_CALENDAR_24", "actions/24/view-pim-calendar");
    public static final AntuIcon ACTION_VIEW_PIM_CONTACTS_24 = create("ACTION_VIEW_PIM_CONTACTS_24", "actions/24/view-pim-contacts");
    public static final AntuIcon ACTION_VIEW_PIM_JOURNAL_24 = create("ACTION_VIEW_PIM_JOURNAL_24", "actions/24/view-pim-journal");
    public static final AntuIcon ACTION_VIEW_PIM_MAIL_24 = create("ACTION_VIEW_PIM_MAIL_24", "actions/24/view-pim-mail");
    public static final AntuIcon ACTION_VIEW_PIM_NEWS_24 = create("ACTION_VIEW_PIM_NEWS_24", "actions/24/view-pim-news");
    public static final AntuIcon ACTION_VIEW_PIM_NOTES_24 = create("ACTION_VIEW_PIM_NOTES_24", "actions/24/view-pim-notes");
    public static final AntuIcon ACTION_VIEW_PIM_TASKS_24 = create("ACTION_VIEW_PIM_TASKS_24", "actions/24/view-pim-tasks");
    public static final AntuIcon ACTION_VIEW_PIM_TASKS_PENDING_24 = create("ACTION_VIEW_PIM_TASKS_PENDING_24", "actions/24/view-pim-tasks-pending");
    public static final AntuIcon ACTION_VIEW_PRESENTATION_24 = create("ACTION_VIEW_PRESENTATION_24", "actions/24/view-presentation");
    public static final AntuIcon ACTION_VIEW_PREVIEW_24 = create("ACTION_VIEW_PREVIEW_24", "actions/24/view-preview");
    public static final AntuIcon ACTION_VIEW_PREVIEW_32 = create("ACTION_VIEW_PREVIEW_32", "actions/32/view-preview");
    public static final AntuIcon ACTION_VIEW_PROCESS_ALL_24 = create("ACTION_VIEW_PROCESS_ALL_24", "actions/24/view-process-all");
    public static final AntuIcon ACTION_VIEW_PROCESS_ALL_TREE_24 = create("ACTION_VIEW_PROCESS_ALL_TREE_24", "actions/24/view-process-all-tree");
    public static final AntuIcon ACTION_VIEW_PROCESS_OWN_24 = create("ACTION_VIEW_PROCESS_OWN_24", "actions/24/view-process-own");
    public static final AntuIcon ACTION_VIEW_PROCESS_SYSTEM_24 = create("ACTION_VIEW_PROCESS_SYSTEM_24", "actions/24/view-process-system");
    public static final AntuIcon ACTION_VIEW_PROCESS_TREE_24 = create("ACTION_VIEW_PROCESS_TREE_24", "actions/24/view-process-tree");
    public static final AntuIcon ACTION_VIEW_PROCESS_USERS_24 = create("ACTION_VIEW_PROCESS_USERS_24", "actions/24/view-process-users");
    public static final AntuIcon ACTION_VIEW_READERMODE_24 = create("ACTION_VIEW_READERMODE_24", "actions/24/view-readermode");
    public static final AntuIcon ACTION_VIEW_READERMODE_ACTIVE_24 = create("ACTION_VIEW_READERMODE_ACTIVE_24", "actions/24/view-readermode-active");
    public static final AntuIcon ACTION_VIEW_REFRESH_24 = create("ACTION_VIEW_REFRESH_24", "actions/24/view-refresh");
    public static final AntuIcon ACTION_VIEW_REFRESH_32 = create("ACTION_VIEW_REFRESH_32", "actions/32/view-refresh");
    public static final AntuIcon ACTION_VIEW_RESOURCE_CALENDAR_24 = create("ACTION_VIEW_RESOURCE_CALENDAR_24", "actions/24/view-resource-calendar");
    public static final AntuIcon ACTION_VIEW_RESTORE_24 = create("ACTION_VIEW_RESTORE_24", "actions/24/view-restore");
    public static final AntuIcon ACTION_VIEW_RIGHT_CLOSE_24 = create("ACTION_VIEW_RIGHT_CLOSE_24", "actions/24/view-right-close");
    public static final AntuIcon ACTION_VIEW_RIGHT_CLOSE_32 = create("ACTION_VIEW_RIGHT_CLOSE_32", "actions/32/view-right-close");
    public static final AntuIcon ACTION_VIEW_RIGHT_NEW_24 = create("ACTION_VIEW_RIGHT_NEW_24", "actions/24/view-right-new");
    public static final AntuIcon ACTION_VIEW_RIGHT_NEW_32 = create("ACTION_VIEW_RIGHT_NEW_32", "actions/32/view-right-new");
    public static final AntuIcon ACTION_VIEW_SERVICES_SCRIPTED_AMAROK_24 = create("ACTION_VIEW_SERVICES_SCRIPTED_AMAROK_24", "actions/24/view-services-scripted-amarok");
    public static final AntuIcon ACTION_VIEW_SIDETREE_24 = create("ACTION_VIEW_SIDETREE_24", "actions/24/view-sidetree");
    public static final AntuIcon ACTION_VIEW_SORT_ASCENDING_24 = create("ACTION_VIEW_SORT_ASCENDING_24", "actions/24/view-sort-ascending");
    public static final AntuIcon ACTION_VIEW_SORT_ASCENDING_32 = create("ACTION_VIEW_SORT_ASCENDING_32", "actions/32/view-sort-ascending");
    public static final AntuIcon ACTION_VIEW_SORT_DESCENDING_24 = create("ACTION_VIEW_SORT_DESCENDING_24", "actions/24/view-sort-descending");
    public static final AntuIcon ACTION_VIEW_SORT_DESCENDING_32 = create("ACTION_VIEW_SORT_DESCENDING_32", "actions/32/view-sort-descending");
    public static final AntuIcon ACTION_VIEW_SPLIT_LEFT_RIGHT_24 = create("ACTION_VIEW_SPLIT_LEFT_RIGHT_24", "actions/24/view-split-left-right");
    public static final AntuIcon ACTION_VIEW_SPLIT_TOP_BOTTOM_24 = create("ACTION_VIEW_SPLIT_TOP_BOTTOM_24", "actions/24/view-split-top-bottom");
    public static final AntuIcon ACTION_VIEW_STATISTICS_24 = create("ACTION_VIEW_STATISTICS_24", "actions/24/view-statistics");
    public static final AntuIcon ACTION_VIEW_STATISTICS_48 = create("ACTION_VIEW_STATISTICS_48", "actions/48/view-statistics");
    public static final AntuIcon ACTION_VIEW_TABLE_OF_CONTENTS_LTR_24 = create("ACTION_VIEW_TABLE_OF_CONTENTS_LTR_24", "actions/24/view-table-of-contents-ltr");
    public static final AntuIcon ACTION_VIEW_TABLE_OF_CONTENTS_RTL_24 = create("ACTION_VIEW_TABLE_OF_CONTENTS_RTL_24", "actions/24/view-table-of-contents-rtl");
    public static final AntuIcon ACTION_VIEW_TASK_24 = create("ACTION_VIEW_TASK_24", "actions/24/view-task");
    public static final AntuIcon ACTION_VIEW_TASK_ADD_24 = create("ACTION_VIEW_TASK_ADD_24", "actions/24/view-task-add");
    public static final AntuIcon ACTION_VIEW_TASK_CHILD_24 = create("ACTION_VIEW_TASK_CHILD_24", "actions/24/view-task-child");
    public static final AntuIcon ACTION_VIEW_TASK_CHILD_ADD_24 = create("ACTION_VIEW_TASK_CHILD_ADD_24", "actions/24/view-task-child-add");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_24 = create("ACTION_VIEW_TIME_SCHEDULE_24", "actions/24/view-time-schedule");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_BASELINED_24 = create("ACTION_VIEW_TIME_SCHEDULE_BASELINED_24", "actions/24/view-time-schedule-baselined");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_BASELINED_ADD_24 = create("ACTION_VIEW_TIME_SCHEDULE_BASELINED_ADD_24", "actions/24/view-time-schedule-baselined-add");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_BASELINED_REMOVE_24 = create("ACTION_VIEW_TIME_SCHEDULE_BASELINED_REMOVE_24", "actions/24/view-time-schedule-baselined-remove");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_CALCULUS_24 = create("ACTION_VIEW_TIME_SCHEDULE_CALCULUS_24", "actions/24/view-time-schedule-calculus");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_CHILD_INSERT_24 = create("ACTION_VIEW_TIME_SCHEDULE_CHILD_INSERT_24", "actions/24/view-time-schedule-child-insert");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_EDIT_24 = create("ACTION_VIEW_TIME_SCHEDULE_EDIT_24", "actions/24/view-time-schedule-edit");
    public static final AntuIcon ACTION_VIEW_TIME_SCHEDULE_INSERT_24 = create("ACTION_VIEW_TIME_SCHEDULE_INSERT_24", "actions/24/view-time-schedule-insert");
    public static final AntuIcon ACTION_VIEW_WEB_BROWSER_DOM_TREE_24 = create("ACTION_VIEW_WEB_BROWSER_DOM_TREE_24", "actions/24/view-web-browser-dom-tree");
    public static final AntuIcon ACTION_VIEWIMAGE_24 = create("ACTION_VIEWIMAGE_24", "actions/24/viewimage");
    public static final AntuIcon ACTION_VISIBILITY_24 = create("ACTION_VISIBILITY_24", "actions/24/visibility");
    public static final AntuIcon ACTION_WHITEBALANCE_24 = create("ACTION_WHITEBALANCE_24", "actions/24/whitebalance");
    public static final AntuIcon ACTION_WINDOW_24 = create("ACTION_WINDOW_24", "actions/24/window");
    public static final AntuIcon ACTION_WINDOW_CLOSE_24 = create("ACTION_WINDOW_CLOSE_24", "actions/24/window-close");
    public static final AntuIcon ACTION_WINDOW_CLOSE_32 = create("ACTION_WINDOW_CLOSE_32", "actions/32/window-close");
    public static final AntuIcon ACTION_WINDOW_DUPLICATE_24 = create("ACTION_WINDOW_DUPLICATE_24", "actions/24/window-duplicate");
    public static final AntuIcon ACTION_WINDOW_NEW_24 = create("ACTION_WINDOW_NEW_24", "actions/24/window-new");
    public static final AntuIcon ACTION_WINDOW_NEXT_24 = create("ACTION_WINDOW_NEXT_24", "actions/24/window-next");
    public static final AntuIcon ACTION_WINDOW_PREVIOUS_24 = create("ACTION_WINDOW_PREVIOUS_24", "actions/24/window-previous");
    public static final AntuIcon ACTION_X_CLEMENTINE_ARTIST_24 = create("ACTION_X_CLEMENTINE_ARTIST_24", "actions/24/x-clementine-artist");
    public static final AntuIcon ACTION_X_SHAPE_CONNECTION_24 = create("ACTION_X_SHAPE_CONNECTION_24", "actions/24/x-shape-connection");
    public static final AntuIcon ACTION_X_SHAPE_FORMULA_24 = create("ACTION_X_SHAPE_FORMULA_24", "actions/24/x-shape-formula");
    public static final AntuIcon ACTION_X_SHAPE_IMAGE_24 = create("ACTION_X_SHAPE_IMAGE_24", "actions/24/x-shape-image");
    public static final AntuIcon ACTION_X_SHAPE_TEXT_24 = create("ACTION_X_SHAPE_TEXT_24", "actions/24/x-shape-text");
    public static final AntuIcon ACTION_X_ZOOM_IN_24 = create("ACTION_X_ZOOM_IN_24", "actions/24/x-zoom-in");
    public static final AntuIcon ACTION_X_ZOOM_OUT_24 = create("ACTION_X_ZOOM_OUT_24", "actions/24/x-zoom-out");
    public static final AntuIcon ACTION_XML_ATTRIBUTE_DELETE_24 = create("ACTION_XML_ATTRIBUTE_DELETE_24", "actions/24/xml-attribute-delete");
    public static final AntuIcon ACTION_XML_ELEMENT_NEW_24 = create("ACTION_XML_ELEMENT_NEW_24", "actions/24/xml-element-new");
    public static final AntuIcon ACTION_XML_NODE_DELETE_24 = create("ACTION_XML_NODE_DELETE_24", "actions/24/xml-node-delete");
    public static final AntuIcon ACTION_XML_NODE_DUPLICATE_24 = create("ACTION_XML_NODE_DUPLICATE_24", "actions/24/xml-node-duplicate");
    public static final AntuIcon ACTION_XML_TEXT_NEW_24 = create("ACTION_XML_TEXT_NEW_24", "actions/24/xml-text-new");
    public static final AntuIcon ACTION_XY_CURVE_24 = create("ACTION_XY_CURVE_24", "actions/24/xy-curve");
    public static final AntuIcon ACTION_XY_EQUATION_CURVE_24 = create("ACTION_XY_EQUATION_CURVE_24", "actions/24/xy-equation-curve");
    public static final AntuIcon ACTION_XY_FIT_CURVE_24 = create("ACTION_XY_FIT_CURVE_24", "actions/24/xy-fit-curve");
    public static final AntuIcon ACTION_Y_ZOOM_IN_24 = create("ACTION_Y_ZOOM_IN_24", "actions/24/y-zoom-in");
    public static final AntuIcon ACTION_Y_ZOOM_OUT_24 = create("ACTION_Y_ZOOM_OUT_24", "actions/24/y-zoom-out");
    public static final AntuIcon ACTION_ZOOM_1_TO_2_24 = create("ACTION_ZOOM_1_TO_2_24", "actions/24/zoom-1-to-2");
    public static final AntuIcon ACTION_ZOOM_2_TO_1_24 = create("ACTION_ZOOM_2_TO_1_24", "actions/24/zoom-2-to-1");
    public static final AntuIcon ACTION_ZOOM_24 = create("ACTION_ZOOM_24", "actions/24/zoom");
    public static final AntuIcon ACTION_ZOOM_DOUBLE_SIZE_24 = create("ACTION_ZOOM_DOUBLE_SIZE_24", "actions/24/zoom-double-size");
    public static final AntuIcon ACTION_ZOOM_DRAW_24 = create("ACTION_ZOOM_DRAW_24", "actions/24/zoom-draw");
    public static final AntuIcon ACTION_ZOOM_FIT_BEST_24 = create("ACTION_ZOOM_FIT_BEST_24", "actions/24/zoom-fit-best");
    public static final AntuIcon ACTION_ZOOM_FIT_DRAWING_24 = create("ACTION_ZOOM_FIT_DRAWING_24", "actions/24/zoom-fit-drawing");
    public static final AntuIcon ACTION_ZOOM_FIT_HEIGHT_24 = create("ACTION_ZOOM_FIT_HEIGHT_24", "actions/24/zoom-fit-height");
    public static final AntuIcon ACTION_ZOOM_FIT_PAGE_24 = create("ACTION_ZOOM_FIT_PAGE_24", "actions/24/zoom-fit-page");
    public static final AntuIcon ACTION_ZOOM_FIT_SELECTION_24 = create("ACTION_ZOOM_FIT_SELECTION_24", "actions/24/zoom-fit-selection");
    public static final AntuIcon ACTION_ZOOM_FIT_WIDTH_24 = create("ACTION_ZOOM_FIT_WIDTH_24", "actions/24/zoom-fit-width");
    public static final AntuIcon ACTION_ZOOM_HALF_SIZE_24 = create("ACTION_ZOOM_HALF_SIZE_24", "actions/24/zoom-half-size");
    public static final AntuIcon ACTION_ZOOM_IN_24 = create("ACTION_ZOOM_IN_24", "actions/24/zoom-in");
    public static final AntuIcon ACTION_ZOOM_IN_X_24 = create("ACTION_ZOOM_IN_X_24", "actions/24/zoom-in-x");
    public static final AntuIcon ACTION_ZOOM_IN_Y_24 = create("ACTION_ZOOM_IN_Y_24", "actions/24/zoom-in-y");
    public static final AntuIcon ACTION_ZOOM_NEXT_24 = create("ACTION_ZOOM_NEXT_24", "actions/24/zoom-next");
    public static final AntuIcon ACTION_ZOOM_ORIGINAL_24 = create("ACTION_ZOOM_ORIGINAL_24", "actions/24/zoom-original");
    public static final AntuIcon ACTION_ZOOM_OUT_24 = create("ACTION_ZOOM_OUT_24", "actions/24/zoom-out");
    public static final AntuIcon ACTION_ZOOM_OUT_X_24 = create("ACTION_ZOOM_OUT_X_24", "actions/24/zoom-out-x");
    public static final AntuIcon ACTION_ZOOM_OUT_Y_24 = create("ACTION_ZOOM_OUT_Y_24", "actions/24/zoom-out-y");
    public static final AntuIcon ACTION_ZOOM_PREVIOUS_24 = create("ACTION_ZOOM_PREVIOUS_24", "actions/24/zoom-previous");
    public static final AntuIcon ACTION_ZOOM_SELECT_24 = create("ACTION_ZOOM_SELECT_24", "actions/24/zoom-select");
    public static final AntuIcon ACTION_ZOOM_SELECT_FIT_24 = create("ACTION_ZOOM_SELECT_FIT_24", "actions/24/zoom-select-fit");
    public static final AntuIcon ACTION_ZOOM_SELECT_X_24 = create("ACTION_ZOOM_SELECT_X_24", "actions/24/zoom-select-x");
    public static final AntuIcon ACTION_ZOOM_SELECT_Y_24 = create("ACTION_ZOOM_SELECT_Y_24", "actions/24/zoom-select-y");

}
