import { HighlightDirection, MatchingOptions, TrivialComponent } from "./TrivialCore";
import { TeamAppsEvent } from "../util/TeamAppsEvent";
export interface TrivialTreeBoxConfig<E> {
    /**
     * Calculates a unique value for an entry. Used to identify nodes in the tree.
     */
    idFunction?: (entry: E) => number | string;
    /**
     * Rendering function used to display a _suggested_ entry
     * (i.e. an entry displayed in the dropdown).
     *
     * @param entry
     * @param depth the depth of the entry in the tree
     * @return HTML string
     * @default Using the `image2LinesTemplate` from `TrivialCore`.
     */
    entryRenderingFunction: (entry: E, depth: number) => string;
    /**
     * The initially selected entry. (Caution: use `selectedEntries` for `TrivialTagBox`).
     */
    selectedEntry?: E;
    /**
     * The initial list of suggested entries.
     */
    entries?: E[];
    /**
     * Used for highlighting suggested entries. Also used by the default filtering functions int `TrivialCore`.
     *
     * @default `{ matchingMode: 'contains', ignoreCase: true, maxLevenshteinDistance: 1 }`
     */
    matchingOptions?: MatchingOptions;
    /**
     * Property used to retrieve the children of a node.
     *
     * @default `'children'`
     */
    childrenProperty?: string;
    /**
     * Property name or function used to determine whether a node has children that need to be lazy-loaded.
     *
     * @default `'hasLazyChildren'`
     */
    lazyChildrenFlag?: string | ((entry: E) => boolean);
    /**
     * Function for retrieving children of a node.
     *
     * @param node
     * @param resultCallback
     */
    lazyChildrenQueryFunction?: (entry: E) => Promise<E[]>;
    /**
     * Property used to determine whether a node is expanded or not.
     *
     * Note: This is subject to being replaced by a function in future versions.
     * @default `'expanded'`
     */
    expandedProperty?: string;
    /**
     * The ID of the initially selected entry in the tree.
     */
    selectedEntryId?: number | string;
    /**
     * Animation duration in milliseconds for expand and collapse animations.
     */
    animationDuration?: number;
    /**
     * Whether or not to show the expander controls for parent nodes.
     */
    showExpanders?: boolean;
    /**
     * Whether or not to expand a node when it is selected.
     */
    toggleExpansionOnClick?: boolean;
    /**
     * Special mode that allows only one path to be expanded.
     * Expands all ancestor nodes of the selected node, as well as the selected node itself.
     * Collapses all others.
     */
    enforceSingleExpandedPath?: boolean;
    /**
     * Html string defining what to display when the list of results from the `queryFunction` is empty.
     */
    noEntriesTemplate?: string;
    /**
     * HTML string defining the spinner to be displayed while lazy children are being retrieved.
     */
    spinnerTemplate?: string;
    /**
     * The indentation in pixels used.
     *
     * If null, use the default value given by the CSS rules.
     * If number, indent each depth/level by the given number of pixels.
     * If array, indent each depth by the corresponding value in the array.
     * If function, indent each depth by the result of the given function (with depth starting at 0).
     */
    indentation?: null | number | number[] | ((depth: number) => number);
    /**
     * Function that decides whether a node can be selected.
     */
    selectableDecider?: (entry: E) => boolean;
    selectOnHover?: boolean;
}
export declare class TrivialTreeBox<E> implements TrivialComponent {
    private config;
    readonly onSelectedEntryChanged: TeamAppsEvent<E>;
    readonly onNodeExpansionStateChanged: TeamAppsEvent<{
        node: E;
        expanded: boolean;
    }>;
    private $componentWrapper;
    private $tree;
    private entries;
    private selectedEntryId;
    constructor(options: TrivialTreeBoxConfig<E>);
    private createEntryElement;
    private create$ChildrenWrapper;
    private setNodeExpanded;
    private setChildren;
    private renderChildren;
    setEntries(newEntries: E[]): void;
    private createEntryWrapper;
    private findEntries;
    private findPathToFirstMatchingNode;
    private findEntryById;
    private setSelectedEntry;
    setSelectedEntryById(nodeId: number | string, fireEvents?: boolean): void;
    private minimallyScrollTo;
    private markSelectedEntry;
    private fireChangeEvents;
    selectNextEntry(direction: HighlightDirection, rollover?: boolean, selectableOnly?: boolean, matcher?: (entry: E) => boolean, fireEvents?: boolean, originalEvent?: unknown): E;
    private getNextVisibleEntry;
    highlightTextMatches(searchString: string): void;
    getSelectedEntry(): E | null;
    private getSelectedEntryWrapper;
    revealSelectedEntry(animate?: boolean): void;
    setSelectedNodeExpanded(expanded: boolean): boolean;
    updateChildren(parentNodeId: string | number, children: E[]): void;
    updateNode(node: E, recursive?: boolean): void;
    removeNode(nodeId: string | number): void;
    addNode(parentNodeId: number | string, node: E): void;
    addOrUpdateNode(parentNodeId: number | string, node: E, recursiveUpdate?: boolean): void;
    destroy(): void;
    getMainDomElement(): HTMLElement;
}
