/*!
Trivial Components (https://github.com/trivial-components/trivial-components)

Copyright 2016 Yann Massard (https://github.com/yamass) and other contributors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
import { EditingMode, HighlightDirection, QueryFunction, TrivialComponent, RenderingFunction } from "./TrivialCore";
import { TrivialEvent } from "./TrivialEvent";
import { TrivialTreeBoxConfig } from "./TrivialTreeBox";
export interface TrivialTagComboBoxConfig<E> extends TrivialTreeBoxConfig<E> {
    /**
     * Calculates the value to set on the original input.
     *
     * @param entries the list of selected entries
     * @return the string to set as the value of the original input
     */
    inputValueFunction?: (entries: E[]) => string;
    /**
     * Rendering function used to display a _selected_ entry
     * (i.e. an entry inside the editor area of the component, not the dropdown).
     *
     * @param entry
     * @return HTML string
     * @default `wrapWithDefaultTagWrapper(entryRenderingFunction(entry))`
     */
    selectedEntryRenderingFunction?: RenderingFunction<E>;
    /**
     * Initially selected entries.
     *
     * @default `[]`
     */
    selectedEntries?: E[];
    /**
     * Performance setting. Defines the maximum number of entries until which text highlighting is performed.
     * Set to `0` to disable text highlighting.
     *
     * @default `100`
     */
    textHighlightingEntryLimit?: number;
    /**
     * Used to retrieve the entries ("suggestions") to be displayed in the dropdown box.
     *
     * @see QueryFunction
     * @default creates a client-side query function using the provided [[entries]]
     */
    queryFunction?: QueryFunction<E>;
    /**
     * Whether or not to provide auto-completion.
     *
     * @default `true`
     */
    autoComplete?: boolean;
    /**
     * The number of milliseconds to wait until auto-completion is performed.
     *
     * @default `0`
     */
    autoCompleteDelay?: number;
    /**
     * Generates an autocompletion string for the current input of the user and currently highlighted entry in the dropdown.
     *
     * @param editorText the current text input from the user
     * @param entry the currently highlighted entry in the dropdown
     * @return The _full_ string (not only the completion part) to apply for auto-completion.
     * @default best effort implementation using entry properties
     */
    autoCompleteFunction?: (editorText: string, entry: E) => string;
    /**
     * List of characters that, when entered by the user, trigger the creation of a tag/entry.
     *
     * @default `[",", ";"]`
     */
    freeTextSeparators?: string[];
    /**
     * Creates an entry (object) from a string entered by the user.
     *
     * @param freeText the text entered by the user
     * @default `{ displayValue: freeText, _isFreeTextEntry: true }`
     */
    freeTextEntryFactory?: (freeText: string) => E | any;
    /**
     * The trigger is the button on the right side of the component that can be clicket to open the dropdown.
     *
     * @default `true`
     */
    showTrigger?: boolean;
    editingMode?: EditingMode;
    /**
     * It `true`, opening the dropdown will be delayed until the result callback of the [[queryFunction]] is called.
     *
     * @default `false`
     */
    showDropDownOnResultsOnly?: boolean;
    /**
     * Function deciding whether an entry entered by the user is complete or not (partial).
     * A partial entry is an entry that needs more input from the user's side.
     *
     * @param entry
     * @return `true` if the entry is considered complete, `false` if not
     * @default always returns `true`
     */
    tagCompleteDecider?: (entry: E) => boolean;
    /**
     * Used to merge the current partial tag with the newly selected by the user.
     * (composite tags feature)
     *
     * @param partialEntry the current partial entry
     * @param newEntryPart the entry selected/entered by the user
     * @return a new entry that will replace the current partial entry. This may in turn be a partial or complete entry.
     * @default always returns the `newEntryPart`
     */
    entryMerger?: (partialEntry: E, newEntryPart: E) => E;
    /**
     * Whether or not a partial tag should be removed when the component
     * looses the focus.
     *
     * @default `true`
     */
    removePartialTagOnBlur?: boolean;
    /**
     * Decides whether the user's input/selection is accepted or not.
     * This can serve many purposes, including
     *  * implementing duplicate prevention
     *  * limiting the number of selectable tags
     *  * allowing or disallowing free text entries (this was previously possible using the now removed `allowFreeText` option)
     *  * allowing only free text entries of a certain form
     *  * ...
     *
     * @param entry the entry to be accepted or not
     * @return `true` if the entry is accepted, `false` if not
     * @default accepting all non-free-text entries
     */
    selectionAcceptor?: (entry: E) => boolean;
    /**
     * HTML string defining the spinner to be displayed while entries are being retrieved.
     */
    spinnerTemplate?: string;
}
export declare class TrivialTagComboBox<E> implements TrivialComponent {
    private config;
    private $spinners;
    private $originalInput;
    private $tagComboBox;
    private $dropDown;
    private popper;
    private $trigger;
    private $editor;
    private $dropDownTargetElement;
    private $tagArea;
    readonly onSelectedEntryChanged: TrivialEvent<E[]>;
    readonly onFocus: TrivialEvent<void>;
    readonly onBlur: TrivialEvent<void>;
    private treeBox;
    private _isDropDownOpen;
    private entries;
    private selectedEntries;
    private blurCausedByClickInsideComponent;
    private autoCompleteTimeoutId;
    private doNoAutoCompleteBecauseBackspaceWasPressed;
    private listBoxDirty;
    private repositionDropDownScheduler;
    private editingMode;
    private usingDefaultQueryFunction;
    private currentPartialTag;
    constructor(originalInput: Element, options?: TrivialTagComboBoxConfig<E>);
    private cancelPartialTag;
    private findNearestTag;
    private updateListBoxEntries;
    updateEntries(newEntries: E[], highlightDirection?: HighlightDirection): void;
    private removeTag;
    private query;
    private showSpinner;
    private hideSpinner;
    private fireChangeEvents;
    private addSelectedEntry;
    private repositionDropDown;
    openDropDown(): void;
    closeDropDown(): void;
    private getNonSelectedEditorValue;
    private autoCompleteIfPossible;
    private isDropDownNeeded;
    private insertAtIndex;
    private doIgnoringBlurEvents;
    setEditingMode(newEditingMode: EditingMode): void;
    setSelectedEntries(entries: E[], forceAcceptance?: boolean): void;
    getSelectedEntries(): E[];
    getDropDownComponent(): TrivialComponent;
    getCurrentPartialTag(): E;
    focus(): void;
    getEditor(): Element;
    setShowTrigger(showTrigger: boolean): void;
    isDropDownOpen(): boolean;
    destroy(): void;
    getMainDomElement(): Element;
}
