/*
 * Decompiled with CFR 0.152.
 */
package org.teamapps.cluster.core;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.net.Socket;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.teamapps.cluster.core.Cluster;
import org.teamapps.cluster.core.ClusterNode;
import org.teamapps.cluster.crypto.AesCipher;
import org.teamapps.cluster.message.protocol.ClusterAvailableServicesUpdate;
import org.teamapps.cluster.message.protocol.ClusterConnectionRequest;
import org.teamapps.cluster.message.protocol.ClusterConnectionResult;
import org.teamapps.cluster.message.protocol.ClusterMessageFilePart;
import org.teamapps.cluster.message.protocol.ClusterNewLeaderInfo;
import org.teamapps.cluster.message.protocol.ClusterNewPeerInfo;
import org.teamapps.cluster.message.protocol.ClusterNodeData;
import org.teamapps.cluster.message.protocol.ClusterServiceBroadcastMessage;
import org.teamapps.cluster.message.protocol.ClusterServiceMethodRequest;
import org.teamapps.cluster.message.protocol.ClusterServiceMethodResult;
import org.teamapps.message.protocol.file.FileData;
import org.teamapps.message.protocol.file.FileDataReader;
import org.teamapps.message.protocol.file.FileDataType;
import org.teamapps.message.protocol.file.FileDataWriter;
import org.teamapps.message.protocol.file.GenericFileData;
import org.teamapps.message.protocol.message.Message;

public class ClusterConnection
implements FileDataWriter,
FileDataReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final int MAX_MESSAGE_SIZE = 100000000;
    private final ArrayBlockingQueue<Message> messageQueue = new ArrayBlockingQueue(100000);
    private final ClusterNodeData remoteHostAddress;
    private final AesCipher aesCipher;
    private final File tempDir;
    private final boolean incomingConnection;
    private volatile boolean connected;
    private Cluster cluster;
    private Socket socket;
    private DataInputStream dataInputStream;
    private DataOutputStream dataOutputStream;
    private ClusterNode clusterNode;
    private long lastMessageTimestamp;
    private long sentBytes;
    private long receivedBytes;
    private long sentMessages;
    private long receivedMessages;

    public ClusterConnection(Cluster cluster, Socket socket) {
        this.cluster = cluster;
        this.socket = socket;
        this.aesCipher = new AesCipher(cluster.getClusterConfig().getClusterSecret());
        this.tempDir = cluster.getTempDir();
        this.connected = true;
        this.remoteHostAddress = new ClusterNodeData().setHost(socket.getRemoteSocketAddress().toString()).setPort(socket.getPort());
        this.incomingConnection = true;
        this.handleSocket(socket);
        this.startReaderThread();
        this.startWriterThread();
    }

    public ClusterConnection(Cluster cluster, ClusterNodeData peerNode, ClusterConnectionRequest clusterConnectionRequest) {
        this.cluster = cluster;
        this.aesCipher = new AesCipher(cluster.getClusterConfig().getClusterSecret());
        this.remoteHostAddress = peerNode;
        this.tempDir = cluster.getTempDir();
        this.incomingConnection = false;
        this.connect(peerNode);
        if (this.connected) {
            this.startReaderThread();
            this.startWriterThread();
            this.writeDirectMessage(clusterConnectionRequest);
        }
    }

    private void connect(ClusterNodeData clusterNode) {
        try {
            this.socket = new Socket(clusterNode.getHost(), clusterNode.getPort());
            this.connected = true;
            this.handleSocket(this.socket);
        }
        catch (IOException e) {
            this.close();
        }
    }

    private void handleSocket(Socket socket) {
        try {
            socket.setKeepAlive(true);
            socket.setTcpNoDelay(true);
            this.dataInputStream = new DataInputStream(socket.getInputStream());
            this.dataOutputStream = new DataOutputStream(socket.getOutputStream());
        }
        catch (IOException e) {
            this.close();
        }
    }

    private void startReaderThread() {
        Thread thread = new Thread(() -> {
            while (this.connected) {
                try {
                    int messageSize = this.dataInputStream.readInt();
                    if (messageSize > 0 && messageSize < 100000000) {
                        byte[] data = new byte[messageSize];
                        this.dataInputStream.readFully(data);
                        byte[] messageData = this.aesCipher.decrypt(data);
                        Message message = new Message(messageData, (FileDataReader)this);
                        switch (message.getMessageDefUuid()) {
                            case "cluster.clusterServiceMethodRequest": {
                                this.handleClusterServiceMethodRequest(ClusterServiceMethodRequest.remap(message));
                                break;
                            }
                            case "cluster.clusterServiceMethodResult": {
                                this.handleClusterServiceMethodResult(ClusterServiceMethodResult.remap(message));
                                break;
                            }
                            case "cluster.clusterServiceBroadcastMessage": {
                                this.handleClusterServiceBroadcastMessage(ClusterServiceBroadcastMessage.remap(message));
                                break;
                            }
                            case "cluster.clusterMessageFilePart": {
                                this.handleClusterMessageFilePart(ClusterMessageFilePart.remap(message));
                                break;
                            }
                            case "cluster.clusterAvailableServicesUpdate": {
                                this.handleClusterAvailableServicesUpdate(ClusterAvailableServicesUpdate.remap(message));
                                break;
                            }
                            case "cluster.clusterNewPeerInfo": {
                                this.handleClusterNewPeerInfo(ClusterNewPeerInfo.remap(message));
                                break;
                            }
                            case "cluster.clusterNewLeaderInfo": {
                                this.handleClusterNewLeaderInfo(ClusterNewLeaderInfo.remap(message));
                                break;
                            }
                            case "cluster.clusterConnectionRequest": {
                                this.handleClusterConnectionRequest(ClusterConnectionRequest.remap(message));
                                break;
                            }
                            case "cluster.clusterConnectionResult": {
                                this.handleClusterConnectionResult(ClusterConnectionResult.remap(message));
                                break;
                            }
                            case "cluster.clusterNodeShutDownInfo": {
                                LOGGER.info("Cluster node {} - cluster peer is shutting down {}:{}", new Object[]{this.cluster.getLocalNode().getNodeId(), this.remoteHostAddress.getHost(), this.remoteHostAddress.getPort()});
                                this.close();
                            }
                        }
                        this.receivedBytes += (long)(messageSize + 4);
                        ++this.receivedMessages;
                        this.lastMessageTimestamp = System.currentTimeMillis();
                        continue;
                    }
                    this.close();
                }
                catch (Exception e) {
                    LOGGER.info("Cluster node [{}]: close connection to {} due to read error: {}", new Object[]{this.cluster.getLocalNode().getNodeId(), this.clusterNode.getNodeData().getNodeId(), e.getMessage()});
                    this.close();
                }
            }
        });
        thread.setName("connection-reader-" + this.socket.getInetAddress().getHostAddress() + "-" + this.socket.getPort());
        thread.setDaemon(true);
        thread.start();
    }

    private void handleClusterServiceBroadcastMessage(ClusterServiceBroadcastMessage broadcastMessage) {
        this.cluster.handleServiceBroadcastMessage(broadcastMessage, this.clusterNode);
    }

    private void handleClusterNewLeaderInfo(ClusterNewLeaderInfo newLeaderInfo) {
        this.cluster.handleClusterNewLeaderInfo(newLeaderInfo, this.clusterNode);
    }

    private void handleClusterNewPeerInfo(ClusterNewPeerInfo newPeerInfo) {
        this.cluster.handleClusterNewPeerInfo(newPeerInfo, this.clusterNode);
    }

    private void handleClusterAvailableServicesUpdate(ClusterAvailableServicesUpdate availableServicesUpdate) {
        this.cluster.handleClusterAvailableServicesUpdate(availableServicesUpdate, this.clusterNode);
    }

    private void handleClusterConnectionRequest(ClusterConnectionRequest request) {
        ClusterConnectionResult connectionResult = this.cluster.handleConnectionRequest(request, this);
        this.writeDirectMessage(connectionResult);
    }

    private void handleClusterConnectionResult(ClusterConnectionResult result) {
        this.cluster.handleConnectionResult(result, result.getLocalNode(), this);
    }

    private void handleClusterServiceMethodRequest(ClusterServiceMethodRequest request) {
        this.cluster.handleServiceMethodExecutionRequest(request, this.clusterNode);
    }

    private void handleClusterServiceMethodResult(ClusterServiceMethodResult result) {
        this.cluster.handleServiceMethodExecutionResult(result, this.clusterNode);
    }

    private void startWriterThread() {
        Thread thread = new Thread(() -> {
            while (this.connected) {
                try {
                    Message message = this.messageQueue.poll(30L, TimeUnit.SECONDS);
                    if (message == null) continue;
                    byte[] bytes = message.toBytes((FileDataWriter)this);
                    byte[] data = this.aesCipher.encrypt(bytes);
                    this.writeData(data);
                }
                catch (InterruptedException message) {
                }
                catch (Exception e) {
                    LOGGER.info("Cluster node [{}]: close connection to {} due to write error: {}", new Object[]{this.cluster.getLocalNode().getNodeId(), this.clusterNode.getNodeData().getNodeId(), e.getMessage()});
                    this.close();
                }
            }
        });
        thread.setDaemon(true);
        thread.setName("connection-writer-" + this.socket.getInetAddress().getHostAddress() + "-" + this.socket.getPort());
        thread.start();
    }

    public void writeMessage(Message message) {
        if (!this.messageQueue.offer(message)) {
            LOGGER.warn("Cluster node [{}]: error: connection message queue is full: {}:{}", new Object[]{this.cluster.getLocalNode().getNodeId(), this.remoteHostAddress.getHost(), this.remoteHostAddress.getPort()});
            this.close();
        }
    }

    private void writeDirectMessage(Message message) {
        try {
            if (this.connected) {
                byte[] bytes = message.toBytes((FileDataWriter)this);
                byte[] data = this.aesCipher.encrypt(bytes);
                this.writeData(data);
            }
        }
        catch (Exception e) {
            this.close();
        }
    }

    private synchronized void writeData(byte[] bytes) {
        try {
            this.dataOutputStream.writeInt(bytes.length);
            this.dataOutputStream.write(bytes);
            this.dataOutputStream.flush();
            this.sentBytes += (long)(bytes.length + 4);
            ++this.sentMessages;
            this.lastMessageTimestamp = System.currentTimeMillis();
        }
        catch (Exception e) {
            this.close();
        }
    }

    private void handleClusterMessageFilePart(ClusterMessageFilePart filePart) {
        try {
            long length = this.appendFileTransferData(filePart.getFileId(), filePart.getData(), filePart.isInitialMessage());
            if (filePart.isLastMessage() && length != filePart.getTotalLength()) {
                LOGGER.error("Cluster node {} - wrong cluster message file size, expected: {}, actual: {}", new Object[]{this.cluster.getLocalNode().getNodeId(), filePart.getTotalLength(), length});
                this.close();
            }
        }
        catch (IOException e) {
            this.close();
        }
    }

    public FileData writeFileData(FileData fileData) throws IOException {
        int maxContentSize;
        if (fileData.getType() == FileDataType.CLUSTER_STORE) {
            return fileData;
        }
        String fileId = UUID.randomUUID().toString().replace("-", ".");
        long fileLength = fileData.getLength();
        int parts = (int)((fileLength - 1L) / (long)(maxContentSize = 10000000)) + 1;
        if (parts == 1) {
            byte[] bytes = fileData.toBytes();
            ClusterMessageFilePart messageFilePart = new ClusterMessageFilePart().setFileId(fileId).setTotalLength(fileLength).setLastMessage(true).setData(bytes);
            this.writeDirectMessage(messageFilePart);
        } else {
            DataInputStream dis = new DataInputStream(new BufferedInputStream(fileData.getInputStream()));
            for (int i = 0; i < parts; ++i) {
                boolean lastMessage;
                int length = maxContentSize;
                boolean bl = lastMessage = i + 1 == parts;
                if (lastMessage) {
                    length = (int)(fileLength - (long)(i * maxContentSize));
                }
                byte[] bytes = new byte[length];
                dis.readFully(bytes);
                ClusterMessageFilePart messageFilePart = new ClusterMessageFilePart().setFileId(fileId).setTotalLength(fileLength).setLastMessage(lastMessage).setData(bytes);
                this.writeDirectMessage(messageFilePart);
            }
            dis.close();
        }
        return new GenericFileData(fileData.getType(), fileData.getFileName(), fileData.getLength(), fileId);
    }

    public FileData readFileData(FileDataType type, String fileName, long length, String descriptor, boolean encrypted, String encryptionKey) throws IOException {
        if (type == FileDataType.CLUSTER_STORE) {
            return new GenericFileData(type, fileName, length, descriptor, encrypted, encryptionKey);
        }
        File clusterTransferFile = this.getClusterTransferFile(descriptor);
        if (!clusterTransferFile.exists() || clusterTransferFile.length() != length) {
            throw new RuntimeException("Error reading file data:" + fileName + ", " + length + ", " + String.valueOf(clusterTransferFile));
        }
        return FileData.create((File)clusterTransferFile, (String)fileName);
    }

    private File getClusterTransferFile(String fileId) {
        return new File(this.tempDir, fileId + ".tmp");
    }

    private long appendFileTransferData(String fileId, byte[] bytes, boolean initialData) throws IOException {
        File file = this.getClusterTransferFile(fileId);
        BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(file, !initialData), 32000);
        bos.write(bytes);
        bos.close();
        return file.length();
    }

    public void close() {
        if (!this.connected) {
            return;
        }
        LOGGER.info("Cluster node {} - closed connection {}:{}", new Object[]{this.cluster.getLocalNode().getNodeId(), this.remoteHostAddress.getHost(), this.remoteHostAddress.getPort()});
        try {
            this.connected = false;
            if (this.socket != null) {
                this.socket.close();
            }
        }
        catch (Exception exception) {
        }
        finally {
            this.socket = null;
            this.dataOutputStream = null;
            this.dataInputStream = null;
            if (this.clusterNode != null) {
                this.clusterNode.handleConnectionClosed();
            }
        }
    }

    public void setClusterNode(ClusterNode clusterNode) {
        this.clusterNode = clusterNode;
    }

    public boolean isConnected() {
        return this.connected;
    }

    public long getLastMessageTimestamp() {
        return this.lastMessageTimestamp;
    }

    public long getSentBytes() {
        return this.sentBytes;
    }

    public long getReceivedBytes() {
        return this.receivedBytes;
    }

    public long getSentMessages() {
        return this.sentMessages;
    }

    public long getReceivedMessages() {
        return this.receivedMessages;
    }

    public boolean isIncomingConnection() {
        return this.incomingConnection;
    }
}

