/*-
 * ========================LICENSE_START=================================
 * TeamApps Commons
 * ---
 * Copyright (C) 2022 - 2023 TeamApps.org
 * ---
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.teamapps.commons.databinding;

import org.teamapps.commons.event.Event;

import java.util.function.Consumer;
import java.util.function.Supplier;

public final class DataBindings {

	/**
	 * Creates an {@link ObservableValue}, the value of which will always be determined by the given {@code valueSupplier}.
	 * The given {@link Event} will be used only for triggering the onChange method, not for determining its value.
	 *
	 * @param changeEvent   Event that will trigger the onChange event.
	 * @param valueSupplier Supplier for the value of the created ObservableValue.
	 * @param <T>           type of value
	 * @return the created ObservableValue
	 */
	public static <T> ObservableValue<T> createObservableValueFromEmptyEvent(Event<?> changeEvent, Supplier<T> valueSupplier) {
		return new ObservableValue<>() {
			private final Event<T> eventWithData;

			{
				eventWithData = new Event<>();
				changeEvent.addListener(o -> eventWithData.fire(valueSupplier.get()));
			}

			@Override
			public Event<T> onChange() {
				return eventWithData;
			}

			@Override
			public T get() {
				return valueSupplier.get();
			}
		};
	}

	/**
	 * Creates an {@link ObservableValue} that takes its value from the values emitted by the specified {@link Event}.
	 * These values will be cached. Note that initially, the cached value is going to be {@code null}.
	 *
	 * @param event Event the values of the created {@link ObservableValue} will be taken from.
	 * @param <T>   type of value
	 * @return the created ObservableValue
	 */
	public static <T> ObservableValue<T> createObservableValueFromEvent(Event<T> event) {
		return createObservableValueFromEvent(event, null);
	}

	/**
	 * Creates an {@link ObservableValue} that takes its value from the values emitted by the specified {@link Event}.
	 * These values will be cached.
	 *
	 * @param event        Event the values of the created {@link ObservableValue} will be taken from.
	 * @param initialValue The initial value (before the first occurrence of {@code event}).
	 * @param <T>          type of value
	 * @return the created ObservableValue
	 */
	public static <T> ObservableValue<T> createObservableValueFromEvent(Event<T> event, T initialValue) {
		return new ObservableValue<>() {

			private T lastSeenValue = initialValue;

			{
				event.addListener(t -> this.lastSeenValue = t);
			}

			@Override
			public Event<T> onChange() {
				return event;
			}

			@Override
			public T get() {
				return lastSeenValue;
			}
		};
	}

	/**
	 * Creates a {@link MutableValue} from a {@link Consumer}. Since both interfaces are virtually the same, this is a trivial operation.
	 *
	 * @param consumer the consumer.
	 * @param <T>      value type
	 * @return the {@link MutableValue}
	 */
	public static <T> MutableValue<T> createMutableValue(Consumer<T> consumer) {
		return consumer::accept;
	}

	public static <T> void bindOneWay(ObservableValue<T> observableValue, MutableValue<T> mutableValue) {
		mutableValue.set(observableValue.get()); // initialize value
		observableValue.onChange().addListener(value -> mutableValue.set(observableValue.get()));
	}

	public static <T> void bindTwoWays(TwoWayBindableValue<T> bindable1, TwoWayBindableValue<T> bindable2) {
		LoopInterruptingConsumerBuilder consumerBuilder = new LoopInterruptingConsumerBuilder();
		bindable1.onChange().addListener(consumerBuilder.create(aVoid -> bindable2.set(bindable1.get())));
		bindable2.onChange().addListener(consumerBuilder.create(aVoid -> bindable1.set(bindable2.get())));
		bindable2.set(bindable1.get());
	}

	static class LoopInterruptingConsumerBuilder {
		private final ThreadLocal<Boolean> processing = new ThreadLocal<>();

		public <EVENT_DATA> Consumer<EVENT_DATA> create(Consumer<EVENT_DATA> handler) {
			return (eventData) -> {
				if (!isAlreadyProcessing()) {
					processing.set(true);
					try {
						handler.accept(eventData);
					} finally {
						processing.set(false);
					}
				}
			};
		}

		private boolean isAlreadyProcessing() {
			return processing.get() != null && processing.get();
		}
	}
}
