/*-
 * ========================LICENSE_START=================================
 * TeamApps Commons
 * ---
 * Copyright (C) 2022 - 2023 TeamApps.org
 * ---
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.teamapps.commons.util.collections;

import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;

public final class ByKeyComparisonResult<A, B, K> {
	private final List<A> aEntriesNotInB;
	private final List<A> aEntriesInB;
	private final List<B> bEntriesNotInA;
	private final List<B> bEntriesInA;
	private final Map<K, A> aByKey;
	private final Map<K, B> bByKey;
	private final Function<A, K> keyExtractorA;
	private final Function<B, K> keyExtractorB;

	public ByKeyComparisonResult(
			List<A> aEntriesNotInB,
			List<A> aEntriesInB,
			List<B> bEntriesNotInA,
			List<B> bEntriesInA,
			Map<K, A> aByKey,
			Map<K, B> bByKey,
			Function<A, K> keyExtractorA,
			Function<B, K> keyExtractorB
	) {
		this.aEntriesNotInB = aEntriesNotInB;
		this.aEntriesInB = aEntriesInB;
		this.bEntriesNotInA = bEntriesNotInA;
		this.bEntriesInA = bEntriesInA;
		this.aByKey = aByKey;
		this.bByKey = bByKey;
		this.keyExtractorA = keyExtractorA;
		this.keyExtractorB = keyExtractorB;
	}

	public A getA(B b) {
		return aByKey.get(keyExtractorB.apply(b));
	}

	public B getB(A a) {
		return bByKey.get(keyExtractorA.apply(a));
	}

	public List<A> getAEntriesNotInB() {
		return aEntriesNotInB;
	}

	public List<A> getAEntriesInB() {
		return aEntriesInB;
	}

	public List<B> getBEntriesNotInA() {
		return bEntriesNotInA;
	}

	public List<B> getBEntriesInA() {
		return bEntriesInA;
	}

	public List<Pair<A, B>> getPairs() {
		return aEntriesInB.stream()
				.map(a -> new Pair<>(a, getB(a)))
				.toList();
	}

	public void forEachExisting(BiConsumer<A, B> sameKeyConsumer) {
		for (A a : aEntriesInB) {
			B b = getB(a);
			sameKeyConsumer.accept(a, b);
		}
	}

	public void forEachNew(Consumer<B> newKeyConsumer) {
		for (B b : bEntriesNotInA) {
			newKeyConsumer.accept(b);
		}
	}

	public void forEachRemoved(Consumer<A> removedKeyConsumer) {
		for (A a : aEntriesNotInB) {
			removedKeyConsumer.accept(a);
		}
	}

	public boolean isDifferent() {
		return !bEntriesNotInA.isEmpty() || !aEntriesNotInB.isEmpty();
	}
}
