/*-
 * ========================LICENSE_START=================================
 * TeamApps Commons
 * ---
 * Copyright (C) 2022 - 2023 TeamApps.org
 * ---
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.teamapps.commons.util.collections;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.invoke.MethodHandles;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

public class CollectionUtil {

	private static final Logger LOGGER = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());

	public static <A, B, K> ByKeyComparisonResult<A, B, K> compareByKey(Collection<A> collectionA, Collection<B> collectionB, Function<A, K> keyExtractorA, Function<B, K> keyExtractorB) {
		return compareByKey(collectionA, collectionB, keyExtractorA, keyExtractorB, false);
	}

	public static <A, B, K> ByKeyComparisonResult<A, B, K> compareByKey(Collection<A> collectionA, Collection<B> collectionB, Function<A, K> keyExtractorA, Function<B, K> keyExtractorB, boolean enforceUniqueKeys) {
		final List<A> aEntriesNotInB = new ArrayList<>();
		final List<A> aEntriesInB = new ArrayList<>();
		final List<B> bEntriesNotInA = new ArrayList<>();
		final List<B> bEntriesInA = new ArrayList<>();
		Map<K, A> aByKey;
		Map<K, B> bByKey;

		if (enforceUniqueKeys) {
			aByKey = collectionA.stream().collect(Collectors.toMap(keyExtractorA, a -> a));
			bByKey = collectionB.stream().collect(Collectors.toMap(keyExtractorB, b -> b));
		} else {
			aByKey = collectionA.stream().collect(Collectors.toMap(keyExtractorA, a -> a, (key1, key2) -> {
				LOGGER.warn("collectionA contains entries with non-unique key \"" + keyExtractorA.apply(key1) + "\": " + key1 + ", " + key2);
				return key1;
			}));
			bByKey = collectionB.stream().collect(Collectors.toMap(keyExtractorB, b -> b, (key1, key2) -> {
				LOGGER.warn("collectionB contains entries with non-unique key \"" + keyExtractorB.apply(key1) + "\": " + key1 + ", " + key2);
				return key1;
			}));
		}

		for (A a : collectionA) {
			if (bByKey.containsKey(keyExtractorA.apply(a))) {
				aEntriesInB.add(a);
			} else {
				aEntriesNotInB.add(a);
			}
		}
		for (B b : collectionB) {
			if (aByKey.containsKey(keyExtractorB.apply(b))) {
				bEntriesInA.add(b);
			} else {
				bEntriesNotInA.add(b);
			}
		}

		return new ByKeyComparisonResult<>(aEntriesNotInB, aEntriesInB, bEntriesNotInA, bEntriesInA, aByKey, bByKey, keyExtractorA, keyExtractorB);
	}

}
