/*
 * Decompiled with CFR 0.152.
 */
package org.teamapps.media.exec;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.teamapps.media.exec.ExternalResource;

public class CommandLineExecutor {
    private static final Logger logger = LoggerFactory.getLogger(CommandLineExecutor.class);
    private AtomicLong longId = new AtomicLong();
    private File workingDirectory;
    private String binaryPath;
    private String name;

    public CommandLineExecutor(ExternalResource resource) {
        this(null, resource);
    }

    public CommandLineExecutor(File workingDirectory, ExternalResource resource) {
        this.workingDirectory = this.createWorkingDir(workingDirectory);
        this.binaryPath = resource.createBinary(this.workingDirectory);
        this.name = this.getClass().getSimpleName();
    }

    public CommandLineExecutor(String binaryPath) {
        this.binaryPath = binaryPath;
        this.workingDirectory = this.createWorkingDir(null);
        this.name = this.getClass().getSimpleName();
    }

    public boolean executeCommand(String commandArgs, int maxRuntimeInSeconds, boolean showLogs) {
        try {
            Process process = Runtime.getRuntime().exec(this.binaryPath + " " + commandArgs);
            if (showLogs) {
                this.showLogs(process.getErrorStream(), true);
                this.showLogs(process.getInputStream(), false);
            }
            if (!process.waitFor(maxRuntimeInSeconds, TimeUnit.SECONDS)) {
                process.destroy();
                process.destroyForcibly();
                return false;
            }
            return true;
        }
        catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public boolean executeCommand(int maxRuntimeInSeconds, boolean showLogs, String ... commandArgs) {
        try {
            String[] cmdArray = new String[commandArgs.length + 1];
            cmdArray[0] = this.binaryPath;
            for (int i = 1; i <= commandArgs.length; ++i) {
                cmdArray[i] = commandArgs[i - 1];
            }
            Process process = Runtime.getRuntime().exec(cmdArray);
            if (showLogs) {
                this.showLogs(process.getErrorStream(), true);
                this.showLogs(process.getInputStream(), false);
            }
            if (!process.waitFor(maxRuntimeInSeconds, TimeUnit.SECONDS)) {
                process.destroy();
                process.destroyForcibly();
                return false;
            }
            return true;
        }
        catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public boolean executeCommand(String commandArgs, int maxRuntimeInSeconds, StringBuilder info, StringBuilder error) {
        try {
            Process process = Runtime.getRuntime().exec(this.binaryPath + " " + commandArgs);
            this.readLogs(process.getInputStream(), info);
            this.readLogs(process.getErrorStream(), error);
            if (!process.waitFor(maxRuntimeInSeconds, TimeUnit.SECONDS)) {
                process.destroy();
                process.destroyForcibly();
                return false;
            }
            return true;
        }
        catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public CompletableFuture<Void> executeCommandAsync(String commandArgs, int maxRuntimeInSeconds) {
        return this.executeCommandAsync(commandArgs, maxRuntimeInSeconds, null);
    }

    public CompletableFuture<Void> executeCommandAsync(String commandArgs, int maxRuntimeInSeconds, Executor executor) {
        Runnable runnable = () -> {
            StringBuilder logsStringBuilder = new StringBuilder();
            try {
                String command = this.binaryPath + " " + commandArgs;
                Process process = Runtime.getRuntime().exec(command);
                this.readLogs(process.getErrorStream(), logsStringBuilder);
                if (!process.waitFor(maxRuntimeInSeconds, TimeUnit.SECONDS)) {
                    process.destroy();
                    process.destroyForcibly();
                    throw new TimeoutException(logsStringBuilder.toString());
                }
            }
            catch (Exception e) {
                throw new RuntimeException(logsStringBuilder.toString(), e);
            }
        };
        if (executor != null) {
            return CompletableFuture.runAsync(runnable, executor);
        }
        return CompletableFuture.runAsync(runnable);
    }

    public CompletableFuture<File> executeCommandAsync(String commandArgs, File expectedResultFile, boolean expectFileWithContent, int maxRuntimeInSeconds, Executor executor) {
        StringBuilder logsBuilder = new StringBuilder();
        StringBuilder errorLogsBuilder = new StringBuilder();
        Function<Void, File> resultFileFunction = aVoid -> {
            if (!expectedResultFile.exists() || expectFileWithContent && expectedResultFile.length() == 0L) {
                throw new RuntimeException("Error: missing result file with executable args:" + commandArgs + "\n" + logsBuilder.toString() + errorLogsBuilder.toString());
            }
            return expectedResultFile;
        };
        Runnable runnable = this.createExecutionRunnable(commandArgs, maxRuntimeInSeconds, logsBuilder, errorLogsBuilder);
        return executor != null ? CompletableFuture.runAsync(runnable, executor).thenApply(resultFileFunction) : CompletableFuture.runAsync(runnable).thenApply(resultFileFunction);
    }

    public CompletableFuture<File> executeCommandAsync(String commandArgs, File path, String partOfResultFileName, boolean expectFileWithContent, int maxRuntimeInSeconds, Executor executor) {
        StringBuilder logsBuilder = new StringBuilder();
        StringBuilder errorLogsBuilder = new StringBuilder();
        Function<Void, File> resultFileFunction = aVoid -> {
            File expectedResultFile = Arrays.stream(path.listFiles()).filter(file -> file.getName().contains(partOfResultFileName)).findAny().orElse(null);
            if (expectedResultFile == null || !expectedResultFile.exists() || expectFileWithContent && expectedResultFile.length() == 0L) {
                throw new RuntimeException("Error: missing result file with executable args:" + commandArgs + "\n" + logsBuilder.toString() + errorLogsBuilder.toString());
            }
            return expectedResultFile;
        };
        Runnable runnable = this.createExecutionRunnable(commandArgs, maxRuntimeInSeconds, logsBuilder, errorLogsBuilder);
        return executor != null ? CompletableFuture.runAsync(runnable, executor).thenApply(resultFileFunction) : CompletableFuture.runAsync(runnable).thenApply(resultFileFunction);
    }

    private Runnable createExecutionRunnable(String commandArgs, int maxRuntimeInSeconds, StringBuilder logsBuilder, StringBuilder errorLogsBuilder) {
        return () -> {
            try {
                String command = this.binaryPath + " " + commandArgs;
                Process process = Runtime.getRuntime().exec(command);
                this.readLogs(process.getInputStream(), logsBuilder);
                this.readLogs(process.getErrorStream(), errorLogsBuilder);
                if (!process.waitFor(maxRuntimeInSeconds, TimeUnit.SECONDS)) {
                    process.destroy();
                    process.destroyForcibly();
                    throw new TimeoutException(errorLogsBuilder.toString());
                }
            }
            catch (Exception e) {
                throw new RuntimeException(errorLogsBuilder.toString(), e);
            }
        };
    }

    private void readLogs(InputStream is, StringBuilder sb) {
        Thread thread = new Thread(() -> {
            try {
                String line;
                BufferedReader reader = new BufferedReader(new InputStreamReader(is));
                while ((line = reader.readLine()) != null) {
                    sb.append(line).append("\n");
                }
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        });
        thread.setDaemon(true);
        thread.setName("cli-output-reader");
        thread.start();
    }

    public void showLogs(final InputStream is, final boolean errorLog) {
        Thread thread = new Thread(new Runnable(){

            @Override
            public void run() {
                try {
                    String line;
                    BufferedReader reader = new BufferedReader(new InputStreamReader(is));
                    while ((line = reader.readLine()) != null) {
                        if (errorLog) {
                            logger.warn(CommandLineExecutor.this.name + ":" + line);
                            continue;
                        }
                        logger.info(CommandLineExecutor.this.name + ":" + line);
                    }
                }
                catch (IOException e) {
                    e.printStackTrace();
                }
            }
        });
        thread.setDaemon(true);
        thread.setName("cli-log-reader");
        thread.start();
    }

    public File createWorkingDir(File workingDirectory) {
        if (workingDirectory == null || !workingDirectory.getParentFile().exists()) {
            try {
                return File.createTempFile("temp", "tmp").getParentFile();
            }
            catch (IOException e) {
                e.printStackTrace();
                return null;
            }
        }
        if (!workingDirectory.exists()) {
            workingDirectory.mkdir();
        }
        return workingDirectory;
    }

    public File getWorkingDirectory() {
        return this.workingDirectory;
    }

    public String getWorkingPath() {
        return this.workingDirectory.getPath();
    }

    public String getBinaryPath() {
        return this.binaryPath;
    }

    public String getNextFileName(String name, String suffix) {
        return name + System.currentTimeMillis() + "-" + this.longId.incrementAndGet() + suffix;
    }
}

