/*
 * Decompiled with CFR 0.152.
 */
package org.teamapps.ux.servlet.resourceprovider;

import jakarta.servlet.ServletException;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServlet;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.teamapps.ux.resource.Resource;
import org.teamapps.ux.servlet.resourceprovider.ResourceProvider;

public class ResourceProviderServlet
extends HttpServlet {
    private static final Logger LOGGER = LoggerFactory.getLogger(ResourceProviderServlet.class);
    private static final long ONE_SECOND_IN_MILLIS = TimeUnit.SECONDS.toMillis(1L);
    private static final String ETAG = "W/\"%s-%s\"";
    private static final Pattern RANGE_PATTERN = Pattern.compile("^bytes=[0-9]*-[0-9]*(,[0-9]*-[0-9]*)*$");
    private static final String MULTIPART_BOUNDARY = UUID.randomUUID().toString();
    private static final String CONTENT_DISPOSITION_HEADER = "%s;filename=\"%2$s\"; filename*=UTF-8''%2$s";
    private final ResourceProvider resourceProvider;

    public ResourceProviderServlet(ResourceProvider resourceProvider) {
        this.resourceProvider = resourceProvider;
    }

    protected void doHead(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.doRequest(request, response, true);
    }

    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.doRequest(request, response, false);
    }

    private void doRequest(HttpServletRequest request, HttpServletResponse response, boolean head) throws IOException {
        Resource resource;
        response.reset();
        try {
            resource = this.resourceProvider.getResource(request.getServletPath(), request.getPathInfo(), request.getSession().getId());
        }
        catch (IllegalArgumentException e) {
            LOGGER.info("Got an IllegalArgumentException from ResourceProvider. Interpreting it as 400 Bad Request. " + request.getServletPath() + request.getPathInfo() + " for HTTP session " + request.getSession().getId(), (Throwable)e);
            response.sendError(400);
            return;
        }
        if (resource == null) {
            this.handleNotFound(request, response);
            return;
        }
        if (this.preconditionFailed(request, resource)) {
            response.sendError(412);
            return;
        }
        this.setCacheHeaders(response, resource, resource.getExpires().getTime());
        if (this.notModified(request, resource)) {
            response.setStatus(304);
            return;
        }
        List<Range> ranges = this.getRanges(request, resource);
        if (ranges == null) {
            response.setHeader("Content-Range", "bytes */" + resource.getLength());
            response.sendError(416);
            return;
        }
        if (!ranges.isEmpty()) {
            response.setStatus(206);
        } else {
            ranges.add(new Range(0L, resource.getLength() - 1L));
        }
        String contentType = this.setContentHeaders(response, resource, ranges);
        if (head) {
            return;
        }
        this.writeContent(response, resource, ranges, contentType);
    }

    protected void handleNotFound(HttpServletRequest request, HttpServletResponse response) throws IOException {
        response.sendError(404);
    }

    protected String getAttachmentName(HttpServletRequest request, File file) {
        return file.getName();
    }

    private boolean preconditionFailed(HttpServletRequest request, Resource resource) {
        String match = request.getHeader("If-Match");
        long unmodified = request.getDateHeader("If-Unmodified-Since");
        return match != null ? !ResourceProviderServlet.matches(match, this.getETag(resource)) : unmodified != -1L && ResourceProviderServlet.modified(unmodified, resource.getLastModified().getTime());
    }

    private void setCacheHeaders(HttpServletResponse response, Resource resource, long expires) {
        ResourceProviderServlet.setCacheHeaders(response, expires);
        response.setHeader("ETag", this.getETag(resource));
        response.setDateHeader("Last-Modified", resource.getLastModified().getTime());
    }

    private boolean notModified(HttpServletRequest request, Resource resource) {
        String noMatch = request.getHeader("If-None-Match");
        long modified = request.getDateHeader("If-Modified-Since");
        return noMatch != null ? ResourceProviderServlet.matches(noMatch, this.getETag(resource)) : modified != -1L && !ResourceProviderServlet.modified(modified, resource.getLastModified().getTime());
    }

    private List<Range> getRanges(HttpServletRequest request, Resource resource) {
        ArrayList<Range> ranges = new ArrayList<Range>(1);
        String rangeHeader = request.getHeader("Range");
        if (rangeHeader == null) {
            return ranges;
        }
        if (!RANGE_PATTERN.matcher(rangeHeader).matches()) {
            return null;
        }
        String ifRange = request.getHeader("If-Range");
        if (ifRange != null && !ifRange.equals(this.getETag(resource))) {
            try {
                long ifRangeTime = request.getDateHeader("If-Range");
                if (ifRangeTime != -1L && ResourceProviderServlet.modified(ifRangeTime, resource.getLastModified().getTime())) {
                    return ranges;
                }
            }
            catch (IllegalArgumentException ifRangeHeaderIsInvalid) {
                LOGGER.debug("If-Range header is invalid. Let's just return full file then.", (Throwable)ifRangeHeaderIsInvalid);
                return ranges;
            }
        }
        for (String rangeHeaderPart : rangeHeader.split("=")[1].split(",")) {
            Range range = this.parseRange(rangeHeaderPart, resource.getLength());
            if (range == null) {
                return null;
            }
            ranges.add(range);
        }
        return ranges;
    }

    private Range parseRange(String range, long length) {
        long start = ResourceProviderServlet.parseLong(range, 0, range.indexOf(45));
        long end = ResourceProviderServlet.parseLong(range, range.indexOf(45) + 1, range.length());
        if (start == -1L) {
            start = length - end;
            end = length - 1L;
        } else if (end == -1L || end > length - 1L) {
            end = length - 1L;
        }
        if (start > end) {
            return null;
        }
        return new Range(start, end);
    }

    private String setContentHeaders(HttpServletResponse response, Resource resource, List<Range> ranges) {
        String contentType = resource.getMimeType();
        String filename = resource.getName();
        response.setHeader("Content-Disposition", String.format(CONTENT_DISPOSITION_HEADER, resource.isAttachment() ? "attachment" : "inline", ResourceProviderServlet.encodeURI(filename)));
        response.setHeader("Accept-Ranges", "bytes");
        if (ranges.size() == 1) {
            Range range = ranges.get(0);
            response.setContentType(contentType);
            response.setHeader("Content-Length", String.valueOf(range.length));
            if (response.getStatus() == 206) {
                response.setHeader("Content-Range", "bytes " + range.start + "-" + range.end + "/" + resource.getLength());
            }
        } else {
            response.setContentType("multipart/byteranges; boundary=" + MULTIPART_BOUNDARY);
        }
        return contentType;
    }

    private void writeContent(HttpServletResponse response, Resource resource, List<Range> ranges, String contentType) throws IOException {
        ServletOutputStream output = response.getOutputStream();
        if (ranges.size() == 1) {
            Range range = ranges.get(0);
            IOUtils.copyLarge((InputStream)resource.getInputStream(), (OutputStream)output, (long)range.start, (long)range.length);
        } else {
            for (Range r : ranges) {
                output.println();
                output.println("--" + MULTIPART_BOUNDARY);
                output.println("Content-Type: " + contentType);
                output.println("Content-Range: bytes " + r.start + "-" + r.end + "/" + resource.getLength());
                IOUtils.copyLarge((InputStream)resource.getInputStream(), (OutputStream)output, (long)r.start, (long)r.length);
            }
            output.println();
            output.println("--" + MULTIPART_BOUNDARY + "--");
        }
    }

    public static String encodeURL(String string) {
        if (string == null) {
            return null;
        }
        return URLEncoder.encode(string, StandardCharsets.UTF_8);
    }

    public static String encodeURI(String string) {
        if (string == null) {
            return null;
        }
        return ResourceProviderServlet.encodeURL(string).replace("+", "%20").replace("%21", "!").replace("%27", "'").replace("%28", "(").replace("%29", ")").replace("%7E", "~");
    }

    public static void setCacheHeaders(HttpServletResponse response, long expires) {
        if (expires > 0L) {
            response.setHeader("Cache-Control", "public,max-age=" + expires + ",must-revalidate");
            response.setDateHeader("Expires", System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(expires));
            response.setHeader("Pragma", "");
        } else {
            ResourceProviderServlet.setNoCacheHeaders(response);
        }
    }

    public static void setNoCacheHeaders(HttpServletResponse response) {
        response.setHeader("Cache-Control", "no-cache,no-store,must-revalidate");
        response.setDateHeader("Expires", 0L);
        response.setHeader("Pragma", "no-cache");
    }

    private String getETag(Resource resource) {
        return String.format(ETAG, ResourceProviderServlet.encodeURL(resource.getName()), resource.getLastModified());
    }

    private static boolean matches(String matchHeader, String eTag) {
        Object[] matchValues = matchHeader.split("\\s*,\\s*");
        Arrays.sort(matchValues);
        return Arrays.binarySearch(matchValues, eTag) > -1 || Arrays.binarySearch(matchValues, "*") > -1;
    }

    private static boolean modified(long modifiedHeader, long lastModified) {
        return modifiedHeader + ONE_SECOND_IN_MILLIS <= lastModified;
    }

    private static long parseLong(String value, int beginIndex, int endIndex) {
        String substring = value.substring(beginIndex, endIndex);
        return substring.isEmpty() ? -1L : Long.parseLong(substring);
    }

    private static boolean accepts(String acceptHeader, String toAccept) {
        Object[] acceptValues = acceptHeader.split("\\s*([,;])\\s*");
        Arrays.sort(acceptValues);
        return Arrays.binarySearch(acceptValues, toAccept) > -1 || Arrays.binarySearch(acceptValues, toAccept.replaceAll("/.*$", "/*")) > -1 || Arrays.binarySearch(acceptValues, "*/*") > -1;
    }

    private static class Range {
        private final long start;
        private final long end;
        private final long length;

        public Range(long start, long end) {
            this.start = start;
            this.end = end;
            this.length = end - start + 1L;
        }
    }
}

