/*
 * Decompiled with CFR 0.152.
 */
package org.teamapps.universaldb.transaction;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.MappedByteBuffer;
import java.util.Iterator;
import java.util.zip.GZIPOutputStream;
import org.agrona.concurrent.AtomicBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.teamapps.universaldb.schema.Schema;
import org.teamapps.universaldb.transaction.ByteArrayTransactionReader;
import org.teamapps.universaldb.transaction.ClusterTransaction;
import org.teamapps.universaldb.transaction.TransactionIdHandler;
import org.teamapps.universaldb.transaction.TransactionPacket;
import org.teamapps.universaldb.transaction.TransactionRequest;
import org.teamapps.universaldb.util.MappedStoreUtil;

public class TransactionStore
implements TransactionIdHandler {
    private static final int TIMESTAMP_FIRST_SYSTEM_START_POS = 0;
    private static final int TIMESTAMP_SYSTEM_START_POS = 8;
    private static final int TIMESTAMP_SHUTDOWN_POS = 16;
    private static final int LAST_TRANSACTION_ID_POS = 24;
    private static final int CURRENT_TRANSACTION_ID_POS = 32;
    private static final int CURRENT_TRANSACTION_FILE_ID_POS = 40;
    private static final int CURRENT_TRANSACTION_FILE_POSITION_POS = 44;
    private static final int TRANSACTIONS_COUNT_POS = 48;
    public static final long MAX_TRANSACTION_FILE_SIZE = Integer.MAX_VALUE;
    private static final int CLUSTER_NODE_TYPE_POS = 56;
    private static final int CLUSTER_CONFIG_POS = 1000;
    private static final int SCHEMA_POS = 30000;
    private static final Logger log = LoggerFactory.getLogger(TransactionStore.class);
    private File path;
    private AtomicBuffer buffer;
    private static int storeSize = 200000;
    private long timestampFirstSystemStart;
    private long timestampSystemStart;
    private long timestampShutdown;
    private long lastTransactionId;
    private long currentTransactionId;
    private int currentTransactionFileId;
    private int currentTransactionFilePosition;
    private long transactionCount;
    private File currentTransactionFile;
    private DataOutputStream currentTransactionOutputStream;
    private Schema schema;

    public TransactionStore(File path) throws IOException {
        this.path = new File(path, "transaction-log");
        File file = new File(this.path, "database.stat");
        this.buffer = MappedStoreUtil.createAtomicBuffer(file, storeSize);
        this.init();
    }

    private void init() throws IOException {
        this.timestampFirstSystemStart = this.buffer.getLong(0);
        this.timestampSystemStart = this.buffer.getLong(8);
        this.timestampShutdown = this.buffer.getLong(16);
        this.lastTransactionId = this.buffer.getLong(24);
        this.currentTransactionId = this.buffer.getLong(32);
        this.currentTransactionFileId = this.buffer.getInt(40);
        this.currentTransactionFilePosition = this.buffer.getInt(44);
        this.transactionCount = this.buffer.getLong(48);
        this.timestampSystemStart = System.currentTimeMillis();
        this.buffer.putLong(8, this.timestampSystemStart);
        if (this.timestampFirstSystemStart == 0L) {
            this.buffer.putLong(0, this.timestampSystemStart);
        }
        this.currentTransactionFile = TransactionStore.getTransactionFileByFileId(this.currentTransactionFileId, this.path, false);
        this.currentTransactionOutputStream = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.currentTransactionFile, true)));
        if (this.currentTransactionId == 0L) {
            this.currentTransactionId = 8L;
            if (this.currentTransactionFile.length() == 0L) {
                this.currentTransactionOutputStream.writeLong(System.currentTimeMillis());
            }
            this.currentTransactionFilePosition = 8;
        }
        this.schema = this.loadSchema();
        log.info("Start UniversalDB with path:" + this.path.getParentFile().getPath());
        log.info("Start transaction-store, file-id:" + this.currentTransactionFileId + ", file-pos:" + this.currentTransactionFilePosition + ", transaction-count:" + this.transactionCount);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                this.currentTransactionOutputStream.flush();
                log.info("Shutting down transaction-store, file-id:" + this.currentTransactionFileId + ", file-pos:" + this.currentTransactionFilePosition + ", transaction-count:" + this.transactionCount);
            }
            catch (IOException e) {
                e.printStackTrace();
            }
        }));
    }

    public Iterator<byte[]> getTransactions(long startTransaction, long lastTransaction) {
        try {
            return new ByteArrayTransactionReader(startTransaction, this.lastTransactionId, this.path);
        }
        catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    public void synchronizeTransaction(ClusterTransaction transaction) throws IOException {
        TransactionRequest transactionRequest = transaction.createRequest();
        this.executeTransaction(transactionRequest);
    }

    public Schema loadSchema() throws IOException {
        int len = this.buffer.getInt(30000);
        if (len == 0) {
            return null;
        }
        byte[] bytes = new byte[len];
        this.buffer.getBytes(30004, bytes);
        return new Schema(new DataInputStream(new ByteArrayInputStream(bytes)));
    }

    public void saveSchema(Schema schema) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        schema.writeSchema(new DataOutputStream(bos));
        byte[] bytes = bos.toByteArray();
        this.buffer.putInt(30000, bytes.length);
        this.buffer.putBytes(30004, bytes);
        this.setSchema(schema);
    }

    public void setSchema(Schema schema) {
        this.schema = schema;
    }

    public synchronized void executeTransaction(TransactionRequest transactionRequest) throws IOException {
        if (transactionRequest.isExecuted()) {
            long transactionId = transactionRequest.getTransaction().getTransactionId();
            if (this.currentTransactionId != transactionId) {
                throw new RuntimeException("Cannot execute transaction with unexptected transaction id, expected" + this.currentTransactionId + ", actual:" + transactionId);
            }
            transactionRequest.executeResolvedTransaction(this);
        } else {
            transactionRequest.executeUnresolvedTransaction(this);
        }
        this.writeTransaction(transactionRequest);
    }

    private void writeTransaction(TransactionRequest transactionRequest) throws IOException {
        if (!transactionRequest.isExecuted()) {
            throw new RuntimeException("Cannot store transaction that has not been executed!");
        }
        TransactionPacket packet = transactionRequest.getPacket();
        byte[] bytes = packet.writePacketBytes();
        this.checkTransactionFilePosition(transactionRequest.getTransaction());
        ++this.transactionCount;
        this.currentTransactionOutputStream.write(1);
        this.currentTransactionOutputStream.writeInt(bytes.length);
        this.currentTransactionOutputStream.write(bytes);
        this.currentTransactionOutputStream.flush();
        this.lastTransactionId = this.currentTransactionId;
        this.buffer.putLong(24, this.lastTransactionId);
        int size = bytes.length + 5;
        if (TransactionStore.newTransactionFileRequired(this.currentTransactionFilePosition, size)) {
            File fileToZip = this.currentTransactionFile;
            File zipFile = TransactionStore.getTransactionFileByFileId(this.currentTransactionFileId, this.path, true);
            File previousFile = null;
            if (this.currentTransactionFileId > 0) {
                previousFile = TransactionStore.getTransactionFileByFileId(this.currentTransactionFileId - 1, this.path, false);
            }
            ++this.currentTransactionFileId;
            this.currentTransactionFilePosition = 8;
            this.currentTransactionOutputStream.close();
            this.currentTransactionFile = TransactionStore.getTransactionFileByFileId(this.currentTransactionFileId, this.path, true);
            this.currentTransactionOutputStream = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(this.currentTransactionFile, true)));
            this.currentTransactionOutputStream.writeLong(System.currentTimeMillis());
            this.compressTransactionFile(fileToZip, zipFile, previousFile);
        } else {
            this.currentTransactionFilePosition += size;
            this.buffer.putInt(44, this.currentTransactionFilePosition);
        }
        this.buffer.putInt(40, this.currentTransactionFileId);
        this.buffer.putInt(44, this.currentTransactionFilePosition);
        this.buffer.putLong(48, this.transactionCount);
        this.currentTransactionId = TransactionStore.createTransactionIndex(this.currentTransactionFileId, this.currentTransactionFilePosition);
        this.buffer.putLong(32, this.currentTransactionId);
    }

    private void checkTransactionFilePosition(ClusterTransaction transaction) {
        long transactionId = transaction.getTransactionId();
        int fileId = TransactionStore.getTransactionFileId(transactionId);
        int filePosition = TransactionStore.getTransactionFilePosition(transactionId);
        if (this.currentTransactionFileId != fileId || this.currentTransactionFilePosition != filePosition) {
            throw new RuntimeException("Wrong transaction with id: " + transactionId + ", expected file-id:" + this.currentTransactionFileId + ", actual file-id:" + fileId + ", expected file-position:" + this.currentTransactionFilePosition + ", actual position:" + filePosition);
        }
    }

    private void compressTransactionFile(File fileToZip, File zipFile, File previousFile) {
        new Thread(() -> {
            byte[] buffer = new byte[8096];
            try (GZIPOutputStream gzipOut = new GZIPOutputStream(new BufferedOutputStream(new FileOutputStream(zipFile)));){
                int read;
                BufferedInputStream in = new BufferedInputStream(new FileInputStream(fileToZip));
                while ((read = in.read(buffer)) > 0) {
                    gzipOut.write(buffer, 0, read);
                }
                in.close();
                gzipOut.finish();
                gzipOut.close();
                if (previousFile != null && previousFile.exists() && new File(previousFile.getParentFile(), previousFile.getName().substring(0, previousFile.getName().length() - 4) + ".bgz").exists()) {
                    previousFile.delete();
                }
                log.info("Compressed transaction file:" + fileToZip.getName());
            }
            catch (IOException ex) {
                ex.printStackTrace();
            }
        }).start();
    }

    @Override
    public long getAndCommitNextTransactionId() {
        return this.currentTransactionId;
    }

    @Override
    public void commitTransactionId(long id) {
    }

    @Override
    public long getLastCommittedTransactionId() {
        return this.lastTransactionId;
    }

    protected static boolean newTransactionFileRequired(int filePosition, int packetSize) {
        return 1L + (long)filePosition + (long)packetSize >= Integer.MAX_VALUE;
    }

    protected static long createTransactionIndex(int fileId, int filePosition) {
        return (long)fileId << 32 | (long)filePosition & 0xFFFFFFFFL;
    }

    protected static int getTransactionFileId(long index) {
        return (int)(index >> 32);
    }

    protected static int getTransactionFilePosition(long index) {
        return (int)index;
    }

    protected static File getTransactionFileByFileId(int fileId, File path, boolean compressed) {
        if (!compressed) {
            return new File(path, "transactions-" + fileId + ".bin");
        }
        return new File(path, "transactions-" + fileId + ".bgz");
    }

    public void close() {
        this.timestampShutdown = System.currentTimeMillis();
        this.buffer.putLong(16, this.timestampShutdown);
        MappedByteBuffer mappedByteBuffer = (MappedByteBuffer)this.buffer.byteBuffer();
        mappedByteBuffer.force();
    }

    public void drop() {
        File file = new File(this.path, "database.stat");
        MappedStoreUtil.deleteBufferAndData(file, this.buffer);
    }

    public long getCurrentTransactionId() {
        return this.currentTransactionId;
    }

    public long getLastTransactionId() {
        return this.lastTransactionId;
    }

    public Schema getSchema() {
        return this.schema;
    }

    public long getTimestampFirstSystemStart() {
        return this.timestampFirstSystemStart;
    }

    public long getTimestampSystemStart() {
        return this.timestampSystemStart;
    }

    public long getTimestampShutdown() {
        return this.timestampShutdown;
    }

    public int getCurrentTransactionFileId() {
        return this.currentTransactionFileId;
    }

    public int getCurrentTransactionFilePosition() {
        return this.currentTransactionFilePosition;
    }

    public long getTransactionCount() {
        return this.transactionCount;
    }
}

