/*
 * Copyright 2016-2023 the original author.All rights reserved.
 * Kingstar(honeysoft@126.com)
 * The license,see the LICENSE file.
 */

package org.teasoft.honey.osql.autogen;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

import org.teasoft.bee.osql.DatabaseConst;
import org.teasoft.bee.osql.exception.BeeErrorNameException;
import org.teasoft.honey.osql.core.HoneyConfig;
import org.teasoft.honey.osql.core.HoneyContext;
import org.teasoft.honey.osql.core.HoneyUtil;
import org.teasoft.honey.osql.core.Logger;
import org.teasoft.honey.osql.core.NameTranslateHandle;
import org.teasoft.honey.osql.name.NameUtil;
import org.teasoft.honey.osql.util.AnnoUtil;
import org.teasoft.honey.osql.util.NameCheckUtil;
import org.teasoft.honey.util.EntityUtil;
import org.teasoft.honey.util.SqlKeyCheck;
import org.teasoft.honey.util.StringUtils;

/**
 * @author Kingstar
 * @since  2.0
 */
public class DdlToSql {
	
	private static final String BE_REPLACE_WITH_TYPE = "It will be replace with type: ";
	private static final String NOT_RELATIVE_COLUMN = " has not the relative database column type!";
	private static final String THE_JAVA_TYPE = "The java type:";
	private static final String CREATE_TABLE = "CREATE TABLE ";
	// private static Map<String, String> java2DbType = Java2DbType.getJava2DbType(HoneyContext.getDbDialect());
	private static String LINE_SEPARATOR = System.getProperty("line.separator"); // 换行符
//	private static PreparedSql preparedSql = BeeFactoryHelper.getPreparedSql();
	private static Map<String, String> pkStatement = new HashMap<>();
	private static Map<String, String> pkStringStatement = new HashMap<>();
	private static String java_lang_String = "java.lang.String";

//	private static SuidRich suidRich = BF.getSuidRich();
	
	static {
		initPkStatement();
		initStringPkStatement();
	}
	
	/**
	 * 根据Javabean生成数据库表建表语句,Javabean无需配置过多的字段信息.此方法只考虑通用情况,若有详细需求,不建议采用
	 * <br>According to the statement of creating database table generated by JavaBean, JavaBean does 
	 * <br> not need to configure too much field information. This method only considers the general 
	 * <br>situation, and is not recommended if there are detailed requirements
	 * @param entity Javabean entity.
	 * @return 建表语句.create table string.
	 * @deprecated
	 */
	public static <T> String toCreateTableSQL(T entity) {
		return toCreateTableSQL(entity.getClass(), null);
	}
	
	public static <T> String toCreateTableSQL(Class<T> entityClass) {
		String tableName = _toTableNameByClass(entityClass);
		return toCreateTableSQL(entityClass, tableName);
	}
	

	public static <T> String toCreateTableSQL(Class<T> entityClass, String tableName) {

		if (HoneyUtil.isSQLite()) {
			return toCreateTableSQLForSQLite(entityClass, tableName);
		} else if (HoneyUtil.isMysql()) {
			return toCreateTableSQLForMySQL(entityClass, tableName);
		} else if (DatabaseConst.H2.equalsIgnoreCase(HoneyContext.getDbDialect())) {
			return toCreateTableSQLForH2(entityClass, tableName);
		} else if (DatabaseConst.PostgreSQL.equalsIgnoreCase(HoneyContext.getDbDialect())) {
			return toCreateTableSQLForPostgreSQL(entityClass, tableName);
		} else if (HoneyUtil.isSqlServer()) {
			return toCreateTableSQLForSQLSERVER(entityClass, tableName);
		} else {
			// ORACLE,Cassandra ...
			return _toCreateTableSQL(entityClass, tableName);
		}
	}

	// ORACLE, Cassandra
	private static <T> String _toCreateTableSQL(Class<T> entityClass, String tableName) {

		if (tableName == null) tableName = _toTableNameByClass(entityClass);
		StringBuilder sqlBuffer = new StringBuilder();
		sqlBuffer.append(CREATE_TABLE + tableName + " (").append(LINE_SEPARATOR);
		Field fields[] = entityClass.getDeclaredFields();
		for (int i = 0; i < fields.length; i++) {
			if (isSkipField(fields[i])) {
				if (i == fields.length - 1)
					sqlBuffer.delete(sqlBuffer.length() - 5, sqlBuffer.length() - 2);
				continue;
			}
			sqlBuffer.append(_toColumnName(fields[i].getName(), entityClass)).append("  ");

			String type = getType(fields[i]);

			if (HoneyUtil.isCassandra()) {
				if (EntityUtil.isList(fields[i]) || EntityUtil.isSet(fields[i])) {
					Class<?> clazz = EntityUtil.getGenericType(fields[i]);
					String type0 = getType(clazz);
					type = type + "<" + type0 + ">";
				} else if (EntityUtil.isMap(fields[i])) {
					Class<?>[] classes = EntityUtil.getGenericTypeArray(fields[i]);
					String type1 = getType(classes[0]);
					String type2 = getType(classes[1]);
					type = type + "<" + type1 + "," + type2 + ">";
				}

			}

			sqlBuffer.append(type);

			if (isPrimaryKey(fields[i])) sqlBuffer.append(" PRIMARY KEY");
			if (i != fields.length - 1)
				sqlBuffer.append(",  ");
			else
				sqlBuffer.append("  ");
			sqlBuffer.append(LINE_SEPARATOR);
		}
		sqlBuffer.append(" )");

		return sqlBuffer.toString();

	}

	private static Map<String, String> getJava2DbType() {  //可能返回null
		return Java2DbType.getJava2DbType(HoneyContext.getDbDialect());
	}
	
	private static String getType(Field field) {
		String type = getJava2DbType().get(field.getType().getName());
		if (type == null) {
			Logger.warn(THE_JAVA_TYPE + type + NOT_RELATIVE_COLUMN);
			type = getJava2DbType().get(java_lang_String);
			Logger.warn(BE_REPLACE_WITH_TYPE + type);
		}

		return type;
	}

	private static String getType(Class<?> c) {
		String name = "";
		if (c == null) {
			Logger.warn("The Class is null,it will be replace with " + java_lang_String);
			name = java_lang_String;
		} else {
			name = c.getName();
		}
		String type = getJava2DbType().get(name);
		if (type == null) {
			if (EntityUtil.isCustomBean(name) && c != null) {
//				type=c.getSimpleName();
				type = NameUtil.firstLetterToLowerCase(c.getSimpleName());
			} else {
				Logger.warn(THE_JAVA_TYPE + name + NOT_RELATIVE_COLUMN);
				type = getJava2DbType().get(java_lang_String);
				Logger.warn(BE_REPLACE_WITH_TYPE + type);
			}
		}

		return type;
	}

	// SQLite
	private static <T> String toCreateTableSQLForSQLite(Class<T> entityClass, String tableName) {
		return toCreateTableSQLComm(entityClass, tableName, DatabaseConst.SQLite);
	}

	// MySQL
	private static <T> String toCreateTableSQLForMySQL(Class<T> entityClass, String tableName) {
		if (tableName == null) tableName = _toTableNameByClass(entityClass);
		StringBuilder sqlBuffer = new StringBuilder();
		sqlBuffer.append(CREATE_TABLE + tableName + " (").append(LINE_SEPARATOR);
		Field fields[] = entityClass.getDeclaredFields();
		for (int i = 0; i < fields.length; i++) {
			if (isSkipField(fields[i])) {
				if (i == fields.length - 1)
					sqlBuffer.delete(sqlBuffer.length() - 5, sqlBuffer.length() - 2);
				continue;
			}
			sqlBuffer.append(_toColumnName(fields[i].getName(), entityClass)).append("  ");

			if (isPrimaryKey(fields[i])) {
				String pkSt = "bigint(20) PRIMARY KEY NOT NULL AUTO_INCREMENT";
				if (String.class.equals(fields[i].getType())) {
					String type = getJava2DbType().get(fields[i].getType().getName());
					if (type != null) {
						pkSt = "varchar(255) PRIMARY KEY NOT NULL";
					}
				}
				sqlBuffer.append(pkSt);
			} else {
				String type = getJava2DbType().get(fields[i].getType().getName());
				if (type == null) {
					Logger.warn(THE_JAVA_TYPE + type + NOT_RELATIVE_COLUMN);
					type = getJava2DbType().get(java_lang_String);
					Logger.warn(BE_REPLACE_WITH_TYPE + type);
				}
				sqlBuffer.append(type);

				if ("timestamp".equalsIgnoreCase(type) || "datetime".equalsIgnoreCase(type)) {
					sqlBuffer.append(" DEFAULT CURRENT_TIMESTAMP");
				} else {
					sqlBuffer.append(" DEFAULT NULL");
				}
			}

			if (i != fields.length - 1)
				sqlBuffer.append(",  ");
			else
				sqlBuffer.append("  ");
			sqlBuffer.append(LINE_SEPARATOR);
		}
		sqlBuffer.append(" )");

		return sqlBuffer.toString();

	}

//	H2
	private static <T> String toCreateTableSQLForH2(Class<T> entityClass, String tableName) {
		return toCreateTableSQLComm(entityClass, tableName, DatabaseConst.H2);
	}

	// Comm: H2,SQLite,PostgreSQL
	private static <T> String toCreateTableSQLComm(Class<T> entityClass, String tableName,
			String databaseName) {
		if (tableName == null) tableName = _toTableNameByClass(entityClass);
		StringBuilder sqlBuffer = new StringBuilder();
		sqlBuffer.append(CREATE_TABLE + tableName + " (").append(LINE_SEPARATOR);
		Field fields[] = entityClass.getDeclaredFields();
		for (int i = 0; i < fields.length; i++) {
			if (isSkipField(fields[i])) {
				if (i == fields.length - 1)
					sqlBuffer.delete(sqlBuffer.length() - 5, sqlBuffer.length() - 2);
				continue;
			}
			sqlBuffer.append(_toColumnName(fields[i].getName(), entityClass)).append("  ");

			if (isPrimaryKey(fields[i])) {
				if (!String.class.equals(fields[i].getType()))
					sqlBuffer.append(getPrimaryKeyStatement(databaseName));// different
				else
					sqlBuffer.append(getStringPrimaryKeyStatement(databaseName));
			} else {

				String type = getJava2DbType().get(fields[i].getType().getName());
				if (type == null) {
					Logger.warn(THE_JAVA_TYPE + type + NOT_RELATIVE_COLUMN);
					type = getJava2DbType().get(java_lang_String);
					Logger.warn(BE_REPLACE_WITH_TYPE + type);
				}
				sqlBuffer.append(type);
				if ("timestamp".equalsIgnoreCase(type) || "datetime".equalsIgnoreCase(type)) {
					sqlBuffer.append(" DEFAULT CURRENT_TIMESTAMP");
				} else {
					sqlBuffer.append(" DEFAULT NULL");
				}
			}

			if (i != fields.length - 1)
				sqlBuffer.append(",  ");
			else
				sqlBuffer.append("  ");
			sqlBuffer.append(LINE_SEPARATOR);
		}
		sqlBuffer.append(" )");

		return sqlBuffer.toString();
	}

//	PostgreSQL
	private static <T> String toCreateTableSQLForPostgreSQL(Class<T> entityClass, String tableName) {
		return toCreateTableSQLComm(entityClass, tableName, DatabaseConst.PostgreSQL);
	}

	// SQLSERVER
	private static <T> String toCreateTableSQLForSQLSERVER(Class<T> entityClass, String tableName) {
		if (tableName == null) tableName = _toTableNameByClass(entityClass);
		StringBuilder sqlBuffer = new StringBuilder();
		sqlBuffer.append(CREATE_TABLE + tableName + " (").append(LINE_SEPARATOR);
		Field fields[] = entityClass.getDeclaredFields();
		boolean hasCurrentTime = false;
		for (int i = 0; i < fields.length; i++) {
			if (isSkipField(fields[i])) {
				if (i == fields.length - 1)
					sqlBuffer.delete(sqlBuffer.length() - 5, sqlBuffer.length() - 2);
				continue;
			}
			sqlBuffer.append(_toColumnName(fields[i].getName(), entityClass)).append("  ");

			if (isPrimaryKey(fields[i])) {
				String pkSt = "bigint PRIMARY KEY NOT NULL";
				if (String.class.equals(fields[i].getType())) {
					String type = getJava2DbType().get(fields[i].getType().getName());
					if (type != null) {
						pkSt = pkSt.replace("bigint", type);
					}
				}
				sqlBuffer.append(pkSt);
			} else {
				String type = getJava2DbType().get(fields[i].getType().getName());
				if (type == null) {
					Logger.warn(THE_JAVA_TYPE + type + NOT_RELATIVE_COLUMN);
					type = getJava2DbType().get(java_lang_String);
					Logger.warn(BE_REPLACE_WITH_TYPE + type);
				}
//				sqlBuffer.append(type);

				if (("timestamp".equalsIgnoreCase(type))) {
					if (!hasCurrentTime) {
						sqlBuffer.append(type);
						sqlBuffer.append(" ");
						hasCurrentTime = true;
					} else {
						sqlBuffer.append("datetime DEFAULT NULL");
					}
				} else {
					sqlBuffer.append(type);
					sqlBuffer.append(" DEFAULT NULL");
				}
			}

			if (i != fields.length - 1)
				sqlBuffer.append(",  ");
			else
				sqlBuffer.append("  ");
			sqlBuffer.append(LINE_SEPARATOR);
		}
		sqlBuffer.append(" )");

		return sqlBuffer.toString();

	}
	
	@SuppressWarnings("rawtypes")
	private static String _toColumnName(String fieldName, Class entityClass) {
		String name = NameTranslateHandle.toColumnName(fieldName, entityClass);
		if (SqlKeyCheck.isKeyWord(name)) {
			Logger.warn("The '" + name + "' is Sql Keyword. Do not recommend!");
		}
		return name;
	}

	private static boolean isSkipField(Field field) {
//		if (field != null) {
//			if ("serialVersionUID".equals(field.getName())) return true;
//			if (field.isSynthetic()) return true;
//			if (field.isAnnotationPresent(JoinTable.class)) return true;
//		}
//		return false;
		return HoneyUtil.isSkipField(field);
	}
	
	private static boolean isPrimaryKey(Field field) {
		if ("id".equalsIgnoreCase(field.getName())) return true;
//	    if (field.isAnnotationPresent(PrimaryKey.class)) return true;//V1.11
//	    return false;
		return AnnoUtil.isPrimaryKey(field);
	}
	
	private static String getPrimaryKeyStatement(String databaseName) {
		return pkStatement.get(databaseName.toLowerCase());
	}

	private static String getStringPrimaryKeyStatement(String databaseName) {
		return pkStringStatement.get(databaseName.toLowerCase());
	}
	
	private static void initStringPkStatement() {
		pkStringStatement.put(DatabaseConst.H2.toLowerCase(),
				"varchar(255) PRIMARY KEY NOT NULL");
		pkStringStatement.put(DatabaseConst.SQLite.toLowerCase(),
				" VARCHAR2(255) PRIMARY KEY NOT NULL");
		pkStringStatement.put(DatabaseConst.PostgreSQL.toLowerCase(),
				"varchar(255) PRIMARY KEY NOT NULL");
		pkStringStatement.put("", "");
		pkStringStatement.put(null, "");
	}
	
	private static void initPkStatement() {
		pkStatement.put(DatabaseConst.H2.toLowerCase(), "bigint PRIMARY KEY NOT NULL");
		pkStatement.put(DatabaseConst.SQLite.toLowerCase(), " INTEGER PRIMARY KEY NOT NULL");
		pkStatement.put(DatabaseConst.PostgreSQL.toLowerCase(), "bigserial NOT NULL");
		pkStatement.put("", "");
		pkStatement.put(null, "");
	}

	@SuppressWarnings("rawtypes")
	private static String _toTableNameByClass(Class c){
		return NameTranslateHandle.toTableName(c.getName());
	}
	
	public static <T> String toDropTableSql(Class<T> entityClass) {
		String tableName = _toTableNameByClass(entityClass);
		String sql0 = "";
		if (HoneyUtil.isOracle() || HoneyUtil.isSqlServer()) {
			sql0 = "DROP TABLE " + tableName;
		} else {
			sql0 = " DROP TABLE IF EXISTS " + tableName;
		}

		return sql0;
	}

	public static <T> String toDropTableSimpleSql(Class<T> entityClass) {
		String tableName = _toTableNameByClass(entityClass);
		String sql0 = "";
		sql0 = "DROP TABLE " + tableName;
		return sql0;
	}
	
	
	private static String transferField(String fields, Class c) {
		String str[] = fields.split(",");
		String columns = "";
		for (int i = 0; i < str.length; i++) {
			if (i != 0) columns += ",";
			columns += _toColumnName(str[i].trim(), c);
		}

		return columns;
	}

	private static void checkField(String fields) {
		NameCheckUtil.checkName(fields);
	}
	public static <T> String toPrimaryKeySql(Class<T> entityClass, String fields, String keyName) {
//		alter table tableName add constraint pk_name primary key (id,pid) --添加主键约束

		String PREFIX = "pk_";
		String typeTip = "normal";

		if (StringUtils.isBlank(fields)) {
			throw new BeeErrorNameException(
					"Create " + typeTip + " index, the fields can not be empty!");
		}
		checkField(fields);
		String tableName = _toTableNameByClass(entityClass);

		String columns = transferField(fields, entityClass);

		if (StringUtils.isBlank(keyName)) {
			keyName = PREFIX + tableName + "_" + columns.replace(",", "_");
		} else {
			checkField(keyName);
		}

		String indexSql = "ALTER TABLE " + tableName + " ADD CONSTRAINT " + keyName
				+ " PRIMARY KEY (" + columns + ")";
		
		return indexSql;
	}
	
	public static <T> String toIndexSql(Class<T> entityClass, String fields, String indexName, String PREFIX,
			String IndexTypeTip, String IndexType) {
//		String PREFIX = "idx_";
//		String IndexTypeTip = "normal";
//		String IndexType = ""; //normal will empty

		if (StringUtils.isBlank(fields)) {
			throw new BeeErrorNameException(
					"Create " + IndexTypeTip + " index, the fields can not be empty!");
		}
		checkField(fields);
		String tableName = _toTableNameByClass(entityClass);

		String columns = transferField(fields, entityClass);

		if (StringUtils.isBlank(indexName)) {
			indexName = PREFIX + tableName + "_" + columns.replace(",", "_");
		} else {
			checkField(indexName);
		}

		String indexSql = "CREATE " + IndexType + "INDEX " + indexName + " ON " + tableName
				+ "(" + columns + ")";
		
		return indexSql;
		
	}
	
	//该方法并不能保证是准确的语句
	public static <T> String toDropIndexSql(Class<T> entityClass, String indexName) {
		String tableName = _toTableNameByClass(entityClass);

		if (indexName != null) {
			String dropSql = "";
			if (HoneyUtil.isSqlServer()) {
				dropSql = "DROP INDEX table_name.index_name";
			} else if (HoneyUtil.isOracle() || HoneyUtil.isSQLite() || DatabaseConst.DB2
					.equalsIgnoreCase(HoneyConfig.getHoneyConfig().getDbName())) {
				dropSql = "DROP INDEX index_name";
			} else if (HoneyUtil.isMysql() || DatabaseConst.MsAccess
					.equalsIgnoreCase(HoneyConfig.getHoneyConfig().getDbName())) {
//				用于 MS Access 等的 DROP INDEX 语法：
				dropSql = "DROP INDEX index_name ON table_name";
			} else {
				dropSql = "DROP INDEX index_name";
			}

			return dropSql.replace("table_name", tableName).replace("index_name", indexName);

		} else {
			return "drop index all on " + tableName; // sql server等
			//不支持这种语法的DB有(只列举部分):mysql
		}
	}

}
