/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.i18n.pdo;

import java.io.Serializable;
import java.util.Objects;
import org.tentackle.common.Compare;
import org.tentackle.session.ClassId;
import org.tentackle.pdo.PersistentDomainObject;
import org.tentackle.pdo.Plural;
import org.tentackle.pdo.Singular;
import org.tentackle.session.TableName;


/*
 * @> $mapping
 *
 * # resource bundle
 * name := $classname
 * id := $classid
 * table := $tablename
 * alias := bndl
 * integrity := composite
 *
 * ## attributes
 * [root, remote, tracked, tableserial]
 *
 * String(128)    name            bname              the resource bundle name [key]
 * String(8)      locale          blocale            the locale, null if default [key, mapnull]
 *
 * ## indexes
 * unique index udk := bname, blocale
 *
 * ## validations
 * name: @NotNull
 *
 * ## relations
 * StoredBundleKey:
 *  relation = composite list,
 *  select = eager,
 *  name = Keys,
 *  method = BundleId
 *
 * @<
 */


/**
 * A resource bundle stored in the database.
 * <p>
 * Important: modifications of the bundle should only be done via
 * {@link #setTranslation(java.lang.String, java.lang.String)} and never by adding or modifying a bundle key
 * directly. Bundles are always loaded eagerly, i.e. including all components..
 *
 * @author harald
 */
@TableName(/**/"bundle"/**/) // @wurblet < Inject --string $tablename
@ClassId(/**/8/**/) // @wurblet < Inject $classid
@Singular("bundle")
@Plural("bundles")
public interface StoredBundle extends PersistentDomainObject<StoredBundle>, StoredBundleDomain, StoredBundlePersistence {

  // @wurblet(fold=expanded) modelComment ModelComment

  //<editor-fold defaultstate="expanded" desc="code 'modelComment' generated by wurblet ModelComment">//GEN-BEGIN:modelComment

  /*
   * -------------------------------------------------------------------------------------------------------------------
   *
   * StoredBundle is not referenced by any other entity
   *
   *
   * StoredBundle is a root entity
   *
   *
   * StoredBundle is a composite with the components:
   *    + StoredBundleKey via bkey.bundleId as keys [1:N]
   *
   *
   * StoredBundle is not referencing other entities
   *
   *
   * Components of StoredBundle are not deeply referenced
   *
   *
   * StoredBundle is not extended
   *
   * -------------------------------------------------------------------------------------------------------------------
   */

  //</editor-fold>//GEN-END:modelComment

  // @wurblet uniqueDomainKey UniqueDomainKey

  //<editor-fold defaultstate="collapsed" desc="code 'uniqueDomainKey' generated by wurblet UniqueDomainKey">//GEN-BEGIN:uniqueDomainKey

  /**
   * The unique domain key for StoredBundle.
   */
  class StoredBundleUDK implements Serializable, Comparable<StoredBundleUDK> {

    private static final long serialVersionUID = 1L;

    /** the resource bundle name. */
    public final String name;

    /** the locale, null if default. */
    public final String locale;

    /**
     * Creates a unique domain key.
     *
     * @param name the resource bundle name
     * @param locale the locale, null if default
     */
    public StoredBundleUDK(String name, String locale) {
      this.name = name;
      this.locale = locale;
    }

    /**
     * Creates a unique domain key for a pdo.
     *
     * @param pdo the pdo
     */
    public StoredBundleUDK(StoredBundle pdo) {
      this(pdo.getName(), pdo.getLocale());
    }

    @Override
    public int hashCode() {
      int hash = 7;
      hash = 79 * hash + Objects.hashCode(name);
      hash = 79 * hash + Objects.hashCode(locale);
      return hash;
    }

    @Override
    public boolean equals(Object obj) {
      if (obj == null) {
        return false;
      }
      if (getClass() != obj.getClass()) {
        return false;
      }
      StoredBundleUDK other = (StoredBundleUDK) obj;
      if (!Objects.equals(name, other.name)) {
        return false;
      }
      if (!Objects.equals(locale, other.locale)) {
        return false;
      }
      return true;
    }

    @Override
    public int compareTo(StoredBundleUDK other) {
      int rv = other == null ? 1 : 0;
      if (rv == 0) {
        rv = Compare.compare(name, other.name);
        if (rv == 0) {
          rv = Compare.compare(locale, other.locale);
        }
      }
      return rv;
    }
  }

  //</editor-fold>//GEN-END:uniqueDomainKey

}
