/*
 * Tentackle - https://tentackle.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


package org.tentackle.i18n.pdo;

import org.tentackle.bind.Bindable;
import org.tentackle.misc.TrackedList;
import org.tentackle.pdo.DomainKey;
import org.tentackle.pdo.PersistentObject;
import org.tentackle.session.Persistent;
import org.tentackle.validate.validator.NotNull;

/**
 * Persistence interface for stored bundles.
 *
 * @author harald
 */
public interface StoredBundlePersistence extends PersistentObject<StoredBundle> {

  // @wurblet fieldlengths ColumnLengths

  //<editor-fold defaultstate="collapsed" desc="code 'fieldlengths' generated by wurblet ColumnLengths">//GEN-BEGIN:fieldlengths


  /** maximum number of characters for 'name'. */
  int CL_NAME = 128;

  /** maximum number of characters for 'locale'. */
  int CL_LOCALE = 8;

  //</editor-fold>//GEN-END:fieldlengths

  // @wurblet membernames AttributeNames

  //<editor-fold defaultstate="collapsed" desc="code 'membernames' generated by wurblet AttributeNames">//GEN-BEGIN:membernames


  /** relation name for 'composite list of StoredBundleKey keys via StoredBundleKey#bundleId (Keys)'. */
  String RN_KEYS = "keys";

  /** attribute name for database column 'bname'. */
  String AN_NAME = "name";

  /** attribute name for database column 'blocale'. */
  String AN_LOCALE = "locale";

  //</editor-fold>//GEN-END:membernames

  // @wurblet methods Methods

  //<editor-fold defaultstate="collapsed" desc="code 'methods' generated by wurblet Methods">//GEN-BEGIN:methods


  /**
   * Gets the attribute name.
   *
   * @return the resource bundle name
   */
  @Persistent(ordinal=1, comment="the resource bundle name")
  @NotNull
  @DomainKey
  String getName();

  /**
   * Sets the attribute name.
   *
   * @param name the resource bundle name
   */
  void setName(String name);

  /**
   * Gets the attribute locale.
   *
   * @return the locale, null if default
   */
  @Persistent(ordinal=2, comment="the locale, null if default")
  @DomainKey
  String getLocale();

  /**
   * Sets the attribute locale.
   *
   * @param locale the locale, null if default
   */
  void setLocale(String locale);

  /**
   * Selects StoredBundle by its unique domain key.
   *
   * @param name the resource bundle name
   * @param locale the locale, null if default
   * @return the StoredBundle, null if no such PDO
   */
  StoredBundle selectByUniqueDomainKey(String name, String locale);

  //</editor-fold>//GEN-END:methods

  // @wurblet relations Relations

  //<editor-fold defaultstate="collapsed" desc="code 'relations' generated by wurblet Relations">//GEN-BEGIN:relations


  /**
   * Gets composite list of StoredBundleKey keys via StoredBundleKey#bundleId (Keys).
   *
   * @return keys Keys
   */
  @Bindable
  @Persistent(ordinal=0, comment="Keys")
  TrackedList<StoredBundleKey> getKeys();

  /**
   * Returns whether Keys is loaded.
   *
   * @return true if getKeys() invoked at least once
   */
  @Bindable
  boolean isKeysLoaded();

  //</editor-fold>//GEN-END:relations

  /**
   * Selects from cache by unique domain key but does not load from db if not in cache.
   *
   * @param udk the unique domain key
   * @return the pdo, null if not in cache
   */
  StoredBundle selectCachedOnlyByUniqueDomainKey(StoredBundle.StoredBundleUDK udk);

  /**
   * Selects via cache by unique domain key.
   *
   * @param udk the unique key
   * @return the pdo, null if no such object
   */
  StoredBundle selectCachedByUniqueDomainKey(StoredBundle.StoredBundleUDK udk);

  /**
   * Finds all bundles (one for each locale) for a given name.
   *
   * @param name the bundle name
   * @return the localized bundles
   */
  TrackedList<StoredBundle> findByName(String name);

  /**
   * Find the bundle by its name and locale.
   *
   * @param name the bundle name
   * @param locale the locale
   * @return the bundle
   */
  StoredBundle findByNameAndLocale(String name, String locale);

  /**
   * Selects via remote cache, if session is remote.
   *
   * @param udk the unique key
   * @return the pdo, null if no such object
   */
  StoredBundle selectByUniqueDomainKeyForCache(StoredBundle.StoredBundleUDK udk);

}
